/*
  Copyright (C) 2005-2012 Julian Roigt, Jean-Philippe Domenger, Achille Braquelaire, Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <sstream>

#include "iseClustererCvKMeans.hpp"
#include "iseDistanceComputer.hpp"

using namespace ise;

ClustererCvKMeans::ClustererCvKMeans(size_t numberClasses) 
  : ClustererExplicit(numberClasses)
  , m_centers()
{  
}

const std::string
ClustererCvKMeans::uri() const
{
  std::stringstream ss(std::ios::in|std::ios::out);
  ss << __FILE__ << "_" << m_numberClasses;
  std::string uri = ss.str();
  return uri;
}

void
ClustererCvKMeans::compute(const cv::Mat &instances, cv::Mat &labels)
{
  assert(instances.channels() == 1);
  buildClusters(instances, &labels);
}

void
ClustererCvKMeans::compute(const std::vector<double*> &instances, size_t dim)
{
  const size_t height = instances.size();
  cv::Mat mat(instances.size() , dim, CV_32F);
  for (size_t y = 0 ; y < height ; ++y) {
    float *buf = mat.ptr<float>(y);
    for (size_t x = 0 ; x < dim ; ++x) {
      buf[x] = instances[y][x];
    }
  }
  buildClusters(mat, NULL);
}

void
ClustererCvKMeans::buildClusters(const cv::Mat &instances, cv::Mat *labels)
{
  size_t dim = instances.cols;
  
  cv::Mat centers;
  centers.create(m_numberClasses, dim, CV_32F);
  
  cv::Mat l_labels;
  if (labels != NULL) {
    labels->create(instances.rows, 1, CV_32S);
    *labels = cv::Scalar(0);
    l_labels = cv::Mat(instances.rows, 1, CV_32S, labels->data); //no copy
  }
  else {
    l_labels = cv::Mat::zeros(instances.rows, 1, CV_32S);
  }
  
  cv::TermCriteria term(1, 10, 0.1); //(type, max_iter, epsilon)
  
  cv::Mat fInstances;
  if (instances.type() == CV_32F) 
    fInstances = cv::Mat(instances.rows, instances.cols, CV_32F, instances.data); //no copy
  else
    instances.convertTo(fInstances, CV_32F);

  
  cv::kmeans(fInstances,
             m_numberClasses,
             l_labels,
             term,
             3,
             cv::KMEANS_PP_CENTERS,
#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION < 230
             &centers
#else
	     centers
#endif
	     );

  //- copy centers in m_centers, with float to double conversion
  m_numberClasses = centers.rows;
  m_centers.resize(m_numberClasses);
  std::vector<double> c(dim);
  for (size_t i = 0 ; i < m_numberClasses ; ++i) {
    const float *r = centers.ptr<float>(i);
    for (size_t j = 0 ; j < dim ; ++j) {
      c[j] = *r; //convert from float to double
      ++r;
    }
    m_centers[i] = c;
  }
}

ise::Clusterer::LabelType
ClustererCvKMeans::getCluster(const double* instance, size_t dim) const
{
  assert(m_centers.size() > 0);

  ise::DistanceComputer distComputer;
  distComputer.setDistanceType(ise::EUCLIDEAN);

  std::vector<double> query(instance, instance+dim);

  //TODO:OPTIM:PARALLEL : parallel_for !!
  double distToCentroid = distComputer.getDistance(query, m_centers[0]); 
  LabelType cluster = 0;
  for (size_t j = 1 ; j < m_centers.size() ; ++j) {
    const double currentValue = distComputer.getDistance(query, m_centers[j]);
    if (currentValue < distToCentroid) {
      distToCentroid = currentValue;
      cluster = j;
    }
  }

  return cluster;
}
