/*
  Templated vector class.
  Copyright (C) 2004-2009 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
 * @file girlToolsVector.hpp
 * @brief Generic vector class.
 */
#ifndef GIRLTOOLS_VECTOR_HPP
#define GIRLTOOLS_VECTOR_HPP

#include <cassert>
#include <cstddef> //size_t
#include <girl/girlCommon.hpp>

namespace girl {

  namespace tools {

    template <class T, size_t N>
    class GIRL_API vector 
    {
    public:

      /** 
       * @brief Default constructor.
       *
       * @warning elements are not initialized.
       */
      vector()
      {

      }

      /**
       * @brief Constructor.
       *
       * Initialize all elements with @a t.
       */
      vector(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] = t;
      }

      /**
       * @brief Copy constructor.
       */
      vector(const vector<T, N> &v)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] = v.m_data[i];
      }

      /**
       * @brief Assignment operator.
       */
      vector &operator=(const vector<T, N> &v)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] = v.m_data[i];
	return *this;
      }

      /**
       * @brief Assignment operator.
       *
       * Assign @a t to all elements.
       */
      vector &operator=(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] = t;
	return *this;
      }

      /**
       * @brief Const accessor.
       */
      const T &operator[](size_t i) const
      {
	assert(i<N);
	return m_data[i];
      }
    
      /**
       * @brief Accessor.
       */
      T &operator[](size_t i)
      {
	assert(i<N);
	return m_data[i];
      }
    
      /**
       * @brief Add @a t to each element of vector.
       */
      vector &operator+=(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] += t;
	return *this;
      }
    
      /**
       * @brief Add @a v to vector.
       */
      vector &operator+=(const vector &v)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] += v.m_data[i];
	return *this;
      }
    
    
      /**
       * @brief Subtract @a t to each element of vector.
       */
      vector &operator-=(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] -= t;
	return *this;
      }
    
      /**
       * @brief Subtract @a v to vector.
       */
      vector &operator-=(const vector &v)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] -= v.m_data[i];
	return *this;
      }
    
      /**
       * @brief Multiply by @a t each element of vector.
       */
      vector &operator*=(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] *= t;
	return *this;
      }
    
      /**
       * @brief Divide by @a t each element of vector.
       */
      vector &operator/=(const T &t)
      {
	for (size_t i=0; i<N; ++i)
	  m_data[i] /= t;
	return *this;
      }
    

    protected:
      T m_data[N];

    };


    template <class T, size_t N>
    vector<T, N> operator+(const vector<T, N> &v1, const vector<T, N> &v2)
    {
      vector<T, N> v(v1);
      v += v2;
      return v;
    }

    template <class T, size_t N>
    vector<T, N> operator-(const vector<T, N> &v1, const vector<T, N> &v2)
    {
      vector<T, N> v(v1);
      v -= v2;
      return v;
    }

    template <class T, size_t N>
    vector<T, N> operator*(const vector<T, N> &v1, const T &t)
    {
      vector<T, N> v(v1);
      v *= t;
      return v;
    }

    template <class T, size_t N>
    vector<T, N> operator*(const T &t, const vector<T, N> &v1)
    {
      return v1*t;
    }

  





    typedef double Flt;


    class GIRL_API Flt2 : public vector<Flt, 2>
    {
    public:

      Flt2(Flt a = 0.0, Flt b = 0.0)
      {
	m_data[0] = a;
	m_data[1] = b;
      }

      Flt2(const vector<Flt, 2> &v)
      : vector<Flt, 2>(v)
      {
      }


    };

    class GIRL_API Flt3 : public vector<Flt, 3>
    {
    public:
    
      Flt3(Flt a = 0.0, Flt b = 0.0, Flt c = 0.0)
      {
	m_data[0] = a;
	m_data[1] = b;
	m_data[2] = c;
      }

      Flt3(const vector<Flt, 3> &v)
      : vector<Flt, 3>(v)
      {
      }


    };

    typedef int Int;
  

    class GIRL_API Int2 : public vector<Int, 2>
    {
    public:
    
      Int2(Int a = 0, Int b = 0)
      {
	m_data[0] = a;
	m_data[1] = b;
      }

      Int2(const vector<Int, 2> &v)
      : vector<Int, 2>(v)
      {
      }
      


    };
  
    class GIRL_API Int3 : public vector<Int, 3>
    {
    public:
    
      Int3(Int a = 0, Int b = 0, Int c = 0)
      {
	m_data[0] = a;
	m_data[1] = b;
	m_data[2] = c;
      }

      Int3(const vector<Int, 3> &v)
      : vector<Int, 3>(v)
      {
      }
      

    };
    

  }

}

#endif /* ! GIRLTOOLS_VECTOR_HPP */
