/*
  Functions to compute information on regions of a toger.
  Copyright (C) 2004-2010 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
 * @file girlToolsRegionInfos.hpp
 * @brief Some functions to get information on regions.
 */
#ifndef GIRLTOOLS_REGIONINFOS_HPP
#define GIRLTOOLS_REGIONINFOS_HPP

#include <girl/girlToger.hpp>
#include <girlTools/girlToolsImage.hpp>

namespace girl {

  namespace tools {

    /**
     * Iterate all (inter-pixel) points of contour.
     */
    class GIRL_API BoundaryPointIterator : public girl::toger::directionIterator
    {
    public:

      /**
       * Overload this method to travers points of boundary once.
       */
      virtual void processPoint(const girl::point &p) = 0;


      virtual void processFirstPoint(const girl::point &p)
      {
	m_cpt = p;
      }

      virtual void processDirection(girl::direction d)
      {
	processPoint(m_cpt);
	m_cpt.move(d);
	
      }
      
      protected:
	girl::point m_cpt;

    };

    /**
     * Iterate only on (inter-pixel) points of contour where 
     * there is a change of direction.
     */
    class GIRL_API BoundaryChangePointIterator : public girl::toger::directionIterator
    {
    public:
      BoundaryChangePointIterator()
	: m_cpt(), m_prevDir(girl::INVALID)
      {}

      /**
       * Overload this method to travers points of boundary 
       * (where there is a change of direction) once
       */
      virtual void processPoint(const girl::point &p) = 0;

      virtual void processFirstPoint(const girl::point &p)
      {
	m_prevDir = girl::INVALID;
	m_cpt = p;
      }

      virtual void processDirection(girl::direction d)
      {
	if (d != m_prevDir) {
	  processPoint(m_cpt);
	  m_prevDir = d;
	}
	m_cpt.move(d);
	
      }
      
      protected:
	girl::point m_cpt;
	girl::direction m_prevDir;

    };


    
    /**
     * @brief Get number of children in a region @a r in toger @a tog.
     *
     * It is O(N) in number of holes. 
     * However it does not do any memory allocation.
     */
    extern GIRL_API size_t getNumberOfChildren(const girl::toger &tog, girl::region r);

    /**
     * @brief Compute perimeter of region @a r in toger @a tog.
     */
    extern GIRL_API size_t computePerimeter(const girl::toger &tog, girl::region r);
    
    /**
     * @brief Compute area of region @a r in toger @a tog.
     */
    extern GIRL_API size_t computeArea(const girl::toger &tog, girl::region r);

    /**
     * @brief Compute area of regions @a regions in toger @a tog.
     */
    extern GIRL_API size_t computeArea(const girl::toger &tog, const girl::regionDeque &regions);

    /**
     * @brief Compute axis-aligned bounding box of region @a r in toger @a tog.
     * @a p1 and @a p2 are respectively top left and down right point included in bounding box.
     */
    extern GIRL_API void computeBoundingBox(const girl::toger &tog, girl::region r, girl::point &p1, girl::point &p2);


    

    
    /**
     * @brief Get toger from image. 
     * 
     * @a img may be of type CV_8UC1 or CV_8UC3
     *
     */
    extern GIRL_API void getTogerFromImageLabel(const cv::Mat &img, girl::toger &tog, girl::regionDeque &regs);

   }

}

#endif /* ! GIRLTOOLS_REGIONINFOS_HPP */
