/*
  Functions to compute information on regions of a toger.
  Copyright (C) 2004-2010 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "girlToolsRegionInfos.hpp"

#include <opencv2/imgproc/imgproc.hpp>

#include <girl/girlHashFunc.hpp>
#include <girl/girlUnorderedSet.hpp>
#include <algorithm>
#include <iostream>

namespace {
class PerimeterComputer : public girl::toger::directionIterator
{
public:
  PerimeterComputer()
    : m_value(0)
  {
  }

  /*
  void reset()
  {
    m_value = 0;
  }
  */

  virtual void processDirection(girl::direction)
  {
    ++m_value;
  }

  size_t value() const
  {
    return m_value;
  }

protected:
  size_t m_value;

};
}

size_t
girl::tools::computePerimeter(const girl::toger &tog, girl::region r)
{
  PerimeterComputer ra;
  tog.iterateOnFaceBoundary(r, ra);
  return ra.value();
}

namespace {
class AreaComputer : public girl::toger::pointIterator
{
public:
  AreaComputer()
    : m_value(0)
  {
  }

  /*
  void reset()
  {
    m_value = 0;
  }
  */

  virtual void processPoint(const girl::point &)
  {
    ++m_value;
  }

  size_t value() const
  {
    return m_value;
  }

protected:
  size_t m_value;

};
}


size_t
girl::tools::getNumberOfChildren(const girl::toger &tog, girl::region r)
{
  size_t nbChildren = 0;
  for (girl::face f = tog.firstChild(r); f!=0; f = tog.nextSibling(f))
    ++nbChildren;
  return nbChildren;
}

size_t
girl::tools::computeArea(const girl::toger &tog, girl::region r)
{
  AreaComputer ra;
  tog.iterateOnRegionDomain(r, ra);
  return ra.value();
}

size_t
girl::tools::computeArea(const girl::toger &tog, const girl::regionDeque &regions)
{
  AreaComputer ra;
  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it)
    tog.iterateOnRegionDomain(*it, ra);
  return ra.value();
}


namespace {
class BoundingBoxComputer : public girl::toger::pointIterator
{
public:
  explicit BoundingBoxComputer(girl::point pt)
    : m_p1(pt)
    , m_p2(pt)
  {
  }

  virtual void processPoint(const girl::point &pt)
  {
    if (pt.x() < m_p1.x())
      m_p1.setX(pt.x());
    else if (pt.x() > m_p2.x())
      m_p2.setX(pt.x());

    if (pt.y() < m_p1.y())
      m_p1.setY(pt.y());
    else if (pt.y() > m_p2.y())
      m_p2.setY(pt.y());
  }

  const girl::point &p1() const
  {
    return m_p1;
  }

  const girl::point &p2() const
  {
    return m_p2;
  }

protected:
  girl::point m_p1, m_p2;

};
}

void
girl::tools::computeBoundingBox(const girl::toger &tog, 
                                girl::region r,
                                girl::point &p1, girl::point &p2)
{
  BoundingBoxComputer bbc(tog.anyPoint(r));
  tog.iterateOnRegionDomain(r, bbc);
  p1 = bbc.p1();
  p2 = bbc.p2();
}





template <typename T>
class ImageSplitter : public girl::toger::sameRegion
{
public:
  explicit ImageSplitter(const cv::Mat &img)
    : m_img(img)
  {}

  virtual bool same(const girl::point &p, girl::direction dir)
  {
    girl::point p2 = p;
    p2.move(dir);
    return m_img.at<T>(p.y(), p.x()) == m_img.at<T>(p2.y(), p2.x());
  }

protected:
  const cv::Mat &m_img;
};


void
girl::tools::getTogerFromImageLabel(const cv::Mat &img, girl::toger &togr, girl::regionDeque &regs)
{
  regs.clear();

  girl::toger tog(img.cols, img.rows);
  switch (img.type()) {
  case CV_8UC1:
    {
      ImageSplitter<unsigned char> sp(img);
      tog.splitRegion(1, sp, regs);
    }
    break;
  case CV_8UC3:
    {
      ImageSplitter<cv::Vec3b> sp(img);
      tog.splitRegion(1, sp, regs);
    }
    break;
  default:
    std::cerr<<"ERROR: girl::tools::getTogerFromImageLabel image type not handled\n";
  }
 
  togr = tog;
}


