#ifndef GIRLTOOLS_PAINTER_HPP
#define GIRLTOOLS_PAINTER_HPP

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>

#include <girl/girlToger.hpp>
#include <girl/euclideanPathSet.hpp>

#include <cassert>
#include <iostream>//DEBUG

namespace girl
{

  namespace tools
  {

    typedef enum {DISCRETE_PATH, EUCLIDEAN_PATH} DrawMode;

    class GIRL_API CvPainter
    {
    public:
      /**
       * @brief Constructor
       *
       * @warning @a img must stay valid the whole lifetime of CvPainter.
       */
       CvPainter(cv::Mat &img, const cv::Scalar &color, int thickness)
	 : m_img(img),
	   m_color(color),
	   m_thickness(thickness),
	   m_x(0), m_y(0)
      {}

      void drawLine(const girl::point &p1, const girl::point &p2)
      {
	cv::line(m_img, cv::Point(p1.x()+m_x, p1.y()+m_y), cv::Point(p2.x()+m_x, p2.y()+m_y), m_color, m_thickness);
      } 
      
      void drawLine(const girl::TPoint<float> &p1, const girl::TPoint<float> &p2)
      {
	//TODO: take into account float coords.
	//see parameter @a shift of cv::line()
	// cf http://docs.opencv.org/modules/core/doc/drawing_functions.html?highlight=line#cv.Line
	
	cv::line(m_img, cv::Point(p1.x()+m_x, p1.y()+m_y), cv::Point(p2.x()+m_x, p2.y()+m_y), m_color, m_thickness);
      } 
      
      void setShift(int x, int y)
      {
	m_x = x;
	m_y = y;
      }
      
    protected:
      cv::Mat &m_img;
      cv::Scalar m_color;
      int m_thickness;
      int m_x;
      int m_y;
    };
    
    /**
     * @brief Draw boundaries of all regions of toger @a tog,
     * i.e., all darts, in discrete draw mode.
     */
    GIRL_API void drawToger(CvPainter &p, const girl::toger &tog);
    
    /**
     * @brief Draw all darts of toger @a tog marked by @a mark 
     * in discrete draw mode.
     */
    GIRL_API void drawToger(CvPainter &p, const girl::toger &tog, girl::toger::markFlags mark);
    
    /**
     * @brief Draw boundaries of all regions of toger @a tog,
     * i.e., all darts, with euclidean paths, defined by points in @a eps.
     */
    GIRL_API void drawToger(CvPainter &p, const girl::toger &tog, const girl::euclideanPathSet &eps);

    /**
     * Draw all darts of toger @a tog marked by @a mark in euclidean draw mode.
     */
    GIRL_API void drawToger(CvPainter &p, const girl::toger &tog, const girl::euclideanPathSet &eps, 
			    girl::toger::markFlags mark);
  

  
    /**
     * @brief Draw boundaries of regions @a regions of toger @a tog
     * in discrete draw mode.
     */  
    GIRL_API void drawRegions(CvPainter &p, const girl::toger &tog, const girl::regionDeque &regions);

    /**
     * @brief Draw boundaries of regions @a regions of toger @a tog 
     * marked by @a mark in discrete draw mode.
     */
    GIRL_API void drawRegions(CvPainter &p, const girl::toger &tog, const girl::regionDeque &regions, 
			      girl::toger::markFlags mark);

    /**
     * @brief Draw boundaries of regions @a regions of toger @a tog,
     * with euclidean paths, defined by points in @a eps.
     */
    GIRL_API void drawRegions(CvPainter &p, const girl::toger &tog, const girl::regionDeque &regions, 
			      const girl::euclideanPathSet &eps);

    /**
     * @brief Draw boundaries of regions @a regions of toger @a tog,
     * marked by @a mark, with euclidean paths, defined by points in @a eps.
     */
    GIRL_API void drawRegions(CvPainter &p, const girl::toger &tog, const girl::regionDeque &regions, 
			      const girl::euclideanPathSet &eps, girl::toger::markFlags mark);

    /**
     * @brief Draw segments of darts @a darts of toger @a tog
     * in discrete draw mode.
     */  
    GIRL_API void drawDarts(CvPainter &p, const girl::toger &tog, const girl::dartDeque &darts);

    /**
     * @brief Draw segments of darts @a darts of toger @a tog, marked by @a mark 
     * in discrete draw mode.
     */
    GIRL_API void drawDarts(CvPainter &p, const girl::toger &tog, const girl::dartDeque &darts, 
			    girl::toger::markFlags mark);

    /**
     * @brief Draw segments of darts @a darts of toger @a tog
     * with euclidean paths, defined by points in @a eps.
     */  
    GIRL_API void drawDarts(CvPainter &p, const girl::toger &tog, const girl::dartDeque &darts, 
			    const girl::euclideanPathSet &eps);

    /**
     * @brief Draw segments of darts @a darts of toger @a tog, marked by @a mark 
     * with euclidean paths, defined by points in @a eps.
     */
    GIRL_API void drawDarts(CvPainter &p, const girl::toger &tog, const girl::dartDeque &darts, 
			    const girl::euclideanPathSet &eps, girl::toger::markFlags mark);
    



    /**
     * Use this class to paint the domain of a region with a specified color on the
     * cv::Mat given in the constructor (e.g. creating a mask of the region)
     */
    template<typename T>
    class Painter : public girl::toger::pointIterator {

    public:
      Painter(cv::Mat &mat, cv::Scalar color, int shiftX = 0, int shiftY = 0) :
	m_mat(mat),
	m_color(color),
	m_shiftX(shiftX),
	m_shiftY(shiftY)
      {
	//B//assert(m_mat.elemSize1() == sizeof(T));
	assert(m_mat.elemSize() == sizeof(T));
	m_size = m_mat.elemSize();
	//m_channels = m_mat.channels();
	setColor(color);
      }

      virtual ~Painter() {}

      void processPoint(const girl::point &pt)
      {
	//  cv::circle(m_mat, cv::Point(pt.x() - m_shiftX, pt.y() - m_shiftY), 0, m_color);
	T* buffer = &m_mat.at<T>(pt.y() - m_shiftY, pt.x() - m_shiftX);
	//B//for (size_t i = 0; i < m_channels; ++i)
	memcpy(buffer, m_buf, m_size);
      }

      void setColor(const cv::Scalar &color)
      {
	m_color = color;
	//cv::scalarToRawData(color, m_buf, m_mat.type());
	//scalarToRawData is private in OpenCV 3.0.0
	
	const cv::Scalar& s = color;
	void* _buf = m_buf;
	int type = m_mat.type();
	int unroll_to = 0;
	
	int i, depth = CV_MAT_DEPTH(type), cn = CV_MAT_CN(type);
	CV_Assert(cn <= 4);
	switch(depth)
	  {
	  case CV_8U:
	    {
	      uchar* buf = (uchar*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<uchar>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_8S:
	    {
	      schar* buf = (schar*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<schar>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_16U:
	    {
	      ushort* buf = (ushort*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<ushort>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_16S:
	    {
	      short* buf = (short*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<short>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_32S:
	    {
	      int* buf = (int*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<int>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_32F:
	    {
	      float* buf = (float*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<float>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	    }
	    break;
	  case CV_64F:
	    {
	      double* buf = (double*)_buf;
	      for(i = 0; i < cn; i++)
		buf[i] = cv::saturate_cast<double>(s.val[i]);
	      for(; i < unroll_to; i++)
		buf[i] = buf[i-cn];
	      break;
	    }
	  default:
	    assert(false);
	    //CV_Error("Unsupported format");
	  }  
      }
	

    private:
      cv::Mat &m_mat;
      cv::Scalar &m_color;
      int m_shiftX;
      int m_shiftY;
      double m_buf[4];
      size_t m_size;
      //size_t m_channels;
    };

  }

}

#endif /* ! GIRLTOOLS_PAINTER_HPP */
