/*
  ImageFeature class.
  Copyright (C) 2004-2009 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "girlToolsImageFeature.hpp"

#include <cassert>
#include <iostream>


using namespace girl::tools;

ImageFeature::ImageFeature()
  : ImageFeatureAbstract(),
    m_isNull(true),
    m_uri(),
    m_imageFeatureImp()//1, 1, CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, 0))
{
  
}

ImageFeature::ImageFeature(int pWidth, int pHeight, int channels)
  : ImageFeatureAbstract(),
    m_isNull(pWidth<=0 || pHeight<=0),
    m_imageFeatureImp(new cv::Mat(cv::Mat::zeros(pHeight, pWidth, CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, channels))))
{
  
}

ImageFeature::ImageFeature(const ImageFeature &img)
  : ImageFeatureAbstract(),
    m_isNull(img.m_isNull),
    m_imageFeatureImp(img.m_imageFeatureImp)
{

}

ImageFeature::ImageFeature(const cv::Mat &mat)
  : ImageFeatureAbstract(),
    m_isNull(mat.empty()),
    m_imageFeatureImp(new cv::Mat(
        cv::Mat::zeros(mat.rows,
                       mat.cols,
                       CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, mat.channels()))
        )
      )
{
  mat.convertTo(*m_imageFeatureImp.get(), CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, mat.channels()));
}

void
ImageFeature::clone(ImageFeature &dest)
{
  dest.m_isNull = m_isNull;
  dest.m_imageFeatureImp = girl::shared_ptr<cv::Mat>(new cv::Mat(m_imageFeatureImp->clone()));
}

int 
ImageFeature::width() const
{
  return m_imageFeatureImp->cols;
}

int 
ImageFeature::height() const
{
  return m_imageFeatureImp->rows;
}

int 
ImageFeature::bytesPerLine() const
{
  return m_imageFeatureImp->step;
}

int 
ImageFeature::channels() const
{
  return m_imageFeatureImp->channels();
}


bool
ImageFeature::isNull() const
{
  return m_isNull;
}

ImageFeature &
ImageFeature::operator=(const ImageFeature &img)
{
  if (&img != this) {
    m_isNull = img.m_isNull;
    m_imageFeatureImp = img.m_imageFeatureImp;
  }
  return *this;
}

void
ImageFeature::copy()
{
  m_imageFeatureImp = girl::shared_ptr<cv::Mat>(new cv::Mat(m_imageFeatureImp->clone()));
}


ImageFeature::type *
ImageFeature::bits()
{
  return m_imageFeatureImp->ptr<ImageFeature::type>();
}

const ImageFeature::type *
ImageFeature::bits() const
{
  return m_imageFeatureImp->ptr<ImageFeature::type>();
}

ImageFeature::type *
ImageFeature::scanLine(int i)
{
  return m_imageFeatureImp->ptr<ImageFeature::type>(i); 
}

const ImageFeature::type *
ImageFeature::scanLine(int i) const
{
  return m_imageFeatureImp->ptr<ImageFeature::type>(i);
}


PixelFromImage<ImageFeature::type>
ImageFeature::getPixel(int x, int y)
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::getPixel on null imageFeature"<<std::endl;
    return PixelFromImage<ImageFeature::type>();
  }
  return PixelFromImage<ImageFeature::type>(*m_imageFeatureImp.get(), x, y);
}

const PixelFromImage<ImageFeature::type>
ImageFeature::getPixel(int x, int y) const
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::getPixel on null imageFeature"<<std::endl;
    return PixelFromImage<ImageFeature::type>();
  }
  return PixelFromImage<ImageFeature::type>(*m_imageFeatureImp.get(), x, y);
}

ImageFeature::type*
ImageFeature::getPixelPtr(int x, int y)
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::getPixel on null imageFeature"<<std::endl;
    return NULL;
  }
  return &(m_imageFeatureImp->at<ImageFeature::type>(y, x*m_imageFeatureImp->channels()));
}

const ImageFeature::type *
ImageFeature::getPixelPtr(int x, int y) const
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::getPixel on null imageFeature"<<std::endl;
    return NULL;
  }
  return &(m_imageFeatureImp->at<ImageFeature::type>(y, x*m_imageFeatureImp->channels()));
}

void
ImageFeature::setPixel(int x, int y, const PixelFromImage<ImageFeature::type> &p)
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::setPixel on null imageFeature"<<std::endl;
    return;
  }
  for (int i=0; i<p.channels(); ++i)
    m_imageFeatureImp->at<ImageFeature::type>(y, x*channels()+i) = p[i];
}

void
ImageFeature::setPixel(int x, int y, const PixelStandAlone<ImageFeature::type> &p)
{
  if (m_isNull) {
    std::cerr<<"ImageFeature::setPixel on null imageFeature"<<std::endl;
    return;
  }
  for (int i=0; i<p.channels(); ++i)
    m_imageFeatureImp->at<ImageFeature::type>(y, x*channels()+i) = p[i];
}

void
ImageFeature::create(int width, int height, int channels)
{
  //release();

  if ( channels == 0)
    m_isNull = true;
  else
    m_isNull = false;

 // m_imageFeatureImp.create(height, width, CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, channels));
  // REMI: Initialize matrix elements to zero please
  m_imageFeatureImp = girl::shared_ptr<cv::Mat>(new cv::Mat(cv::Mat::zeros(height, width, CV_MAKETYPE(cv::DataDepth<ImageFeature::type>::value, channels))));
}

void
ImageFeature::release()
{
  m_imageFeatureImp->release();
  m_isNull = true;
}
