/*
  Image class.
  Copyright (C) 2004-2011 Boris Mansencal  boris.mansencal@labri.fr
 
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef GIRLTOOLS_IMAGE_HPP
#define GIRLTOOLS_IMAGE_HPP

#include <opencv2/core/core.hpp>

#include <string>

#include <girl/girlSharedPtr.hpp>
#include <girl/girlCommon.hpp>

#include "girlToolsPixel.hpp"

namespace girl {

  namespace tools {
    
    class Pixel;

    /**
     * Image in RGBA format
     */
    class GIRL_API Image {
      
    public:
      
      typedef unsigned short type;
      
    public:
      
      Image();
      
      explicit Image(const std::string &fileName);

      Image(int width, int height, int channels = 4);
      
      Image(const Image &); //Warning: no deep copy !

      explicit Image(const cv::Mat &mat);

      /**
       * @brief Save image to file @a fileName.
       *
       * File extension indicates the image format to use.
       * Supported image formats are those of OpenCV (JPEG, PNG, ...)
       *
       * @return true on success, false otherwise.
       */
      bool save(const std::string &fileName) const;

      void clone(Image &dest) const;
      
      Image &operator=(const Image &);

      int width() const;
      int height() const;
      int bytesPerLine() const;

      int channels() const;
      
      bool isNull() const;
      
      type *bits();
      const type *bits() const;
      
      type *scanLine(int i);
      const type *scanLine(int i) const;
      
      bool isContinuous() const { return m_imageImp.isContinuous(); }

      inline int formatSize() const { return channels(); }
      
      PixelFromImage<type> getPixel(int x, int y);
      const PixelFromImage<type> getPixel(int x, int y) const;
      
      /**
       * @warning @a p.channels() must be equal to image channels().
       */
      void getPixelCopy(int x, int y, Pixel &p);

      
      /**
       * Set pixel at position (@a x; @a y) to @a p.
       *
       * @warning @a x must be in [0; width()[ and y in [0; height()[.
       * @warning @a p.channels() must be equal to image channels().
       */
      void setPixel(int x, int y, const PixelFromImage<type> &p);

      /**
       * Set pixel at position (@a x; @a y) to @a p.
       *
       * @warning @a x must be in [0; width()[ and y in [0; height()[.
       * @warning @a p.channels() must be equal to image channels().
       */
      void setPixel(int x, int y, const PixelStandAlone<type> &p);

      cv::Mat* cvMat() { return &m_imageImp; }

      inline const cv::Mat* cvMat() const { return &m_imageImp; }

      void setTo(type r, type g, type b);

    protected:
      
      void copy();

    protected:

      bool m_isNull;
      cv::Mat m_imageImp;
      
    };

    class GIRL_API Pixel : public PixelStandAlone<Image::type>
    {
    public:
      Pixel(Image::type r=0, Image::type g=0, Image::type b=0, Image::type a=0)
	: PixelStandAlone<Image::type>(4)
      {	
	m_data[0] = r;
	m_data[1] = g;
	m_data[2] = b;
	m_data[3] = a;
      }
      
      Pixel(const PixelFromImage<Image::type> &p) 
	: PixelStandAlone<Image::type>(p)
      {	
      }
      
      inline explicit Pixel(const std::vector<Image::type> &values) 
	: PixelStandAlone<Image::type>(values) 
      {}

      inline Pixel(const PixelStandAlone<Image::type> &pixel) 
	: PixelStandAlone<Image::type>(pixel) 
      {}

      inline explicit Pixel(const Image::type *p) 
	: PixelStandAlone<Image::type>(p, 4) 
      {}
      
      inline Image::type a() const { return m_data[0]; }
      inline Image::type b() const { return m_data[1]; }
      inline Image::type c() const { return m_data[2]; }
      inline Image::type d() const { return m_data[3]; }
      
    };

  }

}

#endif /* ! GIRLTOOLS_IMAGE_HPP */ 
