/*
  Image class.
  Copyright (C) 2004-2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "girlToolsImage.hpp"

#include <cassert>
#include <iostream>

#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#if CV_MAJOR_VERSION*100+CV_MINOR_VERSION*10+CV_SUBMINOR_VERSION >= 261
#include <opencv2/imgproc/types_c.h> //for CV_RGB2BGR & CV_BGR2RGB
#endif


using namespace girl::tools;

Image::Image()
: m_isNull(true),
  m_imageImp()//0, 0, CV_MAKETYPE(cv::DataDepth<Image::type>::value, 4))
{
  //B: an assert is triggered in OpenCV (svn version) when cv::Mat is build with null dimensions and type

}


Image::Image(const std::string &fileName)
  : m_isNull(false),
    m_imageImp(cv::imread(fileName/*, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR*/))
{
  if (m_imageImp.channels() == 3)
    cv::cvtColor(m_imageImp, m_imageImp, CV_BGR2RGB);

  m_imageImp.convertTo(m_imageImp, CV_MAKETYPE(cv::DataDepth<Image::type>::value, m_imageImp.channels()));

  m_isNull = (width()<=0 || height()<=0);
}

Image::Image(int pWidth, int pHeight, int channels)
: m_isNull(pWidth<=0 || pHeight<=0),
  m_imageImp(pHeight, pWidth, CV_MAKETYPE(cv::DataDepth<Image::type>::value, channels))
{
 // std::cerr << m_imageImp.channels() << std::endl;
  // m_imageImp = cv::Mat(pHeight, pWidth, CV_MAKETYPE(cv::DataDepth<Image::type>::value, channels));
}

Image::Image(const Image &img)
: m_isNull(img.m_isNull),
  m_imageImp(img.m_imageImp)
{

}

Image::Image(const cv::Mat &mat)
: m_isNull(false)
, m_imageImp()
{
  mat.convertTo(m_imageImp, CV_MAKETYPE(cv::DataDepth<Image::type>::value, mat.channels())); 
}


/**
* Only 8-bit (or 16-bit in the case of PNG, JPEG 2000 and TIFF) single-channel or 3-channel (with ‘BGR’ channel order) images can be saved using imwrite(...).
*/
bool
Image::save(const std::string &fileName) const
{
  const int channels = 3;
  const int type = CV_MAKETYPE(cv::DataDepth<unsigned char>::value, channels);
  if (type == m_imageImp.type()) {
    //return imwrite(fileName, m_imageImp);
    cv::Mat tmp;
    cv::cvtColor(m_imageImp, tmp, CV_RGB2BGR);
    return imwrite(fileName, tmp);
  }
  else {
    cv::Mat tmp;
    m_imageImp.convertTo(tmp, type);
    cv::cvtColor(tmp, tmp, CV_RGB2BGR);
    return imwrite(fileName, tmp);
  }
}

void
Image::clone(Image &dest) const
{
  dest.m_isNull = m_isNull;
  dest.m_imageImp = m_imageImp.clone();
}

int 
Image::width() const
{
  return m_imageImp.cols;
}

int 
Image::height() const
{
  return m_imageImp.rows;
}

int 
Image::bytesPerLine() const
{
  return m_imageImp.cols*sizeof(type)*channels();
}

int
Image::channels() const
{
  return m_imageImp.channels();
}


bool
Image::isNull() const
{
  return m_isNull;
}

Image &
Image::operator=(const Image &img)
{
  if (&img != this) {
    m_isNull = img.m_isNull;
    m_imageImp = img.m_imageImp;
  }
  return *this;
}

void
Image::copy()
{
  m_imageImp = m_imageImp.clone();
}


Image::type *
Image::bits()
{
  return m_imageImp.ptr<Image::type>();
}

const Image::type *
Image::bits() const
{
  return m_imageImp.ptr<Image::type>();
}

Image::type *
Image::scanLine(int i)
{
  return m_imageImp.ptr<Image::type>(i); 
}

const Image::type *
Image::scanLine(int i) const
{
  return m_imageImp.ptr<Image::type>(i);
}

PixelFromImage<Image::type>
Image::getPixel(int x, int y)
{
  if (GIRL_unlikely(m_isNull)) {
    std::cerr<<"Image::getPixel on null image"<<std::endl;
    return PixelFromImage<Image::type>();
  }
  return PixelFromImage<Image::type>(m_imageImp, x, y);
}

const PixelFromImage<Image::type>
Image::getPixel(int x, int y) const
{
  if (GIRL_unlikely(m_isNull)) {
    std::cerr<<"Image::getPixel on null image"<<std::endl;
    return PixelFromImage<Image::type>();
  }
  return PixelFromImage<Image::type>(m_imageImp, x, y);
}

void
Image::getPixelCopy(int x, int y, Pixel &p)
{
  if (GIRL_unlikely(m_isNull)) {
    std::cerr<<"Image::getPixelCopy on null image"<<std::endl;
    return;
  }
  assert(p.channels() >= channels());
  const size_t chans = channels();
  const int xc = x*chans;
  for (size_t i=0; i<chans; ++i)
    p[i] = m_imageImp.at<Image::type>(y, xc+i);
}

void
Image::setPixel(int x, int y, const PixelFromImage<Image::type> &p)
{
  if (GIRL_unlikely(m_isNull)) {
    std::cerr<<"ImageFeature::setPixel on null imageFeature"<<std::endl;
    return;
  }
  assert(p.channels() >= channels());
  const size_t chans = channels();
  const int xc = x*chans;
  for (size_t i=0; i<chans; ++i)
    m_imageImp.at<Image::type>(y, xc+i) = p[i];
}

void
Image::setPixel(int x, int y, const PixelStandAlone<Image::type> &p)
{
  if (GIRL_unlikely(m_isNull)) {
    std::cerr<<"ImageFeature::setPixel on null imageFeature"<<std::endl;
    return;
  }
  assert(p.channels() >= channels());
  const size_t chans = channels();
  const int xc = x*chans;
  for (size_t i=0; i<chans; ++i)
    m_imageImp.at<Image::type>(y, xc+i) = p[i];
}

void
Image::setTo(type r, type g, type b)
{
  cv::Scalar sc(r, g, b, 0);
  m_imageImp.setTo(sc);
}
