/*
  Copyright (C) 2005-2010 Boris Mansencal  boris.mansencal@labri.fr

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/
#ifndef TOGERPAINTER_HPP
#define TOGERPAINTER_HPP

#ifdef TIMING_PAINTER
#include <girl/girlTimer.hpp>
#include <iostream>
#endif //TIMING_PAINTER


/*
  Contexter must have the following interface:
  + void set();
  + void unset();

  SegmentDrawer must inherit girl::tools::BoundaryChangePointIterator 
  and have the following interface:
  + void begin();
  + void end();

  SegmentEPDrawer must inherit girl::tools::BoundaryPointIterator
  and have the following interface:
  + void begin();
  + void end();

*/


template <typename SegmentDrawer>
void
drawFacePositiveDartsT(SegmentDrawer &sd,
		       const girl::toger &tog, girl::face f)
{
  //version without mark

  const girl::dart canonicD = tog.beta(f);
  girl::dart d = canonicD;
  
  do
    {
      if (! tog.isFaceInfinite(tog.lambda(tog.alpha(d)))) {
	if (d > 0) {
	  sd.begin();
	  tog.iterateOnSegment(d, sd);
	  sd.end();
	}
      }
      else {
	sd.begin();
	tog.iterateOnSegment(d, sd);
	sd.end();
      }
            
      d = tog.phi(d);

    }
  while (d != canonicD);
}


template <typename SegmentDrawer>
void
drawFacePositiveDartsT(SegmentDrawer &sd,
		       const girl::toger &tog, girl::face f, 
		       girl::toger::markFlags mark)
{
  //version with mark

  const girl::dart canonicD = tog.beta(f);
  girl::dart d = canonicD;
  
  do
    {
      if (tog.isDartMarked(d, mark)) {

	if (! tog.isFaceInfinite(tog.lambda(tog.alpha(d)))) {
	  if (d > 0) {
	    sd.begin();
	    tog.iterateOnSegment(d, sd);
	    sd.end();
	  }
	}
	else {
	  sd.begin();
	  tog.iterateOnSegment(d, sd);
	  sd.end();
	}

      }
            
      d = tog.phi(d);

    }
  while (d != canonicD);
}

template <typename SegmentDrawer>
void
drawFaceT(SegmentDrawer &sd,
	  const girl::toger &tog, girl::face f)
{
  sd.begin();
  tog.iterateOnFaceBoundary(f, sd);
  sd.end();
}
  
template <typename SegmentDrawer>
void
drawFaceT(SegmentDrawer &sd,
	  const girl::toger &tog, girl::face f, 
	  girl::toger::markFlags mark)
{
  const girl::dart canonicD = tog.beta(f);
  girl::dart d = canonicD;
  
  do
    {
      if (tog.isDartMarked(d, mark)) {
	
	sd.begin();
	tog.iterateOnSegment(d, sd);
	sd.end();
      }
            
      d = tog.phi(d);

    }
  while (d != canonicD);
}

template <typename SegmentDrawer>
inline 
void
drawSegmentT(SegmentDrawer &sd,
	     const girl::toger &tog, girl::dart d)
{
  sd.begin();
  tog.iterateOnSegment(d, sd);
  sd.end();

}

template <typename SegmentDrawer>
void
drawSegmentT(SegmentDrawer &sd,
	     const girl::toger &tog, girl::dart d,
	     girl::toger::markFlags mark)
{
  if (tog.isDartMarked(d, mark)) {
    drawSegmentT<SegmentDrawer>(sd, tog, d);
  }
}
  


  
template <typename SegmentDrawer, typename Contexter>
void
drawTogerT(SegmentDrawer &sd, Contexter &ctx, const girl::toger &tog)
{

#ifdef STATS_PAINTER
  STATS_number_points = 0;
#endif

#ifdef TIMING_PAINTER
  girl::Timer t;
  t.start();
#endif

  ctx.set();
 
  girl::regionDeque regions;
  tog.allRegions(regions);
  
  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it )
    drawFacePositiveDartsT<SegmentDrawer>(sd, tog, *it);


  ctx.unset();

#ifdef TIMING_PAINTER
  const size_t time = t.stop();
  std::cerr<<"TogerPainter::drawToger(): time="<<time<<"ms"<<std::endl;
#endif

#ifdef STATS_PAINTER
  std::cerr<<"TogerPainter::drawToger(): "<<STATS_number_points
	   <<" points"<<std::endl;
#endif
}


template <typename SegmentDrawer, typename Contexter>
void
drawTogerT(SegmentDrawer &sd, Contexter &ctx, const girl::toger &tog, girl::toger::markFlags mark)
{
#ifdef STATS_PAINTER
  STATS_number_points = 0;
#endif

#ifdef TIMING_PAINTER
  girl::Timer t;
  t.start();
#endif

  ctx.set();
 
  girl::regionDeque regions;
  tog.allRegions(regions);
  
  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it )
    drawFacePositiveDartsT<SegmentDrawer>(sd, tog, *it, mark);

  ctx.unset();


#ifdef TIMING_PAINTER
  const size_t time = t.stop();
  std::cerr<<"TogerPainter::drawToger(): time="<<time<<"ms"<<std::endl;
#endif

#ifdef STATS_PAINTER
  std::cerr<<"TogerPainter::drawToger(): "<<STATS_number_points
	   <<" points"<<std::endl;
#endif
}

template <typename SegmentDrawer, typename Contexter>
void
drawRegionsT(SegmentDrawer &sd, Contexter &ctx,
	     const girl::toger &tog, 
	     const girl::regionDeque &regions)
{
#ifdef STATS_PAINTER
  STATS_number_points = 0;
#endif

#ifdef TIMING_PAINTER
  girl::Timer t;
  t.start();
#endif

  ctx.set();

  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();

  for ( ; it != itEnd; ++it)
    drawFaceT<SegmentDrawer>(sd, tog, *it); //[can not draw only positive darts !]

  ctx.unset();

#ifdef TIMING_PAINTER
  const size_t time = t.stop();
  std::cerr<<"TogerPainter::drawRegions(): time="<<time<<"ms"<<std::endl;
#endif

#ifdef STATS_PAINTER
  std::cerr<<"TogerPainter::drawToger(): "<<STATS_number_points
	   <<" points"<<std::endl;
#endif
}

template <typename SegmentDrawer, typename Contexter>
void
drawRegionsT(SegmentDrawer &sd, Contexter &ctx,
	     const girl::toger &tog, 
	     const girl::regionDeque &regions, 
	     girl::toger::markFlags mark)
{

#ifdef STATS_PAINTER
  STATS_number_points = 0;
#endif

#ifdef TIMING_PAINTER
  girl::Timer t;
  t.start();
#endif

  ctx.set();

  girl::regionDeque::const_iterator it = regions.begin();
  const girl::regionDeque::const_iterator itEnd = regions.end();

  for ( ; it != itEnd; ++it)
    drawFaceT<SegmentDrawer>(sd, tog, *it, mark); //[can not draw only positive darts !]

  ctx.unset();

#ifdef TIMING_PAINTER
  const size_t time = t.stop();
  std::cerr<<"TogerPainter::drawRegions(): time="<<time<<"ms"<<std::endl;
#endif

#ifdef STATS_PAINTER
  std::cerr<<"TogerPainter::drawToger(): "<<STATS_number_points
	   <<" points"<<std::endl;
#endif
}


template <typename SegmentDrawer, typename Contexter>
void
drawDartsT(SegmentDrawer &sd, Contexter &ctx,
	   const girl::toger &tog, 
	   const girl::dartDeque &darts)
{
  ctx.set();

  girl::dartDeque::const_iterator it = darts.begin();
  const girl::dartDeque::const_iterator itEnd = darts.end();
  for ( ; it != itEnd; ++it) {
    drawSegmentT<SegmentDrawer>(sd, tog, *it);
  }

  ctx.unset();
}

template <typename SegmentDrawer, typename Contexter>
void
drawDartsT(SegmentDrawer &sd, Contexter &ctx,
	   const girl::toger &tog, 
	   const girl::dartDeque &darts,
	   girl::toger::markFlags mark)
{
  ctx.set();

  girl::dartDeque::const_iterator it = darts.begin();
  const girl::dartDeque::const_iterator itEnd = darts.end();
  for ( ; it != itEnd; ++it) {
    drawSegmentT<SegmentDrawer>(sd, tog, *it, mark);
  }

  ctx.unset();
}



#endif /* TOGERPAINTER_HPP */
