/*
  Region based topological and geometrical image representation
  Copyright (C) 2004-2011 Gabriel de Dietrich, Achille Braquelaire, Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlTogerImpl.hpp
 * @brief Region based topological and geometrical image representation (toger implementation).
 */

#ifndef TOGERIMPL_HPP
#define TOGERIMPL_HPP

#include "girlBitmap.hpp"
#include "girlMap.hpp"
#include "girlGe2to.hpp"
#include "girlScanlineData.hpp"

#include "girlPendingEdgeFinder.hpp"

#include "girlHashMap.hpp"
#include "girlUnorderedSet.hpp"


//ZOOM
#include "euclideanPath.hpp" //for girl::REAL_TYPE
class PointTransformer;


namespace girl {

  //ZOOM
  class euclideanPointSet;

  struct LessPoint //TODO:OPTIM:use a hashMap
  {
    bool
    operator()(const girl::point &p1, const girl::point &p2) const
    {
      return (p1.x() < p2.x() || (p1.x() == p2.x() && p1.y() < p2.y()));
    }
  };

  typedef std::map<girl::point, girl::point, LessPoint> MapPoint2Point; //TODO:OPTIM:use a hashMap




  class togerImpl 
  {

  public:

    /**
     * @brief Destructor.
     */
    ~togerImpl();

    /**
     * @brief Constructor.
     */
    togerImpl(POINT_TYPE width, POINT_TYPE height);

    /**
     * @brief Copy constructor.
     */
    togerImpl(const togerImpl &o);

    /**
     * @brief Assignment operator
     */
    togerImpl &operator=(const togerImpl &o);


    /**
     * @return toger width.
     */
    inline POINT_TYPE width() const;

    /**
     * @return toger height.
     */
    inline POINT_TYPE height() const;


    void splitRegion(girl::region r, 
		     girl::toger::sameRegion &same_r, 
		     girl::regionDeque &regs);
  
    girl::region mergeRegions(girl::region r1, girl::region r2);

    bool insertPaths(const girl::pathDeque &paths, 
		     girl::regionDeque &newRegions,
		     girl::regionDeque &updatedRegions);

    bool insertPath(const girl::path &path, 
		    girl::regionDeque &newRegions,
		    girl::regionDeque &updatedRegions);
    
    bool insertContours(const girl::contourDeque &contours, 
			girl::regionDeque &newRegions,
			girl::regionDeque &updatedRegions);

    bool insertContour(const girl::contour &contour, 
		       girl::regionDeque &newRegions,
		       girl::regionDeque &updatedRegions);



    void addObserver(girl::toger::observer *o);

    void removeObserver(girl::toger::observer *o);

    void removeAllObservers();

    void getObservers(std::deque<girl::toger::observer *> &observers) const;

    void addObservers(const std::deque<girl::toger::observer *> &observers);


    bool belongsTo(const girl::point &p, girl::region r) const;

    girl::region findRegion(const girl::point &p) const;

    void findRegions(const girl::point &p1, const girl::point &p2, 
		     girl::regionDeque &regions) const;

    void findRegions(const girl::point &p1, const girl::point &p2, 
		     girl::regionUSet &regions) const;

    void findRegions(const girl::path &path,
		     girl::regionDeque &regions) const;

    girl::dart findDart(const girl::point &p) const;

    girl::point anyPoint(girl::region r) const;

    bool inSameRegion(const girl::point &p, girl::direction dir) const;

    void regionDomain(girl::region r, std::vector<girl::point> &scan) const; //????

    void regionBoundary(girl::region r, girl::contourDeque &b) const;

    void regionOuterBoundary(girl::region r, girl::contour &c) const;

    void regionsCommonBoundary(girl::region r1, girl::region r2, 
			       girl::pathDeque &b) const;

    void segment(girl::dart d, girl::path &p) const;

    size_t segmentLength(girl::dart d) const;

    bool isRegionValid(girl::region r) const;

    bool isFaceValid(girl::face f) const;

    girl::face lambda(girl::dart d) const;

    girl::dart beta(girl::face f) const;

    girl::dart alpha(girl::dart d) const;

    girl::dart sigma(girl::dart d) const;

    girl::dart phi(girl::dart d) const;

    bool isFaceInfinite(girl::face f) const;
  
    girl::face parent(girl::face fi) const;

    girl::face firstChild(girl::face f) const;

    girl::face nextSibling(girl::face fi) const;

    girl::dart getDart(const girl::point &pt, girl::direction dir) const;


    bool areNeighbours(girl::region r1, girl::region r2, 
		       girl::toger::neighbouringModes mode, 
		       girl::toger::markFlags m) const;

    void regionNeighbourhood(girl::region r, 
			     girl::toger::neighbouringModes mode, 
			     girl::toger::markFlags m, 
			     girl::regionDeque &regs) const;

    girl::region anyRegionNeighbour(girl::region r, 
				    girl::toger::neighbouringModes mode, 
				    girl::toger::markFlags m) const;

    void allRegions(girl::regionDeque &regs) const;

    size_t numRegions() const;

    void innerRegions(girl::region r, girl::regionDeque &regs) const;

    void iterateOnRegionDomain(girl::region r,
			       girl::toger::pointIterator &fct) const;
  
    void iterateOnFaceBoundary(girl::region r,
			       girl::toger::directionIterator &fct) const;

    void iterateOnSegment(girl::dart d,
			  girl::toger::directionIterator &fct) const;
  
    void xMarkRegionBoundary(girl::region r, girl::toger::markFlags m);

    void unmarkRegionBoundary(girl::region r, girl::toger::markFlags m);

    void xMarkOuterBoundary(girl::region r, girl::toger::markFlags m);

    void unmarkOuterBoundary(girl::region r, girl::toger::markFlags m);

    bool isRegionMarked(girl::region r, girl::toger::markFlags m) const;

    bool isRegionOuterBoundaryMarked(girl::region r, 
				     girl::toger::markFlags m) const;
  
    bool isFaceMarked(girl::face f, girl::toger::markFlags m) const;

    void markAllDarts(girl::toger::markFlags m);
  
    void unmarkAllDarts(girl::toger::markFlags m);
  
    bool isDartMarked(girl::dart d, girl::toger::markFlags m) const;

    void markCommonBoundary(girl::region r1, girl::region r2, 
			    girl::toger::markFlags m);

    void unmarkCommonBoundary(girl::region r1, girl::region r2, 
			      girl::toger::markFlags m);
  
    void saveBitmap(const std::string &filename) const;
  
    bool checkConsistency() const;

    void shrink();

    bool write(girl::io::Writer &w, girl::io::writeFlags flags) const; 

    bool read(girl::io::Reader &r, girl::io::writeFlags flags); 

    bool swapable(const girl::point &p) const;

    void swap(const girl::point &p);

    void zoom(girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight);

    void crop(const girl::point &p1, const girl::point &p2);

    void swap(girl::togerImpl &t);

  protected:

    typedef girl::unorderedSet<girl::face> FaceUnorderedSet;

    void unregisterFromObservers();

    void  setGeToMap(const girl::geoDart &u, girl::dart d);

    bool checkPixelValidity(const girl::point &pt) const;
    bool checkInterPixelValidity(const girl::point &pt) const;

    bool checkFirstInfiniteFace() const;

    bool checkBitmapBorder() const;

    bool checkFaceMinimality(girl::face f) const;

    bool checkMinimality() const;

    bool checkNodeConsistencyWoLight(const girl::point &pt) const;

    bool checkNodeConsistency(const girl::point &pt) const;

    bool checkPtNodeConsistency(const girl::point &p) const;

    bool checkContourContinuity(girl::dart d0) const;

    bool checkFacePositiveOrientationFromDart(girl::dart d) const;

    bool checkFacePositiveOrientation(girl::face r) const;

    bool areDirectNeighbours(girl::dart d, girl::region r2) const;

    bool areOuterNeighbours(girl::dart d, girl::region r2) const;

    bool isDirectCommonBoundaryMarked(girl::dart d, girl::face f, 
				      girl::toger::markFlags mark) const;

    bool isOuterCommonBoundaryMarked(girl::dart d, girl::face f, 
				     girl::toger::markFlags mark) const;

    girl::toger::neighbouringModes kindOfNeighbourhood(girl::region r1, 
						       girl::region r2, 
						       girl::dart &dn, 
						       girl::toger::markFlags mark) const;

    void iterateOnDomainPart(girl::region r, 
			     girl::toger::pointIterator &fct) const;
    
    void iterateOnDomain(girl::region r, girl::toger::pointIterator &fct) const;
    
  public: //DEBUG
    void faceOuterBoundary(girl::face f, girl::contour &c) const;
  protected:

    size_t commonDarts(girl::region r1, girl::region r2, 
		       std::deque<girl::dart> &s, 
		       girl::toger::neighbouringModes &m) const;

    bool isContourPositivelyOrientedFromDart(girl::dart d) const;


    void extendEdge(girl::dart b);

    void suppressArtificialNodes(girl::dart b);

    void suppressConnectionEdges(girl::dart b);

    girl::region suppressEdge_disconnectedLoop(girl::dart b);

    girl::region suppressEdge_connectedLoopConsecutiveDarts(girl::dart b, 
							    girl::dart sab);

    girl::region suppressEdge_connectedLoopUnconsecutiveDarts(girl::dart b);

    girl::region suppressEdge_notLoopConnectingEdge(girl::dart b, 
						    girl::dart sb, 
						    girl::dart sab, 
						    girl::face fb, 
						    girl::face fab);

    girl::region suppressEdge_notLoopNotConnectingEdge(girl::dart b);

    void clearUselessNodes(const girl::point &pt);

    girl::region suppressEdge(girl::dart b);

    girl::region deleteEdges(girl::region r1, girl::region r2, 
			     std::deque<girl::dart> &s);

    void applyMergeDelHooks(girl::region r1, girl::region r2);

    void spreadMark(girl::dart commonDart, girl::region r2, 
		    girl::toger::neighbouringModes m);

    bool seekRegionContour(girl::geoDart &u,
			   girl::toger::sameRegion &same_r,
			   std::deque<girl::geoDart> &head_s);

    void insureNode(const girl::bitmap::iterator &it,
		    bool &isFirstNodeArtificial,
		    const girl::geoDart &startHead,
		    girl::dart &currentDart, 
		    girl::direction previousDir,
		    girl::direction nextDir);
    
    void seekInnerHead(const girl::geoDart &startHead,
		       girl::dart &currentDart, 
		       girl::direction previousDir, 
		       const girl::bitmap::iterator &it, 
		       girl::direction nextDir, 
		       bool &isFirstNodeArtificial,
		       girl::toger::sameRegion &same_r,
		       std::deque<girl::geoDart> &head_s);

    size_t pushNewHeads(girl::face f,
			girl::toger::sameRegion &same_r,
			std::deque<girl::geoDart> &head_s);
    
    girl::dart buildNewSegment(const girl::geoDart &u0, const girl::geoDart &u);

    girl::dart buildTangentFace(girl::geoDart &u0, girl::geoDart &u);

    girl::dart buildSegmentBetweenExistingNodes(const girl::geoDart &u0, 
						const girl::geoDart &u);

    void buildBoundaries(girl::toger::sameRegion &same_r,
			 std::deque<girl::geoDart> &head_s, 
			 std::deque<girl::dart> &dart_q);

    void labellingRegion(girl::dart d, 
			 girl::region firstRegion, 
			 bool &mustStillUseFirstRegion, 
			 girl::regionDeque &found_regions);

    void labellingRegions(std::deque<girl::dart> &dart_q, 
			  girl::region firstRegion, 
			  girl::regionDeque &found_regions);

    void labellingFace(girl::dart d,
		       girl::region firstRegion, 
		       bool &mustStillUseFirstRegion, 
		       girl::regionDeque &foundRegions);

    void labellingFaces(std::deque<girl::dart> &dart_q, 
			girl::region firstRegion, bool mustUseFirstRegion, 
			girl::regionDeque &found_regions);
    
    void applyNewRegionHook(const girl::regionDeque &out_q);

    void applyNewRegionHook(const girl::regionDeque &newRegions, const girl::regionDeque &updatedRegions);


    void getDaugthers(girl::region r, 
		      girl::regionDeque &region_q, 
		      FaceUnorderedSet &h) const;
    
    void invalidateMotherDaughterRelation(girl::region r);

    void segmentDirections(girl::dart d, std::deque<girl::direction> &dq) const;

    void faceOuterBoundaryD(girl::face f,
			    girl::point &startPt, 
			    std::deque<girl::direction> &dq) const;

    void dealWithTangentNode(girl::dart &d,
			     const girl::geoDart &u, 
			     girl::regionDeque &region_q);

    girl::direction seekOuterPath(girl::dart &d,
				  bool &artificialNode, 
				  const girl::geoDart &startHead, 
				  const girl::geoDart &u,
				  girl::toger::sameRegion &same_r);

    void seekInfiniteFaceContour(girl::dart d,
				 girl::toger::sameRegion &same_r,
				 girl::regionDeque &region_q);

    void seekKnownInnerContour(const girl::point &p,
			       girl::region r,
			       FaceUnorderedSet &h);

    girl::dart insertArtificialNode(const girl::point &pt, 
				    girl::direction dir1, 
				    girl::direction dir2);
    
    girl::face seekInnerContour(girl::point &p, girl::region r,
				girl::toger::sameRegion &same_r,
				girl::regionDeque &region_q);
    
    void seekUnknownInnerContour(girl::point &p, girl::region r,
				 girl::toger::sameRegion &same_r,
				 girl::regionDeque &region_q);

    void seekAndQueueInnerRegionsWithHoles(girl::region r,
					   girl::toger::sameRegion &same_r, 
					   girl::regionDeque &region_q, 
					   FaceUnorderedSet &h);
    
    void splitNotSimplyConnectedRegionAux(girl::region r,
					  girl::toger::sameRegion &same_r, 
					  girl::regionDeque &out_q,
					  girl::regionDeque &regionToSplit_q,
					  FaceUnorderedSet &h);

    void splitNotSimplyConnectedRegion(girl::region r,
				       girl::toger::sameRegion &same_r, 
				       girl::regionDeque &out_q);

    void getFaceScanline(girl::face f, std::deque<girl::point> &dq) const;

    void getRegionScanline(girl::region r, std::deque<girl::point> &dq) const;

    void seekAndQueueInnerRegions(girl::region r,
				  girl::toger::sameRegion &same_r, 
				  girl::regionDeque &region_q);

    void splitSimplyConnectedRegion(girl::region r,
				    girl::toger::sameRegion &same_r, 
				    girl::regionDeque &out_q);

    void split(girl::region r,
	       girl::toger::sameRegion &same_r, 
	       girl::regionDeque &out_q);
    
    void getRegionBoundary(girl::region r, girl::contourDeque &q) const;

    void markFace(girl::face f, girl::toger::markFlags flag);

    void xMarkFace(girl::face f, girl::toger::markFlags flag);

    void unmarkFace(girl::face f, girl::toger::markFlags flag);

    void xMarkInnerBoundary(girl::region r, girl::toger::markFlags flag);

    void unmarkInnerBoundary(girl::region r, girl::toger::markFlags flag);

    
    void changeDartDirection(const girl::point &p, 
			     girl::direction dirOld, girl::direction dirNew);

    void transferDarts(const girl::point &pOld, const girl::point &pNew);
    

    void contourFromFace(girl::face f, girl::contour &c) const;
    
    void pathFromDart(girl::dart d, girl::path &p) const;
    
    void swap_oneNode(const girl::point &p, int resn);
    void swap_twoNodes(const girl::point &p, int resn);


    //INSERT

    typedef girl::unorderedSet<girl::dart> DartUnorderedSet;
    typedef std::deque<girl::point> PtsCollection;

    struct ReducedPath
    {
      girl::path path;
      PtsCollection selfIntersects;
    };
    
    typedef std::deque<ReducedPath> ReducedPathCollection;

    bool insertPaths(const girl::pathDeque &paths,
		     bool doPendingEdgesSearch,
		     bool doNewRegionHook,
		     girl::regionDeque &newRegions,
		     girl::regionDeque &updatedRegions);

    bool reducePaths(const girl::pathDeque &paths,
		     ReducedPathCollection &reducedPaths);

    bool addPtToReducePaths(girl::bitmap::iterator &itB, 
			    girl::direction nextDir, 
			    ReducedPath &currPath, ReducedPath &pendingPath,
			    bool &onContour, 
			    bool &pathStartedFromContourPoint, 
			    ReducedPathCollection &reducedPaths);

    bool check_onlyMarkedPaths(const ReducedPathCollection &reducedPaths);

    bool check_pathIntersections(const path &p) const;

    bool check_pathsIntersections(const ReducedPathCollection &reducedPaths) const;

    void unmarkPaths(const ReducedPathCollection &reducedPaths);

    void unmarkPath(const girl::path &p);

    void markPaths(const ReducedPathCollection &reducedPaths);

    void markPath(const girl::path &p);

    void insureNodesOnContacts(const ReducedPathCollection  &reducedPaths, 
			       std::deque<girl::edge> &facesEdges);
    
    void insureNodesOnContacts(const girl::path &p,
			       std::deque<girl::edge> &facesEdges,
			       DartUnorderedSet &alreadyMovedArtificialNodes);

    girl::dart insureNodeAtHead(const girl::bitmap::iterator &it, 
				girl::direction dir,
				DartUnorderedSet &alreadyMovedArtificialNodes);
      
    void getDartForHead(girl::point headPt, girl::direction headDir,
			girl::direction &prevDir, girl::direction &nextDir,
			girl::dart &d, girl::geoDart &u);

    void setNodesOnSelfIntersections(const ReducedPathCollection &reducedPaths);
    void unsetNodesOnSelfIntersections(const ReducedPathCollection &reducedPaths);


    void getLabelsToReuse(const ReducedPathCollection &reducedPaths,
			  const std::deque<girl::edge> &faceEdges, 
			  girl::regionDeque &labelsToReuse,
			  girl::regionDeque &regionsToSeek_q);

    void insertReducedPaths(const ReducedPathCollection &paths,
			    girl::regionDeque &labelsToReuse,
			    girl::regionDeque &regionsToSeek_update_q,
			    girl::regionDeque &regionsToSeek_new_q,
			    girl::regionDeque &out_q);

    void insertReducedPath(const ReducedPath &p,
			   std::deque<girl::dart> &dart_q,
			   girl::regionDeque &regionsToSeek_q,
			    girl::regionDeque &out_q);

    girl::dart insertDartForReducedPath(girl::bitmap::iterator &itB,
					girl::geoDart &u0, girl::geoDart &u);

    void insertLoopWithoutContact(const girl::path &p,
				  girl::regionDeque &regionsToSeek_q,
				  girl::regionDeque &out_q);

    //void buildPath(const girl::path &p, std::deque<girl::dart> &dart_q);

    void traceAndUnmarkPathInBitmap(const girl::path &p);

    void labellingFacesForPaths(std::deque<girl::dart> &dart_q,
				girl::regionDeque &labelsToReuse,
				girl::regionDeque &regionToSeek_update_q,
				girl::regionDeque &regionToSeek_new_q);

    void seekInnerRegionsWithHoles(girl::region r);

    void seekKnownInnerContourS(const girl::point &p, girl::region r);

    void removeUnnecessaryNodes(const ReducedPathCollection &reducedPaths);

    void removeUnnecessaryNodes(const girl::path &p);

    void removeUnnecessaryNode(const girl::point &pt);

    

    //ZOOM

    void fillEPS(girl::euclideanPointSet &eps);

      
    void transferZoomedNode(const girl::point &pt,
			    const girl::direction &dir,
			    const girl::point &zoomedPt,
			    girl::map &m,
			    girl::bitmap &b,
			    girl::ge2to &g);

    void transferBorderDart(girl::geoDart &u1, girl::geoDart &u2,
			    PointTransformer &ptTr,
			    girl::map &m,
			    girl::bitmap &b,
			    girl::ge2to &g);

    void transferBorderDart(girl::dart d,
			    PointTransformer &ptTr,
			    girl::map &m,
			    girl::bitmap &b,
			    girl::ge2to &g);
    
    void insertDiscreteLineInBitmap(const girl::point &pt1,
				    const girl::point &pt2,
				    girl::bitmap &b);
    
    void insertDiscreteLineInBitmap(const std::vector<girl::point> &pts,
				    girl::bitmap &b);

    void transferZoomedPoints(PointTransformer &ptTr,
			      girl::map &m,
			      girl::bitmap &b,
			      girl::ge2to &g);

    void zoomPoints(const girl::euclideanPointSet &eps,
		    const girl::REAL_TYPE sx, 
		    const girl::REAL_TYPE sy,
		    girl::map &m,
		    girl::bitmap &b,
		    girl::ge2to &g);
    
    //crop

    void insertBoundingRectangle(const girl::point &p1,
				 girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight,
				 girl::regionDeque &newRegions,
				 girl::regionDeque &updatedRegions);

    void findRegionsOutside(const girl::point &p1, const girl::point &p2,
			    girl::regionDeque &regionsD) const;

    void findDartsOuterContour(const girl::point &ip1, const girl::point &ip2,
			       girl::hashMap<girl::dart, girl::dart> &dartsOuterContour) const;

    void applyDelRegionHook(const girl::regionDeque &out_q);

  private:

    girl::map m_m;
    girl::bitmap m_b;
    girl::ge2to m_g;

    typedef std::deque<girl::toger::observer *> ObserverCollection;
    ObserverCollection m_observers;

    girl::scanlineData m_s;

    mutable std::deque<girl::point> m_dq; //internal deque [stored here for performances reasons]

    POINT_TYPE m_width;
    POINT_TYPE m_height;

  };

  //-- inline functions ------------

  POINT_TYPE togerImpl::width() const
  {
    return m_width;
  }


  POINT_TYPE togerImpl::height() const
  {
    return m_height;
  }

  inline
  void
  swap(togerImpl &t1, togerImpl &t2)
  {
    t1.swap(t2);
  }

}

namespace std {

  template <>
  inline
  void
  swap(girl::togerImpl &t1, girl::togerImpl &t2)
  {
    t1.swap(t2);
  }

}



#endif /* ! TOGERIMPL_HPP */
