/*
  Region based topological and geometrical image representation
  Copyright (C) 2004-2011 Gabriel de Dietrich, Achille Braquelaire, Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlTogerImpl.hpp"

#include <vector>
#include <algorithm>
#include <sstream>
#include <iostream>
#include <stdexcept> //runtime_error

#include <cassert>
//#include <cstdlib>


#include "girlAssert.hpp"
#include "girlEndian.hpp"
#include "girlGeoDart.hpp"


//impl: we use the pimpl idiom, to not have to include the following headers
// in girlToger.hpp
#include "girlBitmap.hpp"
#include "girlMap.hpp"
#include "girlGe2to.hpp"
#include "girlScanlineData.hpp"
#include "girlHashFunc.hpp"

//ZOOM
#include <limits>
#include "girlUtils.hpp"
#include "girlBresenhamDir.hpp"
#include "euclideanPath.hpp"
#include "euclideanPointSet.hpp"




using namespace girl;


//GIRL_UIRC: Update Inclusion Relation Continuously
// update inclusion relation continuously for split of 
//not simply connected faces,
// instead of first invalidating mother-daugther relation
#define GIRL_UIRC 1 

//GIRL_ONLYD: for building boundaries, only stack d (and not d and alpha(d))
// and relabel using d & alpha(d).
#define GIRL_ONLYD 1


/**
 * Change this constant if any structure 
 * (in girlToger, girlMap, girlBitmap, girlGe2to) is modified.
 */
static const int GIRL_TOGER_BINARY_VERSION = 1;



void 
togerImpl::setGeToMap(const girl::geoDart &u, girl::dart d)
{
  m_g.setDart(u, d);
  m_m.setGe(d, u);
}



/**
 * @brief Check point validity,
 * that is @a pt is in [0;0]x[w-1;h-1]
 */
bool
togerImpl::checkPixelValidity(const girl::point &pt) const
{
  return pt.x()>=0 && pt.x()<m_width && pt.y()>=0 && pt.y()<m_height;
}

/**
 * @brief Check point validity,
 * that is @a pt is in [-1;-1]x[w;h]
 */
bool
togerImpl::checkInterPixelValidity(const girl::point &pt) const
{
  return pt.x()>=-1 && pt.x()<=m_width && pt.y()>=-1 && pt.y()<=m_height;
}


/**
 * @brief Check face -1  consistency, 
 * that is its contour stays on the image border.
 */
bool
togerImpl::checkFirstInfiniteFace() const
{
  bool consistent = true;
  girl::contour c;
  faceOuterBoundary(-1, c);
  girl::point cpt = c.startPt();
  girl::contour::const_dir_iterator it = c.begin();
  const girl::contour::const_dir_iterator itEnd = c.end();
  for ( ; it != itEnd; ++it) {
    if (! (cpt.x()==-1 || cpt.x()==(m_width - 1)
	   || cpt.y()==-1 || cpt.y()==(m_height - 1)))
      {
	std::cerr<<"Contour point ("
		 <<cpt.x()<<"; "<<cpt.y()<<") of infinite face -1 not on image border !"
		 <<std::endl;
	consistent = false;
	break;
      }
    cpt.move(*it);
  }

  return consistent;
}

/**
 * @brief Checks that bitmap border is set 
 * (it should be face -1 border)
 */
bool
togerImpl::checkBitmapBorder() const
{
  bool consistent = true;

  girl::point cpt;

  cpt.setX(-1);
  for (cpt.setY(m_height-1); cpt.y()>-1; cpt.setY(cpt.y()-1)) {
    bitmap::const_iterator it = m_b.getIterator(cpt);
    if (! it.hasUpNeighbour()) {
      std::cerr<<"border point ("
	       <<cpt.x()<<"; "<<cpt.y()<<") has no UP neighbour in bitmap !"
	       <<std::endl;
      consistent = false;
      break;
    }
  }

  cpt.setY(-1);
  for (cpt.setX(-1); cpt.x()<m_width-1; cpt.setX(cpt.x()+1)) {
    bitmap::const_iterator it = m_b.getIterator(cpt);
    if (! it.hasRightNeighbour()) {
      std::cerr<<"border point ("
	       <<cpt.x()<<"; "<<cpt.y()<<") has no RIGHT neighbour in bitmap !"
	       <<std::endl;
      consistent = false;
      break;
    }
  }
 
  cpt.setX(m_width-1);
  for (cpt.setY(-1); cpt.y() < m_height-1; cpt.setY(cpt.y()+1)) {
    bitmap::const_iterator it = m_b.getIterator(cpt);
    if (! it.hasDownNeighbour()) {
      std::cerr<<"border point ("
	       <<cpt.x()<<"; "<<cpt.y()<<") has no DOWN neighbour in bitmap !"
	       <<std::endl;
      consistent = false;
      break;
    }
  }

  cpt.setY(m_height-1);
  for (cpt.setX(m_width-1); cpt.x()>-1; cpt.setX(cpt.x()-1)) {
    bitmap::const_iterator it = m_b.getIterator(cpt);
    if (! it.hasLeftNeighbour()) {
      std::cerr<<"border point ("
	       <<cpt.x()<<"; "<<cpt.y()<<") has no LEFT neighbour in bitmap !"
	       <<std::endl;
      consistent = false;
      break;
    }
  }
 
  return consistent;
}

togerImpl::~togerImpl()
{
  unregisterFromObservers();
}

togerImpl::togerImpl(POINT_TYPE pWidth, POINT_TYPE pHeight)
  : m_m(), m_b(pWidth, pHeight), m_g(pWidth, pHeight),
    m_observers(),
    m_s(pWidth, pHeight),
    m_dq(),
    m_width(pWidth), m_height(pHeight)
{

  //girl::geoDart u(-1, -1, RIGHT);
  girl::geoDart u(pWidth-1, -1, DOWN);
  setGeToMap(u, 1);
  //u.setDir(DOWN);
  u.setDir(LEFT);
  setGeToMap(u, alpha(1));

  girlAssert2(checkBitmapBorder());
  girlAssert2(checkFirstInfiniteFace());
  girlAssert1(checkConsistency());

}

/*
[OBSERVERS_PIMPL]

REM: Observers are not copied when a toger is copied because 
an observer may contain a reference/pointer towards its subject.
And thus if we copied them, an observer referencing the wrong subject could be signaled.

It works in conjunction with the PIMPL idiom used for Toger only because 
observers are signaled on "side effect" operations : thus on toger just copied/not shared !!!

 */


togerImpl::togerImpl(const togerImpl &o)
  : m_m(o.m_m), m_b(o.m_b), m_g(o.m_g), 
    m_observers(), //Warning: observers are not copied, on purpose. cf OBSERVERS_PIMPL
    m_s(o.m_s),
    m_dq(o.m_dq),
    m_width(o.m_width), m_height(o.m_height)

{
  
}

togerImpl &
togerImpl::operator=(const togerImpl &o)
{
  if (this != &o) {
    m_m = o.m_m;
    m_b = o.m_b; 
    m_g = o.m_g; 
    //m_observers = o.m_observers; //Warning: observers are not copied, on purpose. cf OBSERVERS_PIMPL
    m_s = o.m_s;
    m_dq = o.m_dq;
    m_width = o.m_width; 
    m_height = o.m_height;
  }
  return *this;
}


/**
 * @brief Sets position of given geometrical dart @a u  to 
 * the current position of bitmap @a b.
 */
/*
  static
  void
  P_putBitmapPositionInGDart(const girl::bitmap &b, girl::geoDart &u)
  {
  girl::point pt;
  b.getPosition(pt);
  u.setPoint(pt);
  }
*/


/**
 * @brief Checks face @a f minimality in representation,
 * that is that a face has at most one artificial node and 
 * no artificial node if it is not a loop.
 */
bool
togerImpl::checkFaceMinimality(girl::face f) const
{
  bool consistent = true;
  bool hasArtificialNode = false;
  const girl::dart d0 = m_m.getLambda_1(f);
  girl::dart d = d0;
  do
    {
      girl::geoDart u;
      m_m.getGe(d, u);
      assert(d != 0);
      assert(! m_m.dartIsDeleted(d));
      const int bc = m_b.getIterator(u.point()).getPointCardinality();
      const int mc = m_m.getSigmaCycleCardinality(d);

      if (bc != mc) 
	{
	  consistent = false;
	  break;
	}
      if (bc == 2)
	{
	  if (hasArtificialNode == false)
	    {
	      if (m_m.getPhi(d) == d)
		hasArtificialNode = true;
	      else
		{
		  consistent = false;
		  break;
		}
	    }
	  else
	    {
	      consistent = false;
	      break;
	    }
	}

      d = m_m.getPhi(d);
    }
  while (d != d0);

  return consistent;
}

/**
 * @brief Checks overall toger minimality.
 */
bool
togerImpl::checkMinimality() const
{
  bool consistent = true;
  girl::face f = 0;
  while ((f = m_m.getNextExistingFace(f)) != 0)
    {
      if (!checkFaceMinimality(f))
	{
	  consistent = false;
	  break;
	}
    }
  return consistent;
}


/**
 * @brief Checks node consistency at given position,
 * but does not test if neighbour point in direction of dart is 
 * lighted in bitmap.
 */
bool
togerImpl::checkNodeConsistencyWoLight(const girl::point &pt) const
{
  bool consistent = true;
  girl::geoDart u;
  u.setPoint(pt);
  bitmap::const_iterator it = m_b.getIterator(pt);
  girl::dart prevd = 0;
  for (int i=0; i<4; ++i)
    {
      u.setDir( static_cast<eDirection>(i) );
      girl::dart d = m_g.getDart(u);
      if (d != 0)
	{
	  if (! it.isNode())
	    {
	      consistent = false;
	      break;
	    }
	  else
	    {
	      girl::geoDart v;
	      m_m.getGe(d, v);
	      if (! (v == u)) 
		{
		  consistent = false;
		  break;
		}
	      if (prevd != 0) 
		{
		  //warning: implicitly considers directions declared in CCW order...
		  if (m_m.getSigma(prevd) != d) 
		    {
		      consistent = false;
		      break;
		    }
		}
	    }
	  prevd = d;
	}
    }   
  return consistent;
}

/**
 * @brief Checks node consistency at current bitmap position.
 */
bool
togerImpl::checkNodeConsistency(const girl::point &pt) const
{
  bool consistent = true;
  girl::geoDart u;
  u.setPoint(pt);
  bitmap::const_iterator it = m_b.getIterator(pt);
  girl::dart prevd = 0;
  for (int i=0; i<4; ++i)
    {
      u.setDir( static_cast<eDirection>(i) );
      girl::dart d = m_g.getDart(u);
      if (d != 0)
	{
	  if (! it.isNode())
	    {
	      consistent = 0;
	      std::cerr<<"togerImpl::checkNodeConsistency ("<<it.getPosition().x()<<"; "<< it.getPosition().y()<<") not a node in bitnamp"<<std::endl;
	      break;
	    }
	  else
	    {
	      //node in bitmap

	      if (! it.hasNeighbourInDirection( static_cast<eDirection>(i) ))
		{
		  consistent = false;
		  std::cerr<<"togerImpl::checkNodeConsistency at ("<<it.getPosition().x()<<"; "<< it.getPosition().y()<<"), dart "<<d<<" in direction "<<i<<" but no neighbour in bitmap"<<std::endl;
		  break;
		}
	      girl::geoDart v;
	      m_m.getGe(d, v);
	      if (! (v == u))
		{
		  std::cerr<<"togerImpl::checkNodeConsistency at ("<<it.getPosition().x()<<"; "<< it.getPosition().y()<<"), dart "<<d<<" in direction "<<i<<" in ge2to but not this geoDart in map : pt=("<<v.x()<<"; "<<v.y()<<") dir="<<v.dir()<<std::endl;

		  consistent = false;
		  break;
		}
	      if (prevd != 0) 
		{
		  //warning: implicitly considers directions declared in CCW order...
		  if (m_m.getSigma(prevd) != d) 
		    {
		      std::cerr<<"togerImpl::checkNodeConsistency at ("<<it.getPosition().x()<<"; "<< it.getPosition().y()<<"), wrong sigma cycle"<<std::endl;

		      consistent = false;
		      break;
		    }
		}
	    }
	  prevd = d;
	}
    }   
  return consistent;
}

/**
 * @brief Checks node consistency at given point @a p.
 */
bool
togerImpl::checkPtNodeConsistency(const girl::point &p) const
{
  return checkNodeConsistency(p);
}

/**
 * @brief Checks contour continuity, starting from dart @a d0,
 * that is following the contour in bitmap we traverse by all the nodes of 
 * the Phi-cycle.
 *
 * @warning can be costly
 */
bool
togerImpl::checkContourContinuity(girl::dart d0) const
{
  assert(m_m.checkDartPhiCycle(d0));

  bool consistent = true;

  girl::geoDart u, v;
  m_m.getGe(d0, u);

  girl::dart d = d0;
  do
    {
      bitmap::const_iterator it = m_b.getIterator(u.point());
      do
	{
	  it.moveInDirection(u.dir());
	  u.setDir(it.getNextDirection(u.dir()));
	  u.setPoint(it.getPosition()); 
	  //TODO:optim ?
	}
      while (! it.isNode());

      d = m_m.getPhi(d);
      m_m.getGe(d, v);
      if (v.point() != u.point())
	{
	  consistent = false;
	  break;
	} 
      u = v;
    }
  while (d != d0);

  return consistent;
}


#ifndef NDEBUG1 

static
void
DBG_printContour(girl::contour &c)
{
  size_t nbp = 0;
  girl::point pt = c.startPt();
  girl::contour::const_dir_iterator it = c.begin();
  const girl::contour::const_dir_iterator itEnd = c.end();
  for ( ; it!=itEnd; ++it) {
    std::cerr<<"("<<pt.x()<<"; "<<pt.y()<<") ";
    pt.move(*it);
    ++nbp;
  }
  std::cerr<<"\n->"<<nbp<<" contour points\n";
}
/*
static
void
DBG_printRegionContourFromDart(const girl::togerImpl &tog, girl::dart d)
{
  std::cerr<<"DBG_printRegionContourFromDart of "<<d<<std::endl;
  girl::contour c;
  tog.regionOuterBoundary(tog.lambda(d), c);
  DBG_printContour(c);
}

static
void
DBG_printRegionContour(const girl::togerImpl &tog, girl::region r)
{
  std::cerr<<"DBG_printRegionContour of "<<r<<std::endl;
  girl::contour c;
  tog.regionOuterBoundary(r, c);
  DBG_printContour(c);
}
*/

//TODO: A FAIRE AVEC tog ! ou integrer a class toger !!!
/*
  static
  void
  DBG_printPhiCycleFromDart(const girl::toger &tog, girl::dart canonicD)
  {
  girl::dart d = canonicD;
  do
  {
  girl::geoDart u;
  m_m.getGe(d, &u);
  std::cerr<<"dart "<<d<<" at ("<<u.x()<<"; "<<u.y()<<") 
  dir="<<u.dir()<<"  lambda="<<m_m.getLambda(d)<<std::endl;
  d = m_m.getPhi(d);
  }
  while (canonicD != d);
  }


  //TODO: A FAIRE AVEC tog ! ou integrer a class toger !!!
  static
  void
  DBG_printPhiCycle(const girl::toger &tog, girl::face f)
  {
  std::cerr<<"Phi cycle of face "<<f<<std::endl;
  const girl::dart canonicD = m_m.getLambda_1(f);
  DBG_printPhiCycleFromDart(tog, canonicD);
  }
*/

#endif


bool
togerImpl::checkFacePositiveOrientationFromDart(girl::dart d) const
{
  return checkFacePositiveOrientation(lambda(d));
}

bool
togerImpl::checkFacePositiveOrientation(girl::face f) const
{
  bool consistent = true;
  girl::contour c;

  faceOuterBoundary(f, c);

  if (! c.isPositivelyOriented())
    consistent = false;

  return consistent;
}



/**
 * @brief Moves girl::point of given geometrical dart @a u in direction @a d.
 */
static
void
P_moveGDartPoint(girl::geoDart &u, girl::direction d)
{
  girl::point pt = u.point();
  pt.move(d);
  u.setPoint(girl::point(pt.x(), pt.y()));
}

/**
 * interior pixel 
 * with @a d direction towards point @a p
 */
static
void
P_img_LR_pixel(girl::point &p, girl::direction d)
{
  switch (d)
    {
    case RIGHT:
      p.setY(p.y()+1);
      break;
    case UP:
      p.set(p.x()+1, p.y()+1);
      break;
    case LEFT:
      p.setX(p.x()+1);
      break;
    case DOWN:
      break;
    case INVALID:
    default:
      assert(false);
      break;
    }
}

/**
 * exterior pixel 
 * with @a d direction towards point @a p
 */
static
void
P_img_LL_pixel(girl::point &p, girl::direction d)
{
  switch (d)
    {
    case RIGHT:
      break;
    case UP:
      p.setY(p.y()+1);
      break;
    case LEFT:
      p.set(p.x()+1, p.y()+1);
      break;
    case DOWN:
      p.setX(p.x()+1);
      break;
    case INVALID:
    default:
      assert(false);
      break;
    }
}

void
togerImpl::addObserver(girl::toger::observer *o)
{
  removeObserver(o); //to avoid duplicates
  o->registerSubject(this);
  m_observers.push_back(o);
  
}

void
togerImpl::removeObserver(girl::toger::observer *o)
{
  ObserverCollection::iterator it = std::find(m_observers.begin(), m_observers.end(), o);
  if (it != m_observers.end()) {
    m_observers.erase(it);
    o->unregisterSubject(this);
  }
}

void
togerImpl::removeAllObservers()
{
  ObserverCollection::const_iterator it = m_observers.begin();
  const ObserverCollection::const_iterator itEnd = m_observers.end();
  for ( ; it != itEnd; ++it) {
    (*it)->unregisterSubject(this);
  }
  m_observers.clear();
}

void
togerImpl::getObservers(std::deque<girl::toger::observer *> &observers) const
{
  observers = m_observers;
}

void
togerImpl::addObservers(const std::deque<girl::toger::observer *> &observers)
{
  ObserverCollection::const_iterator it = observers.begin();
  const ObserverCollection::const_iterator itEnd = observers.end();
  for ( ; it != itEnd; ++it) {
    addObserver(*it);
  }
}

void
togerImpl::unregisterFromObservers()
{
  ObserverCollection::iterator it = m_observers.begin();
  const ObserverCollection::iterator itEnd = m_observers.end();
  for ( ; it != itEnd; ++it) {
    (*it)->unregisterSubject(this);
  }
}

void
girl::toger::observer::unregisterSubject(girl::togerImpl *tog)
{
  SubjectCollection::iterator it = std::find(m_subjects.begin(), m_subjects.end(), tog);
  if (it != m_subjects.end())
    m_subjects.erase(it);
}

void
girl::toger::observer::registerSubject(girl::togerImpl *tog)
{
  unregisterSubject(tog); //to avoid duplicates
  m_subjects.push_back(tog);
}

void
girl::toger::observer::unregisterSubjects()
{
  for (SubjectCollection::reverse_iterator it = m_subjects.rbegin(); 
       it != m_subjects.rend(); 
       ++it) {
    (*it)->removeObserver(this);
  }
}

bool
togerImpl::isRegionValid(girl::region r) const
{
  return m_m.checkRegion(r);
}

bool
togerImpl::isFaceValid(girl::face f) const
{
  return m_m.checkFace(f);
}

bool
togerImpl::belongsTo(const girl::point &p, girl::region r) const
{
  assert(m_m.checkRegion(r));

  return findRegion(p) == r;
}


girl::region
togerImpl::findRegion(const girl::point &p) const
{
  assert(checkPixelValidity(p));

  const girl::dart d = findDart(p);

  assert(d);
  
  const girl::face f = m_m.getLambda(d);
  const girl::face fm = m_m.faceGetMother(f);

  return fm != GIRL_FINITE_FACE ? fm : f;
}


void
togerImpl::findRegions(const girl::point &p1, const girl::point &p2, 
		       girl::unorderedSet<girl::region> &regions) const
{
  const size_t w = p2.x()-p1.x()+1;
  const size_t h = p2.y()-p1.y()+1;
  if (w==0 || h==0) { //no point
    return;
  }
  if (w==1 && h==1) {//regions reduced to one point
    const girl::region r = findRegion(p1);
    regions.insert(r);
    return;
  }
    

  //algo: one each line of bounding box (defined by p1 & p2), 
  //      do findRegion() only if there is a Up neighbour.
  //REM: is is like doing the bitmap::sound() manually ???

  for (girl::POINT_TYPE y=p1.y(); y<=p2.y(); ++y) {
    bitmap::const_iterator itB = m_b.getIterator(girl::point(p1.x(), y));
    girl::POINT_TYPE lastX = p1.x()-1;

    for (girl::POINT_TYPE x=p1.x(); ; ) {
      if (itB.hasUpNeighbour()) {
	const girl::point &pos = itB.getPosition();

	const girl::region r = findRegion(pos);
	
	assert(r != 0);
	regions.insert(r);

	lastX = pos.x();
      }
      ++x;
      if (x>p2.x())
	break;
      itB.moveRight();
    }
    if (lastX != p2.x()) {

      const girl::region r = findRegion(girl::point(p2.x(), y));

      assert(r != 0);
      regions.insert(r);
    }    

  }

}

#if 0

//T may be girl::regionDeque, std::vector<girl::region>,  ...
template <typename T>
void
P_unique(T &regions)
{
  std::sort(regions.begin(), regions.end());

  regions.erase(std::unique(regions.begin(), regions.end()), 
		regions.end());
}

void
togerImpl::findRegions(const girl::point &p1, const girl::point &p2, 
		       girl::regionDeque &regionsD) const
{
  assert(checkPixelValidity(p1));
  assert(checkPixelValidity(p2));
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());

  const size_t w = p2.x()-p1.x()+1;
  const size_t h = p2.y()-p1.y()+1;
  if (w==0 || h==0) { //no point
    return;
  }
  if (w==1 && h==1) {//regions reduced to one point
    const girl::region r = findRegion(p1);
    regionsD.push_back(r);
    return;
  }
    
  const size_t reserveSize = ((w+1)/2 * (h+1)/2);
  //B: heuristic: should be ok even if w==1 || h==1.

  typedef std::vector<girl::region> regionVector;
  regionVector regions;
  regions.reserve(reserveSize);


  for (girl::POINT_TYPE y=p1.y(); y<=p2.y(); ++y) {
    bitmap::const_iterator itB = m_b.getIterator(girl::point(p1.x(), y));
    girl::POINT_TYPE lastX = p1.x()-1;

    for (girl::POINT_TYPE x=p1.x(); ; ) {
      if (itB.hasUpNeighbour()) {
	const girl::point &pos = itB.getPosition();

	const girl::region r = findRegion(pos);

	assert(r != 0);
	if (regions.empty() || regions.back()!=r)
	  regions.push_back(r);
	lastX = pos.x();
      }
      ++x;
      if (x>p2.x())
	break;
      itB.moveRight();
    }
    if (lastX != p2.x()) {

      const girl::region r = findRegion(girl::point(p2.x(), y));

      assert(r != 0);
      if (regions.empty() || regions.back()!=r)
	regions.push_back(r);
    }    

  }

  P_unique(regions);
  
  regionVector::const_iterator it = regions.begin();
  const regionVector::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it) {
    regionsD.push_back(*it);
  }

}

#else


void
togerImpl::findRegions(const girl::point &p1, const girl::point &p2, 
		       girl::regionDeque &regionsD) const
{
  assert(checkPixelValidity(p1));
  assert(checkPixelValidity(p2));
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());

  typedef girl::unorderedSet<girl::region> RegionsHash;
  RegionsHash regions;
  findRegions(p1, p2, regions);
    
  RegionsHash::const_iterator it = regions.begin();
  const RegionsHash::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it) {
    regionsD.push_back(*it);
  }

}


#endif

void
togerImpl::findRegions(const girl::path &path,
		       girl::regionDeque &regions) const
{
  //assert(path.isValid());

  //std::cerr<<"togerImpl::findRegions path.length()="<<path.length()<<std::endl;

  typedef girl::unorderedSet<girl::region> RegionsHash;
  RegionsHash regionsH;

  if (path.length() > 0) {

    //REM/Impl: the idea is to reduce the number of calls to findRegion(girl::point).
    // We should call findRegion(girl::point) only for points of part of path traversing
    // boundaries in bitmap.

    //- Call findRegions only when path traverse a boundary in bitmap : 
    // for two points on each side of the boundary
    girl::point lastAddedPoint;

    girl::path::const_dir_iterator it = path.begin();
    const girl::path::const_dir_iterator itEnd = path.end();
    girl::point p1 = path.startPt();
    bitmap::const_iterator itB = m_b.getIterator(p1); //always on p1
    for ( ; it != itEnd; ++it) {
      const girl::direction dir = *it;
      //check if there is a boundary between two consecutive pixels
      bool border = false;
      switch (dir) {
      case RIGHT:
	border = itB.hasUpNeighbour();
	itB.moveRight(); //moveInDirection(dir);
	break;
      case UP:
	itB.moveUp(); //moveInDirection(dir);
	border = itB.hasLeftNeighbour();
	break;
      case LEFT:
	itB.moveLeft(); //moveInDirection(dir);
	border = itB.hasUpNeighbour();
	break;
      case DOWN:
	border = itB.hasLeftNeighbour();
	itB.moveDown(); //moveInDirection(dir);
	break;
      case INVALID:
      default:
	assert(false);
      }
      if (border) {
	if (regionsH.empty() || lastAddedPoint != p1) { //(*)
	  const girl::region r1 = findRegion(p1);
	  regionsH.insert(r1);
	}
	
	p1.move(dir);//p1 become p2

	const girl::region r2 = findRegion(p1);
	regionsH.insert(r2);
	lastAddedPoint = p1;//i.e., p2

	//(*) this test avoids to add 
	// two times the same point for 
	// regions reduced to one pixel.
	// Indeed if we traverse a region of one pixel,
	// its pixel is added as p2 on a first pass,
	// and as p1 on the second pass.
      }
      else {
	p1.move(dir);
      }
    }
  }
  if (! regionsH.empty()) {
    //std::cerr<<"togerImpl::findRegions nb found regions="<<regionsH.size()<<std::endl;
    RegionsHash::const_iterator itH = regionsH.begin();
    const RegionsHash::const_iterator itHEnd = regionsH.end();
    for ( ; itH != itHEnd; ++itH) {
      regions.push_back(*itH);
    }
  }
  else {
    regions.push_back(findRegion(path.startPt()));
    //std::cerr<<"togerImpl::findRegions nb found regions="<<1<<std::endl;
  }

}



girl::dart
togerImpl::findDart(const girl::point &p) const
{
  assert(checkPixelValidity(p));

  girl::point contourPt;
  m_b.sound(p, contourPt);
  assert (contourPt.x()!=INVALID || contourPt.y()!=INVALID);
  
  girl::geoDart u;
  m_b.contourTracking(contourPt, u);

  return m_g.getDart(u);
}




girl::point
togerImpl::anyPoint(girl::region r) const
{
  assert(m_m.checkRegion(r));

  const girl::dart d = m_m.getLambda_1(r);
  girl::geoDart u;
  m_m.getGe(d, u);
  girl::point pt = u.point();
  const girl::direction dir = u.dir();
  pt.move(dir);
  P_img_LR_pixel(pt, dir);

  assert(checkPixelValidity(pt));

  return pt;
}

//B:
//This method allows to split (rather efficiently) a toger using an other existing toger.
//This is the only function giving read-access to the bitmap.
// iterateOnRegionDomain & iterateOnFaceBoundary also give read-access, but are more cumbersome to use.
bool
togerImpl::inSameRegion(const girl::point &p, girl::direction dir) const
{
  assert(checkPixelValidity(p));
  //Bitmap is in inter-pixels: we have to check if there is a link between inter-pixel corners
  switch (dir) {
  case RIGHT:
    return ! m_b.getIterator(p).hasUpNeighbour();
    break;
  case UP:
    return ! m_b.getIterator(girl::point(p.x()-1, p.y()-1)).hasRightNeighbour();
    break;
  case LEFT:
    return ! m_b.getIterator(girl::point(p.x()-1, p.y())).hasUpNeighbour();
    break;    
  case DOWN:
    return ! m_b.getIterator(girl::point(p.x()-1, p.y())).hasRightNeighbour();
    break;
    case INVALID:
    default:
      assert(false);
      break;
    }
  assert(false);
  return false;
}

/**
 * @brief Checks if region r1 to which dart @a d belongs has region @a r2 as 
 * DIRECT neighbour.
 */
bool
togerImpl::areDirectNeighbours(girl::dart d, girl::region r2) const
{
  return (m_m.getLambda(m_m.alpha(d)) == r2);
}

/**
 * @brief Checks if region r1 to which dart @a d belongs has region @a r2 as 
 * OUTER neighbour.
 */
bool
togerImpl::areOuterNeighbours(girl::dart d, girl::region r2) const
{
  const girl::face f = m_m.getLambda(m_m.alpha(d));
  return (f != -1 && m_m.faceGetMother(f) == r2);
}


/**
 * @brief Checks if dart @a d of face f1, direct neighbour of face f2, has its 
 * boundary marked by @a mark.
 *
 * @return true if at least a dart is marked.
 * @warning lambda_1(@a d) and @a f must be direct neighbours. 
 */
bool
togerImpl::isDirectCommonBoundaryMarked(girl::dart d, girl::face f, 
					girl::toger::markFlags mark) const
{
  girlAssert1(areDirectNeighbours(d, f));
  
  if (mark == girl::toger::MARK_NONE)
    return false;

  const girl::dart d0 = m_m.getLambda_1(m_m.getLambda(d));
  girl::dart cd = d;
  do 
    {
      if (areDirectNeighbours(cd, f)
	  && m_m.dartIsMarked(cd, mark))//BTODO: alpha ?
	return true;

      cd = m_m.getPhi(cd);
    }
  while (cd != d0);
  
  return false;
}

/**
 * @brief Checks if dart @a d of face f1, outer neighbour of face f2, has its 
 * boundary marked by @a mark.
 *
 * @return true if at least a dart is marked.
 * @warning lambda_1(@a d) and @a f must be outer neighbours. 
 */
bool
togerImpl::isOuterCommonBoundaryMarked(girl::dart d, girl::face f, 
				       girl::toger::markFlags mark) const
{
  girlAssert1(areOuterNeighbours(d, f));

  if (mark == girl::toger::MARK_NONE)
    return false;

  const girl::dart d0 = m_m.getLambda_1(m_m.getLambda(d));
  girl::dart cd = d;
  do 
    {
      if (areOuterNeighbours(cd, f)
	  && m_m.dartIsMarked(cd, mark))//BTODO: alpha ?
	return true;

      cd = m_m.getPhi(cd);
    }
  while (cd != d0);
  
  return false;
}

/**
 * Gives neighbouring mode of two regions @a r1 and @a r2, and 
 * first neighbouring dart @a nd not marked by @a mark.
 *
 * If regions @a r1 and @a r2 are neighbours, returned neighbouring mode 
 * will be DIRECT, OUTER or INNER; and @a d will be the first dart 
 * of the Phi cycle (following canonical dart), not marked by @a mark, and 
 * such as :
 * - @a dn belongs to @a r1 and alpha(@a dn) belongs to @a r2 if neighbouring 
 *  mode is DIRECT,
 * - @a dn belongs to @a r1 and alpha(@a dn) belongs to a daugther of @a r2 if 
 *  neighbouring mode is OUTER,
 * - @a dn belongs to @a r2 and alpha(@a dn) belongs to a daugther of @a r1 if 
 *  neighbouring mode is INNER.
 * @warning d could be canonical dart of its region, so the first dart 
 * of the phi cycle.
 *
 * If regions @a r1 and @a r2 are not neighbours, 0 is returned.
 */
girl::toger::neighbouringModes 
togerImpl::kindOfNeighbourhood(girl::region r1, girl::region r2, girl::dart &dn, girl::toger::markFlags mark) const
{
  girl::dart d, canonicD;
  //first, test DIRECT and OUTER neighbourhood
  canonicD = d = m_m.getLambda_1(r1);
  do
    {
      girlAssert1(m_m.getLambda(d) == r1);
      
      if (areDirectNeighbours(d, r2))
	{
	  if (isDirectCommonBoundaryMarked(d, r2, mark))//BTODO:alpha?
	    {
	      dn = 0;
	      return girl::toger::NEIGHBOUR_NONE;
	    }

	  dn = d;
	  return girl::toger::NEIGHBOUR_DIRECT;
	}

      if (areOuterNeighbours(d, r2))
	{
	  if (isOuterCommonBoundaryMarked(d, r2, mark))//BTODO:alpha?
	    {
	      dn = 0;
	      return girl::toger::NEIGHBOUR_NONE;
	    }

	  dn =d;
	  return girl::toger::NEIGHBOUR_OUTER;
	}
      d = m_m.getPhi(d);
    }
  while (canonicD != d);

  //then test INNER neigbourhood
  //r1 is inner neighbour of r2 only if r2 is outer neighbour of r1
  canonicD = d = m_m.getLambda_1(r2);
  do 
    {
      girlAssert1(m_m.getLambda(d) == r2);
      if (areOuterNeighbours(d, r1))
	{
	  if (isOuterCommonBoundaryMarked(d, r1, mark))//BTODO:alpha?
	    {
	      dn = 0;
	      return girl::toger::NEIGHBOUR_NONE;
	    }

	  dn = d;
	  return girl::toger::NEIGHBOUR_INNER;
	}
      d = m_m.getPhi(d);
    }
  while (canonicD != d);

  dn = 0;
  return girl::toger::NEIGHBOUR_NONE;
}

bool
togerImpl::areNeighbours(girl::region r1, girl::region r2, girl::toger::neighbouringModes mode, girl::toger::markFlags f) const
{
  assert(m_m.checkRegion(r1));
  assert(m_m.checkRegion(r2));

  //TODO: OPTIM ??? avoid temporary ???

  girl::dart d;
  girl::toger::neighbouringModes nm = kindOfNeighbourhood(r1, r2, d, f);
  return ((nm & mode) != 0);
}


void
togerImpl::allRegions(girl::regionDeque &regs) const
{
  regs.clear();

  girl::face f = 0;
  while ((f = m_m.getNextExistingFace(f)) != 0)
    if (! m_m.faceIsInfinite(f))
      regs.push_back(f);
}

size_t 
togerImpl::numRegions() const
{
  //Allow to know the number of regions without any allocation 
  //(contrary to allRegions)
  //Useful for example to quickly know if there are at least 2 regions and 
  // thus if a merge is possible
  //Complexity: O(N) where N is the number of faces in toger.

  size_t num = 0;
  girl::face f = 0;
  while ((f = m_m.getNextExistingFace(f)) != 0)
    if (! m_m.faceIsInfinite(f))
      ++num;
  return num;
}




void
togerImpl::innerRegions(girl::region r, girl::regionDeque &regs) const
{
  assert(m_m.checkRegion(r));
  
  typedef girl::unorderedSet<girl::region> RegionsHash;
  RegionsHash regions;

  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f) {
    const girl::dart canonicalDart = m_m.getLambda_1(f);
    girl::dart d = canonicalDart;
    do {
      const girl::region ri = m_m.getLambda(m_m.alpha(d));
      assert(! m_m.faceIsInfinite(ri));
      assert(ri != 0);
      regions.insert(ri);

      d = m_m.getPhi(d);
    }
    while (d != canonicalDart);
    
    f = m_m.faceGetNextSister(f);
  }

  RegionsHash::const_iterator it = regions.begin();
  const RegionsHash::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it) {
    regs.push_back(*it);
  }
}


void
togerImpl::iterateOnDomainPart(girl::region r, 
			       girl::toger::pointIterator &fct) const
{

  //assert(checkBitmapBorder());
  //assert(checkFirstInfiniteFace());
  //assert(checkConsistency());
  
  const girl::dart canonicalDart = m_m.getLambda_1(r);
  girl::dart d = canonicalDart;

  girl::geoDart u;
  m_m.getGe(canonicalDart, u);
  bitmap::const_iterator it = m_b.getIterator(u.point());

  //assert(m_b.isNode());

  do
    {
      m_m.getGe(d, u);
      girl::direction dir = u.dir();

      do 
	{
	  if (dir == UP) 
	    {
	      girl::point pt = it.getPosition();
	      assert(pt.x() < m_width-1);

	      girl::point mpt = pt;
	      do {
		mpt.setX(mpt.x()+1); //move on pixel on the right
		//assert(mpt.x() < m_width);

		fct.processPoint(mpt);

		it.moveRight();
	      }
	      while (! it.hasUpNeighbour());
	      
	      it.moveTo(pt);
	    }
	  it.moveInDirection(dir);
	  dir = it.getNextDirection(dir);
	}
      while (! it.isNode());
      
      d = m_m.getPhi(d);
    }
  while (d != canonicalDart);

}

//TODO: code duplication: function that traverse a region and its daugther
// could be factorized !!
void
togerImpl::iterateOnDomain(girl::region r, 
			   girl::toger::pointIterator &fct) const
{
  iterateOnDomainPart(r, fct);
  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      iterateOnDomainPart(f, fct);
      f = m_m.faceGetNextSister(f);
    }
}


void
togerImpl::iterateOnRegionDomain(girl::region r,
				 girl::toger::pointIterator &fct) const
{
  assert(m_m.checkRegion(r));

  iterateOnDomain(r, fct);
}

/*
  static
  void
  P_processBoundaryPixels(const girl::point &pt, const girl::direction prevDir, 
  girl::toger::interPixelIterator &fct)
  {
  girl::point pt1, pt2;

  switch (prevDir) {
  case RIGHT:
  pt1.setX(pt.x());
  pt1.setY(pt.y());
  pt2.setX(pt.x());
  pt2.setY(pt.y()+1);
  break;
  case UP:
  pt1.setX(pt.x());
  pt1.setY(pt.y()+1);
  pt2.setX(pt.x()+1);
  pt2.setY(pt.y()+1);
  break;
  case LEFT:
  pt1.setX(pt.x()+1);
  pt1.setY(pt.y()+1);
  pt2.setX(pt.x()+1);
  pt2.setY(pt.y());
  break;
  case DOWN:   
  pt1.setX(pt.x()+1);
  pt1.setY(pt.y());
  pt2.setX(pt.x());
  pt2.setY(pt.y());
  break;
  default:
  assert(false);
  break;
  }

  fct.iterateInterPixel(pt1, pt2);
  }
*/

/**
 * Iterate on face boundary, that is on inter-pixel points.
 *
 * @warning do not consider face @a f eventual daughters.
 *
 */
void
togerImpl::iterateOnFaceBoundary(girl::face f, girl::toger::directionIterator &fct) const
{
  //some code duplication with segmentDirections() and iterateOnSegment()
  //We cannot call iterateOnSegment() because fct.processLastPoint(()
  // would be called on each node and not only once.

  const girl::dart canonicalDart = m_m.getLambda_1(f);
  girl::dart d = canonicalDart;

  girl::geoDart u;
  m_m.getGe(canonicalDart, u);

  const girl::point firstPt = u.point();

  fct.processFirstPoint(firstPt);

  bitmap::const_iterator it = m_b.getIterator(firstPt);

  do
    {
      m_m.getGe(d, u);
      girl::direction dir = u.dir();

      do 
	{
	  fct.processDirection(dir);

	  it.moveInDirection(dir);

	  dir = it.getNextDirection(dir);
	}
      while (! it.isNode());
      
      d = m_m.getPhi(d);
    }
  while (d != canonicalDart);

  assert(it.getPosition() == firstPt);

  fct.processLastPoint(firstPt);
}
  
void 
togerImpl::iterateOnSegment(girl::dart d,
			    girl::toger::directionIterator &fct) const
{
  //some code duplication with segmentDirections()
  
  //impl: does not allocate a segment/path explicitly !
  girlAssert1(m_m.checkDart(d));

  girl::geoDart u;
  m_m.getGe(d, u);

  fct.processFirstPoint(u.point());

  bitmap::const_iterator it = m_b.getIterator(u.point());
  girl::direction dir = u.dir();

  do
    {
      fct.processDirection(dir);
      
      it.moveInDirection(dir);
      dir = it.getNextDirection(dir);
    }
  while (! it.isNode());

  fct.processLastPoint(it.getPosition());
}


girl::region
togerImpl::anyRegionNeighbour(girl::region r, 
			      girl::toger::neighbouringModes mode, 
			      girl::toger::markFlags mark) const
{
  assert(m_m.checkRegion(r));

  //BTODO: mark correctly used ? 

  if (mode & girl::toger::NEIGHBOUR_OUTER)
    {
      const girl::dart canonicD = m_m.getLambda_1(r);
      girl::dart d = canonicD;
      do
	{
	  girl::face f = m_m.getLambda(m_m.alpha(d));
	  if (m_m.faceIsInfinite(f)) 
	    {
	      if (f == -1)
		break;

	      f = m_m.faceGetMother(f);
	      if (! isOuterCommonBoundaryMarked(d, f, mark))
		return f;
	      break;
	    }
	  d = m_m.getPhi(d);
	}
      while (d != canonicD);
    }

  if (mode & girl::toger::NEIGHBOUR_DIRECT)
    {
      //Warning: complexity = O(N2) with N=card(phi*(canonicD)) !
      const girl::dart canonicD = m_m.getLambda_1(r);
      girl::dart d = canonicD;
      do
	{
	  girl::face f = m_m.getLambda(m_m.alpha(d));
	  if (! m_m.faceIsInfinite(f) 
	      && ! isDirectCommonBoundaryMarked(d, f, mark))
	    return f;
	  
	  d = m_m.getPhi(d);
	}
      while (d != canonicD);
    }

  if (mode & girl::toger::NEIGHBOUR_INNER)
    {
      //Warning: complexity 
      girl::face daughter = m_m.faceGetFirstDaughter(r);
      while (daughter)
	{
	  const girl::dart canonicD = m_m.getLambda_1(daughter);
	  girl::dart d = canonicD;
	  do
	    {
	      girl::face f = m_m.getLambda(m_m.alpha(d));
	      girlAssert1(! m_m.faceIsInfinite(f));
	      if (! isOuterCommonBoundaryMarked(m_m.alpha(d), r, mark))
		return f;

	      d = m_m.getPhi(d);
	    }
	  while (d != canonicD);
	  daughter = m_m.faceGetNextSister(daughter);
	}
    }
  
  return 0;
}

#ifndef NDEBUG
static
bool
P_checkRegionUniqueness(const girl::regionDeque &regs)
{
  if (regs.size() <= 1)
    return true;

  for (girl::regionDeque::const_iterator it = regs.begin();
       it != regs.end()-1; ++it) {

    for (girl::regionDeque::const_iterator it2 = it + 1;
	 it2 != regs.end(); ++it2) {

      if (*it == *it2)
	return false;

    }
  }
  return true;
}
#endif

void
togerImpl::regionNeighbourhood(girl::region r, 
			       girl::toger::neighbouringModes mode, 
			       girl::toger::markFlags mark, 
			       girl::regionDeque &regs) const
{
  assert(m_m.checkRegion(r));
  //BTODO: mark correctly used ? 

  if (mode & girl::toger::NEIGHBOUR_OUTER)
    {
      const girl::dart canonicD = m_m.getLambda_1(r);
      girl::dart d = canonicD;
      do
	{
	  girl::face f = m_m.getLambda(m_m.alpha(d));
	  if (m_m.faceIsInfinite(f))
	    {
	      if (f == -1)
		break;

	      f = m_m.faceGetMother(f);
	      assert(m_m.checkFace(f));
	      assert(! m_m.faceIsMarked(f));
	      if (! isOuterCommonBoundaryMarked(d, f, mark))
		{
		  regs.push_back(f);
		  m_m.faceMark(f);
		}
	      break;
	    }
	  d = m_m.getPhi(d);
	}
      while (d != canonicD);
    }
  
  if (mode & girl::toger::NEIGHBOUR_DIRECT)
    {
      //Warning: complexity
      const girl::dart canonicD = m_m.getLambda_1(r);
      girl::dart d = canonicD;
      do
	{
	  girl::face f = m_m.getLambda(m_m.alpha(d));

	  if (! m_m.faceIsInfinite(f) && ! m_m.faceIsMarked(f))
	    if (! isDirectCommonBoundaryMarked(d, f, mark))
	      {
		regs.push_back(f);
		m_m.faceMark(f);
	      }
	  d = m_m.getPhi(d);
	}
      while (d != canonicD);
    }

  if (mode & girl::toger::NEIGHBOUR_INNER)
    {
      //Warning: complexity 
      girl::face daughter = m_m.faceGetFirstDaughter(r);
      while (daughter)
	{
	  girlAssert1(m_m.faceIsInfinite(daughter));
	  const girl::dart canonicD = m_m.getLambda_1(daughter);
	  girl::dart d = canonicD;
	  do
	    {
	      girl::face f = m_m.getLambda(m_m.alpha(d));
	      girlAssert1(! m_m.faceIsInfinite(f));
	      if (! m_m.faceIsMarked(f))
		if (! isOuterCommonBoundaryMarked(m_m.alpha(d), r, mark))
		  {
		    regs.push_back(f);
		    m_m.faceMark(f);
		  }
	      d = m_m.getPhi(d);
	    }
	  while (d != canonicD);
	  daughter = m_m.faceGetNextSister(daughter);
	}
    }

  //we unmark all added regions/faces to @a regs.
  for (girl::regionDeque::const_iterator it = regs.begin();
       it != regs.end(); ++it) {

    const girl::face ftmp = *it;
    assert(m_m.checkFace(ftmp));
    m_m.faceUnmark(ftmp);
    assert(m_m.checkFace(ftmp));
  }    
   
  assert(P_checkRegionUniqueness(regs));
}

/**
 * Gives a stack @a swith darts as elements 
 * such that each dart d belongs to @a r1, and 
 * @a r1 and @a r2 are DIRECT, INNER or OUTER neighbours.
 *
 * Gives the neghbouring mode @a m found.
 * @return the number of common darts.
 *
 * @warning chooses preferably darts from @a r2.
 */
size_t
togerImpl::commonDarts(girl::region r1, girl::region r2, 
		       std::deque<girl::dart> &sd, 
		       girl::toger::neighbouringModes &nm) const
{

  girl::dart d, canonicD;

  //first, test DIRECT and OUTER neighbourhood.  
  //OPTIM: we stop at first dart where we know if we are DIRECT or OUTER
  //  neighbours, and then we test only one of the two conditions.
  //Faster than callng kindOfNeighbourhood() because we don't check mark
  // and we continue from dart where detection was made.
  canonicD = d = m_m.getLambda_1(r1);
  do
    {
      girlAssert1(m_m.getLambda(d) == r1);

      if (areDirectNeighbours(d, r2))
	{
	  sd.push_back(d);
	  nm = girl::toger::NEIGHBOUR_DIRECT;
	  break;
	}
      else if (areOuterNeighbours(d, r2))
	{
	  sd.push_back(d);
	  nm = girl::toger::NEIGHBOUR_OUTER;
	  break;
	}
      d = m_m.getPhi(d);
    }
  while (canonicD != d);
  if (! sd.empty()) {
    assert(sd.back() == d);
    d = m_m.getPhi(d);
    if (nm == girl::toger::NEIGHBOUR_DIRECT) {
      while (canonicD != d) {
	if (areDirectNeighbours(d, r2))
	  sd.push_back(d);
	d = m_m.getPhi(d);
      }
    }
    else {
      assert(nm == girl::toger::NEIGHBOUR_OUTER);
      while (canonicD != d) {
	if (areOuterNeighbours(d, r2))
	  sd.push_back(d);
	d = m_m.getPhi(d);
      }
    }
  }

  //then test INNER neigbourhood
  //r1 is inner neighbour of r2 only if r2 is outer neighbour of r1
  if (sd.empty()) {
    canonicD = d = m_m.getLambda_1(r2);
    do 
      {
	girlAssert1(m_m.getLambda(d) == r2);
	
	if (areOuterNeighbours(d, r1))
	  {
	    sd.push_back(d);
	    nm = girl::toger::NEIGHBOUR_INNER;
	  }
	d = m_m.getPhi(d);
      }
    while (canonicD != d);
    
  }

  return sd.size();

}


/**
 * @brief Tells if the contour of a face, starting from a dart @a d, is 
 * positively oriented.
 */
bool
togerImpl::isContourPositivelyOrientedFromDart(girl::dart d) const
{
  //OPTIM: Traverse bitmap, without building contour explicitly.
  //(some code duplication with regionOuterBoundary, segmentDirections, & girl::contour_isPositivelyOriented)

  int c = 0;

  const girl::dart d0 = d;
  
  girl::geoDart u0;
  m_m.getGe(d, u0);
  assert(u0.dir() != INVALID);

  girl::direction dir0, dirPrev;
  dir0 = dirPrev = u0.dir(); 
  //REM: the first iteration of the loop consider the angle between u.dir() & u.dir(),
  // and as they are the same it doesn't change contour orientation

  do
    {
      girl::geoDart u;
      m_m.getGe(d, u);
      assert(u.dir() != INVALID);
      bitmap::const_iterator it = m_b.getIterator(u.point());
      girl::direction dirN = u.dir();
      assert(dirN != INVALID);

      do
	{
	  if (dirN == dirPrev.next())
	    ++c;
	  else if (dirN == dirPrev.previous())
	    --c;
	  dirPrev = dirN; 

	  it.moveInDirection(dirN);
	  dirN = it.getNextDirection(dirN);
	}
      while (! it.isNode());

      d = m_m.getPhi(d);
    }
  while (d != d0);
  
  if (dir0 == dirPrev.next())
    ++c;
  else if (dir0 == dirPrev.previous())
    --c;
  
  return c > 0;
}


/**
 * @brief Extends edge removing its artificial nodes.
 *
 * See Luc Brun's PhD Thesis, 6.2.3, p131-132.
 */
void
togerImpl::extendEdge(girl::dart d)
{
  girlAssert1(m_m.getSigma(m_m.getSigma(d)) == d);//artifial node

  // b is the only dart of a loop : nothing to do
  girl::dart tmp = m_m.getPhi_1(d);
  if (tmp == d) 
    return;

  // delete all the artificial nodes (in bitmap and girlGe2to) before sigma(b)
  while (tmp != d)
    {
      girl::dart sd = m_m.getSigma(tmp);
      if (m_m.getSigma(sd) != tmp) 
	break;

      girl::geoDart u;
      m_m.getGe(tmp, u);
      m_b.unsetNode(u.point());
      m_g.deleteNode(u.point());

      tmp = m_m.alpha(sd);
    }

  if (tmp == d)
    {
      // There was a loop of artificial nodes
      girl::geoDart u;
      tmp = m_m.getSigma(d);
      m_m.getGe(tmp, u);
      m_g.setDart(u, m_m.alpha(d));
      m_m.simplifyLoop(d);
    }
  else
    {
      // sigma(sigma(tmp)) != tmp => not an artificial node anymore
      girl::geoDart u;
      m_m.getGe(d, u);
      m_b.unsetNode(u.point());
      m_g.deleteNode(u.point());

      m_m.getGe(tmp, u);
      m_g.setDart(u, d);
      m_m.extendEdge(d, tmp);
    }

  //NO girlAssert1(checkConsistency());
}


/**
 * @brief Suppresses artificial nodes, from the given dart (before and after).
 * 
 * Extend edges if |sigma*(d)| == 2 ou |sigma*(alpha(d))| == 2 , 
 * so may change sigma(d) or sigma(alpha(d)) !
 * Does nothing if 
 
 * It's a suppress_edge preliminary stage.
 * See Luc Brun's PhD Thesis, section 6.4.2, algorithm p141, first "if".
 */
void
togerImpl::suppressArtificialNodes(girl::dart d)
{
  // Is node associated to sigma*(d) artificial ? 
  //if (m_m.getSigma(m_m.getSigma(d)) == d)
  if (m_m.nodeIsArtificial(d))
    extendEdge(d);

  // Is node associated to sigma*(alpha(d)) artificial ?
  const girl::dart ad = m_m.alpha(d);
  //if (m_m.getSigma(m_m.getSigma(ad)) == ad)
  if (m_m.nodeIsArtificial(ad))
    extendEdge(ad);
}


/**
 * @brief Suppresses connection edges, from the given dart.
 * May call P_suppressEdge recursively (at most 1)
 *
 * It's a suppress_edge preliminary stage.
 * See Luc Brun's PhD Thesis, section 6.4.2, algorithm p141, second and 
 * third "if".
 */
void
togerImpl::suppressConnectionEdges(girl::dart d)
{
  const girl::dart sd = m_m.getSigma(d);
  const girl::dart sad = m_m.getPhi(d);
  const girl::dart ad = m_m.alpha(d);

  if (sad == d && m_m.getSigma(sd) == ad)
    suppressEdge(sd); //recursive
  else if (sd == ad && m_m.getSigma(sad) == d)
    suppressEdge(sad); //recursive
}

/**
 * @brief Suppresses edge actually : case where edge is a disconnected loop 
 * (sigma(d)=alpha(d) && sigma(alpha(d))=d) 
 *
 * @return the deleted region
 *
 * See Luc Brun's PhD Thesis, section 6.4.3.1.
 */
girl::region
togerImpl::suppressEdge_disconnectedLoop(girl::dart d)
{
  girlAssert1(m_m.getSigma(d)==m_m.alpha(d) 
	      && m_m.getSigma(m_m.alpha(d))==d);

  const girl::dart ad = m_m.alpha(d);

  girl::face fi = m_m.getLambda(d);
  girl::face ff;
  if (! m_m.faceIsInfinite(fi))
    {
      ff = fi;
      fi = m_m.getLambda(ad);
    }
  else
    ff = m_m.getLambda(ad);

  girl::geoDart u;
  
  m_m.getGe(d, u);
  m_b.clearSegment(u);
  m_b.unsetNode(u.point());
  m_g.deleteNode(u.point());
    
  m_m.faceMoveDaughters(ff, fi);// As fi is infinite, daughters will be moved to its mother.
  m_m.faceRemove(ff);
  m_m.faceRemove(fi);
  m_m.removeMap(d); 
  return ff; 
}


/**
 * @brief Suppresses edge actually : case where edge is a loop, tangent to 
 * a face, 
 * with d and alpha(d) consecutive in sigma*(d)
 * (i.e. sigma(d)=alpha(d) or sigma(alpha(d))=d).
 *
 * @return the deleted region.
 *
 * See Luc Brun's PhD Thesis, section 6.4.3.2, 1st case:
 *   cardinality is 4 and, d and alpha(d) are consecutive.
 */
girl::region
togerImpl::suppressEdge_connectedLoopConsecutiveDarts(girl::dart d, 
						      girl::dart sad)
{
  girl::dart ad = m_m.alpha(d);
  if (sad == d) // swap and go...
    {
      girl::dart tmp = d; 
      d = ad; 
      ad = tmp;
    }

  const girl::face fd = m_m.getLambda(d);
  const girl::face fad = m_m.getLambda(ad);
  girl::region del_r = 0;

  if (! m_m.faceIsInfinite(fad)) // cases a and b : fad finite
    {
      m_m.faceMoveDaughters(fad, fd);
      m_m.faceRemove(fad);
      del_r = fad;
    }
  else // fd finite, fad infinite
    {
      m_m.faceMoveDaughters(fd, fad);
      m_m.mergeLambda(fad, fd);
      m_m.faceRemove(fd);
      del_r = fd;
    }

  m_m.removeTangent(d);  

  assert(! m_m.checkFace(del_r));

  return del_r;
}


/**
 * @brief Suppresses edge actually : case where edge d is a loop, tangent to 
 * a face, 
 * with d and alpha(d) not consecutive in sigma*(d) (of cardinality 4)
 * (i.e. sigma(d)!=alpha(d) an sigma(alpha(d))!=d).
 * 
 * @return the deleted region.
 *
 * See Luc Brun's PhD Thesis, section 6.4.3.2, 2nd case:
 * cardinality is 4 and, d and alpha(d) are not consecutive.
 */
girl::region
togerImpl::suppressEdge_connectedLoopUnconsecutiveDarts(girl::dart d)
{
  girl::dart ad = m_m.alpha(d);

  girl::face fd = m_m.getLambda(d);
  girl::face fad;
  if (m_m.faceIsInfinite(fd)) // swap and go...
    {
      girl::dart tmp = d; 
      d = ad; 
      ad = tmp;

      fad = fd;
      fd = m_m.getLambda(ad);
    }
  else
    fad = m_m.getLambda(ad);

  m_m.faceMoveDaughters(fd, fad);
  m_m.faceRemove(fd);
  m_m.removeEdge(d);
  
  return fd;
}


/**
 * @brief Suppresses edge actually : case where edge is not a loop, but 
 * a connecting edge.
 *
 * @return the deleted region.
 *
 * See Luc Brun's PhD Thesis, section 6.4.3.3, 1st case.
 */
girl::region
togerImpl::suppressEdge_notLoopConnectingEdge(girl::dart d, 
					      girl::dart sd, 
					      girl::dart sad, 
					      girl::face fd, 
					      girl::face fad)
{ 
  if (! m_m.faceIsInfinite(fd)) //fd finite
    {
      /*B edge connects a mother to one of its daughter */

      m_m.removeEdge(d);

      const bool contourPositivelyOriented = isContourPositivelyOrientedFromDart(sd);
      if (contourPositivelyOriented)
	{
	  m_m.setLambda_1(fd, sd);
	  girl::face nf = m_m.changeLambdaCycle(sad);
	  m_m.faceAddDaughter(fd, nf);

	  girlAssert1(! checkFacePositiveOrientation(nf));

	  m_m.setLambda_1(nf, sad);
	}
      else
	{
	  m_m.setLambda_1(fad, sad);
	  girl::face nf = m_m.changeLambdaCycle(sd);
	  m_m.faceAddDaughter(fd, nf);

	  girlAssert1(! checkFacePositiveOrientation(nf));

	  m_m.setLambda_1(nf, sd);
	}
    }
  else // fb infinite
    {
      /*B edge connects 2 faces with same mother  */

      m_m.removeEdge(d);

      m_m.setLambda_1(fd, sd);
      girl::face nf = m_m.changeLambdaCycle(sad);
      m_m.faceAddDaughter(m_m.faceGetMother(fd), nf);

      girlAssert1(! checkFacePositiveOrientation(nf));

      m_m.setLambda_1(nf, sad);
    }

  return 0;
}

/**
 * @brief Suppresses edge actually : case where edge is not a loop and 
 * not a connecting edge
 *
 * @return the deleted region.
 *
 * See Luc Brun's PhD Thesis, section 6.4.3.3, 2nd case.
 */
girl::region
togerImpl::suppressEdge_notLoopNotConnectingEdge(girl::dart d)
{
  girl::region del_r = 0;

  const girl::dart ad = m_m.alpha(d);
  const girl::face fd = m_m.getLambda(d);
  const girl::face fad = m_m.getLambda(ad);
  if (! m_m.faceIsInfinite(fd) && ! m_m.faceIsInfinite(fad)) //fd and fad finites
    {
      m_m.faceMoveDaughters(fad, fd);
      m_m.mergeLambda(fd, fad);
      m_m.faceRemove(fad);
      del_r = fad;
    }
  else if (m_m.faceIsInfinite(fd) && ! m_m.faceIsInfinite(fad)) //fd infinite, fad finite
    {
      m_m.faceMoveDaughters(fad, fd);
      m_m.mergeLambda(fd, fad);
      m_m.faceRemove(fad);
      del_r = fad;
    }
  else // fd finite, fad infinite
    {
      m_m.faceMoveDaughters(fd, fad);
      m_m.mergeLambda(fad, fd);
      m_m.faceRemove(fd);
      del_r = fd;
    }
  m_m.removeEdge(d);

  assert(! m_m.checkFace(del_r));

  return del_r;
}


/**
 * @brief clear useless nodes both in map and bitmap 
 * starting from given girl::point @a pt.
 *
 * Used Used after edge deletion, to ensure face minimality
 */
void
togerImpl::clearUselessNodes(const girl::point &pt)
{
  bitmap::const_iterator it = const_cast<const girl::bitmap&>(m_b).getIterator(pt); 
  if (it.isNode())
    {
      girl::geoDart u;
      u.setPoint(pt);

      girl::direction dir = INVALID;
      if (it.hasNeighbourInDirection(UP))
	dir = UP;
      else if (it.hasNeighbourInDirection(RIGHT))
	dir = RIGHT;
      else if (it.hasNeighbourInDirection(DOWN))
	dir = DOWN;
      else if (it.hasNeighbourInDirection(LEFT))
	dir = LEFT;
      assert(dir != INVALID);
      u.setDir(dir);

      const girl::dart d = m_g.getDart(u);
      assert(d!=0);

      girlAssert1(m_b.getIterator(u.point()).getPointCardinality() 
		  == m_m.getSigmaCycleCardinality(d));

      if (m_m.getSigma(m_m.getSigma(d)) == d)
	extendEdge(d);
    } 
}

/**
 * @brief Suppress an edge, from the given dart.
 *
 * @return the deleted region.
 *
 * See Luc Brun's PhD Thesis, section 6.4.2, algorithm p141.
 */
girl::region
togerImpl::suppressEdge(girl::dart d)
{
  //NO girlAssert1(checkConsistency());

  // Preliminary stage

  suppressArtificialNodes(d);

  suppressConnectionEdges(d);

  // Effective suppression

  const girl::dart ad = m_m.alpha(d);
  
  const girl::dart sad = m_m.getPhi(d);
  girl::dart sd_c[4];
  const int sd_l = m_m.getSigmaCycle(d, sd_c);
  assert(sd_l>1);
  const girl::dart sd = sd_c[1];


  girl::region del_r = 0;

  if (sd_l == 2 && sd == ad)
    {
      /*B  6.4.3.1 : n=n' et |sigma*(d)| = 2  */
      return suppressEdge_disconnectedLoop(d);
    }
  else if (sd_l == 4 && (sd == ad || sad == d))
    {
      /*B 6.4.3.2, 1st case : cardinality is 4 and, 
	d and sigma(d) are consecutive */
      del_r = suppressEdge_connectedLoopConsecutiveDarts(d, sad);
    }
  else if (sd_l == 4 && sd_c[2] == ad)
    {
      /*B 6.4.3.2, 2nd case : cardinality is 4 and, 
	d and alpha(d) are not consecutive */
      del_r = suppressEdge_connectedLoopUnconsecutiveDarts(d);
    }
  else
    {
      /*B 6.4.3.3 : not a loop */
      girl::face fd, fad;
      if ((fd = m_m.getLambda(d)) ==
	  (fad = m_m.getLambda(ad))) // connecting edge
	{
	  del_r = suppressEdge_notLoopConnectingEdge(d, sd, sad, fd, fad);
	}
      else // not a connecting edge
	{
	  del_r = suppressEdge_notLoopNotConnectingEdge(d);

	}
    }
  
  girl::geoDart ud, uad;
  m_m.getGe(d, ud);
  m_g.clearDart(ud);
  m_m.getGe(ad, uad);
  m_g.clearDart(uad);

  m_b.clearSegment(uad);//could be ud

  //ensure map minimality after actual edge suppression
  clearUselessNodes(ud.point());
  clearUselessNodes(uad.point());

  assert(! m_m.checkFace(del_r));

  return del_r;
}



/**
 * @brief Deletes edges from common darts of regions @a r1 and @ r2, 
 * stacked in @a s
 * and returns kepts region, i.e., r1 or r2.
 */
girl::region
togerImpl::deleteEdges(girl::region r1, girl::region r2, 
		       std::deque<girl::dart> &sd)
{

  assert(m_m.checkFace(r1));
  assert(m_m.checkFace(r2));

  girl::region del_r = 0;
  while (! sd.empty()) {
    const girl::dart d = sd.back();

    if (m_m.dartIsDeleted(d)) {
      sd.pop_back();
      continue;
    }
    const girl::region r = suppressEdge(d);


    if (r) { 
      assert(del_r == 0);
      del_r = r;
      assert(! m_m.checkFace(del_r));
    }
    //assert(del_r == 0 || ! m_m.checkFace(del_r)); //Not true, face may have been reused
  }

  girl::region keptRegion = r1;
  if (del_r == r1)
    keptRegion = r2;

  assert(m_m.checkFace(keptRegion));
  //assert(! m_m.checkFace(del_r));//Not true, face may have been reused
  girlAssert2(checkConsistency());

  assert(m_m.checkFace(keptRegion));
  girlAssert1(m_m.checkDartPhiCycleLambda(m_m.getLambda_1(keptRegion)));
  girlAssert1(checkFaceMinimality(keptRegion));

  return keptRegion;
}

/**
 * @brief apply merge then del hooks if defined.
 */ 
void
togerImpl::applyMergeDelHooks(girl::region keptRegion, girl::region delRegion)
{
  ObserverCollection::iterator itO = m_observers.begin();
  const ObserverCollection::iterator itOEnd = m_observers.end();
  for ( ; itO != itOEnd; ++itO) {
    girl::toger::observer *o = *itO;
    assert(o);
    if (o->doMergeRegions())
      o->mergeRegions(keptRegion, delRegion);
    if (o->doDeleteRegion())
      o->deleteRegion(delRegion);
  }
}

/**
 * @brief Spreads mark of dart @a commonDart, belonging 
 * to both regions r1 and @a r2, to r2, according to neighbouring mode @a m_m.  
 *
 * If commonDart is not marked, mark of r2 stays unchanged.
 */
void
togerImpl::spreadMark(girl::dart commonDart, girl::region r2, 
		      girl::toger::neighbouringModes nm)
{
  const girl::toger::markFlags fl = m_m.getDartMark(commonDart);
  
  if (fl != girl::toger::MARK_NONE) {
    if (nm == girl::toger::NEIGHBOUR_OUTER 
	|| nm == girl::toger::NEIGHBOUR_DIRECT) {
      xMarkRegionBoundary(r2, fl);
    }
  }
}


girl::region
togerImpl::mergeRegions(girl::region r1, girl::region r2)
{
  assert(r1 != r2);
  assert(m_m.checkRegion(r1));
  assert(m_m.checkRegion(r2));
  assert(! m_m.faceIsInfinite(r1));
  assert(! m_m.faceIsInfinite(r2));
  girlAssert1(m_m.checkDartPhiCycleLambda(m_m.getLambda_1(r1)));
  girlAssert1(m_m.checkDartPhiCycleLambda(m_m.getLambda_1(r1)));
  girlAssert1(checkFacePositiveOrientation(r1));
  girlAssert1(checkFacePositiveOrientation(r2));
  girlAssert1(checkConsistency());
  
  std::deque<girl::dart> sd;
  
  girl::toger::neighbouringModes nm = girl::toger::NEIGHBOUR_NONE;
  
  commonDarts(r1, r2, sd, nm);
  
  girlAssert2((sd.empty() && nm==0 && ! areNeighbours(r1, r2, nm, 
						      girl::toger::MARK_NONE))
	      || (! sd.empty() && areNeighbours(r1, r2, nm, 
						girl::toger::MARK_NONE)));
  
  girl::region keptRegion = 0;
  
  if (! sd.empty()) {
    girl::dart commonDart = sd.back();
    spreadMark(commonDart, r2, nm); 
    
    keptRegion = deleteEdges(r1, r2, sd);
    
    if (keptRegion == r1) {
      girlAssert1(m_m.checkFacePhiCycle(r1));
      girlAssert1(checkFacePositiveOrientation(r1));
      applyMergeDelHooks(r1, r2);
    }
    else {
      assert(keptRegion == r2);
      girlAssert1(m_m.checkFacePhiCycle(r2));
      girlAssert1(checkFacePositiveOrientation(r2));
      applyMergeDelHooks(r2, r1);
    }

    assert(m_m.checkFace(keptRegion));
  }
  
  girlAssert2(checkConsistency());
  
  return keptRegion;
}


/**
 * @brief Seeks outer path at @a curPos.
 *
 * Sets *(@a newNode) to true if a new node is found.
 * Stacks found heads (at most 3).
 * @return out direction (to follow outer region contour).
 * @warning Raises an exception if no outer path is found.
 */ 
static 
girl::direction
P_findOuterPath(const girl::geoDart &curPos,
		girl::toger::sameRegion &same_r, 
		bool &newNode,
		std::deque<girl::geoDart> &head_s)
{
  girl::point p = curPos.point();

  P_img_LL_pixel(p, curPos.dir());
  girl::direction dir = curPos.dir();
  assert(dir != INVALID);

  girl::direction out_dir = INVALID;
  bool pathFound = false; 
  newNode = false;

  for (int i = 3; i; --i) // do it 3 times
    {
      assert(dir != INVALID);

      if (! same_r.same(p, dir))
	{
	  if (! pathFound)
	    {
	      pathFound = true;
	      out_dir = dir.previous();
	    }
	  else
	    {
	      if (! newNode)
		{
		  newNode = true;
		  head_s.push_back( girl::geoDart(curPos.x(), curPos.y(), 
						  out_dir) );
		}
	      head_s.push_back( girl::geoDart(curPos.x(), curPos.y(), 
					      dir.previous()) );
	    }
	}
      p.move(dir);
      dir = dir.next();
    }

  if (GIRL_unlikely(! pathFound)) {
    std::stringstream ss;
    ss<<"girl::toger::splitRegion: dangling edge around ("
      <<p.x()<<";"<<p.y()<<").";
    throw std::runtime_error(ss.str());
  }

  return out_dir;
} 

/**
 * @brief Seeks external region contour from the given head @a u,
 * and stacks found heads if any.
 *
 * Stops on the first node found. 
 * This node can be an existing node or a new found node.
 * @return true if a new node has been created, or false if node on which 
 * seek stopped is an existing node.
 * On exit, @a u contains node position, and direction taken to get to 
 * this position.
 * 
 * @warning If a new node is found, it is not marked at such in the bitmap.
 */
bool
togerImpl::seekRegionContour(girl::geoDart &u,
			     girl::toger::sameRegion &same_r, 
			     std::deque<girl::geoDart> &head_s)
{
  bitmap::iterator it = m_b.getIterator(u.point());
  assert(it.isNode());
  it.writeNeighbourInDirection(u.dir(), 1);
  it.moveInDirection(u.dir());
  
  while (! it.isNode())
    {
      u.setPoint(it.getPosition());

      bool newNode;
      const girl::direction out_dir = P_findOuterPath(u,
						      same_r, 
						      newNode,
						      head_s);

      if (newNode)
	return true;
      
      u.setDir(out_dir);
      it.writeNeighbourInDirection(u.dir(), 1);

#ifndef NDEBUG1
      if (it.getPointCardinality() > 2 
	  && ! it.isNode()) {
	std::cerr<<"introduced bitmap inconsistency at ("
		 <<u.x()<<"; "<<u.y()<<"), last dir="<<u.dir()<<std::endl;
	std::cerr<<"   cardinality="<<it.getPointCardinality()
		 <<"  is_node="<<it.isNode()<<std::endl;
      }
#endif

      it.moveInDirection(u.dir());
    }

  u.setPoint(it.getPosition());
  
  return false;
}


/**
 * @brief Insures there is a node at @a it position.
 * A node will be inserted, if not.
 *
 * @param it   bitmap iterator at current position
 * @param isFirstNodeArtificial  tells if first node is artificial.
 * @param startHead  geoDart on contour
 * Useful only if *(@a isFirstNodeArtificial) is true, then the node 
 * could be moved.
 * @param currentDart  previous dart encountered on contour.
 * @param previousDir  previous direction to get to current point on contour.
 * @param nextDir      next direction from current point on contour.
 */
void
togerImpl::insureNode(const girl::bitmap::iterator &it,
		      bool &isFirstNodeArtificial,
		      const girl::geoDart &startHead,
		      girl::dart &currentDart, 
		      girl::direction previousDir,
		      girl::direction nextDir)
{
  if (isFirstNodeArtificial) {
    //will only move artificial node  
    
    isFirstNodeArtificial = false;
    girlAssert1(checkPtNodeConsistency(startHead.point()));
    m_g.deleteNode(startHead.point());
    m_b.unsetNode(startHead.point());
    //OPTIM: reduce the number of calls to ge2to
    /*
      setGeToMap(girl::geoDart(it.getPosition(), nextDir), currentDart);
      setGeToMap(girl::geoDart(it.getPosition(), previousDir.opposite()), m_m.alpha(currentDart));
    */
    girl::dart darts[4] = {0, 0, 0, 0};
    darts[nextDir] = currentDart;
    darts[previousDir.opposite()] = m_m.alpha(currentDart);
    m_g.setDarts(it.getPosition(), darts);
    m_m.setGe(currentDart, girl::geoDart(it.getPosition(), nextDir));
    m_m.setGe(m_m.alpha(currentDart), girl::geoDart(it.getPosition(), previousDir.opposite()));
  }
  else if (! it.isNode()) {
    //will split an edge (keeping topological marking)
    
    girl::geoDart old_ad_u;
    const girl::dart alpha_d = m_m.alpha(currentDart);
    m_m.getGe(alpha_d, old_ad_u);
    //OPTIM:  reduce the number of calls to ge2to
    /*
      setGeToMap(girl::geoDart(it.getPosition(), previousDir.opposite()), alpha_d);
      currentDart = m_m.splitEdge(currentDart);
      setGeToMap(girl::geoDart(it.getPosition(), nextDir), currentDart);
    */
    currentDart = m_m.splitEdge(currentDart);

    girl::dart darts[4] = {0, 0, 0, 0};
    darts[previousDir.opposite()] = alpha_d;
    darts[nextDir] = currentDart;
    m_g.setDarts(it.getPosition(), darts);
    m_m.setGe(alpha_d, girl::geoDart(it.getPosition(), previousDir.opposite()));
    m_m.setGe(currentDart, girl::geoDart(it.getPosition(), nextDir));

    setGeToMap(old_ad_u, m_m.alpha(currentDart));
  }
  it.setNode();  
}

/**
 * @brief Seeks inner head from position @a currentPos, and stack heads.
 *
 * Insures there is a node at head position.
 *
 * @param startHead first geoDart of face.
 *         Useful only if *(@a isFirstNodeArtificial) is true, then the node
 *         could be moved.
 * @param currentDart previous dart encountered on contour.
 * @param previousDir  previous direction to get to current point on contour.
 * @param it   iterator on bitmap at current contour point.
 * @param nextDir  next direction after current point on contour.
 * @param isFirstNodeArtificial  tells if first node is artificial.
 * @param same_r
 * @param head_s   deque filled with found heads.
 */
void
togerImpl::seekInnerHead(const girl::geoDart &startHead,
			 girl::dart &currentDart, 
			 girl::direction previousDir, 
			 const girl::bitmap::iterator &it, 
			 girl::direction nextDir, 
			 bool &isFirstNodeArtificial,
			 girl::toger::sameRegion &same_r,
			 std::deque<girl::geoDart> &head_s)
{
  girl::direction dir = previousDir;
  girl::point p = it.getPosition();
  assert(p.x() < m_width && p.x() >= -1);
  assert(p.y() < m_height && p.y() >= -1);
  assert(dir != INVALID);
  P_img_LR_pixel(p, dir);
	
  for (int i = (nextDir == previousDir ? 1 : 2); i; --i)
    {
      assert(dir != INVALID);
      assert(p.x() < m_width && p.x() >= -1);
      assert(p.y() < m_height && p.y() >= -1);

      if (! same_r.same(p, dir))
	{
	  insureNode(it, isFirstNodeArtificial, startHead, currentDart, previousDir, nextDir);

#if 0

	  if (isFirstNodeArtificial)
	    {
	      //will only move artificial node  
	      isFirstNodeArtificial = false;
	      girlAssert1(checkPtNodeConsistency(startHead.point()));
	      m_g.deleteNode(startHead.point());
	      m_b.unsetNode(startHead.point());
	      //OPTIM: reduce the number of calls to ge2to
	      /*
		setGeToMap(currentPos, currentDart);
		setGeToMap(girl::geoDart(currentPos.point(), 
		previousDir.opposite()), m_m.alpha(currentDart));
	      */
	      girl::dart darts[4] = {0, 0, 0, 0};
	      darts[currentPos.dir()] = currentDart;
	      darts[previousDir.opposite()] = m_m.alpha(currentDart);
	      m_g.setDarts(currentPos.point(), darts);
	      m_m.setGe(currentDart, currentPos);
	      m_m.setGe(m_m.alpha(currentDart), 
			girl::geoDart(currentPos.point(), 
				      previousDir.opposite()));
	    }
	  else if (! it.isNode())
	    {
	      //will split an edge (keeping topological marking)
	      girl::geoDart old_ad_u;
	      const girl::dart alpha_d = m_m.alpha(currentDart);
	      m_m.getGe(alpha_d, old_ad_u);
	      //OPTIM:  reduce the number of calls to ge2to
	      /*
		setGeToMap(girl::geoDart(currentPos.point(), 
		previousDir.opposite()), alpha_d);
		currentDart = m_m.splitEdge(currentDart);
		setGeToMap(currentPos, currentDart);
	      */
	      currentDart = m_m.splitEdge(currentDart);
	      girl::dart darts[4] = {0, 0, 0, 0};
	      darts[previousDir.opposite()] = alpha_d;
	      darts[currentPos.dir()] = currentDart;
	      m_g.setDarts(currentPos.point(), darts);
	      m_m.setGe(alpha_d, girl::geoDart(currentPos.point(), 
					       previousDir.opposite()));
	      m_m.setGe(currentDart, currentPos);

	      setGeToMap(old_ad_u, m_m.alpha(currentDart));
	    }

	  it.setNode();
#endif

	  girlAssert1(it.checkNodeConsistency());
	  head_s.push_back( girl::geoDart(it.getPosition(), dir.next()) );

	}
      p.move(dir);
      dir = dir.previous();
    }
}


/**
 * @brief Stacks up new heads found on face @a f boundary in stack @a head_s
 *
 * Insures there is a node at head position.
 *
 * Updates bitmap and girlGe2to but do not create darts.
 * (Can switch-on new (artificial) nodes in bitmap or already existing nodes.
 *  Updates girlGe2to only if an artificial node is moved or an existing edge 
 * is splitted,
 * but no dart is created in head's direction. )
 * Heads point inwards the face (which can be finite or infinite)   
 *
 * @return size of the contour of face @a f.
 */
size_t
togerImpl::pushNewHeads(girl::face f,
			girl::toger::sameRegion &same_r,
			std::deque<girl::geoDart> &head_s)
{
  girlAssert1(m_m.checkFacePhiCycle(f));

  const girl::dart canonicD = m_m.getLambda_1(f);
  assert(canonicD!=0);
  girlAssert2(checkContourContinuity(canonicD));
  girl::dart d = canonicD;

  girl::geoDart startHead;
  m_m.getGe(canonicD, startHead);

  //move to next point and store previous direction
  bitmap::iterator itB = m_b.getIterator(startHead.point());
  itB.moveInDirection(startHead.dir());
  girl::direction prevDir = startHead.dir();

  girl::direction prevDir0 = prevDir;
  girl::direction nextDir = girl::INVALID;

  const girl::point p0 = itB.getPosition();
  
  bool isFirstNodeArtificial = m_m.nodeIsArtificial(d);

  size_t contourSize = 0;

  do
    {
      //get next direction
      if (itB.isNode()) {
	d = m_m.getPhi(d);
	girl::geoDart u;
	m_m.getGe(d, u);
	nextDir = u.dir();
      }
      else {
	nextDir = itB.getNextDirection(prevDir);
      }

      const bool couldHaveANewHead = (nextDir != prevDir.next());
      if (couldHaveANewHead)
	{
	  seekInnerHead(startHead,
			d, prevDir, itB, nextDir,
			isFirstNodeArtificial,
			same_r, 
			head_s);
	}
      itB.moveInDirection(nextDir);
      prevDir = nextDir;

      ++contourSize;
    }
  while (itB.getPosition() != p0 || prevDir != prevDir0);
  //stops when we come back were we started, with the same path 
  //(direction must be tested)!

  girlAssert2(m_b.checkConsistency());

  return contourSize;
}


/**
 * @brief Gives next dart in sigma starting from girl::geoDart @a u.
 */
static
girl::dart
P_nextDartInSigma(const girl::ge2to &g, const girl::geoDart &u)
{
  girl::dart darts[4];
  g.getDarts(u.point(), darts);

  girl::dart d = 0;
  girl::direction dir = u.dir();
  for (int i=0; i<4; ++i) {
    dir = dir.next();
    d = darts[dir];
    if (d != 0)
      break;
  }
  //girlAssert1(dir != u.dir());
  return d;

  /*
    girl::dart d = 0;
    girl::geoDart tmp = u;
    do
    {
    tmp.setDir(tmp.dir().next());
    }
    while ((d = m_g.getDart(tmp)) == 0);
    girlAssert1(tmp.dir() != u.dir());
    return d;
  */
}

/**
 * @brief Builds a new segment between @a u0 and @a u.
 *
 * @warning u.dir() points towards u.point().
 * @warning switch-on node in bitmap at @a u.
 * @return newly created dart.
 */
girl::dart
togerImpl::buildNewSegment(const girl::geoDart &u0, const girl::geoDart &u)
{
#ifdef NO_INSERT
  girlAssert1(! m_b.isNode(u.point()));//not yet switched on, except for insertion
#endif
  m_b.setNode(u.point());
  const girl::dart d = P_nextDartInSigma(m_g, u0);
  assert(d);
  return m_m.insertDart(d);
}

/**
 * @brief Builds a tangent face (loop).
 *
 * @warning u.dir() points towards u.point().
 * @warning can modify @a u0 and @a u.
 * @return newly created dart.
 */
girl::dart
togerImpl::buildTangentFace(girl::geoDart &u0, girl::geoDart &u)
{
  girlAssert1(m_b.isNode(u0.point()));
  girlAssert1(u0.dir() != u.dir().opposite());
  
  if (u.dir() == u0.dir().next())
    {
      //change to turn positively

      const girl::direction dir = u.dir();
      u.setDir( u0.dir().opposite() );
      u0.setDir(dir.opposite());
    }
  const girl::dart d1 = m_g.getDart(girl::geoDart(u0.point(), 
						  u0.dir().opposite()));
  girlAssert1(d1 != 0);
  return m_m.insertTangent(d1);
}

/**
 * @brief Builds a new segment between @a u0 and @a u,
 * each being already a node.
 *
 * @warning u.dir() points towards u.point().
 * @return newly created dart.
 */
girl::dart
togerImpl::buildSegmentBetweenExistingNodes(const girl::geoDart &u0, 
					    const girl::geoDart &u)
{
  girlAssert1(m_b.isNode(u.point()));

  //TODO:OPTIM???: get rid of temporary ?

  const girl::dart pos = P_nextDartInSigma(m_g, 
					   girl::geoDart(u.point(), 
							 u.dir().opposite()));
  assert(pos!=0);
  const girl::dart neg = P_nextDartInSigma(m_g, u0);
  assert(neg!=0);

  return m_m.insertSegment(neg, pos);
}


/**
 * @brief Creates internal nodes and thus internal regions, 
 * starting from previously found heads.
 *
 * (May create new inner nodes, and so will stack them up and unstack them in 
 * @a head_s).
 * At exit, stack @a head_s is empty.
 * Will create darts and queue them up (will queue both d and alpha(d)). 
 * @warning created darts are not (yet) labelled by lambda !
 */
void
togerImpl::buildBoundaries(girl::toger::sameRegion &same_r,
			   std::deque<girl::geoDart> &head_s, 
			   std::deque<girl::dart> &dart_q)
{
  while (! head_s.empty())  {//warning: new geoDarts can be pushed to head_s during loop
    
    girl::geoDart u0 = head_s.back(); //not const, may be modified later by buildTangentFace()
    head_s.pop_back();

    bitmap::const_iterator it = const_cast<const girl::bitmap&>(m_b).getIterator(u0.point());
    girlAssert1(it.isNode());

    if (it.hasNeighbourInDirection(u0.dir()) == true) 
      continue;

    girl::dart d;
    girl::geoDart u = u0;

    const bool created_a_node = seekRegionContour(u, same_r, head_s); //modify head_s
    //warning u.dir() points towards u.point().
    if (created_a_node) {
      d = buildNewSegment(u0, u);
    }
    else if (u0.point() == u.point()) {
      d = buildTangentFace(u0, u);
    }
    else {
      d = buildSegmentBetweenExistingNodes(u0, u);
    }
  
    m_m.clearEdgeFaces(d);
    setGeToMap(u0, d);
    setGeToMap(girl::geoDart(u.point(), u.dir().opposite()), m_m.alpha(d));

    girlAssert1(checkPtNodeConsistency(u0.point()));
    girlAssert1(checkPtNodeConsistency(u.point()));
    dart_q.push_back(d);
#ifndef GIRL_ONLYD
    //std::cerr<<"stack alpha d"<<std::endl;
    dart_q.push_back(m_m.alpha(d));
#endif

  }
  
  girlAssert2(m_b.checkConsistency());
  
}


void
togerImpl::labellingRegion(girl::dart d,
			   girl::region firstRegion, 
			   bool &mustStillUseFirstRegion, 
			   girl::regionDeque &foundRegions)
{
  if (m_m.getLambda(d)) {
    girlAssert1(m_m.checkFacePhiCycle(m_m.getLambda(d)));
    return;
  }

  girl::region r;
  if (mustStillUseFirstRegion == true) {
    mustStillUseFirstRegion = false;
    m_m.setLambdaCycle(d, firstRegion);
    r = firstRegion;
  }
  else {
    r = m_m.changeLambdaCycle(d);
  }

  girlAssert1(m_m.checkFacePhiCycle(r));
  girlAssert1(checkContourContinuity(m_m.getLambda_1(r)));
  girlAssert1(checkFacePositiveOrientation(r));

  foundRegions.push_back(r);  
}


/**
 * @brief Labels darts from queue @a dart_q to define regions, 
 * and returns regions in @a found_regions.
 *
 * Will reuse the first region label @a firstRegion on the first found region.
 *
 * @warning given darts must have a null label, and only define regions.  
 * @warning will empty given queue of darts
 *
 */
void
togerImpl::labellingRegions(std::deque<girl::dart> &dart_q, 
			    girl::region firstRegion, 
			    girl::regionDeque &foundRegions)
{
  bool mustStillUseFirstRegion = true;

  while (! dart_q.empty())
    {
      const girl::dart d = dart_q.front();
      dart_q.pop_front();

      labellingRegion(d, firstRegion, mustStillUseFirstRegion, foundRegions);

#ifdef GIRL_ONLYD
      const girl::dart ad = m_m.alpha(d);
      labellingRegion(ad, firstRegion, mustStillUseFirstRegion, foundRegions);
#endif
      
    }
}
/**
 *
 * @param[in, out] mustStillUseFirstRegion tells if @a firstRegion 
 * must be used to label region and is updated if the label was 
 * effectively used.
 */
void
togerImpl::labellingFace(girl::dart d,
			 girl::region firstRegion, 
			 bool &mustStillUseFirstRegion, 
			 girl::regionDeque &foundRegions)
{

  if (m_m.getLambda(d)) {
    girlAssert1(m_m.checkFacePhiCycle(m_m.getLambda(d)));
    return;
  }

  const bool isRegion = isContourPositivelyOrientedFromDart(d); //checkFacePositiveOrientationFromDart(d);
  if (isRegion) {
    //knowing whether it is a region serves two purposes : 
    //checking if we can reuse label & keeping region to later find 
    //its possible daughters

#ifndef NDEBUG  
    //std::cerr<<"* labellingFace  -> region"<<std::endl;
#endif

    girl::region r;
    if (mustStillUseFirstRegion == true)
      {
	mustStillUseFirstRegion = false;
	m_m.setLambdaCycle(d, firstRegion);
	r = firstRegion;
      }
    else
      r = m_m.changeLambdaCycle(d);

    girlAssert1(m_m.checkFacePhiCycle(r));
    girlAssert1(checkContourContinuity(m_m.getLambda_1(r)));
    girlAssert1(checkFacePositiveOrientation(r));

    foundRegions.push_back(r);
  }
  else {

#ifndef NDEBUG
    //std::cerr<<"* labellingFace  -> infinite face"<<std::endl;
#endif

    m_m.changeLambdaCycle(d);

    girlAssert1(m_m.checkFacePhiCycle(m_m.getLambda(d)));
    girlAssert1(checkContourContinuity(d));
    girlAssert1(! checkFacePositiveOrientationFromDart(d));
  }

}

/**
 * @brief Labels darts from @a dart_q to define faces, 
 * and returns found regions (and only regions) in @a found_regions.
 *
 * Will reuse the first region label @a firstRegion on the first 
 * found finite face if @a mustUseFirstRegion is true.
 *
 * @warning given darts must have a null label, and 
 * can define finite or infinite face. 
 * @warning will empty given queue of darts
 */
void
togerImpl::labellingFaces(std::deque<girl::dart> &dart_q, 
			  girl::region firstRegion, 
			  bool mustUseFirstRegion, 
			  girl::regionDeque &foundRegions)
{
  bool mustStillUseFirstRegion = mustUseFirstRegion;

  while (! dart_q.empty()) {
    const girl::dart d = dart_q.front();
    dart_q.pop_front();
    
    assert(d !=0);
    girlAssert1(m_m.checkDart(d));
    
    labellingFace(d, firstRegion, mustStillUseFirstRegion, foundRegions);

#ifdef GIRL_ONLYD
    const girl::dart ad = m_m.alpha(d);
    labellingFace(ad, firstRegion, mustStillUseFirstRegion, foundRegions);
#endif
  }
  
  assert(dart_q.empty());
}

/**
 * Considers that first region of @a out_q is splitted region
 * and other regions are new regions.
 * Will call new and update hooks accordingly.
 */
void
togerImpl::applyNewRegionHook(const girl::regionDeque &out_q)
{
  assert(! out_q.empty()); //at least splitted region

  girl::regionDeque &regions_q = const_cast<girl::regionDeque &>(out_q); //to be able to do a temporary pop...

  const girl::region splittedRegion = regions_q.front();
  regions_q.pop_front();
  girl::regionDeque updatedRegions;
  updatedRegions.push_back(splittedRegion);
  applyNewRegionHook(regions_q, updatedRegions);
  regions_q.push_front(splittedRegion);
}

void
togerImpl::applyNewRegionHook(const girl::regionDeque &newRegions, const girl::regionDeque &updatedRegions)
{
  ObserverCollection::iterator itO = m_observers.begin();
  const ObserverCollection::iterator itOEnd = m_observers.end();
  for ( ; itO != itOEnd; ++itO) {
    girl::toger::observer *o = *itO;
    assert(o);
    if (o->doUpdateRegion())
      o->updateRegion(updatedRegions);
    if (o->doNewRegion())
      o->newRegion(newRegions);
  }
}


/**
 * @brief Adds daughters of region @a r to queue @a region_q and hash_table @a h.
 *
 * @warning add infinite faces to @a region_q !
 */
void
togerImpl::getDaugthers(girl::region r, 
			girl::regionDeque &region_q, 
			FaceUnorderedSet &h) const
{
  girl::face a_child = m_m.faceGetFirstDaughter(r);
  while (a_child)
    {
      girlAssert1(m_m.checkInfiniteFaceConsistency(a_child));

      region_q.push_back(a_child);

      assert(h.find(a_child) == h.end());
      h.insert(a_child);

      a_child = m_m.faceGetNextSister(a_child);
    }
}

/**
 * @brief Invalidates relation between region @a r and all its daughters.
 */
void
togerImpl::invalidateMotherDaughterRelation(girl::region r)
{
  assert(r != 0);
  girlAssert1(checkFacePositiveOrientation(r));

  m_m.faceRemoveAllDaughters(r);
}

/**
 * @brief Puts directions describing geometry of 
 * the segment associated with dart @a d, in @a dq.
 */
void
togerImpl::segmentDirections(girl::dart d, 
			     std::deque<girl::direction> &dq) const
{
  girl::geoDart u;
  m_m.getGe(d, u);
  assert(u.dir() != INVALID);
  bitmap::const_iterator it = m_b.getIterator(u.point());
  girl::direction daux = u.dir();
  assert(daux != INVALID);

  do
    {
      assert(daux != INVALID);
      dq.push_back(daux);
      
      it.moveInDirection(daux);
      daux = it.getNextDirection(daux);
    }
  while (! it.isNode());
}

#ifndef NDEBUG
static 
bool
P_checkContourIsClosed(const girl::point &startPt, 
		       const std::deque<girl::direction> &dq)
{
  girl::point lStartPt = startPt;
  girl::point pt = lStartPt;

  for (std::deque<girl::direction>::const_iterator it = dq.begin();
       it != dq.end(); ++it) {
    pt.move(*it);
  }
  
  return pt == lStartPt;
}

#if 0
static 
void
P_printContourB(const girl::point &startPt, 
		const std::deque<girl::direction> &dq)
{
  girl::point lStartPt = startPt;
  girl::point pt = lStartPt;

  for (std::deque<girl::direction>::const_iterator it = dq.begin();
       it != dq.end(); ++it) {

    std::cerr<<"("<<pt.x()<<"; "<<pt.y()<<") ";
    pt.move(*it);
  }
  std::cerr<<std::endl;
}
#endif

#endif


/**
 * @brief gives face outer boundary, 
 * in the form of a starting pt and a queue of directions. 
 *
 * Unlike girl::togerImpl::regionOuterBoundary or faceOuterBoundary, a contour is not built explicitly.
 *
 * Do not consider eventual daugthers of @a r.
 */
void
togerImpl::faceOuterBoundaryD(girl::region r,
			      girl::point &startPt, 
			      std::deque<girl::direction> &dq) const
{
  const girl::dart canonicD = m_m.getLambda_1(r);

  girlAssert1(m_m.checkDartPhiCycle(canonicD));
  girlAssert1(checkContourContinuity(canonicD));

  girl::geoDart u;
  m_m.getGe(canonicD, u);
  startPt = u.point();
  
  girl::dart d = canonicD;

  do
    {
      segmentDirections(d, dq);

      d = m_m.getPhi(d);
    }
  while (d != canonicD);

  assert(P_checkContourIsClosed(startPt, dq));
}

/**
 * @brief Functor to compare two points.
 */
struct PointComparator
{
  inline bool operator()(const girl::point &p1, const girl::point &p2)
  {
    return (p1.y() != p2.y()) ? p1.y() < p2.y() : p1.x() < p2.x();
  }
};

/**
 * @brief Gives scanline from a starting point and a queue of directions.
 * (faster than to explicitly build a contour object)
 *
 * @a scan is an array of M=2*N girl::point, 
 * such as at 2*i and 2*i+1 are the positions 
 * of the first and last pixels of the line.
 * Lines are sorted in increasing y order. 
 *
 */
static
void
P_regionScanlineFromPointAndDirections(const girl::point &startPt, 
				       std::deque<girl::direction> &dq, 
				       std::vector<girl::point> &scan)
{
  assert(P_checkContourIsClosed(startPt, dq));

  assert(! dq.empty());

  //REM:OPTIM: allocate more than necessary, but allow to allocate all at once.
  scan.reserve(dq.size());

  const girl::point lStartPt = startPt;
  girl::point pt = lStartPt;

  do
    {
      const girl::direction d =  dq.front();
      dq.pop_front();
      pt.move(d);
      if (d.isVertical()) //(d == UP || d == DOWN)
	{	    
	  girl::point p = pt;
	  P_img_LR_pixel(p, d);
	  scan.push_back(p);
	}
    }
  while (! dq.empty());

  std::sort(scan.begin(), scan.end(), PointComparator());
}

/**
 *
 * Called from P_seekInfiniteFaceContour...
 */
void
togerImpl::dealWithTangentNode(girl::dart &d,
			       const girl::geoDart &u, 
			       girl::regionDeque &region_q)
{
  const girl::point upt = u.point();
  const girl::direction ud = u.dir();

  girl::geoDart old_ad_u;
  const girl::dart d1 = m_g.getDart(girl::geoDart(upt, ud.next()));
  
  m_m.getGe(m_m.alpha(d), old_ad_u);
  
  setGeToMap(girl::geoDart(upt, ud.opposite()), m_m.alpha(d));
  d = m_m.splitEdge(d);
  
  const girl::face af = m_m.joinNodes(d, d1);
  girlAssert1(m_m.checkFacePhiCycle(af));
  
  girlAssert1(checkFacePositiveOrientation(af));
  
  region_q.push_back(af);
  setGeToMap(girl::geoDart(upt, ud.previous()), d);
  setGeToMap(old_ad_u, m_m.alpha(d));
  
  girlAssert1(checkNodeConsistencyWoLight(upt)); 
}

/**
 * @brief Seeks outer path from given position.
 *
 * @return found direction.
 *
 * @warning Raises an exception if outer path is not found.
 */
girl::direction
togerImpl::seekOuterPath(girl::dart &d,
			 bool &artificialNode, const girl::geoDart &startHead, 
			 const girl::geoDart &u,
			 girl::toger::sameRegion &same_r)
{
  const girl::point upt = u.point();
  const girl::direction ud = u.dir();
  girl::point pt = u.point();
  assert(pt.x() < m_width && pt.x() >= -1);
  assert(pt.y() < m_height && pt.y() >= -1);

  P_img_LL_pixel(pt, ud);
  girl::direction dir = ud;
  girl::direction outDir = INVALID;

  bool pathFound = false;

  for (int i = 3; i; --i) // do it 3 times
    {
      assert(pt.x() < m_width && pt.x() >= -1);
      assert(pt.y() < m_height && pt.y() >= -1);
      assert(dir != INVALID);

      if (! same_r.same(pt, dir))
	{
	  if (! pathFound)
	    {
	      pathFound = true;
	      outDir = dir.previous();
	    }
	  else if (artificialNode)
	    {
	      artificialNode = false;
	      m_g.deleteNode(startHead.point());
	      m_b.unsetNode(startHead.point());
	      /*
		setGeToMap(girl::geoDart(upt, outDir), d);
		setGeToMap(girl::geoDart(upt, ud.opposite()), m_m.alpha(d));
	      */
	      //B:OPTIM: reduce the number of call to girlGe2to
	      const girl::dart alpha_d = m_m.alpha(d);
	      girl::dart darts[4] = {0, 0, 0, 0};
	      darts[outDir] = d;
	      darts[ud.opposite()] = alpha_d;
	      m_g.setDarts(upt, darts);
	      m_m.setGe(d, girl::geoDart(upt, outDir));
	      m_m.setGe(alpha_d, girl::geoDart(upt, ud.opposite()));
	      
	      m_b.setNode(upt);
	      girlAssert1(checkNodeConsistencyWoLight(upt));
	      break;
	    }
	  else
	    {
	      girl::geoDart old_ad_u;
	      m_m.getGe(m_m.alpha(d), old_ad_u);
	      /*
		setGeToMap(girl::geoDart(upt, ud.opposite()), m_m.alpha(d));
		d = m_m.splitEdge(d);
		setGeToMap(girl::geoDart(upt, outDir), d);	      
	      */
	      //B:OPTIM: reduce the number of call to girlGe2to
	      const girl::dart alpha_d = m_m.alpha(d);
	      m_m.setGe(alpha_d, girl::geoDart(upt, ud.opposite()));
	      d = m_m.splitEdge(d);
	      girl::dart darts[4] = {0, 0, 0, 0};
	      darts[ud.opposite()] = alpha_d;
	      darts[outDir] = d;
	      m_g.setDarts(upt, darts);
	      m_m.setGe(d, girl::geoDart(upt, outDir));

	      setGeToMap(old_ad_u, m_m.alpha(d));
	      m_b.setNode(upt);
	      girlAssert1(checkNodeConsistencyWoLight(upt));
	      break;
	    }
	}
      pt.move(dir);
      dir = dir.next();
    }

  
  if (GIRL_unlikely(pathFound == false)) {
    std::stringstream ss;
    ss<<"girl::toger::splitRegion: dangling edge around ("<<pt.x()<<";"<<pt.y()<<").";
    throw std::runtime_error(ss.str());
  }

  return outDir;
}



/**
 * @brief Seek infinite face contour from given dart @a d.
 *
 * @warning face doesn't exist yet, @a d is only its first dart.
 * Update bitmap, map and girlGe2to for the infinite face.
 * Adds the found finite face to queue @a q !
 */
void
togerImpl::seekInfiniteFaceContour(girl::dart d,
				   girl::toger::sameRegion &same_r,
				   girl::regionDeque &region_q)
{
  bool artificialNode = true;
  
  girl::geoDart startHead, u;
  m_m.getGe(d, startHead);
  u = startHead;
  
  bitmap::iterator it = m_b.getIterator(u.point());
  it.writeNeighbourInDirection(u.dir(), 1);
  it.moveInDirection(u.dir());
  P_moveGDartPoint(u, u.dir());
  
  while (u.point() != startHead.point())
    {
      if (it.isNode())
	{
	  dealWithTangentNode(d, u, region_q);

	  u.setDir(u.dir().previous());
	}
      else 
	{
	  const girl::direction outDir = seekOuterPath(d, artificialNode, 
						       startHead, u,
						       same_r);
	  u.setDir(outDir);
	}

      it.writeNeighbourInDirection(u.dir(), 1);
      girlAssert1(checkNodeConsistency(it.getPosition()));
      it.moveInDirection(u.dir());
      P_moveGDartPoint(u, u.dir());
    }

  const girl::face f = m_m.getLambda(d);

#ifndef NDEBUG1
  girlAssert1(m_m.checkFacePhiCycle(f));
  girlAssert1(checkFacePositiveOrientation(f));
#endif

  region_q.push_back(f);
}


/**
 * @brief Traverses contour of a known inner contour,
 * i.e. of a former daughter.
 * 
 * Will update mother-daughter relation : will add found face as 
 * daugther of @a r
 * and will update hash_table : will remove found face from @a h.
 */
void
togerImpl::seekKnownInnerContour(const girl::point &p,
				 girl::region r,
				 FaceUnorderedSet &h)
{
  girlAssert2(const_cast<const girl::bitmap&>(m_b).getIterator(p).hasUpNeighbour());

  const girl::point sp = m_s.firstPairStart();
  if (sp != p)
    {
      //it's an (old) infinite face not already scanned

      girl::geoDart u;
      m_b.contourTracking(p, u); //get first head/node on contour
      const girl::dart d =  m_g.getDart(u);
      assert(d!=0);
      girlAssert1(m_m.checkDart(d));
      girl::face inf = m_m.getLambda(d); //get face

      h.erase(inf);
		  
      assert(inf != 0);
      girlAssert1(m_m.checkFace(inf));

#ifndef GIRL_UIRC 
      m_m.faceAddDaughter(r, inf);
#else
      if (m_m.faceGetMother(inf) != r) {
	m_m.faceRemoveInfiniteFace(inf);
	m_m.faceAddDaughter(r, inf);
      }
#endif

      assert(! checkFacePositiveOrientation(inf));
      girlAssert1(m_m.checkFacePhiCycle(inf));

      std::deque<girl::direction> dq;
      girl::point startPt;
      faceOuterBoundaryD(inf, startPt, dq);
      m_s.add(startPt, dq);
    }
}

/**
 * @brief Inserts an artificial node at position @a pt,
 * with direction @a dir1 & @a dir2.
 *
 * Updates representation (bitmap, map, ge2to). 
 * Only set a node in bitmap, no contour. So bitmap is inconsitant.
 * Inclusion relation is also inconsistant.
 *
 * @return dart d in direction @a dir1 (alpha(d) is in direction @a dir2).
 */
girl::dart
togerImpl::insertArtificialNode(const girl::point &pt, 
				girl::direction dir1, girl::direction dir2)
{
  m_b.setNode(pt);
  const girl::dart d = m_m.insertMap();
  //B:OPTIM: reduce the number of call to girlGe2to
  /*
    setGeToMap(girl::geoDart(pt, dir1), d);
    setGeToMap(girl::geoDart(pt, dir2), m_m.alpha(d));
  */
  const girl::dart alpha_d = m_m.alpha(d);
  girl::dart darts[4] = {0, 0, 0, 0};
  darts[dir1] = d;
  darts[dir2] = alpha_d;
  m_g.setDarts(pt, darts);
  m_m.setGe(d,       girl::geoDart(pt, dir1));
  m_m.setGe(alpha_d, girl::geoDart(pt, dir2));

  return d;
}

/**
 * @brief Seeks a contour of an infinite face inside region @a r.
 *
 * Current bitmap position has different right and left pixels in image, 
 * so it means that we are at the up-left corner of a new region. 
 * Will update bitmap, mother-daugther relation.
 */
girl::face
togerImpl::seekInnerContour(girl::point &p, girl::region r,
			    girl::toger::sameRegion &same_r,
			    girl::regionDeque &region_q)
{
  girl::geoDart u;
  u.set(p.x(), p.y(), RIGHT);
  P_moveGDartPoint(u, UP);//move to border, in bitmap coords

  const girl::dart d = insertArtificialNode(u.point(), RIGHT, DOWN);

  seekInfiniteFaceContour(d, same_r, region_q);
  girl::face inf = m_m.getLambda(m_m.alpha(d));
  m_m.faceAddDaughter(r, inf);

  girlAssert1(m_m.checkFacePhiCycle(inf));
  assert(! checkFacePositiveOrientation(inf));

  return inf;
}

/**
 * @brief Traverses contour of an unknown inner contour,
 * i.e. of a new daughter.
 * 
 * Will update mother-daughter relation.
 */
void
togerImpl::seekUnknownInnerContour(girl::point &p, girl::region r,
				   girl::toger::sameRegion &same_r, 
				   girl::regionDeque &region_q)
{
  girl::face inf = seekInnerContour(p, r, same_r, region_q);
  
  //update scanlineData
  std::deque<girl::direction> dq;
  girl::point startPt;
  faceOuterBoundaryD(inf, startPt, dq);
  m_s.add(startPt, dq);

}


/**
 *
 * //@return true if given face @a r was a finite face (and so 
 * inner regions were sought)
 * //and return false for an infinite face.
 */
void
togerImpl::seekAndQueueInnerRegionsWithHoles(girl::region r,
					     girl::toger::sameRegion &same_r, 
					     girl::regionDeque &region_q, 
					     FaceUnorderedSet &h)
{
  std::deque<girl::direction> dq;
  girl::point startPt;
  faceOuterBoundaryD(r, startPt, dq);

  if (dq.size() < 12)
    //a contour with less than 12 boundary points encloses less than 9 pixels
    // and thus cannot surround regions (see Luc Brun's PhD Thesis)
    return;
  

  m_s.init(startPt, dq);

  assert(P_checkContourIsClosed(startPt, dq));

  std::vector<girl::point> scan;
  P_regionScanlineFromPointAndDirections(startPt, dq, scan);

  for (std::vector<girl::point>::const_iterator it = scan.begin();
       it != scan.end(); it += 2) {

    assert(it->y() == (it+1)->y());

    const girl::point p0 = *it;

    const girl::POINT_TYPE x0 = p0.x();
    const girl::POINT_TYPE x1 = (it+1)->x();

    if (x0+1 == x1)
      //no way there is a new border between these two.
      continue;
    
    bitmap::const_iterator itB = 
      const_cast<const girl::bitmap&>(m_b).getIterator(p0);

    for (girl::point p = p0; p.x() < x1;  )
      {
	assert(p.y() == it->y());
	
	if (itB.hasUpNeighbour())
	  { 
	    seekKnownInnerContour(p, r, h);
	    
	    //go to scanline end point
#ifndef NDEBUG
	    const girl::point scp = m_s.firstPairStart(); 
	    assert(p == scp);
	    const int py = p.y();
#endif
	    p = m_s.firstPairEndAndRemove();
	    assert(py == p.y());
	    itB.moveTo(p);
	    
	  }
	else if (! same_r.same(p, RIGHT))
	  {
	    //a new daughter is found
	    seekUnknownInnerContour(p, r, same_r, region_q);

	    //go to scanline end point
#ifndef NDEBUG
	    const girl::point scp = m_s.firstPairStart(); 
	    assert(p == scp);
	    const int py = p.y();
#endif
	    p = m_s.firstPairEndAndRemove();
	    assert(py == p.y());
	    itB.moveTo(p);
	  }
	
	itB.moveRight();
	p = itB.getPosition();
      }
  }
  
}

/**
 * @brief Splits a not simply connected region (recursive function)
 *
 * @warning first region of @a regionToSplit_q must be the original region @a r.
 */
void    
togerImpl::splitNotSimplyConnectedRegionAux(girl::region r,
					    girl::toger::sameRegion &same_r, 
					    girl::regionDeque &out_q,
					    girl::regionDeque &regionToSplit_q,
					    FaceUnorderedSet &h)
{
  girlAssert1(checkFacePositiveOrientation(r));


  assert(! regionToSplit_q.empty());
  const girl::region fr = regionToSplit_q.front();
  regionToSplit_q.pop_front();
  assert(fr == r);

  std::deque<girl::geoDart> head_s;
  pushNewHeads(fr, same_r, head_s);//modify head_s 

  const bool headCreatedOnOriginalRegion = (! head_s.empty());

  while (! regionToSplit_q.empty()) {
    const girl::region rl = regionToSplit_q.front();
    regionToSplit_q.pop_front();
    
    pushNewHeads(rl, same_r, head_s);//modify head_s 
  }

  //find regions
  std::deque<girl::dart> dart_q;
  girl::regionDeque regionToSeek_q;
  if (! headCreatedOnOriginalRegion) {
    regionToSeek_q.push_back(r);
  }
  if (! head_s.empty()) {
    buildBoundaries(same_r, head_s, dart_q);//empty head_s and fill dart_q
    
    labellingFaces(dart_q, r, headCreatedOnOriginalRegion, regionToSeek_q);//empty dart_q and fill regionToSeek_q //???
  }
  assert(head_s.empty());
  assert(dart_q.empty());


  //deal with inner regions if necessary
  while (! regionToSeek_q.empty()) {
    const girl::region rl = regionToSeek_q.front();
    regionToSeek_q.pop_front();
    
    seekAndQueueInnerRegionsWithHoles(rl, same_r, regionToSplit_q, h);
    //could fill regionToSplit_q, and empty h
    
    girlAssert1(checkFacePositiveOrientation(rl));
    
    out_q.push_back(rl);
  }

  girl::regionDeque lregion_q;
  while (! regionToSplit_q.empty()) {
    const girl::region rl = regionToSplit_q.front();
    regionToSplit_q.pop_front();

    assert(m_m.faceGetFirstDaughter(rl) == 0);

    assert(lregion_q.empty());
    lregion_q.push_back(rl);

    splitNotSimplyConnectedRegionAux(rl, same_r, out_q, lregion_q, h);
    assert(lregion_q.empty());
  }
  
}


/**
 * @brief Splits a not simply connected region @a r.
 *
 * found regions arequeued in @a out_q.
 */
void
togerImpl::splitNotSimplyConnectedRegion(girl::region r,
					 girl::toger::sameRegion &same_r, 
					 girl::regionDeque &out_q)
{
  assert(! m_m.faceIsInfinite(r));
  girlAssert1(checkFacePositiveOrientation(r));

  girl::regionDeque regionToSplit_q;
  regionToSplit_q.push_back(r);

  FaceUnorderedSet h;
 
  getDaugthers(r, regionToSplit_q, h);//update regionToSplit_q and h

  assert(r == regionToSplit_q.front());

#ifndef GIRL_UIRC
  invalidateMotherDaughterRelation(r);
#endif

  splitNotSimplyConnectedRegionAux(r, same_r, out_q, regionToSplit_q, h);
  assert(regionToSplit_q.empty());

  //free not reused (old daughters) labels
  FaceUnorderedSet::iterator it = h.begin();
  const FaceUnorderedSet::iterator itEnd = h.end();
  for ( ; it != itEnd; ++it) {
#ifndef GIRL_UIRC
    m_m.faceFree(*it);
#else
    m_m.faceRemove(*it);
#endif
  }

  girlAssert1(m_m.checkConsistency());
}

/**
 * @brief Entering an inner region border, go to the exiting border.
 * Will update given girl::point @a p.
 */
static
void
P_skipInnerRegionScanline(girl::bitmap::const_iterator &it, girl::point &p)
{
  assert(it.hasUpNeighbour());

  //performance of this function is critic for some cases
  // like "concentric"

  it.moveRightTillUpNeighbour();

  p = it.getPosition();
}


/**
 * Fills the given deque @a dq with points that belongs to the scanline 
 * of face @a f.
 *
 * @warning do not consider face @a f eventual daughters.
 *
 * @warning change bitmap internal position.
 */
void
togerImpl::getFaceScanline(girl::face f, std::deque<girl::point> &dq) const
{
  const girl::dart canonicalDart = m_m.getLambda_1(f);
  girl::dart d = canonicalDart;

  girl::geoDart u;
  m_m.getGe(canonicalDart, u);

  bitmap::const_iterator it = m_b.getIterator(u.point());

  do
    {
      m_m.getGe(d, u);
      girl::direction dir = u.dir();

      do 
	{
	  it.moveInDirection(dir);

	  if (dir.isVertical()) //(dir == UP || dir == DOWN) 
	    {
	      girl::point p;
	      p = it.getPosition();
	      P_img_LR_pixel(p, dir);
	      dq.push_back(p);
	    }
      
	  dir = it.getNextDirection(dir);
	}
      while (! it.isNode());
      
      d = m_m.getPhi(d);
     
    }
  while (d != canonicalDart);
}



/**
 * Fills the given deque @a dq with points that belongs to the scanline 
 * of region @a r and its daughters.
 *
 * @warning points are in undefined order, i.e. not sorted.
 */
void
togerImpl::getRegionScanline(girl::region r, std::deque<girl::point> &dq) const
{  
  getFaceScanline(r, dq);
  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      getFaceScanline(f, dq);
      f = m_m.faceGetNextSister(f);
    }
}

/**
 * fill @a scan with sorted points from @a dq.
 * 
 * @warning empty @a dq and do not delete it.
 *
 * @return number of points.
 */
static 
void
P_sortRegionScanline(std::deque<girl::point> &dq, 
		     std::vector<girl::point> &scan)
{
  const int nbp = dq.size();
  girlAssert1(nbp%2 == 0);
  scan.resize(nbp);

  std::copy(dq.begin(), dq.end(), scan.begin());

  std::sort(scan.begin(), scan.end(), PointComparator());

  dq.clear();
  
  assert(dq.empty());

  //TODO: verifier que c'est plus rapide que de faire un sort direct 
  // sur la deque !!!???
}



/**
 * @brief Seeks inner regions of region @a r.
 *
 * Will update mother-daugther relation, 
 * and queue found regions in @a region_q
 */
void
togerImpl::seekAndQueueInnerRegions(girl::region r,
				    girl::toger::sameRegion &same_r, 
				    girl::regionDeque &region_q)
{
  girlAssert1(checkFacePositiveOrientation(r));

  //int n = regionDomain(r, &scan);
  getFaceScanline(r, m_dq);
  size_t n = m_dq.size();
  if (n < 6) {
    //there can not be any inside region on a domain that spreads on 
    // less than 3 lines
    //(REM: n == 6 may spread on 3 lines, but it is not necessary)
    m_dq.clear();
    assert(m_dq.empty());
    return;
  }
  girl::point p;
  std::vector<girl::point> scan;
  P_sortRegionScanline(m_dq, scan);
  assert(m_dq.empty());


  for (std::vector<girl::point>::const_iterator it = scan.begin();
       it != scan.end(); it+=2) {
    
    //there can not be any inside region on one or two-pixel wide line
    if (it->x() +1 < (it+1)->x()) {
      
      bitmap::const_iterator itB = const_cast<const girl::bitmap&>(m_b).getIterator(*it);
      for (p = *it; p.x() < (it+1)->x(); p.setX(p.x()+1)) {
	
	assert(p.x() >= -1 && p.x() < m_width);
	assert(p.y() >= -1 && p.y() < m_height);
	
	if (itB.hasUpNeighbour()) { 
	  // we touch an already found inner region.
	  P_skipInnerRegionScanline(itB, p); 
	}
	else if (! same_r.same(p, RIGHT)) {
	  //we touch a new inner region.
	  seekInnerContour(p, r, same_r, region_q);
	  
	  P_skipInnerRegionScanline(itB, p); 
	}
	itB.moveRight();
      }
    }
  }
}

/**
 * @brief Splits a simply connected region.
 *
 * @return found regions are queued in @a out_q.
 */
void
togerImpl::splitSimplyConnectedRegion(girl::region r,
				      girl::toger::sameRegion &same_r, 
				      girl::regionDeque &out_q)
{
  girlAssert1(checkFacePositiveOrientation(r));

  std::deque<girl::geoDart> head_s;
  std::deque<girl::dart> dart_q;

  girl::regionDeque regionToSplit_q;
  girl::regionDeque regionToSeek_q;
  
  regionToSplit_q.push_back(r);

  while (! regionToSplit_q.empty()) //warning: may be modified later
    {
      girl::region rl = regionToSplit_q.front();
      regionToSplit_q.pop_front();
      
      girlAssert1(checkFacePositiveOrientation(rl));

      const size_t contourSize = pushNewHeads(rl, same_r, head_s);//modify head_s 

      //find regions
      if (head_s.empty())
	{
	  out_q.push_back(rl);
	      
	  if (contourSize >= 12)
	    seekAndQueueInnerRegions(rl, same_r, regionToSplit_q);//modify regionToSeek_q
	}
      else
	{
	  buildBoundaries(same_r, head_s, dart_q);//modify dart_q (and empty head_s)

	  assert(head_s.empty());
	  
	  labellingRegions(dart_q, rl, regionToSeek_q);//modify regionToSeek_q
	  
	  assert(checkFirstInfiniteFace());
	  
	  //deal with inner regions 
	  if (contourSize >= 12) {
	    while (! regionToSeek_q.empty())
	      {
		rl = regionToSeek_q.front();
		regionToSeek_q.pop_front();
		
		girlAssert1(checkFacePositiveOrientation(rl));
		
		out_q.push_back(rl);
		
		seekAndQueueInnerRegions(rl, same_r, regionToSplit_q);
	      }
	  }
	  else {
	    //regions to small to need to seek inside.
	    //just append regionToSeek_q to out_q.
	    while (! regionToSeek_q.empty())
	      {
		rl = regionToSeek_q.front();
		regionToSeek_q.pop_front();
		
		girlAssert1(checkFacePositiveOrientation(rl));
		
		out_q.push_back(rl);
	      }
	  }
	  
	}
    }
  


}

/**
 * @brief Splits a given region @a r, simply connected or not.
 *
 * @return the number of found regions, queued in @a out_q.
 */
void
togerImpl::split(girl::region r,
		 girl::toger::sameRegion &same_r, 
		 girl::regionDeque &out_q)
{
  girlAssert1(checkFacePositiveOrientation(r));

  const bool isNotSimplyConnected = (m_m.faceGetFirstDaughter(r) != 0);
  if (isNotSimplyConnected)
    splitNotSimplyConnectedRegion(r, same_r, out_q);
  else
    splitSimplyConnectedRegion(r, same_r, out_q);
}

void
togerImpl::splitRegion(girl::region r,
		       girl::toger::sameRegion &same_r, 
		       girl::regionDeque &allRegs)
{
  assert(m_m.checkRegion(r));
  girlAssert1(checkFirstInfiniteFace());
  girlAssert1(checkConsistency());
  girlAssert1(checkFacePositiveOrientation(r));

  girl::regionDeque regs;
  split(r, same_r, regs);
  
  assert(! regs.empty());
  assert(*regs.begin() == r);

  applyNewRegionHook(regs);

  assert(! regs.empty());
  assert(*regs.begin() == r);

  if (allRegs.empty())
    allRegs.swap(regs);
  else
    allRegs.insert(allRegs.end(), regs.begin(), regs.end());

  girlAssert1(checkConsistency());
  girlAssert1(checkFirstInfiniteFace());
}


void
togerImpl::regionDomain(girl::region r, std::vector<girl::point> &scan) const //vector ???
{
  assert(m_m.checkRegion(r));

  assert(m_dq.empty());

  getRegionScanline(r, m_dq);

  P_sortRegionScanline(m_dq, scan);
}


void
togerImpl::regionOuterBoundary(girl::region r, girl::contour &c) const
{
  assert(m_m.checkRegion(r));

  faceOuterBoundary(r, c);
}

void
togerImpl::faceOuterBoundary(girl::face f, girl::contour &c) const
{
  girlAssert1(m_m.checkFacePhiCycle(f));

  contourFromFace(f, c);
}

/**
 * @brief Fills the given queue @a q with contours of given region @a r.
 *
 * @return the number of found contours.
 */
void
togerImpl::getRegionBoundary(girl::region r, std::deque<contour> &q) const
{
  q.push_back(contour()); 
  contourFromFace(r, q.back());

  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      q.push_back(contour()); 
      contourFromFace(f, q.back());

      f = m_m.faceGetNextSister(f);
    }
}

void
togerImpl::regionBoundary(girl::region r, girl::contourDeque &q) const
{
  assert(m_m.checkRegion(r));
  getRegionBoundary(r, q);
}


void
togerImpl::regionsCommonBoundary(girl::region r1, girl::region r2, 
				 //			       girl::toger::neighbouringModes mode, 
				 //			       girl::toger::markFlags m,
				 girl::pathDeque &sp) const
{
  assert(m_m.checkRegion(r1));
  assert(m_m.checkRegion(r2));

  std::deque<girl::dart> ds;
  girl::toger::neighbouringModes nm;

  commonDarts(r1, r2, ds, nm);

  if (ds.empty())
    return;

  //TODO:OPTIM??? if sp was a vector, we could do sp.reserve(ds.size()) !!!

  /*
    //B: I don't understand why there is an "if" here. We do the same thing in both cases !
  
  if (nm == girl::toger::NEIGHBOUR_INNER) {
    for (std::deque<girl::dart>::const_iterator it = ds.begin();
	 it != ds.end(); ++it) {
      sp.push_back(path());
      pathFromDart(*it, sp.back()); 
    }
  }
  else {
    for (std::deque<girl::dart>::const_iterator it = ds.begin();
	 it != ds.end(); ++it) {
      sp.push_back(path());
      pathFromDart(*it, sp.back()); 
    }
  }   
   */

  for (std::deque<girl::dart>::const_iterator it = ds.begin();
       it != ds.end(); ++it) {
    sp.push_back(path());
    pathFromDart(*it, sp.back()); 
  }
 
}

void
togerImpl::segment(girl::dart d, girl::path &p) const
{
  assert(m_m.checkDart(d));

  pathFromDart(d, p);
}

size_t
togerImpl::segmentLength(girl::dart d) const
{
  //impl: does not allocate a segment/path explicitly !
  girlAssert1(m_m.checkDart(d));

  girl::geoDart u;
  m_m.getGe(d, u);
  bitmap::const_iterator it = m_b.getIterator(u.point());
  girl::direction dir = u.dir();

  size_t nbd = 0;
  do
    {
      ++nbd;
      
      it.moveInDirection(dir);
      dir = it.getNextDirection(dir);
    }
  while (! it.isNode());

  return nbd;
}


/**
 * @brief Fills the contour @a c for face @a f.
 *
 */
void
togerImpl::contourFromFace(girl::face f, girl::contour &c) const
{
  girl::dart d = m_m.getLambda_1(f);
  
  girlAssert1(m_m.checkDartPhiCycle(d));
  girlAssert1(checkContourContinuity(d));

  const girl::dart canonicD = d;

  girl::geoDart u;
  m_m.getGe(canonicD, u);
  girl::point startPt = u.point();
  
  std::deque<girl::direction> dq;

  do
    {
      segmentDirections(d, dq);

      d = m_m.getPhi(d);
    }
  while (d != canonicD);

  c.set(startPt, dq);
}


/**
 * @brief Gives start @a start and end @a end points for a given dart @a d, and 
 * directions along this path in @a segment_path.
 *
 * @return the number of directions along the returned path.
 * @warning Move bitmap internal stored position to end point.
 */
void
togerImpl::pathFromDart(girl::dart d, girl::path &p) const
{
  girl::geoDart u;
  m_m.getGe(d, u);
  girl::point startPt = u.point();

  std::deque<girl::direction> dq;
  segmentDirections(d, dq);
  
  m_m.getGe(m_m.getPhi(d), u);
  girl::point endPt = u.point();

  p.set(startPt, endPt, dq);
}


bool
togerImpl::isRegionMarked(girl::region r, girl::toger::markFlags mf) const
{
  const bool outerBoundaryMarked = isRegionOuterBoundaryMarked(r, mf);
  if (outerBoundaryMarked)
    return true;

  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      if (isFaceMarked(f, mf))
	return true;
      f = m_m.faceGetNextSister(f);
    }
  
  return false;
}


bool
togerImpl::isRegionOuterBoundaryMarked(girl::region r, 
				       girl::toger::markFlags mf) const
{
  assert(m_m.checkRegion(r));

  return isFaceMarked(r, mf);
}


bool
togerImpl::isFaceMarked(girl::face f, girl::toger::markFlags mf) const
{
  const girl::dart canonicD = m_m.getLambda_1(f);
  girl::dart d = canonicD;
  do
    {
      if (m_m.dartIsMarked(d, mf))
	return true;
      d = m_m.getPhi(d);
    }
  while (d != canonicD);
  return false;
}


void
togerImpl::markFace(girl::face f, girl::toger::markFlags flag)
{  
  const girl::dart canonicD = m_m.getLambda_1(f);
  girl::dart d = canonicD;
  do
    {
      m_m.markDart(d, flag);
      m_m.markDart(m_m.alpha(d), flag);
      d = m_m.getPhi(d);
    }
  while (d != canonicD);
}

void
togerImpl::xMarkFace(girl::face f, girl::toger::markFlags flag)
{  
  const girl::dart canonicD = m_m.getLambda_1(f);
  girl::dart d = canonicD;
  do
    {
      m_m.xMarkDart(d, flag);
      m_m.xMarkDart(m_m.alpha(d), flag);
      d = m_m.getPhi(d);
    }
  while (d != canonicD);
}


void
togerImpl::unmarkFace(girl::face f, girl::toger::markFlags flag)
{
  const girl::dart canonicD =  m_m.getLambda_1(f);
  girl::dart d = canonicD; 
  do
    {
      m_m.unmarkDart(d, flag);
      m_m.unmarkDart(m_m.alpha(d), flag);
      d = m_m.getPhi(d);
    }
  while (d != canonicD);
}



void
togerImpl::xMarkInnerBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      xMarkFace(f, flag);
      f = m_m.faceGetNextSister(f);
    }
}

void
togerImpl::unmarkInnerBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  girl::face f = m_m.faceGetFirstDaughter(r);
  while (f)
    {
      unmarkFace(f, flag);
      f = m_m.faceGetNextSister(f);
    }
}

void 
togerImpl::xMarkRegionBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  xMarkFace(r, flag);

  xMarkInnerBoundary(r, flag);
}

void
togerImpl::unmarkRegionBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  unmarkFace(r, flag);

  unmarkInnerBoundary(r, flag);
}

void
togerImpl::xMarkOuterBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  xMarkFace(r, flag);
}

void
togerImpl::unmarkOuterBoundary(girl::region r, girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r));
  unmarkFace(r, flag);
}

void
togerImpl::markCommonBoundary(girl::region r1, girl::region r2, 
			      girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r1));
  assert(m_m.checkRegion(r2));

  std::deque<girl::dart> sd;

  girl::toger::neighbouringModes nm;
  commonDarts(r1, r2, sd, nm);

  //TODO:OPTIM???:iterator+clear ?
  while (! sd.empty())
    {
      m_m.markDart(sd.back(), flag);
      sd.pop_back();
      //warning: not symmetrical !!!
    }
}

void
togerImpl::unmarkCommonBoundary(girl::region r1, girl::region r2, 
				girl::toger::markFlags flag)
{
  assert(m_m.checkRegion(r1));
  assert(m_m.checkRegion(r2));

  std::deque<girl::dart> sd;

  girl::toger::neighbouringModes nm;
  commonDarts(r1, r2, sd, nm);

  //TODO:OPTIM???:iterator+clear ?
  while (! sd.empty())
    {
      m_m.unmarkDart(sd.back(), flag);
      sd.pop_back();
      //warning: not symmetrical !!!
    }
}

void
togerImpl::markAllDarts(girl::toger::markFlags flag)
{
  m_m.markAllDarts(flag);
}

void
togerImpl::unmarkAllDarts(girl::toger::markFlags flag)
{
  m_m.unmarkAllDarts(flag);
}

bool
togerImpl::isDartMarked(girl::dart d, girl::toger::markFlags flag) const
{
  return m_m.dartIsMarked(d, flag);
}


void
togerImpl::saveBitmap(const std::string &filename) const
{
  m_b.save(filename);
}

bool
togerImpl::checkConsistency() const
{
  bool consistent = true;
  if (! m_m.checkConsistency())
    {
      std::cerr<<"map is not consistent\n";
      consistent = false;
    }  
  if (! m_b.checkConsistency())
    {
      std::cerr<<"bitmap is not consistent\n";
      consistent = false;
    }
  
  girl::face f = 0;
  while ((f = m_m.getNextExistingFace(f)) != 0)
    {
      bool hasArtificialNode = false;
      int nbd = 0;
      const girl::dart d0 = m_m.getLambda_1(f);
      girl::dart d = d0;
      do 
	{
	  ++nbd;
	  girl::geoDart u;
	  m_m.getGe(d, u);
	  bitmap::const_iterator it = m_b.getIterator(u.point());
	  if (! it.isNode()) 
	    {
	      std::cerr<<"dart "<<d<<" is at ("<<u.x()<<"; "<<u.y()<<") in map but this point is not a node in bitmap\n";
	      consistent = 0;
	    }
	  if (! it.hasNeighbourInDirection(u.dir()))
	    {
	      std::cerr<<"dart "<<d<<" at ("<<u.x()<<"; "<<u.y()<<") with direction "<<u.dir()<<" in map but this point has no neighbour in this direction in bitmap\n";
	      consistent = false;
	    }
	  if (d != m_g.getDart(u))
	    {
	      std::cerr<<"dart at ("<<u.x()<<"; "<<u.y()<<") with direction "<<u.dir()<<" is "<<m_g.getDart(u)<<" and should be "<<d<<std::endl;
	      consistent = false;
	    }

	  const int bc = it.getPointCardinality();
	  const int mc = m_m.getSigmaCycleCardinality(d);

	  if (bc != mc) 
	    {
	      std::cerr<<"dart "<<d<<" at ("<<u.x()<<"; "<<u.y()<<") has a sigma cycle of cardinality "<<mc<<" and corresponding node in bitmap has a cardinality of "<<bc<<std::endl;
	      consistent = false;
	    }
	
	  if (bc == 2)
	    {
	      if (hasArtificialNode == false)
		hasArtificialNode = true;
	      else
		{
		  std::cerr<<"More than on artificial node on face "<<f<<std::endl;
		  consistent = false;
	
		  std::cerr<<"face "<<f<<" : more than one artificial node\n";
#ifndef NDEBUG1
		  girl::contour c;
		  faceOuterBoundary(f, c);
		  DBG_printContour(c);
		  //girl::contour_delete(c);
#endif

		}
	    }


	  d = m_m.getPhi(d);
	}
      while (d != d0);

      if (hasArtificialNode == true && nbd > 1)
	{
	  std::cerr<<"artificial node on face "<<f<<" with a phi cycle of cardinality "<<nbd<<std::endl;
	  consistent = false;
    
	  std::cerr<<"face "<<f<<" : artificial node with phi cardinality "<<nbd<<std::endl;
#ifndef NDEBUG1
	  girl::contour c;
	  faceOuterBoundary(f, c);
	  DBG_printContour(c);
	  //girl::contour_delete(c);
#endif
    
	}

#ifndef NDEBUG
#endif
      if (! m_m.checkDartPhiCycle(d0))
	{
	  std::cerr<<"face "<<f<<" with canonical dart "<<d0<<" has a wrong phi cycle : alpha dart ill positionned\n";
	  consistent = false;
	}
      if (! m_m.checkDartPhiCycleLambda(d0))
	{
	  std::cerr<<"face "<<f<<" with canonical dart "<<d0<<" has a wrong phi cycle : bad lambda labeling\n";
	  consistent = false;
	}
      if (! m_m.checkFaceBetaLambda(f))
	{
	  std::cerr<<"face "<<f<<" has wrong beta/lambda\n";
	  consistent = false;
	}
      
    }  

  for (int j=-1; j<m_height; ++j)
    for (int i=-1; i<m_width; ++i) {
      girl::point p;
      p.setX(i); p.setY(j);
      if (! checkPtNodeConsistency(p)) 
	{
	  std::cerr<<"pt ("<<p.x()<<"; "<<p.y()<<") has wrong pt-node consistency\n";
	  consistent = false;
	  break;
	}
    }
  
  return consistent;
}




girl::dart
togerImpl::alpha(girl::dart d) const
{
  return m_m.alpha(d);
}


girl::dart
togerImpl::sigma(girl::dart d) const
{
  return m_m.getSigma(d);
}


girl::dart
togerImpl::phi(girl::dart d) const
{
  return m_m.getPhi(d);
}


girl::face
togerImpl::lambda(girl::dart d) const
{
  return m_m.getLambda(d);
}


girl::dart
togerImpl::beta(girl::face f) const
{
  return m_m.getLambda_1(f);
}

bool 
togerImpl::isFaceInfinite(girl::face f) const
{
  return m_m.faceGetMother(f) != 0;
}

girl::face
togerImpl::parent(girl::face f) const
{
  return m_m.faceGetMother(f);
}


girl::face
togerImpl::firstChild(girl::face f) const
{
  return m_m.faceGetFirstDaughter(f);
}


girl::face
togerImpl::nextSibling(girl::face fi) const
{
  return m_m.faceGetNextSister(fi);
}

girl::dart
togerImpl::getDart(const girl::point &pt, girl::direction dir) const
{
  return m_g.getDart(girl::geoDart(pt, dir));
}



void
togerImpl::shrink()
{
  typedef girl::hashMap<girl::face, girl::face> FacesMap;
  typedef girl::hashMap<girl::dart, girl::dart> DartsMap;
  
  FacesMap facesMap;
  DartsMap dartsMap;

  //Impl: should be exception safe, even if we modify map before ge2to
  // as there is no allocation in ge2to.

  m_m.shrink(facesMap, dartsMap);
  
  m_g.shrink(dartsMap);

  assert(checkConsistency());

  {
    //rename

    //first check if necessary
    bool doRename = false;
    ObserverCollection::iterator itO = m_observers.begin();
    const ObserverCollection::iterator itOEnd = m_observers.end();
    for ( ; itO != itOEnd; ++itO) {
      girl::toger::observer *o = *itO;
      assert(o);
      if (o->doRenameRegion()) {
	doRename = true;
	break;
      }
    }
    if (doRename) {

      //transfer in std::map thus sorted in increasing order according to old label.
      typedef std::map<girl::face, girl::face> FacesMap2;
      FacesMap2 facesMap2;
      for (FacesMap::const_iterator it = facesMap.begin();
	   it != facesMap.end();
	   ++it) {
	girl::face newF = it->second;
	if (m_m.checkRegion(newF)) { //is a region
	  facesMap2.insert(*it); //facesMap2[it->first] = it->second;
	}
      }
      //Warning : must be traversed in increasing order !!!
      for (FacesMap2::const_iterator it = facesMap2.begin();
	   it != facesMap2.end();
	   ++it) {
	girl::face oldF = it->first;
	girl::face newF = it->second;
	assert(m_m.checkRegion(newF)); //is a region
	
	ObserverCollection::iterator itO = m_observers.begin();
	const ObserverCollection::iterator itOEnd = m_observers.end();
	for ( ; itO != itOEnd; ++itO) {
	  girl::toger::observer *o = *itO;
	  assert(o);
	  if (o->doRenameRegion())
	    o->renameRegion(oldF, newF);
	}
      }
    }
  
  }

      
}


static const int versionMask = 0xffffff;
static const int labelShift = 24;

bool
togerImpl::write(girl::io::Writer &w, girl::io::writeFlags flags) const
{
  //all is saved in little endian

  assert(sizeof(girl::POINT_TYPE) == 2);

  int version = GIRL_TOGER_BINARY_VERSION & versionMask;
  if (flags & girl::io::NO_LABELS_MARKS_CARE) {
    version |= (1 << labelShift);
  }
  version = GIRL_swapLE32(version);
  POINT_TYPE rw = GIRL_swapLE16(m_width);
  POINT_TYPE rh = GIRL_swapLE16(m_height);

  if (w.write(static_cast<const void *>(&version), sizeof(version)) != sizeof(version))
    return false;
  if (w.write(&rw, sizeof(rw)) != sizeof(rw))
    return false;
  if (w.write(&rh, sizeof(rh)) != sizeof(rh))
    return false;

  if (! m_b.write(w, flags))
    return false;

  if ((flags & girl::io::NO_LABELS_MARKS_CARE) == 0) {
    if (! m_m.write(w, flags))
      return false;
    if (! m_g.write(w, flags))
      return false;
  }

  return true;
}

#if 0
class sameRegionForBitmap : public girl::toger::sameRegion
{
public:
  sameRegionForBitmap(const girl::bitmap &btm)
    : m_btm(btm)
  {

  }
  
  virtual bool same(const girl::point &p, girl::direction dir)
  {
    //REM: we use the fact that bitmap uses actually only Right & Up directions...

    switch (dir) {
    case RIGHT:
      return ! m_btm.getIterator(girl::point(p.x(), p.y())).hasUpNeighbour();
    case UP:
      return ! m_btm.getIterator(girl::point(p.x()-1, p.y()-1)).hasRightNeighbour();
    case LEFT:
      return ! m_btm.getIterator(girl::point(p.x()-1, p.y())).hasUpNeighbour();
    case DOWN:
      return ! m_btm.getIterator(girl::point(p.x()-1, p.y())).hasRightNeighbour();
    case INVALID:
    default:
      assert(false);
    }

    return false;
  }

protected:
  
  const girl::bitmap &m_btm;
};
#else
class sameRegionForBitmap : public girl::toger::sameRegion
{
public:
  sameRegionForBitmap(const girl::bitmap &btm)
    : m_btm(btm),
      m_it(btm.getIterator(girl::point(-1, -1)))
  {
    
  }
  
  virtual bool same(const girl::point &p, girl::direction dir)
  {
    //REM: we use the fact that bitmap uses actually only Right & Up directions...
    
    switch (dir) {
    case RIGHT:
      m_it.moveTo(girl::point(p.x(), p.y()));
      return ! m_it.hasUpNeighbour();
    case UP:
      m_it.moveTo(girl::point(p.x()-1, p.y()-1));
      return ! m_it.hasRightNeighbour();
    case LEFT:
      m_it.moveTo(girl::point(p.x()-1, p.y()));
      return ! m_it.hasUpNeighbour();
    case DOWN:
      m_it.moveTo(girl::point(p.x()-1, p.y()));
      return ! m_it.hasRightNeighbour();
    case INVALID:
    default:
      assert(false);
    }
    
    return false;
  }
	   
protected:
	   
  const girl::bitmap &m_btm;
  girl::bitmap::const_iterator m_it;
};



#endif




bool
togerImpl::read(girl::io::Reader &r, girl::io::writeFlags flags)
{
  assert(m_m.checkConsistency());

  assert(sizeof(girl::POINT_TYPE) == 2);

  int version = 0;
  if (r.read(&version, sizeof(int)) != sizeof(int))
    return false;
  version = GIRL_swapLE32(version);
  if ((version & versionMask)!= GIRL_TOGER_BINARY_VERSION)
    return false;
  const bool hasLabels = ((version >> labelShift) & 1) == 0;

  girl::POINT_TYPE rw=0, rh=0;
  if (r.read(&rw, sizeof(rw)) != sizeof(rw))
    return false;
  if (r.read(&rh, sizeof(rh)) != sizeof(rh))
    return false;
  rw = GIRL_swapLE16(rw);
  rh = GIRL_swapLE16(rh);
    
  assert(rw>0);
  assert(rh>0);

  girl::bitmap lb(rw, rh);
  if (! lb.read(r, flags)) 
    return false;
  assert(lb.checkConsistency());

  if (hasLabels) {

    girl::map lm;
    if (! lm.read(r, flags)) 
      return false;
    assert(lm.checkConsistency());

    girl::ge2to lg(rw, rh);
    if (! lg.read(r, flags)) 
      return false;
    assert(lg.checkConsistency());
  

    m_m = lm;
    m_g = lg;

    m_b = lb;
    m_width = rw;
    m_height = rh;
    
    m_s = girl::scanlineData(rw, rh);
  }
  else {
    
    girl::togerImpl togImpl(rw, rh);
    sameRegionForBitmap srfb(lb);
    girl::regionDeque regs;
    togImpl.splitRegion(1, srfb, regs);

    std::swap(*this, togImpl);
  }

  return true;
}


bool
togerImpl::swapable(const girl::point &p) const
{
  assert(checkInterPixelValidity(p));

  return m_b.swapable(p);
}


//@return a bit field with bits DLUR set if there is a dart in this direction
//@return 0 if point is not a node in @a g.
static
int
P_dartDirections(girl::ge2to &g, const girl::point &p)
{
  int res = 0;

  girl::dart darts[4];
  g.getDarts(p, darts);
  /*
    if (darts[RIGHT])
    res |= 1;
    if (darts[UP])
    res |= 1<<1;
    if (darts[LEFT])
    res |= 1<<2;
    if (darts[DOWN])
    res |= 1<<3;
  */

  res |= (darts[RIGHT]) ? 1    : 0;
  res |= (darts[UP])    ? 1<<1 : 0;
  res |= (darts[LEFT])  ? 1<<2 : 0;
  res |= (darts[DOWN])  ? 1<<3 : 0;
  
  /*
    u.d = RIGHT;
    if (girlGe2to_getDart(g, &u) != 0)
    res |= 1;
    u.d = UP;
    if (girlGe2to_getDart(g, &u) != 0)
    res |= 1<<1;
    u.d = LEFT;
    if (girlGe2to_getDart(g, &u) != 0)
    res |= 1<<2;
    u.d = DOWN;
    if (girlGe2to_getDart(g, &u) != 0)
    res |= 1<<3;
  */

  return res;
}


void
togerImpl::changeDartDirection(const girl::point &p, 
			       girl::direction dirOld, girl::direction dirNew)
{
  girl::geoDart u;
  u.setX(p.x());
  u.setY(p.y());
  u.setDir(dirOld);

  const girl::dart d = m_g.replaceDirection(u, dirNew);

  u.setDir(dirNew);
  m_m.setGe(d, u);
}


void
togerImpl::transferDarts(const girl::point &pOld, const girl::point &pNew)
{
  m_g.transferDarts(pOld, pNew);
  girl::dart darts[4];
  m_g.getDarts(pNew, darts);
  girl::geoDart uNew;
  uNew.setX(pNew.x()); 
  uNew.setY(pNew.y()); 
  for (int i=0; i<4; ++i) {
    if (darts[i]) {
      uNew.setDir( static_cast<eDirection>(i) );
      m_m.setGe(darts[i], uNew);
    }
  }
  m_g.deleteNode(pOld);
} 


void
togerImpl::swap_oneNode(const girl::point &p, int resn)
{

#ifdef VERBOSE
  std::cout<<"1 node\n"<<std::endl;
#endif

  int i1 = -1;
  for (int i=0; i<4; ++i) {
    const int v = (resn>>(i<<2)) & 0xf;
    if (v != 0) {
      i1 = i; //index of node
      break;
    }
  }
  assert(i1!=-1);

  const int rn = (resn>>(i1<<2)) & 0xf; //directions for node

  girl::point pts[4];
  pts[0].setX( p.x() );     pts[0].setY( p.y() ); 
  pts[1].setX( p.x() );     pts[1].setY( p.y() - 1 ); 
  pts[2].setX( p.x() - 1 ); pts[2].setY( p.y() - 1 ); 
  pts[3].setX( p.x() - 1 ); pts[3].setY( p.y() ); 

  //find node in ge2to
  girl::geoDart u;
  u.setPoint( pts[i1] );
  int ro = P_dartDirections(m_g, u.point());
  if (ro) { 
    //unique point has not moved,
    //just its darts have been changed.
#ifdef VERBOSE
    std::cout<<"1 node not moved\n";
    std::cout<<"rn="<<rn<<" ro="<<ro<<"\n";
#endif
    int jn=-1, jo=-1;
    for (int i=0; i<4; ++i) {
      const int vn = (rn>>i)&1;
      const int vo = (ro>>i)&1;
      if (vn != vo) {
	if (vn == 1) {
	  assert(jn==-1);
	  jn = i;
	}
	else if (vo == 1) {
	  assert(jo==-1);
	  jo = i;
	}
      } 
    }
    assert(jn!=-1 && jo!=-1 && jn!=jo);
    changeDartDirection(u.point(), static_cast<eDirection>(jo), static_cast<eDirection>(jn));
  }
  else {

    enum dirBit {R=1, U=2, L=4, D=8};

    girl::geoDart u2;
    //unique node has moved
#ifdef VERBOSE
    std::cout<<"1 node moved\n";
#endif
    for (int i=0; i<4; ++i) {
      if (i != i1) {
	u2.setPoint(pts[i]);
	ro = P_dartDirections(m_g, u2.point());
	if (ro) {
	  //we have found where the moved node comes from
	      
	  if (rn == ro) {
	    //node has moved but its darts stay in the same directions
	    transferDarts(pts[i], pts[i1]);
	  }
	  else {
	    //node has moved and its darts have changed their directions
		
	    girl::direction do1 = INVALID, do2 = INVALID; //old directions
	    girl::direction dn1 = INVALID, dn2 = INVALID; //new directions
		
	    switch (ro) {
	    case 0x3: //0011=RU
	      { 
		do1 = RIGHT;
		do2 = UP;
		if (rn&D && rn&L) { //RU -> DL
		  assert(rn == 0xC); //0xC=1100=DL
		  dn1 = DOWN;
		  dn2 = LEFT;
		}
		else if (rn&D) { //RU -> DR|DU
		  dn1 = DOWN;
		  dn2 = (rn&R) ? RIGHT : UP;
		  assert((rn&R && dn2==RIGHT) || (rn&U && dn2==UP));
		}
		else if (rn&L) { //RU -> RL|UL
		  dn1 = (rn&R) ? RIGHT : UP;
		  dn2 = LEFT;
		  assert((rn&R && dn1==RIGHT) || (rn&U && dn1==UP));		      
		}
		else {
		  assert(false);
		}
		break;
	      }
	    case 0x6: //0110=UL -> RD
	      { 
		do1 = UP;
		do2 = LEFT;
		if (rn&R && rn&D) { //UL -> RD
		  assert(rn == 0x9);
		  dn1 = RIGHT;
		  dn2 = DOWN;
		}
		else if (rn&R) { //UL -> RU|RL
		  dn1 = RIGHT;
		  dn2 = (rn&U) ? UP : LEFT;
		  assert((rn&U && dn2==UP) || (rn&L && dn2==LEFT));
		}
		else if (rn&D) { //UL -> UD|LD
		  dn1 = (rn&U) ? UP : LEFT;
		  dn2 = DOWN;
		  assert((rn&U && dn1==UP) || (rn&L && dn1==LEFT));
		}
		else {
		  assert(false);
		}
		break;
	      }
	    case 0x9: //1001=RD -> UL
	      { 
		do1 = RIGHT;
		do2 = DOWN;
		if (rn&U && rn&L) { //RD -> UL
		  assert(rn == 0x6);
		  dn1 = UP;
		  dn2 = LEFT;
		}
		else if (rn&L) {   //RD -> RL|DL
		  dn1 = (rn&R) ? RIGHT : DOWN;
		  dn2 = LEFT;
		  assert((rn&R && dn1==RIGHT) || (rn&D && dn1==DOWN));
		}
		else if (rn&U) {  //RD -> UR|UD
		  dn1 = UP;
		  dn2 = (rn&R) ? RIGHT : DOWN;
		  assert((rn&R && dn2==RIGHT) || (rn&D && dn2==DOWN));
		}
		else {
		  assert(false);
		}
		break;
	      }
	    case 0xC: //1100=LD 
	      { 
		do1 = LEFT;
		do2 = DOWN;
		if (rn&U && rn&R) { //LD -> UR
		  assert(rn == 0x3);
		  dn1 = UP;
		  dn2 = RIGHT;
		}
		else if (rn&U) {//LD -> UL|UD
		  dn1 = UP;
		  dn2 = (rn&L) ? LEFT : DOWN;
		  assert((rn&L && dn2==LEFT) || (rn&D && dn2==DOWN));
		}
		else if (rn&R) { //LD -> LR|DR
		  dn1 = (rn&L) ? LEFT : DOWN;
		  dn2 = RIGHT;
		  assert((rn&L && dn1==LEFT) || (rn&D && dn1==DOWN));
		}
		else {
		  assert(false);
		}
		break;
	      }
	    default:
	      {
		assert(false);
	      }
	    }

	    assert(do1 != INVALID);
	    assert(do2 != INVALID);
	    assert(dn1 != INVALID);
	    assert(dn2 != INVALID);
		  

	    u2.setDir(do1);
	    const girl::dart d1 = m_g.getDart(u2);
	    u.setDir(dn1);
	    assert(m_g.getDart(u) == 0);
	    setGeToMap(u, d1);

	    u2.setDir(do2);
	    const girl::dart d2 = m_g.getDart(u2);
	    u.setDir(dn2);
	    assert(m_g.getDart(u) == 0);
	    setGeToMap(u, d2);
		    
	    m_g.deleteNode(u2.point());
		  
	  }

	  return;
	}


      }
    }
	
  }

}

void
togerImpl::swap_twoNodes(const girl::point &p, int resn)
{

#ifdef VERBOSE
  std::cout<<"2 nodes ++++++++++++++++++++++++++++++++++++++++++++++++\n";
#endif      
  //we know that if there are 2 nodes then they do not move,
  // it is just their darts that change direction.

  enum dirBit {R=1, U=2, L=4, D=8};


  const bool x_y_N = (resn&0xf) != 0;
  const bool x_ym1_N = ((resn>>4)&0xf) != 0;
  const bool xm1_ym1_N = ((resn>>8)&0xf) != 0;
  const bool xm1_y_N = ((resn>>12)&0xf) != 0;

  girl::point pt;

  if (xm1_ym1_N) {

    if (x_ym1_N) { //two nodes on same line (y-1)
      const int x_ym1 = (resn>>4)&0xf; //x_ym1
      if (x_ym1&L) {
	pt.setX( p.x() - 1 ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, DOWN, RIGHT);
	pt.setX( p.x() ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, DOWN, LEFT);
      }
      else {
	assert(x_ym1&D);
	pt.setX( p.x() - 1 ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, RIGHT, DOWN);
	pt.setX( p.x() ); pt.setY( p.y() - 1);
	changeDartDirection(pt, LEFT, DOWN);
      }
    }
    else if (xm1_y_N) { //two nodes on same column (x-1)
      const int xm1_ym1 = (resn>>8)&0xf; //xm1_ym1
      if (xm1_ym1&D) {
	pt.setX( p.x() - 1 ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, RIGHT, DOWN);
	pt.setX( p.x() - 1 ); pt.setY( p.y() );
	changeDartDirection(pt, RIGHT, UP);
      }
      else {
	assert(xm1_ym1&R);
	pt.setX( p.x() - 1 ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, DOWN, RIGHT);
	pt.setX( p.x() - 1 ); pt.setY( p.y() );
	changeDartDirection(pt, UP, RIGHT);
      }
    }
    else { 
      assert(x_y_N); //diagonal (x-1; y-1)->(x; y)

      const int x_y = resn&0xf;
      girl::direction o1 = INVALID, o2 = INVALID;
      girl::direction n1 = INVALID, n2 = INVALID;

      if (x_y&L) { //x_y_L
	o1 = RIGHT;
	n1 = DOWN;
	o2 = UP;
	n2 = LEFT;
      }
      else {
	assert(x_y&U); //x_y_U
	o1 = DOWN;
	n1 = RIGHT;
	o2 = LEFT;
	n2 = UP;
      }
      assert(o1 != INVALID);
      assert(o2 != INVALID);
      assert(n1 != INVALID);
      assert(n2 != INVALID);
		  
      pt.setX( p.x() - 1 ); pt.setY( p.y() - 1);
      changeDartDirection(pt, o1, n1);
      pt.setX( p.x() ); pt.setY( p.y() );
      changeDartDirection(pt, o2, n2);
    }

  }
  else if (x_ym1_N) {

    if (x_y_N) { //two nodes on same column (x)
      const int x_y = resn&0xf;
      if (x_y&U) {
	pt.setX( p.x() ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, LEFT, DOWN);
	pt.setX( p.x() ); pt.setY( p.y() );
	changeDartDirection(pt, LEFT, UP);
      }
      else {
	assert(x_y&L);
	pt.setX( p.x() ); pt.setY( p.y() - 1 );
	changeDartDirection(pt, DOWN, LEFT);
	pt.setX( p.x() ); pt.setY( p.y() );
	changeDartDirection(pt, UP, LEFT);
      }
    }
    else {
      assert(xm1_y_N); //diagonal (x; y-1)->(x-1; y)
      const int x_ym1 = (resn>>4)&0xf;
      girl::direction o1 = INVALID, o2 = INVALID;
      girl::direction n1 = INVALID, n2 = INVALID;
      if (x_ym1&L) { //x_ym1_L
	o1 = DOWN;
	n1 = LEFT;
	o2 = RIGHT;
	n2 = UP;
      }
      else {
	assert(x_ym1&D);
	o1 = LEFT;
	n1 = DOWN;
	o2 = UP;
	n2 = RIGHT;
      }
      assert(o1 != INVALID);
      assert(o2 != INVALID);
      assert(n1 != INVALID);
      assert(n2 != INVALID);		  
	  
      pt.setX( p.x() ); pt.setY( p.y() - 1 );
      changeDartDirection(pt, o1, n1);
      pt.setX( p.x() - 1); pt.setY( p.y() );
      changeDartDirection(pt, o2, n2);
    }

  }
  else {
    //two nodes on same line (y-1)
    assert(x_y_N);
    assert(xm1_y_N);
	
    const int x_y = resn&0xf;
    if (x_y&L) {
      pt.setX( p.x() ); pt.setY( p.y() );
      changeDartDirection(pt, UP, LEFT);
      pt.setX( p.x() - 1 ); pt.setY( p.y() );
      changeDartDirection(pt, UP, RIGHT);
    }
    else { 
      assert(x_y&U);
      pt.setX( p.x() ); pt.setY( p.y() );
      changeDartDirection(pt, LEFT, UP);
      pt.setX( p.x() - 1 ); pt.setY( p.y() );
      changeDartDirection(pt, RIGHT, UP);
    }
  }


}

void
togerImpl::swap(const girl::point &p)
{
  const int resn = m_b.swap(p);
  assert(m_b.checkConsistency());
 
  if (resn != 0) {

    const int nbNodes = resn>>16;
    assert(nbNodes<3 && nbNodes >0);

    if (nbNodes == 1) {
      swap_oneNode(p, resn);	
    }
    else if (nbNodes == 2) {
      swap_twoNodes(p, resn);
    }
    
  }
    
  assert(checkConsistency());
}


//-----------------------------------------------------------------
//------ Insertion ------------------------------------------------

bool
togerImpl::insertPaths(const girl::pathDeque &paths,
		       girl::regionDeque &newRegions,
		       girl::regionDeque &updatedRegions)
{
  const bool doPendingEdgesSearch = true;
  const bool doNewRegionHook = true;

  return insertPaths(paths,
		     doPendingEdgesSearch,
		     doNewRegionHook,
		     newRegions, 
		     updatedRegions);
}

bool
togerImpl::insertPath(const girl::path &path, 
		      girl::regionDeque &newRegions,
		      girl::regionDeque &updatedRegions)
{
  girl::pathDeque paths;
  paths.push_back(path);
  
  return insertPaths(paths, newRegions, updatedRegions);
}

bool
togerImpl::insertContours(const girl::contourDeque &contours, 
			  girl::regionDeque &newRegions,
			  girl::regionDeque &updatedRegions)
{
  const bool doPendingEdgesSearch = false;
  const bool doNewRegionHook = true;

  girl::pathDeque paths;  
  for (girl::contourDeque::const_iterator it = contours.begin();
       it != contours.end();
       ++it) {
    paths.push_back(girl::path(*it));
  }

  return insertPaths(paths,
		     doPendingEdgesSearch,
		     doNewRegionHook,
		     newRegions, 
		     updatedRegions);
}

bool
togerImpl::insertContour(const girl::contour &contour, 
			 girl::regionDeque &newRegions,
			 girl::regionDeque &updatedRegions)
{
  girl::contourDeque contours;
  contours.push_back(contour);
  
  return insertContours(contours, newRegions, updatedRegions);
}



bool
togerImpl::insertPaths(const girl::pathDeque &paths,
		       bool doPendingEdgesSearch,
		       bool doNewRegionHook,
		       girl::regionDeque &newRegions,
		       girl::regionDeque &updatedRegions)
{
  ReducedPathCollection reducedPaths;
  
  const bool reductionOk = reducePaths(paths, reducedPaths);
  if (reductionOk) {
    
#ifdef VERBOSE_INSERT
    std::cerr<<"nb reduced paths = "<<reducedPaths.size()<<std::endl;
#endif
    
    if (reducedPaths.size() == 0)
      return true;
	   


    std::deque<girl::edge> facesEdges;
    
    insureNodesOnContacts(reducedPaths, facesEdges);

    setNodesOnSelfIntersections(reducedPaths);

    assert(reducedPaths.size() == facesEdges.size());


    bool hasPendingEdge = false;
    if (doPendingEdgesSearch)
      hasPendingEdge = pendingEdgeFinder().find(facesEdges);

    if (hasPendingEdge) {
#ifdef VERBOSE_INSERT
      std::cerr<<"-----------------> pending edge found !!!"<<std::endl;
#endif

      unsetNodesOnSelfIntersections(reducedPaths);
      unmarkPaths(reducedPaths);
      removeUnnecessaryNodes(reducedPaths);
      reducedPaths.clear();
      facesEdges.clear();
      assert(m_b.checkNoMark());
      assert(checkConsistency());

      return false;
    }
   
    girl::regionDeque labelsToReuse;
    girl::regionDeque regionsToSeek_new_q;
    //girl::regionDeque regionsToSeek_update_q;
    getLabelsToReuse(reducedPaths, facesEdges, labelsToReuse, updatedRegions);

    insertReducedPaths(reducedPaths, labelsToReuse, updatedRegions, regionsToSeek_new_q, newRegions);

    //seek regions in regionsToSeek_new_q AND add them to newRegions
    girl::regionDeque::const_iterator it = regionsToSeek_new_q.begin();
    girl::regionDeque::const_iterator itEnd = regionsToSeek_new_q.end();
    for ( ; it != itEnd; ++it) {
      girl::region rl = *it;

#ifdef VERBOSE_INSERT
      std::cerr<<"will seek "<<rl<<std::endl;
#endif

      seekInnerRegionsWithHoles(rl);
      
      girlAssert1(checkFacePositiveOrientation(rl));
      
      newRegions.push_back(rl);
    }

    //seek regions in updatedRegions
    it = updatedRegions.begin();
    itEnd = updatedRegions.end();
    for ( ; it != itEnd; ++it) {
      girl::region rl = *it;

#ifdef VERBOSE_INSERT
      std::cerr<<"will seek "<<rl<<std::endl;
#endif

      seekInnerRegionsWithHoles(rl);
      
      girlAssert1(checkFacePositiveOrientation(rl));
      
    }
    
    if (doNewRegionHook)
      applyNewRegionHook(newRegions, updatedRegions);

    girlAssert1(checkConsistency());
    return true;
  }
#ifdef VERBOSE_INSERT
  else
    std::cerr<<"paths reduction failed"<<std::endl;
#endif
  
  return false;
}


/**
 * @brief Builds reduced paths from given paths @a paths.
 *
 * @a reducedPaths is filled with paths build from @a paths
 * that start and stop on boundary points in bitmap, 
 * or with closed contours.
 *
 * We forbid intersections between different inserted paths.
 * Besides, we allow intersection within a path only if it is on the last pt,
 * or if it does not create new regions.
 *
 * Reduced paths are marked in bitmap !
 * Start and end points of reduced paths are not marked.
 *
 * @a facesEdges is filled with pair of faces corresponding to 
 * path extremities :
 * first face of pair is for the path start, second face is for the path end.
 * Paths intersecting no boundaries have 0 for first and second face.
 *
 * @a selfIntersectionPts is filled with self intersection points 
 * corresponding to each path.
 *
 * @return true if @a reducedPaths are valid and so can be inserted,
 * false if not.
 *
 * @warning If true is returned, @a reducedPaths are still marked in 
 * bitmap and need to be unmarked.
 *
 * We forbid intersection between @a paths, with 2 exceptions :
 * - parts of path superimposed on existing boundaries in bitmap are not 
 * considered (so there can be an intersection there)
 * - perfect loops, i.e. path self-intersecting on their last/first point, are 
 * accepted.
 *
 * If @a paths are all super-imposed on existing paths i ntoger, there is nothing to insert,
 * true wille be returned !
 * 
 */
bool
togerImpl::reducePaths(const girl::pathDeque &paths,
		       ReducedPathCollection &reducedPaths)
{
  bool insertionPossible = true; //return value

  bool allPathsAlreadyExist = true; //tells if all @a paths are super-imposed on existing paths

  girl::pathDeque::const_iterator itEnd = paths.end();
  girl::pathDeque::const_iterator it = paths.begin();
  for ( ; it != itEnd; ++it) {
    const girl::path &pth = *it;
    
#ifdef VERBOSE_INSERT
    std::cerr<<" path: startPt=("<<pth.startPt().x()<<"; "<<pth.startPt().y()<<") endPt=("<<pth.endPt().x()<<"; "<<pth.endPt().y()<<") length="<<pth.length()<<std::endl;
#endif

    bool pathStartedFromContourPoint = false;

    ReducedPath pendingPath;
    ReducedPath currPath;

    bool onContour = false; //tells if we are on a contour point in bitmap
    
    girl::path::const_dir_iterator itD = pth.begin();
    const girl::path::const_dir_iterator itDEnd = pth.end();    
    girl::bitmap::iterator itB = m_b.getIterator(pth.startPt());

    for ( ; itD != itDEnd; ++itD) {
      
      //DEBUG
      //const girl::point tmppt = itB.getPosition();
      //std::cerr<<"itB: ("<<tmppt.x()<<", "<<tmppt.y()<<")   nextDir="<<*itD<<" ; "; 

      const girl::direction nextDir = *itD;
      const bool res = addPtToReducePaths(itB, nextDir, currPath, pendingPath,
					  onContour, pathStartedFromContourPoint, 
					  reducedPaths);

      allPathsAlreadyExist &= onContour;

      if (! res) {

#ifdef VERBOSE_INSERT
	std::cerr<<"reducePaths failed at ("<<itB.getPosition().x()<<"; "<<itB.getPosition().y()<<") nextDir="<<nextDir<<std::endl;
#endif
	insertionPossible = false;
	break;
      }
      
    }
    if (! insertionPossible) 
      break;
    
    const girl::direction nextDir(girl::INVALID);
    const bool res = addPtToReducePaths(itB, nextDir, currPath, pendingPath,
					onContour, pathStartedFromContourPoint, 
					reducedPaths);

#ifdef VERBOSE_INSERT
    if (pendingPath.path.length()!=0) {
      std::cerr<<"pendingPath.path.length()="<<pendingPath.path.length()<<" currPath.path.length()="<<currPath.path.length()<<std::endl;
    }
#endif

    assert(pendingPath.path.length()==0);
    if (! res) {
#ifdef VERBOSE_INSERT
      std::cerr<<"reducePaths failed for path at ("<<itB.getPosition().x()<<"; "<<itB.getPosition().y()<<") = end of path."<<std::endl;
#endif
      insertionPossible = false;
      break;
    }

  }



  //B: we could unmark reduced paths here !?
  //but not between each path reduction, because 
  // we need it to test for intersections !!!?
  

  if (! paths.empty() && reducedPaths.empty() && ! allPathsAlreadyExist) {

#ifdef VERBOSE_INSERT
    std::cerr<<"reducePaths failed :  (! paths.empty() && reducedPaths.empty())  => paths.size()="<<paths.size()<<" reducedPaths.size()="<<reducedPaths.size()<<std::endl;
#endif

    insertionPossible = false;
  }

  //   if (needPendingEdgeSearch) {
  //     assert(insertionPossible == true);
    
  //     pendingEdgeFinder finder;
  //     const bool hasPendingEdge = finder.find(faceEdges);
  //     if (hasPendingEdge) {
  //       unmarkPaths(reducedPaths);
  //       reducedPaths.clear();
  //       facesEdges.clear();
  //       assert(checkNoMark());
  //       insertionPossible = false;
  //     }
  //   }
  
  assert(insertionPossible == true || reducedPaths.size() == 0);
  girlAssert1(check_onlyMarkedPaths(reducedPaths));
  girlAssert1(check_pathsIntersections(reducedPaths));

#ifdef VERBOSE_INSERT
  std::cerr<<"togerImpl::reducePaths()   reducedPaths.size()="<<reducedPaths.size()<<std::endl;
#endif

  return insertionPossible;
}


  
/**
 * @brief Add a point to build reduced paths
 *
 * @param itB  iterator at current point on bitmap. It will be moved according to @a nextDir.
 * @param nextDir is the next direction for @a currPath. 
 *  It will be girl::invalid for the last point of path.
 * @param currPath  is the current built path.
 * @param pendingPath  is the pending path
 * @param onContour  tells if we are on an existing contour (in bitmap)
 * @param s1   is the extremity from where current path was started (if it is valid).
 * @param reducedPaths  is the collection of found reduced paths.
 *
 * start and end points of reduced paths are not marked.
 */
bool
togerImpl::addPtToReducePaths(girl::bitmap::iterator &itB, girl::direction nextDir, 
			      ReducedPath &currPath, ReducedPath &pendingPath,
			      bool &onContour, 
			      bool &pathStartedFromContourPoint, 
			      ReducedPathCollection &reducedPaths)
{
  
  if (itB.isMarked()) {
    //point already marked in bitmap so there is an intersection 
    //within this path or between paths.
    
#ifdef POSITIVE_DARTS_ONLY_INSERTION
    //NEW --------------------------------------------------------------------
    if (currPath.path.length() == 0) {
      //start from a marked point 
      
      assert(pendingPath.path.length() == 0);

      //(for example, second loop of two loops starting at the same point)

      assert(nextDir != girl::INVALID);
      if (itB.hasNeighbourInDirection(nextDir)) {
	//path will continue on a contour
	
	//unmarkPath(pendingPath.path);
	//unmarkPath(currPath.path);
	unmarkPaths(reducedPaths);
	reducedPaths.clear();
	return false;
      }
      else {
	//a new path will start
	onContour = false;
	pathStartedFromContourPoint = true;
	currPath.path = path(itB.getPosition());
	currPath.path.add(nextDir);

	//ensure node at start of this path !
	itB.setNode(); //????????

      }

      assert(nextDir != girl::INVALID);
      itB.moveInDirection(nextDir);

    }
    else 
      //NEW end ----------------------------------------------------------------
#endif
      if (currPath.path.endPt() == currPath.path.startPt()
	  && nextDir == girl::INVALID) {
	//loop without any intersection -> no possible pending edge
	reducedPaths.push_back(currPath);
      }
      else if (currPath.path.endPt() == pendingPath.path.startPt()
	       && nextDir == girl::INVALID) {
	//loop with intersection(s) -> no possible pending edge
	currPath.path.append(pendingPath.path);
	currPath.selfIntersects.insert(currPath.selfIntersects.end(),
				       pendingPath.selfIntersects.begin(), pendingPath.selfIntersects.end());
	reducedPaths.push_back(currPath);
	pendingPath = ReducedPath(); //reset
      }
      else if (nextDir != girl::INVALID
	       && currPath.path.belongs(itB.getPosition()) //self intersection //Warning: O(N) !!! [*]
	       && (*currPath.path.rbegin()).opposite().previous() == nextDir) { //currPath.path.rbegin()==prevDir
	//suitable inner loop, inside contour
	  
	// [*] : as we currently have only one mark,
	// we cannot distinguish between intersections with a previous path
	// and self intersection.

	assert(itB.isMarked());
	assert(! itB.isContourPoint());

	currPath.selfIntersects.push_back(itB.getPosition());

	currPath.path.add(nextDir);	
	//itB.mark(); //unnecessary, already marked
	itB.moveInDirection(nextDir);
      }
#ifdef POSITIVE_DARTS_ONLY_INSERTION
    //NEW --------------------------------------------------------------------------------
      else if (nextDir == girl::INVALID
	       && itB.getPosition() == currPath.path.endPt()) {
	//last point of current segment

	//ensure node at start of this path !
	//itB.setNode();
	assert(itB.isNode()); //????????

	reducedPaths.push_back(currPath);

      }
    //NEW end --------------------------------------------------------------------------------
#endif
      else {
#ifdef VERBOSE_INSERT
	std::cerr<<"nextDir="<<nextDir<<std::endl;
	std::cerr<<"currPath.path.startPt=("<<currPath.path.startPt().x()<<"; "<<currPath.path.startPt().y()<<") endPt="<<currPath.path.endPt().x()<<"; "<<currPath.path.endPt().y()<<")  length="<<currPath.path.length()<<std::endl;
	std::cerr<<"pendingPath.path.startPt=("<<pendingPath.path.startPt().x()<<"; "<<pendingPath.path.startPt().y()<<") endPt="<<pendingPath.path.endPt().x()<<"; "<<pendingPath.path.endPt().y()<<")  length="<<pendingPath.path.length()<<std::endl;
	std::cerr<<"---> !!!  intersection at ("<<itB.getPosition().x()<<", "<<itB.getPosition().y()<<") nextDir="<<nextDir<<std::endl;
	if (currPath.path.length() > 0)
	  std::cerr<<"   currPath.path.belongs(itB.getPosition())="<<currPath.path.belongs(itB.getPosition())<<" (*currPath.path.rbegin()).opposite().previous()="<<(*currPath.path.rbegin()).opposite().previous()<<" =?= nextDir="<<nextDir<<std::endl;
	else 
	  std::cerr<<"   currPath.path.belongs(itB.getPosition())="<<currPath.path.belongs(itB.getPosition())<<std::endl;
#endif

	unmarkPath(pendingPath.path);
	unmarkPath(currPath.path);
	unmarkPaths(reducedPaths);
	reducedPaths.clear();
	return false;
      }
    
    
  }
  else {
    //point not yet marked in bitmap

    if (itB.isContourPoint()) {

      if (! onContour) { //we just hit a contour point

	if (currPath.path.length() > 0) {

	  if (pathStartedFromContourPoint) {
	    //we end an already started path
	    
	    reducedPaths.push_back(currPath);
	  }
	  else {
	    //curent path was not started on a contour point
	    //so it is the pending path
	    
	    assert(pendingPath.path.length() == 0);
	    pendingPath = currPath;
	  }	
	  currPath = ReducedPath(); //reset
	  assert(currPath.path.length()==0);
	}  
	 
      }
      //DEBUG
      else {
	//onContour
	assert(currPath.path.length() == 0);
      }

      if (nextDir != girl::INVALID) {
	assert(currPath.path.length() == 0);
	if (itB.hasNeighbourInDirection(nextDir)) {
	  //path will continue on a contour
	  onContour = true;
	  //pathStartedFromPoint = false;
	}
	else {
	  //a new path will start
	  onContour = false;
	  pathStartedFromContourPoint = true;
	  currPath.path = path(itB.getPosition());
	  currPath.path.add(nextDir);
	}
      }
      
    }
    else {
      //not on a contour point
      
      if (nextDir != girl::INVALID) {
	if (currPath.path.length() == 0) {
	  currPath.path = path(itB.getPosition());
	}
	currPath.path.add(nextDir);
	itB.mark();
      }
      else {
	if (currPath.path.length() > 0) {
	  //pending path
	  unmarkPath(currPath.path);
	}

      }
    }

    if (nextDir != girl::INVALID)
      itB.moveInDirection(nextDir);
    
  }

  return true;
}


/**
 * Checks that only paths in @paths are marked in bitmap.
 */
bool
togerImpl::check_onlyMarkedPaths(const ReducedPathCollection &paths)
{
  //impl: cannot be 'const'

  unmarkPaths(paths);
  const bool res = m_b.checkNoMark();
  markPaths(paths);
  return res;
}

/**
 * Checks that only path start and end intersect a contour in bitmap.
 *
 * Also valid for closed path !
 */
bool
togerImpl::check_pathIntersections(const path &p) const
{
  assert(p.length() > 0);
  
  const bool closedPath = p.isClosed();

  bitmap::const_iterator itBe = m_b.getIterator(p.endPt());
  if (! itBe.isContourPoint() && ! closedPath)
    return false;

  bitmap::const_iterator itB = m_b.getIterator(p.startPt());
  if (! itB.isContourPoint() && ! closedPath)
    return false;
  path::const_dir_iterator itD = p.begin();
  path::const_dir_iterator itDEnd = p.end();
  itB.moveInDirection(*itD);
  ++itD;
  for ( ; itD != itDEnd; ++itD) {
    if (itB.isContourPoint())
      return false;
    itB.moveInDirection(*itD);
  }
  return true;
}

/**
 * Checks that only paths start and end intersect a contour in bitmap.
 *
 * Also valid for closed paths !
 */
bool
togerImpl::check_pathsIntersections(const ReducedPathCollection &paths) const
{
  for (ReducedPathCollection::const_iterator it = paths.begin();
       it != paths.end(); ++it)
    if (! check_pathIntersections(it->path))
      return false;
  return true;
}

/**
 * unmark all paths from @a paths in bitmap.
 */
void
togerImpl::unmarkPaths(const ReducedPathCollection &paths)
{
  for (ReducedPathCollection::const_iterator it = paths.begin();
       it != paths.end(); ++it)    
    unmarkPath(it->path);
}

/**
 * unmark path @a p in bitmap.
 * 
 * Does not unmark start and end points for path starting on a contour point. 
 * Does not unmark end pt for path not starting on a contour point 
 * (they are necessarily closed loops without intersections and so 
 * the last point is the first one and has already been unmarked)
 */
void
togerImpl::unmarkPath(const girl::path &p)
{
  if (p.length() == 0)
    return;

  girl::bitmap::iterator itB = m_b.getIterator(p.startPt());
  girl::path::const_dir_iterator it = p.begin();
  const girl::path::const_dir_iterator itEnd = p.end();

  if (! itB.isContourPoint()) {
    assert(itB.isMarked());
    itB.unmark();
  }
  itB.moveInDirection(*it);
  ++it;
  for ( ; it != itEnd; ++it) {
    itB.unmark();
    itB.moveInDirection(*it);
  }

  /*
    if (itB.isContourPoint()) {
    itB.moveInDirection(*it);
    ++it;
    }
    for ( ; it != itEnd; ++it) {
    itB.unmark();
    itB.moveInDirection(*it);
    }
  */
}

/**
 * mark all paths from @a paths in bitmap.
 */
void
togerImpl::markPaths(const ReducedPathCollection &paths)
{
  for (ReducedPathCollection::const_iterator it = paths.begin();
       it != paths.end(); ++it)    
    markPath(it->path);
}

/**
 * unmark path @a p, except start and end points, in bitmap.
 *
 * Does not mark start and end points for path starting on a contour point. 
 * Does not mark end pt for path not starting on a contour point 
 * (they are necessarily closed loops without intersections and so 
 * the last point is the first one and has already been marked)
 */
void
togerImpl::markPath(const girl::path &p)
{
  //code duplication with unmarkPath

  if (p.length() == 0)
    return;

  girl::bitmap::iterator itB = m_b.getIterator(p.startPt());
  girl::path::const_dir_iterator it = p.begin();
  const girl::path::const_dir_iterator itEnd = p.end();


  if (! itB.isContourPoint()) {
    itB.mark();
  }
  itB.moveInDirection(*it);
  ++it;
  for ( ; it != itEnd; ++it) {
    itB.mark();
    itB.moveInDirection(*it);
  }

  /*
    if (itB.isContourPoint()) {
    itB.moveInDirection(*it);
    ++it;
    }
    for ( ; it != itEnd; ++it) {
    itB.mark();
    itB.moveInDirection(*it);
    }
  */
}

void
togerImpl::insureNodesOnContacts(const ReducedPathCollection &reducedPaths, 
				 std::deque<girl::edge> &facesEdges)
{
  DartUnorderedSet alreadyMovedArtificialNodes; 
  //store artificial node (actually one of the two dart) moved by paths,
  // so we do not move the same node several times for different paths.

  ReducedPathCollection::const_iterator itEnd = reducedPaths.end();
  ReducedPathCollection::const_iterator it = reducedPaths.begin();
  for ( ; it != itEnd; ++it) {  
    insureNodesOnContacts(it->path, facesEdges, alreadyMovedArtificialNodes);
  }
}

/**
 * Insure there are nodes at path @a p extremities,
 * if @a p is not a loop without intersection.
 *
 * Moved artificial nodes positive dart are added to @a alreadyMovedArtificialNodes.
 * Edges between different faces are added to @a facesEdges.
 */
void
togerImpl::insureNodesOnContacts(const girl::path &p,
				 std::deque<girl::edge> &facesEdges,
				 DartUnorderedSet &alreadyMovedArtificialNodes)
{
  girl::point startPt = p.startPt();
  girl::direction startDir = *(p.begin());

  bitmap::iterator itB = m_b.getIterator(startPt);
  assert(! itB.hasNeighbourInDirection(startDir));
  if (itB.isContourPoint()) {
    //path is not a loop without intersection

    const girl::dart d1 = insureNodeAtHead(itB, startDir, alreadyMovedArtificialNodes);
    const girl::face f1 = m_m.getLambda(d1);

    girl::point endPt = p.endPt();
    if (endPt != startPt) {
      bitmap::iterator itBb = m_b.getIterator(endPt);
      assert(itBb.isContourPoint());
      
      girl::direction oendDir = (*p.rbegin()).opposite();
      assert(! itBb.hasNeighbourInDirection(oendDir));
      
      const girl::dart d2 = insureNodeAtHead(itBb, oendDir, alreadyMovedArtificialNodes);

      facesEdges.push_back(girl::edge(f1, m_m.getLambda(d2)));
    }
    else {
      facesEdges.push_back(girl::edge(f1, f1));
    }
  }
  else {
    facesEdges.push_back(girl::edge(0, 0));
  }
}

/**
 * Insure node at head, described by position of @a it and direction @a dir.
 *
 * Moved artificial node positive dart is added to @a alreadyMovedArtificialNodes.
 */
girl::dart
togerImpl::insureNodeAtHead(const girl::bitmap::iterator &it, 
			    girl::direction dir,
			    DartUnorderedSet &alreadyMovedArtificialNodes)
{
  
  girl::geoDart startHead;
  girl::dart d;
  girl::direction prevDir;
  girl::direction nextDir;

  getDartForHead(it.getPosition(), dir, 
		 prevDir, nextDir, d, startHead);

  bool isFirstNodeArtificial = m_m.nodeIsArtificial(d);
  if (isFirstNodeArtificial) {
    if (alreadyMovedArtificialNodes.find(abs(d)) != alreadyMovedArtificialNodes.end()) {
      isFirstNodeArtificial = false;
    }
    else {
      alreadyMovedArtificialNodes.insert(abs(d));
    }
  }

  insureNode(it, isFirstNodeArtificial, startHead, d, prevDir, nextDir);

  return d;
}

/**
 * Get dart of face from which start the head,
 * defined by @a headPt & @a headDir.
 *
 * @param headPt  point of new head
 * @param headDir direction of new head
 * @param prevDir returned previous direction on contour to get to @a headPt
 * @param nextDir returned next direction on contour from @a headPt
 * @param d       returned found dart
 * @param u       returned found geoDart
 */
void
togerImpl::getDartForHead(girl::point headPt, girl::direction headDir,
			  girl::direction &prevDir, girl::direction &nextDir,
			  girl::dart &d, girl::geoDart &u)
{
  girl::bitmap::iterator itB = m_b.getIterator(headPt);
  assert(itB.isContourPoint());

  d = 0;

  if (itB.isNode()) {
    nextDir = headDir.previous();
#ifdef NDEBUG
    for ( ; ! itB.hasNeighbourInDirection(nextDir); nextDir = nextDir.previous())
      ;
#else
    for ( ; ! itB.hasNeighbourInDirection(nextDir) && nextDir != headDir; nextDir = nextDir.previous())
      ;
    assert(nextDir != headDir);
#endif

    prevDir = headDir.next();
#ifdef NDEBUG
    for ( ; ! itB.hasNeighbourInDirection(prevDir); prevDir = prevDir.next())
      ;
#else
    for ( ; ! itB.hasNeighbourInDirection(prevDir) && prevDir != headDir; prevDir = prevDir.next())
      ;
    assert(prevDir != headDir);
#endif
    prevDir = prevDir.opposite();
    u.set(headPt, nextDir);
    d = m_g.getDart(u);
  }
  else {
    
    m_b.contourTracking(headPt, u);
    d = m_g.getDart(u);
    
    itB = m_b.getIterator(u.point());
    itB.moveInDirection(u.dir());
    prevDir = u.dir();

    while (itB.getPosition() != headPt) {
      assert(! itB.isNode());
      
      prevDir = itB.getNextDirection(prevDir);
      itB.moveInDirection(prevDir);
    }
    nextDir = itB.getNextDirection(prevDir);
    girl::direction dir = prevDir.opposite();
    dir = dir.previous();
  
    for (int i = 2; i; --i) {
      if (itB.hasNeighbourInDirection(dir)) {
	//we found a neighbour before finding headDir
	//  so we were following the face from the wrong dart
	assert(dir != headDir);
	assert(dir == nextDir);
	d = m_m.alpha(d);
	m_m.getGe(d, u);
	nextDir = prevDir.opposite();
	prevDir = dir.opposite();
	break;
      }
      else {
	if (dir == headDir)
	  break;
      }
      assert(dir != headDir);
      dir = dir.previous();
    }
  
  }

}


void
togerImpl::setNodesOnSelfIntersections(const ReducedPathCollection &reducedPaths)
{
  ReducedPathCollection::const_iterator itC = reducedPaths.begin();
  const ReducedPathCollection::const_iterator itCEnd = reducedPaths.end();
  for ( ; itC != itCEnd; ++itC) {

    PtsCollection::const_iterator it = itC->selfIntersects.begin();
    const PtsCollection::const_iterator itEnd = itC->selfIntersects.end();
    for ( ; it != itEnd; ++it) {
      
      m_b.setNode(*it);

    }
  }
}

void
togerImpl::unsetNodesOnSelfIntersections(const ReducedPathCollection &reducedPaths)
{
  ReducedPathCollection::const_iterator itC = reducedPaths.begin();
  const ReducedPathCollection::const_iterator itCEnd = reducedPaths.end();
  for ( ; itC != itCEnd; ++itC) {

    PtsCollection::const_iterator it = itC->selfIntersects.begin();
    const PtsCollection::const_iterator itEnd = itC->selfIntersects.end();
    for ( ; it != itEnd; ++it) {
      
      m_b.unsetNode(*it);

    }
  }
}




/**
 * Get region labels to reuse in @a labelsToReuse, 
 * get regions to search for possible holes/daugther in regionsToSeek_q
 *
 * Touched regions are kept in @a labelsToReuse in order to be reused when
 * regions are relabelled..
 *
 * Fills @a regionsToSeek_q with regions that are not touched
 * but may contain new infinite face and so will have to be sought.
 *
 */
void
togerImpl::getLabelsToReuse(const ReducedPathCollection &reducedPaths,
			    const std::deque<girl::edge> &faceEdges,
			    girl::regionDeque &labelsToReuse,
			    girl::regionDeque &regionsToSeek_q)
{
  //Edges between infinite faces (different or the same)
  // can build new infinite faces for which mother 
  // would not be necessary scanned. 
  // So we add their mother to @a regionstoSeek.
  // However, if there is an edge between a region and an infinite face,
  // the region has not to be scanned (the infinite face cannot build a 
  // new infinite face) and must not be scanned (because the region will 
  // probably have its name changed). 
  //
  //We also keep labels of regions to reuse them (path insertion can only 
  // produce more regions) and free labels of touched infinite faces from 
  // map (it updates inclusion relation), 

  assert(reducedPaths.size() == faceEdges.size());

  FaceUnorderedSet labelsToReuseH;
  FaceUnorderedSet regionsToSeek;
  FaceUnorderedSet regionsToNotSeek;

  ReducedPathCollection::const_iterator itP = reducedPaths.begin();
  std::deque<girl::edge>::const_iterator it = faceEdges.begin();
  const std::deque<girl::edge>::const_iterator itEnd = faceEdges.end();
  for ( ; it != itEnd ; ++it, ++itP) {
    const girl::face f1 = it->first;
    const girl::face f2 = it->second;

    if (f1 == 0) {
      //this a loop without intersection
      //we must keep its face to find it.
      assert(f2 == 0);
      const girl::path &p = itP->path;
      assert(p.startPt() == p.endPt());
      
      girl::region r = findRegion(p.startPt());
      assert(r != 0);
      regionsToSeek.insert(r);

      //TODO:OPTIM:
      // For loop without intersection, we will scan their "pseudo-mother" 
      //(the mother find here is not necessarily their final mother).
      // We could probably spare this scan... 
      // We would have to :
      // - deal first with paths concerning loops without intersections
      // - keep all their "pseudo-mother" in a list
      // - for each such path : insert it (to get their new label), attach it 
      // to its "pseudo-mother",
      // puts this "pseudo-mother" in the list of final modified regions but 
      // not to regions to scan
      // (obviously the internal region of the infinite face would be scanned)
      // - we would finally have to check uniqueness of regions in the list of 
      // final modified regions.


    }
    else {
      
      labelsToReuseH.insert(f1);
      labelsToReuseH.insert(f2);
      
      if (m_m.faceIsInfinite(f1)) {
	if (f2 != f1) {
	  if (m_m.faceIsInfinite(f2)) {
	    assert(m_m.faceGetMother(f1) == m_m.faceGetMother(f2));
	    
	    regionsToSeek.insert(m_m.faceGetMother(f1));
	  }
	  else {
	    //f1 infinite, not f2
	    
	    assert(m_m.faceGetMother(f1) == f2);
	    
	    regionsToNotSeek.insert(f2);
	  }
	  
	}
	else {
	  //f2==f1 infinite
	  
	  regionsToSeek.insert(m_m.faceGetMother(f1));
	}
      }
      else if (m_m.faceIsInfinite(f2)) {
	assert(m_m.faceGetMother(f2) == f1);
	
	regionsToNotSeek.insert(f1);
      }
    }

  }

  //- remove regionsToNotSeek from regionsToSeek
  FaceUnorderedSet::iterator itH = regionsToNotSeek.begin();
  FaceUnorderedSet::iterator itHEnd = regionsToNotSeek.end();
  for ( ; itH != itHEnd; ++itH) {
    regionsToSeek.erase(*itH);
  }

  //- remove labelsToReuseH from regionsToSeek 
  // (they will be re-added later, when the label is reused)
  FaceUnorderedSet::iterator itH2 = labelsToReuseH.begin();
  FaceUnorderedSet::iterator itH2End = labelsToReuseH.end();
  for ( ; itH2 != itH2End; ++itH2) {
    regionsToSeek.erase(*itH2);
  }
  //B: ok ???



  for (itH= regionsToSeek.begin(), itHEnd = regionsToSeek.end();
       itH != itHEnd; ++itH) 
    regionsToSeek_q.push_back(*itH);


#ifdef VERBOSE_INSERT
  {
    girl::regionDeque::const_iterator itD = regionsToSeek_q.begin();
    const girl::regionDeque::const_iterator itDEnd = regionsToSeek_q.end();
    for ( ; itD!= itDEnd; ++itD) {
      if (labelsToReuseH.find(*itD) != labelsToReuseH.end()) {
	std::cerr<<"Error: region "<<*itD<<" is in the list of regions to seek and also on labels which will be freed !"<<std::endl;
	exit(11);
      }
    }
  }
#endif

  for (itH= labelsToReuseH.begin(), itHEnd = labelsToReuseH.end();
       itH != itHEnd; ++itH) {
    const girl::face f = *itH;
    if (! m_m.faceIsInfinite(f)) {
      labelsToReuse.push_back(f);
    }
    else {
      m_m.faceRemove(f);
    }
  }

#ifdef VERBOSE_INSERT
  std::cerr<<labelsToReuse.size()<<" labels of regions to reuse : "<<std::endl<<"  "; 
  for (girl::regionDeque::const_iterator itD = labelsToReuse.begin();
       itD != labelsToReuse.end(); ++itD)
    std::cerr<<*itD<< " ";
  std::cerr<<std::endl;

  std::cerr<<regionsToSeek_q.size()<<" regions to seek (because internal infinite faces linked)  : "<<std::endl<<"  "; 
  for (girl::regionDeque::const_iterator itD = regionsToSeek_q.begin();
       itD != regionsToSeek_q.end(); ++itD)
    std::cerr<<*itD<< " ";
  std::cerr<<std::endl;

#endif

  
  //TODO:OPTIM:
  //Edges that have the same extremities are inside a region or outside an 
  // infintie face.
  //Edges inside a region split this region.
  //If there is only one edge splitting a region, i.e., the region is splitted 
  // in two new regions, 
  // we could only scan one of these two new regions. We would eventually find 
  // daugthers of original region
  // to reparent but that's all !


}


/**
 * Inserts all reduced paths from @a paths.
 *
 * Empties @a labelsToReuse
 * Fills @a regionsToSeek_update_q with regions to update.
 * Fills @a regionsToSeek_new_q with newly created regions to seek.
 * Fills @a out_q with newly created regions (too small to be sought).
 */
void
togerImpl::insertReducedPaths(const ReducedPathCollection &paths,
			      girl::regionDeque &labelsToReuse,
			      girl::regionDeque &regionsToSeek_update_q,
			      girl::regionDeque &regionsToSeek_new_q,
			      girl::regionDeque &out_q)
{
  std::deque<girl::dart> dart_q;

  ReducedPathCollection::const_iterator itP = paths.begin();
  const ReducedPathCollection::const_iterator itPEnd = paths.end();
  
  for ( ; itP != itPEnd ; ++itP)
    insertReducedPath(*itP, dart_q, regionsToSeek_new_q, out_q);

  labellingFacesForPaths(dart_q, labelsToReuse, regionsToSeek_update_q, regionsToSeek_new_q);
}

/**
 * Inserts reduced path @a p
 *
 *
 * Updates representation (bitmap, map, ge2to).
 *
 * Adds created darts to @a dart_q. 
 * Regions with these darts will have to be labelled.
 *
 * Adds regions that are loops without intersections 
 *  - directly in out_q if their length < 12
 *  - else directly in regionsToSeek_q 
 * (their darts are already labelled by lambda).
 *
 */
void
togerImpl::insertReducedPath(const ReducedPath &p,
			     std::deque<girl::dart> &dart_q,
			     girl::regionDeque &regionsToSeek_q,
			     girl::regionDeque &out_q)
{

  girl::bitmap::iterator itB = m_b.getIterator(p.path.startPt());

  if (! itB.isContourPoint()
      && p.selfIntersects.size() == 0) {
    
    //loop, without self intersection points

    insertLoopWithoutContact(p.path, regionsToSeek_q, out_q);

  }
  else {
    //path with intersectons
    //with borders or self

    //- go to the first node (keep number of dirs)
    girl::path::const_dir_iterator it = p.path.begin();
    const girl::path::const_dir_iterator itEnd = p.path.end();
    size_t nbDirsBeforeNode = 0;
    for ( ; it != itEnd; ++it, ++nbDirsBeforeNode) {
      if (itB.isNode())
	break;
      itB.moveInDirection(*it);
    }
    
    assert(itB.isNode());

    girl::geoDart u0(itB.getPosition(), *it);
    itB.writeNeighbourInDirection(*it, true);
    itB.moveInDirection(*it);
    girl::direction prevDir = *it;
    ++it;

    for ( ; it != itEnd; ++it) {
      
      if (itB.isNode()) {
	
	girl::geoDart u(itB.getPosition(), prevDir); //warning u.dir() points towards u.point().
	
	const girl::dart d = insertDartForReducedPath(itB, u0, u);

	dart_q.push_back(d);
#ifndef GIRL_ONLYD
	//std::cerr<<"stack alpha d"<<std::endl;
	dart_q.push_back(m_m.alpha(d));
#endif

	u0 = girl::geoDart(itB.getPosition(), *it);
	  
      }
      else {
	//no a node
	assert(itB.isMarked());

	itB.unmark();
      }

      //itB.unmark();
      itB.writeNeighbourInDirection(*it, true);
      itB.moveInDirection(*it);
      prevDir = *it;
      
    }

    //unmark & trace path to first node
    if (nbDirsBeforeNode > 0) {
      it = p.path.begin();
      itB = m_b.getIterator(p.path.startPt());
      for (size_t i=0; i<nbDirsBeforeNode; ++i) {
	itB.writeNeighbourInDirection(*it, true);
	prevDir = *it;
	itB.moveInDirection(*it);
	itB.unmark();
	++it;
      }
    }

    //deal with last point 
    if (itB.isNode()) {
      girl::geoDart u(itB.getPosition(), prevDir); //warning u.dir() points towards u.point().
	
      const girl::dart d = insertDartForReducedPath(itB, u0, u);

      dart_q.push_back(d);
#ifndef GIRL_ONLYD
      //std::cerr<<"stack alpha d"<<std::endl;
      dart_q.push_back(m_m.alpha(d));
#endif
    }

  }

}


girl::dart
togerImpl::insertDartForReducedPath(girl::bitmap::iterator &itB,
				    girl::geoDart &u0, girl::geoDart &u)
{
  //some kind of code duplication with buildBoundaries()
  
  girl::dart d = 0;
  if (itB.isMarked()) {    //Node && marked implies inner loop !
    
    if (u0.point() == u.point()) {
      //we unmarked this point but did not pass through,
      //we may have started on this point 
      assert(itB.getPointCardinality()==2);
      d = m_m.insertCycle(); 
      //and not insertMap : we do not want that d be labelled by a new lambda
      //indeed, even if we unlabel dart afterwards, the allocated face would not be freed !

      //d = buildTangentFace(u0, u);
    }
    else {

      if (m_b.getIterator(u0.point()).getPointCardinality() == 1) {
	// node at u0 does not have any dart yet 
	d = m_m.insertSingleDart();

      }	
      else if (m_g.getNbDarts(u.point()) > 0) {
	//this is a node where we added a single dart
	assert(m_g.getNbDarts(u.point()) == 1); //???
	assert(m_g.getNbDarts(u0.point()) > 0);
	d = buildSegmentBetweenExistingNodes(u0, u);

      }
      else {
	
	d = buildNewSegment(u0, u);

      }
    }

    itB.unmark();
  }
  else if (u0.point() == u.point()) {
    d = buildTangentFace(u0, u);

    itB.unmark();
  }
  else {
    d = buildSegmentBetweenExistingNodes(u0, u);

    itB.unmark();
  }
  m_m.clearEdgeFaces(d);

  setGeToMap(u0, d);
  setGeToMap(girl::geoDart(u.point(), u.dir().opposite()), m_m.alpha(d));
	
  girlAssert1(checkPtNodeConsistency(u0.point())); //???
  girlAssert1(checkPtNodeConsistency(u.point()));  //???
  
  assert(d);
  assert(lambda(d)==0);
  assert(lambda(m_m.alpha(d))==0);

  return d;
}



/**
 * Insert path @a p that do not intersect any boundary in bitmap.
 *
 * Will add created region to @a regionsToSeek_q or out_q.
 * If its length < 12, region has not to be scanned and is added to out_q.
 * Else region has to be scanned and is added to regionsToSeek_q.
 */
void
togerImpl::insertLoopWithoutContact(const girl::path &p,
				    girl::regionDeque &regionsToSeek_q,
				    girl::regionDeque &out_q)
{
  assert(p.length() > 0);
  assert(p.startPt() == p.endPt());
  assert(check_pathIntersections(p));

  traceAndUnmarkPathInBitmap(p);

  const girl::dart d = insertArtificialNode(p.startPt(), *(p.begin()), (*(p.rbegin())).opposite());
  //this new cycle is labelled by lambda !

  //[we do not reuse one of the label to reuse but we should not have to !]

  assert(m_m.getLambda(d) != 0);
  assert(m_m.getLambda(m_m.alpha(d)) != 0);

  const bool isRegion = isContourPositivelyOrientedFromDart(d);

  girl::region r = 0;
  if (isRegion) {
    r = m_m.getLambda(d);
  }
  else {
    r = m_m.getLambda(m_m.alpha(d));
  }
  
  if (p.length() >= 12)
    regionsToSeek_q.push_back(r);
  else 
    out_q.push_back(r);
}



/**
 * Builds path in representation.
 *
 * Updates representation (bitmap, map, ge2to).
 *
 * Add darts to @a dart_q to relabel regions.
 */
/*
  void
  togerImpl::buildPath(const girl::path &p,
  std::deque<girl::dart> &dart_q)
  {
  traceAndUnmarkPathInBitmap(p);

  girl::geoDart u0(p.startPt(), *(p.begin()));
  girl::geoDart u1(p.endPt(), *(p.rbegin())); //u1.dir() points towards u1.point().
  girl::dart d;
  if (! p.isClosed()) {
  //non closed path
    
  d = buildSegmentBetweenExistingNodes(u0, u1);
  }
  else {
  //closed path
   
  d = buildTangentFace(u0, u1);
  }

  m_m.clearEdgeFaces(d);
  setGeToMap(u0, d);
  setGeToMap(girl::geoDart(u1.point(), u1.dir().opposite()), m_m.alpha(d));
  dart_q.push_back(d);
  #ifndef GIRL_ONLYD
  dart_q.push_back(m_m.alpha(d));
  #endif

  }
*/

/**
 * Trace & unmark path in bitmap
 */
void
togerImpl::traceAndUnmarkPathInBitmap(const girl::path &p)
{
  assert(p.length() > 0);

  girl::bitmap::iterator itB = m_b.getIterator(p.startPt());
  girl::path::const_dir_iterator it = p.begin();
  const girl::path::const_dir_iterator itEnd = p.end();

  assert(! itB.hasNeighbourInDirection(*it));
  itB.writeNeighbourInDirection(*it, true);
  if (! itB.isContourPoint()) {
    assert(itB.isMarked());
    itB.unmark();
  }
  itB.moveInDirection(*it);
  ++it;
  for ( ; it != itEnd; ++it) {
    assert(! itB.isNode());
    assert(itB.isMarked());
    assert(! itB.hasNeighbourInDirection(*it));
    itB.unmark();
    itB.writeNeighbourInDirection(*it, true);
    itB.moveInDirection(*it);
  }    

  /*
    assert(! itB.hasNeighbourInDirection(*it));
    itB.writeNeighbourInDirection(*it, true);
    if (itB.isContourPoint()) {
    assert(itB.isNode());
    assert(! itB.isMarked());
    itB.moveInDirection(*it);
    ++it;
    }
    for ( ; it != itEnd; ++it) {
    assert(! itB.isNode());
    assert(itB.isMarked());
    assert(! itB.hasNeighbourInDirection(*it));
    itB.unmark();
    itB.writeNeighbourInDirection(*it, true);
    itB.moveInDirection(*it);
    }
  */

}

/**
 *
 * Empty @a labelsToReuse
 * Fills @a regionsToSeek_update_q with regions to update.
 * Fills @a regionsToSeek_new_q with newly created regions to seek.
 */ 
void
togerImpl::labellingFacesForPaths(std::deque<girl::dart> &dart_q,
				  girl::regionDeque &labelsToReuse,
				  girl::regionDeque &regionsToSeek_update_q,
				  girl::regionDeque &regionsToSeek_new_q)
{
  bool mustUseRegion = false;
  girl::region r = 0;

  while (! dart_q.empty()) {

    const girl::dart d = dart_q.front();
    dart_q.pop_front();
    assert(d != 0);
    girlAssert1(m_m.checkDart(d));

    girl::regionDeque *regionsToSeek_q = &regionsToSeek_new_q;

    if (mustUseRegion == false) {
      if (! labelsToReuse.empty()) {
	r = labelsToReuse.front();
	labelsToReuse.pop_front();
	mustUseRegion = true;
	regionsToSeek_q = &regionsToSeek_update_q;
      }
      else {
	regionsToSeek_q = &regionsToSeek_new_q;
      }
    }
    else {
      regionsToSeek_q = &regionsToSeek_update_q;
    }

#ifdef VERBOSE_INSERT
    std::cerr<<"nb of labels to reuse : "<<labelsToReuse.size()<<std::endl;
    if (mustUseRegion)
      std::cerr<<" must use region "<<r<<std::endl;
    else
      std::cerr<<" must not use any region"<<std::endl;
#endif

    labellingFace(d, r, mustUseRegion, *regionsToSeek_q);

#ifdef VERBOSE_INSERT
    std::cerr<<"dart d="<<d<<" was labeled "<<m_m.getLambda(d)<<std::endl;
#endif

#ifdef GIRL_ONLYD
    const girl::dart ad = m_m.alpha(d);

    if (mustUseRegion == false) {
      if (! labelsToReuse.empty()) {
	r = labelsToReuse.front();
	labelsToReuse.pop_front();
	mustUseRegion = true;
	regionsToSeek_q = &regionsToSeek_update_q;
      }
      else {
	regionsToSeek_q = &regionsToSeek_new_q;
      }
    }
    else {
      regionsToSeek_q = &regionsToSeek_update_q;
    }

#ifdef VERBOSE_INSERT
    std::cerr<<"nb of labels to reuse : "<<labelsToReuse.size()<<std::endl;
    if (mustUseRegion)
      std::cerr<<" must use region "<<r<<std::endl;
    else
      std::cerr<<" must not use any region"<<std::endl;
#endif

    labellingFace(ad, r, mustUseRegion, *regionsToSeek_q);

#ifdef VERBOSE_INSERT
    std::cerr<<"dart ad="<<ad<<" was labeled "<<m_m.getLambda(ad)<<std::endl;
#endif

#endif //GIRL_ONLYD

  }
  
  assert(dart_q.empty());
  assert(labelsToReuse.empty());
}

/*
  void
  togerImpl::labellingFace(girl::dart d,
  girl::regionDeque &foundRegions)
  {

  if (m_m.getLambda(d)) {
  girlAssert1(m_m.checkFacePhiCycle(m_m.getLambda(d)));
  return;
  }

  const bool isRegion = isContourPositivelyOrientedFromDart(d); //checkFacePositiveOrientationFromDart(d);

  if (isRegion) {
    
  if (

  girl::face f = m_m.changeLambdaCycle(d);  

  girlAssert1(m_m.checkFacePhiCycle(m_m.getLambda(d)));
  girlAssert1(checkContourContinuity(d));

  if (isRegion) {
  girlAssert1(checkFacePositiveOrientation(f));
  
  foundRegions.push_back(f);
  }
  }
*/


void
togerImpl::seekInnerRegionsWithHoles(girl::region r)
{
  //TODO!!! some code duplication with seekAndQueueInnerRegionsWithHoles !!!

  std::deque<girl::direction> dq;
  girl::point startPt;
  faceOuterBoundaryD(r, startPt, dq);

  if (dq.size() < 12)
    //a contour with less than 12 boundary points encloses less than 9 pixels
    // and thus cannot surround regions (see Luc Brun's PhD Thesis)
    return;
  

  m_s.init(startPt, dq);

  assert(P_checkContourIsClosed(startPt, dq));

  std::vector<girl::point> scan;
  P_regionScanlineFromPointAndDirections(startPt, dq, scan);

  for (std::vector<girl::point>::const_iterator it = scan.begin();
       it != scan.end(); it += 2) {

    assert(it->y() == (it+1)->y());

    const girl::point p0 = *it;

    const girl::POINT_TYPE x0 = p0.x();
    const girl::POINT_TYPE x1 = (it+1)->x();

    if (x0+1 == x1)
      //no way there is a new border between these two.
      continue;
    
    bitmap::const_iterator itB = 
      const_cast<const girl::bitmap&>(m_b).getIterator(p0);

    for (girl::POINT_TYPE x = x0; x < x1; ) {
      
      if (itB.hasUpNeighbour()) {

	girl::point p = itB.getPosition();
	
	seekKnownInnerContourS(p, r);
	  
	p = m_s.firstPairEndAndRemove();
	
	itB.moveTo(p);
	x = p.x();
      }
      
      itB.moveRight();
      ++x;

    }

  }

}


void
togerImpl::seekKnownInnerContourS(const girl::point &p, girl::region r)
{
  //TODO!!! some code duplication with seekKnownInnerContour !!!
  //except h.erase(inf) [& mother/daugther update ???]

  girlAssert2(const_cast<const girl::bitmap&>(m_b).getIterator(p).hasUpNeighbour());

  const girl::point sp = m_s.firstPairStart();
  if (sp != p)
    {
      //it's an (old) infinite face not already scanned

      girl::geoDart u;
      m_b.contourTracking(p, u); //get first head/node on contour
      const girl::dart d =  m_g.getDart(u);
      assert(d!=0);
      girlAssert1(m_m.checkDart(d));
      girl::face inf = m_m.getLambda(d); //get face

      assert(inf != 0);
      girlAssert1(m_m.checkFace(inf));

      if (m_m.faceGetMother(inf) != r) {
	m_m.faceRemoveInfiniteFace(inf);
	m_m.faceAddDaughter(r, inf);
      }
      //m_m.faceAddDaughter(r, inf);

      assert(! checkFacePositiveOrientation(inf));
      girlAssert1(m_m.checkFacePhiCycle(inf));

      std::deque<girl::direction> dq;
      girl::point startPt;
      faceOuterBoundaryD(inf, startPt, dq);
      m_s.add(startPt, dq);
    }  

}



void
togerImpl::removeUnnecessaryNodes(const ReducedPathCollection &reducedPaths)
{
  ReducedPathCollection::const_iterator it = reducedPaths.begin();
  const ReducedPathCollection::const_iterator itEnd = reducedPaths.end();
  for ( ; it != itEnd; ++it)
    removeUnnecessaryNodes(it->path);
  
}

void
togerImpl::removeUnnecessaryNodes(const girl::path &p)
{
  girl::point startPt = p.startPt();
  girl::point endPt = p.endPt();
  
  removeUnnecessaryNode(startPt);
  removeUnnecessaryNode(endPt);
}

void
togerImpl::removeUnnecessaryNode(const girl::point &pt)
{
  
  bitmap::iterator itB = m_b.getIterator(pt);
  const int card = itB.getPointCardinality();
  assert(card >= 2);
  if (card == 2) {
    
    girl::dart darts[4];
    m_g.getDarts(pt, darts);
    girl::dart d1 = 0;
    for (int i=0; i<3; ++i) {
      d1 = darts[i];
      if (d1)
	break;
    }

    if (m_m.getPhi(d1) != d1) { 
      //there is another node on face -> useless artificial node 
      
      m_g.deleteNode(pt);
      itB.unsetNode();
      m_m.removeUnnecessaryArtificialNode(d1);
      
    }
  }
  
}


void
togerImpl::swap(togerImpl &t)
{
  std::swap(m_m, t.m_m);
  std::swap(m_b, t.m_b);
  std::swap(m_g, t.m_g);
  
  std::swap(m_observers, t.m_observers);

  std::swap(m_s, t.m_s);

  std::swap(m_dq, t.m_dq);

  std::swap(m_width, t.m_width);
  std::swap(m_height, t.m_height);
}



//-----------------------------------------------------------------
//--- zoom --------------------------------------------------------


void
togerImpl::fillEPS(euclideanPointSet &eps)
{
  //some code duplication with EuclideanPathSet::initFromRegions()...

  girl::face f = 0;
  girl::contour c;
  euclideanPath ep;
  euclideanPointSet::euclideanShiftValues vals;
  while ((f = m_m.getNextExistingFace(f)) != 0) {
    if (! m_m.faceIsInfinite(f)) { //f is a region
      contourFromFace(f, c);
      
      ep.set(c);
      
      euclideanPath::const_iterator it = ep.begin();
      const euclideanPath::const_iterator itEnd = ep.end();
      for ( ; it != itEnd; ++it) {
	
	it.getShifts(vals[0], vals[1]);
	
	eps.addShifts(it.getPoint(), vals);
      }
    }
  }
  eps.finalize();
}





/*
 return true if x is an half int
 that is, x = n + 0.5; where n is an integer.
 */
static inline
bool
P_isHalfInt(girl::REAL_TYPE x)
{
  return ( fabs(x - floor(x-0.5) - 0.5) <= std::numeric_limits<girl::REAL_TYPE>::epsilon() );
}


static
void
P_getPoint(const euclideanPointSet &eps,
	 const girl::point &pt, 
	 girl::REAL_TYPE &x, girl::REAL_TYPE &y)
{
  //TODO: CODE CUPLICATION with euclideanPathSet::getPoint() !!!


  //TODO: get rid off of 'f' in "0.5f" !!!!!!!!!

  euclideanPointSet::euclideanShiftValues vals;
  eps.getShifts(pt, vals); 

  x = static_cast<girl::REAL_TYPE>(pt.x()) + euclideanPath::realValue(vals[0]);
  y = static_cast<girl::REAL_TYPE>(pt.y()) - euclideanPath::realValue(vals[1]);

}



class PointTransformer : public girl::toger::directionIterator
{
public:

  PointTransformer(const euclideanPointSet &eps,
		   const girl::REAL_TYPE sx, 
		   const girl::REAL_TYPE sy,
		   girl::POINT_TYPE w, girl::POINT_TYPE h,
		   girl::POINT_TYPE nw, girl::POINT_TYPE nh,
		   const girl::bitmap &b)
    : m_eps(eps),
      m_sx(sx),
      m_sy(sy),
      m_width(w), m_height(h),
      m_newWidth(nw), m_newHeight(nh),
      m_b(b)
  {}
  

  virtual void processFirstPoint(const girl::point &pt)
  {
    m_firstPoint = pt;
    m_prevPoint = pt;
    m_currentPoint = pt;
    m_firstDir = girl::INVALID;
    m_oppLastDir = girl::INVALID;
    m_prevPointZoomed = zoomPoint(pt);
    m_zoomedPoints.clear();
    m_zoomedPoints.push_back(m_prevPointZoomed);

    // if (m_face == 587)
    //   std::cerr<<"processFirstPoint(): "<<pt<<std::endl;

    

  }
  
  virtual void processDirection(girl::direction d)
  {
    if (m_firstDir == girl::INVALID) {
      m_firstDir = d; //TODO:OPTIM:we could optimize this test out, if we duplicate code of iterateOnSegment() ...
    }


    m_currentPoint.move(d);
    m_oppLastDir = d;
#if 1
    processCurrentPoint();
#endif

    if (m_prevPointZoomed != m_zoomedPoints.back()) //ambiguity
       m_zoomedPoints.push_back(m_prevPointZoomed);
    // if (m_face == 587)
    //   std::cerr<<"processDirection(): "<<d<<std::endl;

  }
  
  virtual void processLastPoint(const girl::point &pt)
  {


    m_lastPoint = pt;
    m_oppLastDir = m_oppLastDir.opposite();

#if 0
    processCurrentPoint();
#endif


    //if (pt != m_firstPoint)
    //processCurrentPoint();

    // if (m_face == 587)
    //   std::cerr<<"processLastPoint(): "<<pt<<std::endl;
  }

protected:

  void processCurrentPoint()
  {

    //const girl::point pt1 = m_prevPoint;
    const girl::point pt2 = m_currentPoint;
      
    //const girl::point zpt1 = m_prevPointZoomed;
    const girl::point zpt2 = zoomPoint(pt2);
      
    m_prevPoint = pt2;
    m_prevPointZoomed = zpt2;
  }
  


  virtual girl::point zoomPoint(const girl::point &p1) const
  {
    if (isBorderPoint(p1))
      return zoomBorderPoint(p1);

    //if (m_b.getIterator(p1).isNode()) {
    girl::bitmap::const_iterator itB = m_b.getIterator(p1);
    if (itB.isNode() && itB.getPointCardinality()>2) { //true node 
      return zoomDiscretePoint(p1);
    }


    girl::REAL_TYPE x1, y1;
    P_getPoint(m_eps, p1, x1, y1);
    
    girl::REAL_TYPE xz1, yz1;
	xz1 = static_cast<girl::REAL_TYPE>( (x1+0.5)*m_sx+0.5 );
    yz1 = static_cast<girl::REAL_TYPE>( (y1+0.5)*m_sy+0.5 );

    xz1 = static_cast<girl::REAL_TYPE>( girl::nearByInt(xz1)+0.5*(m_sx-1)-1 );
    yz1 = static_cast<girl::REAL_TYPE>( girl::nearByInt(yz1)+0.5*(m_sy-1)-1 );

    //girl::point zp1(int(xz1), int(yz1)); 	
    //girl::point zp1(girl::nearByInt(xz1), girl::nearByInt(yz1)); 	
   
    girl::point zp1(roundToInt(xz1, p1.x()),
		    roundToInt(yz1, p1.y()));


    assert(zp1.x()>=-1);
    assert(zp1.y()>=-1);

    

    return zp1;
  }

protected:

  //TODO: could be exterior to class !!!
  //TODO: first parameter could be template !!?????
  inline girl::POINT_TYPE discreteScale(girl::POINT_TYPE v, girl::REAL_TYPE s) const
  {
    return static_cast<girl::POINT_TYPE>( girl::nearByInt((v+1)*s)-1 );
  }

  //round to nearest int if it is not an half int 
  //else, round towards discrete zoom point 
  //
  // zx is zoomed euclidean coordinate
  // x is original coordinate
  //TODO: could be exterior to class !!!
  inline int roundToInt(girl::REAL_TYPE zx, girl::POINT_TYPE x) const
  {
    if (! P_isHalfInt(zx)) {
      //return int(zx);
      return static_cast<int>( girl::nearByInt(zx) );
    }
    else {
      const girl::POINT_TYPE zxd = discreteScale(x, m_sx);
      if (zx < zxd) {
	return (girl::POINT_TYPE(zx)+1);
      }
      else {
	return (girl::POINT_TYPE(zx));
      }
    }
  }


  girl::point zoomDiscretePoint(const girl::point &pt) const
  {
    girl::POINT_TYPE x = pt.x();
    girl::POINT_TYPE y = pt.y();
    
    girl::POINT_TYPE zx = discreteScale(x, m_sx);
    girl::POINT_TYPE zy = discreteScale(y, m_sy);
    
    const girl::point zpt(zx, zy);

    assert(zpt.x()>=-1);
    assert(zpt.y()>=-1);
    assert(zpt.x()<m_newWidth);
    assert(zpt.y()<m_newHeight);

    return zpt;
  }

public:

  
  

  girl::point zoomBorderPoint(const girl::point &pt) const
  {
    assert(isBorderPoint(pt));

    const girl::point zpt = zoomDiscretePoint(pt);
    
    assert(isZoomedBorderPoint(zpt));

    return zpt;
  }


  bool isBorderPoint(const girl::point &pt) const
  {
    return (pt.x() == -1 || pt.y() == -1 || pt.x() == m_width-1 || pt.y() == m_height-1);
  }

  bool isZoomedBorderPoint(const girl::point &pt) const
  {
    return (pt.x() == -1 || pt.y() == -1 || pt.x() == m_newWidth-1 || pt.y() == m_newHeight-1);
  }


public:
  const euclideanPointSet &m_eps;
  const girl::REAL_TYPE m_sx;
  const girl::REAL_TYPE m_sy;
  const girl::POINT_TYPE m_width;
  const girl::POINT_TYPE m_height;
  const girl::POINT_TYPE m_newWidth;
  const girl::POINT_TYPE m_newHeight;

  girl::point m_firstPoint;
  girl::point m_lastPoint;
  std::vector<girl::point> m_zoomedPoints;
  girl::direction m_firstDir;
  girl::direction m_oppLastDir;
  
  const girl::bitmap &m_b;

protected:
  girl::point m_prevPoint;
  girl::point m_prevPointZoomed;
  girl::point m_currentPoint;
};


void
togerImpl::transferZoomedNode(const girl::point &pt,
			      const girl::direction & /* dir */,
			      const girl::point &zoomedPt,
			      girl::map &m,
			      girl::bitmap &b,
			      girl::ge2to &g)
{
  girl::dart darts[4];
  m_g.getDarts(pt, darts);

  g.setDarts(zoomedPt, darts);
  for (size_t i=0; i<4; ++i) 
    if (darts[i] != 0) {
      const girl::geoDart gd(zoomedPt, girl::direction(static_cast<girl::eDirection>(i)));
      m.setGe(darts[i], gd);
    }
  
  b.setNode(zoomedPt);


  
}


void
togerImpl::insertDiscreteLineInBitmap(const girl::point &pt1,
				      const girl::point &pt2,
				      girl::bitmap &b)
{
  assert(pt1 !=  pt2);

  //TODO:OPTIM: put bresenham line construction directly in Bitmap ?
  // we would avoid a temporary deque<direction> ...

  girl::bitmap::iterator itB = b.getIterator(pt1);


  typedef std::deque<girl::direction> DirDeque;
  DirDeque dirs;
  BresenhamDirs(pt1, pt2, dirs);
  
  //girl::point pt = pt1;

  DirDeque::const_iterator it = dirs.begin();
  const DirDeque::const_iterator itEnd = dirs.end();
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    itB.writeNeighbourInDirection(d, true);
    itB.moveInDirection(d);
    //pt.move(d); //TODO:useless...
  }


}


void
togerImpl::insertDiscreteLineInBitmap(const std::vector<girl::point> &pts,
				      girl::bitmap &b)
{
  assert(pts.size() > 2);

  girl::bitmap::iterator itB = b.getIterator(pts[0]);


  const size_t numPts = pts.size();

  typedef std::deque<girl::direction> DirDeque;
  DirDeque allDirs;
  
  BresenhamDirs(pts[0], pts[1], allDirs);

  DirDeque dirs;
  for (size_t i=1; i<numPts-1; ++i) { //start from 1, stop at numPts-2
    assert(dirs.empty());
    BresenhamDirs(pts[i], pts[i+1], dirs);
    
    while (! allDirs.empty()
	   && ! dirs.empty()
	   && allDirs.back() == dirs.front().opposite()) {
      allDirs.pop_back();
      dirs.pop_front();
    }
    while (! dirs.empty()) {
      allDirs.push_back(dirs.front());
      dirs.pop_front();
    }

  }

  girl::point pt = pts[0];

  //DirDeque::const_iterator it = allDirs.begin();
  //const DirDeque::const_iterator itEnd = allDirs.end();
  DirDeque::iterator it = allDirs.begin();
  const DirDeque::iterator itEnd = allDirs.end();
  for ( ; it != itEnd; ++it) {

    girl::direction d = *it;

    girl::bitmap::iterator itBt = itB;
    itBt.moveInDirection(d);
    if (itBt.getPointCardinality() > 1) {
      //point already used in an other contour
      //if we trace, we change topology

      DirDeque::iterator itN = it;
      ++itN;
      if (itN != itEnd) {
	//swap two following directions
	*it = *itN;
	*itN = d;
	d = *it;

      }
      
    }


    itB.writeNeighbourInDirection(d, true);
    itB.moveInDirection(d);
    pt.move(d); //TODO:useless...

  }
  
}





static
bool
P_checkZoomedPts(const girl::point &pt1, girl::direction dir1,
		 const girl::point &pt2, girl::direction dir2)
{

  const size_t manhattanDist = abs(pt1.x()-pt2.x()) + abs(pt1.y()-pt2.y());
  if (manhattanDist == 1) {
    if (dir1 != dir2.opposite())
      return false;
  }
  else if (manhattanDist == 2) {
    if (pt1.x()!=pt2.x() && pt1.y()!=pt2.y() && dir1 == dir2.opposite())
      return false;
  }
  
  return true;
}

/*
 Transfer dart defined by @a u1 & @a u2 on bitmap border,
 from current toger to zoomed toger,
 defined by @a m, @a b and @a g.
 */
void
togerImpl::transferBorderDart(girl::geoDart &u1, girl::geoDart &u2,
			      PointTransformer &ptTr,
			      girl::map &m,
			      girl::bitmap &b,
			      girl::ge2to &g)
{
  //Border is already marked in destination/zoomed bitmap  @a b.
  // thus we only need to transfer nodes at dart extremities.
  //Besides this nodes must stay on contour, 
  // thus we use discrete transformation. 

  const girl::point zp1 = ptTr.zoomBorderPoint(u1.point());
  transferZoomedNode(u1.point(), u1.dir(), zp1, m, b, g);
  if (u1.point() != u2.point()) { 
    const girl::point zp2 = ptTr.zoomBorderPoint(u2.point());
    transferZoomedNode(u2.point(), u2.dir(), zp2, m, b, g);
  }

}

/*
 Transfer dart @a d on bitmap border,
 from current toger to zoomed toger,
 defined by @a m, @a b and @a g.
 */
void
togerImpl::transferBorderDart(girl::dart d,
			      PointTransformer &ptTr,
			      girl::map &m,
			      girl::bitmap &b,
			      girl::ge2to &g)
{

  girl::geoDart u1, u2;
  m_m.getGe(d, u1);
  m_m.getGe(-d, u2);

  transferBorderDart(u1, u2, ptTr, m, b, g);
}

/*
 Transfer zoomed points in @a ptTr,
 from current toger to zoomed toger,
 defined by @a m, @a b and @a g.

 */
void
togerImpl::transferZoomedPoints(PointTransformer &ptTr,
				girl::map &m,
				girl::bitmap &b,
				girl::ge2to &g)
{
  
  transferZoomedNode(ptTr.m_firstPoint, ptTr.m_firstDir, ptTr.m_zoomedPoints.front(), m, b, g);
  if (ptTr.m_lastPoint != ptTr.m_firstPoint)
    transferZoomedNode(ptTr.m_lastPoint, ptTr.m_oppLastDir, ptTr.m_zoomedPoints.back(), m, b, g);


  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#if 1

  const size_t numPoints = ptTr.m_zoomedPoints.size();
  assert(numPoints >= 2);

  b.getIterator(ptTr.m_zoomedPoints.front()).writeNeighbourInDirection(ptTr.m_firstDir, true);
  ptTr.m_zoomedPoints.front().move(ptTr.m_firstDir);

  if (numPoints == 2) {
    girl::point &p1 = ptTr.m_zoomedPoints.front();
    girl::point &p2 = ptTr.m_zoomedPoints.back();
    P_checkZoomedPts(p1, ptTr.m_firstDir, 
		     p2, ptTr.m_oppLastDir);
    if (p1 != p2) {
      b.getIterator(p2).writeNeighbourInDirection(ptTr.m_oppLastDir, true);
      p2.move(ptTr.m_oppLastDir);
      if (p1 != p2)
	insertDiscreteLineInBitmap(p1, p2, b);
    }
  }
  else {
    b.getIterator(ptTr.m_zoomedPoints.back()).writeNeighbourInDirection(ptTr.m_oppLastDir, true);
    ptTr.m_zoomedPoints.back().move(ptTr.m_oppLastDir);
    
    insertDiscreteLineInBitmap(ptTr.m_zoomedPoints, b);
  }
  
#endif

}


void
togerImpl::zoomPoints(const euclideanPointSet &eps,
		      const girl::REAL_TYPE sx, 
		      const girl::REAL_TYPE sy,
		      girl::map &m,
		      girl::bitmap &b,
		      girl::ge2to &g)
{  

  PointTransformer ptTr(eps, sx, sy, width(), height(), b.width(), b.height(), this->m_b);

  girl::face f = 0;
  while ((f = m_m.getNextExistingFace(f)) != 0) {
    if (! m_m.faceIsInfinite(f)) { //f is a region
      const girl::dart canonicD = beta(f);
      girl::dart d = canonicD;
      do {
	girl::region lad = lambda(alpha(d));
	if (! isFaceInfinite(lad)) {
	  if (d > 0) {

	    girl::geoDart u1;
	    m_m.getGe(d, u1);
	    girl::bitmap::const_iterator it1 = (static_cast<const girl::bitmap &>(m_b)).getIterator(u1.point());
	    if (it1.isBorderPoint()) {
	      it1.moveInDirection(u1.dir());
	      if (it1.isBorderPoint()) { //we stay on border

		girl::geoDart u2;
		m_m.getGe(-d, u2);

		//border dart
		transferBorderDart(u1, u2, ptTr, m, b, g);
	      }
	      else {
		iterateOnSegment(d, ptTr);
		transferZoomedPoints(ptTr, m, b, g);
	      }
	    }
	    else {
		
	      iterateOnSegment(d, ptTr);
	      transferZoomedPoints(ptTr, m, b, g);
	
	    }

	  }
	}
	else {
	  if (lad != -1) { //not outside face


	    iterateOnSegment(d, ptTr);
	    transferZoomedPoints(ptTr, m, b, g);
	  }
	  else { //outside face: we avoid to bend outer countour...

	    transferBorderDart(d, ptTr, m, b, g);
	    
	  }

	}
	
	d = phi(d);
      }
      while (d != canonicD);

    }
  }

} 

void
togerImpl::zoom(girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight)
{
  assert((newWidth > m_width && newHeight >= m_height)
	 || (newWidth >= m_width && newHeight > m_height));
  

  euclideanPointSet eps(m_width, m_height);
  fillEPS(eps);

  const girl::REAL_TYPE sx = static_cast<girl::REAL_TYPE>(newWidth) / static_cast<girl::REAL_TYPE>(m_width);
  const girl::REAL_TYPE sy = static_cast<girl::REAL_TYPE>(newHeight) / static_cast<girl::REAL_TYPE>(m_height);

  girl::map lm = m_m; //copy of original
  girl::bitmap lb(newWidth, newHeight); //empty

  lb.setBorderArtificialNode(false);


  girl::ge2to lg(newWidth, newHeight); //empty

  zoomPoints(eps, sx, sy, lm, lb, lg);

  //B: On pourrait faire les modifs directement dans la map d'origine ?????

  assert(lm.checkConsistency());
  assert(lb.checkConsistency());
  //assert(lg.checkConsistency());

  m_m = lm;
  m_b = lb;
  m_g = lg;
  m_width = newWidth;
  m_height = newHeight;
    
  m_s = girl::scanlineData(newWidth, newHeight);
  

  //DEBUG !!!!
  if (! checkConsistency()) {
    std::cerr<<"Toger : inconsistent !!!!!!!!"<<std::endl;
    exit(12);
  }
  if (! checkMinimality()) {
    std::cerr<<"Toger is not minimal !!!!!!!!"<<std::endl;
    exit(13);
  }



  girlAssert1(checkConsistency());
}

//-----------------------------------------------------------------
//--- crop --------------------------------------------------------

static
void
P_initRectangle(girl::POINT_TYPE width, girl::POINT_TYPE height, 
		std::deque<girl::direction> &dirs)
{
  dirs.clear();
  for (girl::POINT_TYPE i=0; i<width; ++i) 
    dirs.push_back(girl::RIGHT);
  for (girl::POINT_TYPE i=0; i<height; ++i) 
    dirs.push_back(girl::DOWN);
  for (girl::POINT_TYPE i=0; i<width; ++i) 
    dirs.push_back(girl::LEFT);
  for (girl::POINT_TYPE i=0; i<height; ++i) 
    dirs.push_back(girl::UP);

}

void
togerImpl::insertBoundingRectangle(const girl::point &p1,
				   girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight,
				   girl::regionDeque &newRegions,
				   girl::regionDeque &updatedRegions)
{
    std::deque<girl::direction> dirs;
    P_initRectangle(newWidth, newHeight, dirs);
    
    girl::point rectStartPt(p1.x()-1, p1.y()-1); //transfer to interpixel

    girl::path pth(rectStartPt, dirs);
    assert(pth.isClosed());
    
    girl::pathDeque paths;
    paths.push_back(pth);
    

    const bool doPendingEdgesSearch = false; //we know we insert a closed contour
    const bool doNewRegionHook = false; //we don't want to call New/UpdateRegionHooks during crop !

#ifndef NDEBUG
    const bool res = 
#endif

      insertPaths(paths,
		  doPendingEdgesSearch,
		  doNewRegionHook,
		  newRegions, updatedRegions);


    assert(res);
}

/*
void
togerImpl::deleteNodesInGe2to(const girl::point &p1, const girl::point &p2)
{
  girl::point zp1(p1.x()-1, p1.y()-1); //inter-pixel
  girl::point zp2(p2.x()-1, p2.y()-1); //inter-pixel

  bitmap::const_iterator it = m_b.getIterator(zp1);

  for (POINT_TYPE y=zp1.y(); y<=zp2.y(); ++y) {
    it.moveTo(0, y);
    for (POINT_TYPE x=zp1.x(); x<=zp2.x(); ++x) {
      if (it.isNode()) {
	m_g.deleteNode(girl::point(x, y));
      }
      it.moveRight();
    }
  }
  
}

void
togerImpl::cropGe2to(const girl::point &p1, const girl::point &p2)
{
  //remove all references to points outside crop rectangle

  if (p1.y() > 0)
    deleteNodesInGe2to(girl::point(0, 0), girl::point(width()-1, p1.y()-1));
  if (p1.x() > 0)
    deleteNodesInGe2to(girl::point(0, p1.y()), girl::point(p1.x()-1, p2.y()));
  if (p2.x() < width()-1)
    deleteNodesInGe2to(girl::point(p2.x()+1, p1.y()), girl::point(width()-1, p2.y()));
  if (p2.y() < height-1)
    deleteNodesInGe2to(girl::point(0, p1.y()+1), girl::point(width()-1, height()-1));

    //TODO: also resize ge2to !!!

}
*/

void
togerImpl::findRegionsOutside(const girl::point &p1, const girl::point &p2,
			      girl::regionDeque &regionsD) const
{
  typedef girl::unorderedSet<girl::region> RegionsHash;
  RegionsHash regions;

  if (p1.y() > 0)
    findRegions(girl::point(0, 0), girl::point(width()-1, p1.y()-1), regions);
  if (p1.x() > 0)
    findRegions(girl::point(0, p1.y()), girl::point(p1.x()-1, p2.y()), regions);
  if (p2.x() < width()-1)
    findRegions(girl::point(p2.x()+1, p1.y()), girl::point(width()-1, p2.y()), regions);
  if (p2.y() < height()-1)
    findRegions(girl::point(0, p2.y()+1), girl::point(width()-1, height()-1), regions);
  
  RegionsHash::const_iterator it = regions.begin();
  const RegionsHash::const_iterator itEnd = regions.end();
  for ( ; it != itEnd; ++it) {
    regionsD.push_back(*it);
  }

}


/*
 Gather darts of outer contour of box defined by inter-pixels points @a ip1 and @æ ip2.
 Fill @a dartsOuterContour with darts of outer contour and their new previous dart in phi.
 */
void
togerImpl::findDartsOuterContour(const girl::point &ip1, const girl::point &ip2,
				 girl::hashMap<girl::dart, girl::dart> &dartsOuterContour) const
{

  girl::dart firstDart = 0;
  girl::dart lastDart = 0;

  bitmap::const_iterator it = m_b.getIterator(ip1);
  
  for (POINT_TYPE y=ip1.y(); y<ip2.y(); ++y) {
    if (it.isNode()) {
      const girl::dart d = m_g.getDart(girl::geoDart(it.getPosition(), girl::DOWN));
      if (d != 0) {
	if (firstDart == 0) {
	  assert(lastDart == 0);
	  firstDart = d;
	}
	dartsOuterContour[d] = lastDart;
	lastDart = d;
      }

    }
    it.moveDown();
  }
  
  //std::cerr<<"it.getPosition()=("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<")  ip1.x()="<<ip1.x()<<", ip2.y()="<<ip2.y()<<std::endl;

  assert(it.getPosition() == (girl::point(ip1.x(), ip2.y())));
  
  for (POINT_TYPE x=ip1.x(); x<ip2.x(); ++x) {
    if (it.isNode()) {
      const girl::dart d = m_g.getDart(girl::geoDart(it.getPosition(), girl::RIGHT));
      if (d != 0) {
	if (firstDart == 0) {
	  assert(lastDart == 0);
	  firstDart = d;
	}
	dartsOuterContour[d] = lastDart;
	lastDart = d;
      }
    }
    it.moveRight();
  }
  
  //std::cerr<<"it.getPosition()=("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<")  ip2.x()="<<ip2.x()<<", ip2.y()="<<ip2.y()<<std::endl;

  assert(it.getPosition() == (girl::point(ip2.x(), ip2.y())));
  
  for (POINT_TYPE y=ip2.y(); y>ip1.y(); --y) {
    if (it.isNode()) {
      const girl::dart d = m_g.getDart(girl::geoDart(it.getPosition(), girl::UP));
      if (d != 0) {
	if (firstDart == 0) {
	  assert(lastDart == 0);
	  firstDart = d;
	}
	dartsOuterContour[d] = lastDart;
	lastDart = d;
      }
    }
    it.moveUp();
  }
  
  //std::cerr<<"it.getPosition()=("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<")  ip2.x()="<<ip2.x()<<", ip1.y()="<<ip1.y()<<std::endl;

  assert(it.getPosition() == (girl::point(ip2.x(), ip1.y())));
  
  for (POINT_TYPE x=ip2.x(); x>ip1.x(); --x) {
    if (it.isNode()) {
      const girl::dart d = m_g.getDart(girl::geoDart(it.getPosition(), girl::LEFT));
      if (d != 0) {
	if (firstDart == 0) {
	  assert(lastDart == 0);
	  firstDart = d;
	}
	dartsOuterContour[d] = lastDart;
	lastDart = d;
      }
    }
    it.moveLeft();
  }

  
  assert(firstDart != 0);
  assert(lastDart != 0);

  if (lastDart != 0) {
    //there were several darts on contour
    
    dartsOuterContour[firstDart] = lastDart;
  }
  else {
    //there was only one artificial node 

    assert(dartsOuterContour.size() == 1);

    dartsOuterContour[firstDart] = firstDart;
  }

  assert(! dartsOuterContour.empty());

}

/**
 * @return apply del region hook on regions in @a out_q.
 */
void
togerImpl::applyDelRegionHook(const girl::regionDeque &out_q)
{

  ObserverCollection::iterator itO = m_observers.begin();
  const ObserverCollection::iterator itOEnd = m_observers.end();
  for ( ; itO != itOEnd; ++itO) {
    girl::toger::observer *o = *itO;
    assert(o);
    if (o->doDeleteRegion()) {
      for (girl::regionDeque::const_iterator it = out_q.begin();
	   it != out_q.end(); ++it) {
	o->deleteRegion(*it);
      }
    }
  }
}

void
togerImpl::crop(const girl::point &p1, const girl::point &p2)
{
  assert(checkPixelValidity(p1));
  assert(checkPixelValidity(p2));
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());

  const size_t newWidth = p2.x()-p1.x()+1;
  const size_t newHeight = p2.y()-p1.y()+1;
  if (newWidth==0 || newHeight==0) { //no point
    return;
  }

  girlAssert2(checkConsistency());

  girl::regionDeque newRegions;
  girl::regionDeque updatedRegions;
  insertBoundingRectangle(p1, newWidth, newHeight, newRegions, updatedRegions);

  girl::regionDeque regionsOutside;
  findRegionsOutside(p1, p2, regionsOutside);

  const girl::point ip1(p1.x()-1, p1.y()-1); //inter-pixel
  const girl::point ip2(p2.x(), p2.y()); //inter-pixel

  girl::hashMap<girl::dart, girl::dart> dartsNewOuterContour;
  findDartsOuterContour(ip1, ip2, dartsNewOuterContour);

  typedef std::deque<girl::point> PointDeque;
  PointDeque nodesToRemove;
  typedef std::map<girl::dart, girl::geoDart> Dart2GeoDartMap;
  Dart2GeoDartMap movedGeoDarts;

  m_m.crop(ip1, regionsOutside, dartsNewOuterContour, nodesToRemove, movedGeoDarts);

  m_width = newWidth;
  m_height = newHeight;
    

  for (PointDeque::const_iterator it = nodesToRemove.begin();
       it != nodesToRemove.end();
       ++it) {
    m_b.makeNode(*it, false);
    m_g.deleteNode(*it);
  }

  for (Dart2GeoDartMap::const_iterator it = movedGeoDarts.begin();
       it != movedGeoDarts.end();
       ++it) {

    m_g.setDart(it->second, it->first);
  }

  m_b.crop(p1, p2);

  m_g.crop(ip1, ip2);


  m_s = girl::scanlineData(newWidth, newHeight);


  //remove regionsOutside from updatedRegions
  //TODO:OPTIM? O(N2) algo ...
  for (regionDeque::iterator it = updatedRegions.begin(); it!=updatedRegions.end(); ) {
    
    regionDeque::const_iterator itD = std::find(regionsOutside.begin(), regionsOutside.end(), *it);
    if (itD != regionsOutside.end())
      it = updatedRegions.erase(it);
    else 
      ++it;
  }

  //DEBUG ????
  for (regionDeque::iterator it = updatedRegions.begin(); it!=updatedRegions.end(); ) {
    
    regionDeque::const_iterator itD = std::find(regionsOutside.begin(), regionsOutside.end(), *it);
    if (itD != regionsOutside.end()) {
      std::cerr<<"togerImpl::crop(): region "<<*it<<" found in updatedRegions and in regionsOutside !"<<std::endl;
      assert(false);

      it = updatedRegions.erase(it);
    }
    else 
      ++it;
  }
  



  //DEBUG
  /*
  {
    std::cerr<<"deleted regions:\n";
    for (regionDeque::iterator it = regionsOutside.begin(); it!=regionsOutside.end(); ++it)
      std::cerr<<*it<<" ";
    std::cerr<<std::endl;
    std::cerr<<"new regions:\n";
    for (regionDeque::iterator it = modifiedRegions.begin(); it!=modifiedRegions.end(); ++it)
      std::cerr<<*it<<" ";
    std::cerr<<std::endl;
  }
  */

  applyNewRegionHook(newRegions, updatedRegions);

  applyDelRegionHook(regionsOutside);

  //DEBUG !!!!
  /*
  if (! checkConsistency()) {
    std::cerr<<"Toger : inconsistent !!!!!!!!"<<std::endl;
    exit(12);
  }
  if (! checkMinimality()) {
    std::cerr<<"Toger is not minimal !!!!!!!!"<<std::endl;
    exit(13);
  }
  */

  girlAssert1(checkConsistency());
  /*
  */

}

