/*
  Region based topological and geometrical image representation
  Copyright (C) 2004-2011 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlToger.hpp
 * @brief Region based topological and geometrical image representation.
 */

/**
 * @mainpage
 *
 * GIRL stands for General Image Representation Library.
 *
 * GIRL is free software, it is distributed under GNU LGPL license 
 * (version 2.1 or later). 
 * See file COPYING for license.
 *
 * GIRL implements a model of planar maps to represent the regions 
 * of a segmented image. It provides efficient access to most of 
 * topological and geometrical features required by segmentation algorithms.
 * See file doc/image_representation.pdf for more details.
 *
 * See README file for details on installation, use, and contact information.
 *
 */

#ifndef GIRLTOGER_HPP
#define GIRLTOGER_HPP

#include <girl/girlCommon.hpp>

#include <deque>
#include <vector>
#include <string>

#include <girl/girlSharedPtr.hpp>
#include <girl/girlUnorderedSet.hpp>

#include <girl/girlFlags.hpp>
#include <girl/girlPoint.hpp>
#include <girl/girlDirection.hpp>
#include <girl/girlIO.hpp>
#include <girl/girlContour.hpp>
#include <girl/girlPath.hpp>

namespace girl {

  /**
   * @brief dart id class.
   *
   * Dart identifer, roughly equivalent to int.
   */
  struct GIRL_API dart
  {
    /**
     * @brief Constructor.
     */
    dart(int d = 0)
      : m_d(d)
    {
    }

    /**
     * @brief int conversion operator.
     */
    operator int () const
    {
      return m_d;
    }

    int m_d;
  };


  //typedef int face;
  //typedef face region;

  /**
   * @brief face id class.
   *
   * Face identifer, roughly equivalent to int.
   */
  struct GIRL_API face
  {
    /**
     * @brief Constructor.
     */
    face(int f = 0)
      : m_f(f)
    {
    }

    /**
     * @brief int conversion operator.
     */
    operator int () const
    {
      return m_f;
    }

    face &operator++()
    {
      ++m_f;
      return *this;
    }

    int m_f;
  };
  /**
   * @brief A region is a face.
   *
   * Actually, a region is a finite face, with potentially holes, that are infinite faces.
   */
  typedef face region;

}

#include <girl/girlHashFunc.hpp>

namespace girl {


  /**
   * @brief region collection type.
   *
   * @warning unicity is not ensured.
   */
  typedef std::deque<girl::region> regionDeque;

  /**
   * @brief region unordered set.
   */
  typedef girl::unorderedSet<girl::region> regionUSet;

  /**
   * @brief dart collection type.
   */
  typedef std::deque<girl::dart> dartDeque;

  /**
   * @brief path collection type.
   */
  typedef std::deque<girl::path> pathDeque;

  /**
   * @brief contour collection type.
   */
  typedef std::deque<girl::contour> contourDeque;


  class togerImpl;

  /**
   * @class toger girlToger.hpp girl/girlToger.hpp
   *
   * @brief Toger class.
   *
   * Region based topological and geometrical image representation.
   */
  class GIRL_API toger 
  {

  public:

    /**
     * Neighbourhing modes :
     * - @a NEIGHBOUR_NONE: two regions are not neighbours.
     * - @a NEIGHBOUR_DIRECT: a region r2 is a direct neighbour of a 
     *  region r1 if the intersection of their outer boundaries is not empty.
     * - @a NEIGHBOUR_INNER: a region r2 is an inner neighbour of a region 
     *  r1 if the outer boundary of r2 intersects the internal boundary of r1. 
     * - @a NEIGHBOUR_OUTER: a region r2 is an outer neighbour of region r1 
     *  if r1 is an inner neighbour of r2.
     * - @a NEIGHBOUR_ANY: is an OR'ed combination of @a NEIGHBOUR_DIRECT, 
     *  @a NEIGHBOUR_INNER and @a NEIGHBOUR_OUTER.
     */
    typedef enum neighbouringMode {
      NEIGHBOUR_NONE = 0,
      NEIGHBOUR_DIRECT = 1,
      NEIGHBOUR_INNER = 2,
      NEIGHBOUR_OUTER = 4,
      NEIGHBOUR_ANY = 7
    } neighbouringMode;


    /**
     * Marks used to mark darts.
     * - @a MARK_NONE is used to specify no mark,
     * - @a MARK_i, with i in [O;7] defines 8 differents marks. 
     * - @a MARK_ANY is an OR'ed combination of all MARK_i.
     */
    typedef enum markFlag {
      MARK_NONE = 0,
      MARK_0 = 1,
      MARK_1 = 2,
      MARK_2 = 4,
      MARK_3 = 8,
      MARK_4 = 16,
      MARK_5 = 32,
      MARK_6 = 64,
      MARK_7 = 128,
      MARK_ANY = 255
    } markFlag;

    GIRL_DECLARE_FLAGS(neighbouringModes, neighbouringMode)
    GIRL_DECLARE_FLAGS(markFlags, markFlag)
    


    //side effects functions

    /**
     * @brief Constructor.
     *
     * Initialize the representation with one region, labelled 1,
     * that fill the box defined by @a width & @a height.
     *
     * @a width and @a height are the size of the image.
     *
     * @warning both @a width and @a height must be strictly superior to 0.
     *
     * toger has a Copy-On-Write policy.
     * @warning observers are not copied when a togere is copied.
     */
    toger(POINT_TYPE width=1, POINT_TYPE height=1);


    /**
     * @return toger width.
     */
    POINT_TYPE width() const;

    /**
     * @return toger height.
     */
    POINT_TYPE height() const;


    /**
     * @brief Splitting functor.
     *
     * Returns true if and only if pixel at @a p and neighbour pixel in 
     * @a dir direction belong to the same region.
     * @a user_data allows to pass a context to the function.
     */
    class GIRL_API sameRegion
    {
    public:
      /**
       * @brief Destructor.
       */
      virtual ~sameRegion() {}

      /**
       * @brief Method used to decide if a pixel and its neighbour 
       * are in the same region or not.
       */
      virtual bool same(const girl::point &p, girl::direction dir) = 0;
    };

    /**
     * @brief Splits a given region @a r of representation, 
     * using @a same_r as a partition criterion.
     *
     * Will call @a same() of @a same_r to decide if two 
     * neighbour pixels belong to the same region or not.
     * 
     * Gives the found regions in @a regions.
     * Given region @a r is reused and will be the first region in @a regions.
     *
     * Returned regions are in no specific order (except @a r being first)
     * (no geometrical or topological information should be inferred of 
     * regions order in @a regions)
     *
     * For each added observer o:
     * o.updateRegion() will be called once with @a r (as @a r is reused),
     * o.newRegion() will be called once for each other region (in @a regions).
     */
    void splitRegion(girl::region r, 
		     girl::toger::sameRegion &same_r, 
		     girl::regionDeque &regions);
  
    /**
     * @brief Merges two regions @a r1 and @a r2, in representation.
     *
     * If merge is possible, i.e., regions are neighbours, final region,
     * result of the merge, will be labelled as one of the two incoming regions.
     * The region label not reused will be freed.
     *
     * For each added observer o:
     * o.mergeRegions() will be called first with the kept region as first argument 
     *  and the deleted region as second argument.
     * o.deleteRegion() will then be called, with the deleted region.
     *
     * @return the kept region or 0 if regions are not neighbours and thus were not merged.
     */
    girl::region mergeRegions(girl::region r1, girl::region r2);


    /**
     * @brief Inserts a set of paths @a paths in representation.
     *
     * @return true if insertion was successful, else false.
     *
     * If insertion is succesfull,
     * @a newRegions is filled with newly created regions,
     * @a updatedRegions is filled with modified existing regions.
     *
     * For each added observer o:
     * o.updateRegion() will be called once with updated regions.
     * o.newRegion() will be called once with newly created regions.
     */
    bool insertPaths(const girl::pathDeque &paths,
		     girl::regionDeque &newRegions,
		     girl::regionDeque &updatedRegions);

    /**
     * @brief Inserts a path @a path in representation.
     *
     * @see insertPaths.
     */
    bool insertPath(const girl::path &path,
		    girl::regionDeque &newRegions,
		    girl::regionDeque &updatedRegions);

    /**
     * @brief Inserts a set of contours @a contours in representation.
     *
     * @see insertPaths.
     */
    bool insertContours(const girl::contourDeque &contours,
			girl::regionDeque &newRegions,
			girl::regionDeque &updatedRegions);
    
    /**
     * @brief Inserts a contour @a contour in representation.
     *
     * @see insertPaths.
     */
    bool insertContour(const girl::contour &contour,
			girl::regionDeque &newRegions,
			girl::regionDeque &updatedRegions);
    


    /**
     * @brief Base class for toger observer.
     *
     * Inherit from this class,
     * overload its methods to define new behaviour,
     * and overload the corresponding do*() method to return true.
     *
     */
    class GIRL_API observer
    {
    public:

      virtual ~observer() 
      {
	unregisterSubjects();
      } 

      /**
       * @brief Tells if newRegion() method must be called.
       */
      virtual bool doNewRegion() const { return false; }

      /**
       * @brief Called when a new region is created.
       */
      virtual void newRegion(const girl::regionDeque & /* regions */) {}
      
      /**
       * @brief Tells if updateRegion() method must be called.
       */
      virtual bool doUpdateRegion() const { return false; }

       /**
       * @brief Called when an existing region is updated.
       */
      virtual void updateRegion(const girl::regionDeque & /* regions */) {}

      /**
       * @brief Tells if mergeRegions() method must be called.
       */
      virtual bool doMergeRegions() const { return false; }

      /**
       * @brief Called when two regions are merged.
       *
       * @a keptRegion is the region that will be kept.
       * @a deletedRegion is the deleted region.
       *
       * @warning when this function is called, region @a deletedRegion does 
       * not exist anymore in the toger.
       */
      virtual void mergeRegions(girl::region /* keptRegion */, girl::region /* deletedRegion */) {}

      /**
       * @brief Tells if deleteRegion() method must be called.
       */
      virtual bool doDeleteRegion() const { return false; }

      /**
       * @brief Called when a region @a r is deleted.
       *
       * It will be called after mergeRegions().
       */
      virtual void deleteRegion(girl::region /* r */) {}

      /**
       * @brief Tells if renameRegion() method must be called.
       */
      virtual bool doRenameRegion() const { return false; }

      /**
       * @brief Called when a region is renamed.
       *
       * We garantee that @a newName <= @a oldName
       * and method will be called in increasing region order.
       */
      virtual void renameRegion(girl::region /* oldRegion */, girl::region /* newRegion */) {}

    protected:

      typedef std::vector<girl::togerImpl *> SubjectCollection;

      void unregisterSubject(girl::togerImpl *tog);

      void registerSubject(girl::togerImpl *tog);

      void unregisterSubjects();

      friend class girl::togerImpl; //to be able to call unregisterSubject

      SubjectCollection m_subjects;
    };


    /**
     * @brief Add an observer @a o to representation.
     *
     * @warning does not take ownership of @a o. 
     * @a o must remain valid as long as it remains an observer.
     * @warning It does not call @a o->newRegions() on existing regions in representation.
     */
    void addObserver(girl::toger::observer *o);

    /**
     * @brief Remove an observer @a o from representation.
     *
     * Does not delete @a o.
     */
    void removeObserver(girl::toger::observer *o);

    /**
     * @brief Remove all observers from representation.
     *
     * Does not delete any observer.
     */
    void removeAllObservers();

    /**
     * @brief Get Observers in @a observers.
     */
    void getObservers(std::deque<girl::toger::observer *> &observers) const;

    /**
     * @brief Add observers @a observers.
     *
     * @sa addObserver.
     */
    void addObservers(const std::deque<girl::toger::observer *> &observers);
    

    // point inclusion and region localization
  
    /**
     * @brief Checks if a point @a p belongs to region @a r,
     * according to representation.
     */
    bool belongsTo(const girl::point &p, girl::region r) const;


    /** 
     * @brief Returns the region containing the point @a p,
     * according to representation.
     *
     * Will sound the internal image representation from left to right, 
     * starting at given point @a p, in order to find 
     * a contour point that belongs to a region.
     *
     * @warning We must have 0<=p1.x()<width() and 0<=p1.y()<height().
     *
     * @sa inSameRegion. 
     */
    girl::region findRegion(const girl::point &p) const;

    /** 
     * @brief Returns @a regions containing the set of points in 
     * the box defined by @a p1 and @a p2, according to representation.
     *
     * @warning We must have 0<=p1.x()<=p2.x()<width() 
     * and 0<=p1.y()<=p2.y()<height().
     *
     * All points (x; y) such as p1.x()<=x<=p2.x() and p1.y()<=y<=p2.y()
     * will be traversed.
     *
     * @warning Returned regions are in no particular order. 
     * Each returned region is unique, no matter how many pixels 
     * belonged to this region in the specified box. 
     * If the method is called with a non empty @a regions,
     * new regions will be added and all will be unique.
     */
    void findRegions(const girl::point &p1, const girl::point &p2, 
		     girl::regionDeque &regions) const;

    /** 
     * @brief Returns @a regions containing the set of points in 
     * the box defined by @a p1 and @a p2, according to representation.
     *
     * @warning We must have 0<=p1.x()<=p2.x()<width() 
     * and 0<=p1.y()<=p2.y()<height().
     *
     * All points (x; y) such as p1.x()<=x<=p2.x() and p1.y()<=y<=p2.y()
     * will be traversed.
     *
     * @warning Returned regions are in no particular order. 
     * Each returned region is unique, no matter how many pixels 
     * belonged to this region in the specified box. 
     * If the method is called with a non empty @a regions,
     * new regions will be added and all will be unique.
     */
    void findRegions(const girl::point &p1, const girl::point &p2, 
		     girl::regionUSet &regions) const;



    /** 
     * @brief Returns @a regions containing points of 
     * @a path, according to representation.
     *
     * @warning all points of path @a path must be in the representation,
     * that is, for a point pt of path,
     * 0<=pt.x()<width() and 0<=pt.y()<height().
     *
     * @warning Returned regions are in no particular order. 
     * Each returned region is unique, no matter how many pixels 
     * belonged to this region in the specified path. 
     * If the method is called with a non empty @a regions,
     * new regions will be added and all will be unique.
     */
    void findRegions(const girl::path &path,
		     girl::regionDeque &regions) const;


    /**
     * @brief Gives the dart at the right of point p.
     *
     * Will sound the internal image representation from left to right, 
     * starting at given point @a p, in order to find 
     * a contour point that belongs to a region.
     *
     * @sa girlToger_findRegion.
     * @sa girlToger_segment.
     */
    girl::dart findDart(const girl::point &p) const;



    /**
     * @brief Gives a point/pixel belonging to the region @a r.
     *
     * While there is no side effect on toger, the point is 
     * assured to stay the same for a given region.
     */
    girl::point anyPoint(girl::region r) const;

    /**
     * @brief Allows to say if two neighbour points/pixels are in the same region.
     *
     * Tells if points/pixels @a p & p.move(@a dir) are in the same region.
     * Faster than two calls to findRegion().
     *
     * @warning We must have 0<=p.x()<width() and 0<=p.y()<height().
     * 
     * @sa findRegion.
     */
    bool inSameRegion(const girl::point &p, girl::direction dir) const;



    //geometrical fonctions


    /**
     * @brief Gives the scanline of the region@a r.
     *
     * return 2*N points, such as two consecutive points, 2*i and 2*i+1, are 
     * extremities of a line of the domain. We then have 
     * scan[2*i].x()<=scan[2*i+1].x() && scan[2*i].y()==scan[2*i+1].y().
     * Besides point are ordered in increasing x and y. Thus we have
     * scan[2*i].y()<scan[2*i+2].y() or 
     * scan[2*i].y()==scan[2*i+2].y() && scan[2*i].x()<scan[2*i+2].x().
     */
    void regionDomain(girl::region r, std::vector<girl::point> &scan) const;


    /**
     * @brief Gives whole boundary of region @a r, 
     * that is both external and internal region boundaries.
     *
     * The first one is the external contour.
     *
     */
    void regionBoundary(girl::region r, girl::contourDeque &b) const;

  
    /**
     * @brief Gives external boundary of region @a r.
     *
     * @a c is filled with boundary of region @a r.
     *
     */
    void regionOuterBoundary(girl::region r, girl::contour &c) const;


    /**
     * @brief Gives the common boundary paths between regions @a r1 and @a r2.
     *
     * Paths are oriented from @a r1.
     *
     */
    void regionsCommonBoundary(girl::region r1, girl::region r2, 
			       pathDeque &commonSegments) const;


    /**
     * @brief Gives the path which describes the geometry of the 
     * segment associated with dart @a d.
     *
     * @sa segmentLength.
     */
    void segment(girl::dart d, girl::path &p) const;

    /**
     * @brief Gives the length of path which describes the geometry of 
     * the segment associated with dart @a d.
     *
     * @sa segment.
     */
    size_t segmentLength(girl::dart d) const;



    // low level topological functions
  
    /**
     * @brief Tells if region @a r is valid in representation.
     *
     * O(1).
     */
    bool isRegionValid(girl::region r) const;

    /**
     * @brief Tells if face @a f is valid in representation.
     *
     * O(1).
     */
    bool isFaceValid(girl::face f) const;


    /**
     * @brief Gives face corresponding to dart @a b in representation.
     */
    girl::face lambda(girl::dart d) const;

    /**
     * @brief Gives canonical dart corresponding to face @a f in representation.
     */
    girl::dart beta(girl::face f) const;


    /**
     * @brief Gives image by alpha of dart @a d in representation.
     */
    girl::dart alpha(girl::dart d) const;

    /**
     * @brief Gives image by sigma of dart @a d in representation.
     */
    girl::dart sigma(girl::dart d) const;

    /**
     * @brief Gives image by phi of dart @a d in representation.
     */
    girl::dart phi(girl::dart d) const;



    /**
     * @brief Checks if the given face @a f is infinite
     */
    bool isFaceInfinite(girl::face f) const;
  
    /**
     * @brief Gives the finite face containing the infinite face @a f.
     *
     * regions all have 0 for mother ; infinite faces have a mother != 0.
     */
    girl::face parent(girl::face f) const;


    /**
     * @brief Gives the first face contained in @a f.
     *
     * @return 0 if none.
     *
     * Allows to traverse children of a face with @a nextSibling(girl::face).
     * Commonly used code to traverse children of region r of toger tog is:
     * @code
     * for (girl::face f = tog.firstChild(r); f!=0; f=tog.nextSibling(f)) {
     *   ...
     * }
     * @endcode
     *
     * @sa nextSibling().
     */
    girl::face firstChild(girl::face f) const;
  

    /**
     * @brief Gives the next sibling of face @a fi in the inclusion tree.
     *
     * @return 0 if none.
     * 
     * @warning @a fi must be an infinite face.
     * @sa firstChild().
     */
    girl::face nextSibling(girl::face fi) const;


    /**
     * @brief Gives dart at geometrical node (@a pt, @a dir).
     *
     * @a pt must be an inter-pixel point.
     *
     * @return 0 if there is no dart at this point and/or direction.
     */
    girl::dart getDart(const girl::point &pt, girl::direction dir) const;


  
    /**
     * @brief Checks if two regions @a r1 and @a r2 are neighbours,
     * according to neighbourhood mode @a mode and mark flags @a f, in the 
     * representation.
     *
     * @a mode is an OR'ed combination of girl::toger::neighbouringMode values.
     * @a m is an OR'ed combination of markFlag.
     * If @a f is different than MARK_NONE, region @a r1 and @a r2, 
     * neighbours according to specified neighbouring mode @a m, 
     * are reported as neighbours only if their common boundary is NOT marked 
     * by @a f.   
     *
     * @sa regionNeighbourhood().
     */
    bool areNeighbours(girl::region r1, 
		       girl::region r2, 
		       girl::toger::neighbouringModes mode, 
		       girl::toger::markFlags m) const;


    /**
     * @brief Gives all regions neighbour of a region @a r according to 
     * the given neighbouring mode @a kind and flags @a f, in the 
     * representation.
     *
     * @a kind is an OR'ed combination of the neighbouring modes.
     * @a m is an OR'ed combination of markFlag.
     * If @a f is different than MARK_NONE, two regions, neighbours according 
     * to specified neighbouring mode @a m, are reported as neighbours only if
     * their common boundary is NOT marked by @a f.   
     *
     * @a regions will hold the found regions
     */
    void regionNeighbourhood(girl::region r, 
			     girl::toger::neighbouringModes mode, 
			     girl::toger::markFlags m, 
			     regionDeque &regions) const;


    /**
     * @brief Gives one region neighbour of a region @a r according to 
     * the given neighbouring mode @a kind and flags @a f, in the 
     * representation.
     *
     * @a kind is an OR'ed combination of the neighbouring modes.
     * @a m is an OR'ed combination of markFlag.
     * If @a f is different than MARK_NONE, two regions, neighbours according
     * to specified neighbouring mode @a m, are reported as neighbours only if
     * their common boundary is NOT marked by @a f.   
     *
     * @return the found region if any or 0.
     */
    girl::region anyRegionNeighbour(girl::region r, 
				    girl::toger::neighbouringModes mode, 
				    girl::toger::markFlags m) const;


    /**
     * @brief Gives all regions of the representation.
     *
     * Gives regions in @a regions.
     * @warning nothing must be inferred on the order of regions in @a regions.
     * @warning @a regions is cleared before being filled, 
     * thus eventually it contains only regions of toger.
     *
     * @sa numRegions.
     */
    void allRegions(regionDeque &regions) const;

    /**
     * @brief Gives the number of regions in the representation.
     *
     * Complexity: O(N) (but no memory allocation).
     * @return the number of regions (without any allocation)
     *
     * @sa allRegions.
     */
    size_t numRegions() const;

    /**
     * @brief Gives all regions touching the inner boundary of region @a r 
     * in representation.
     *
     * It gives all regions touching borders of daugthers/holes of region @a r.
     * @warning it is not necessarily all the regions inside daugthers/holes of
     * region @a r.
     *
     * Gives returned regions in @a regions.
     * @warning nothing must be inferred on the order of regions in @a regions.
     */
    void innerRegions(girl::region r, regionDeque &regions) const;



    /**
     * @brief Base class of function object used to iterate on 
     * region domain.
     *
     */
    class GIRL_API pointIterator
    {
    public:
      /**
       * @brief Destructor.
       */
      virtual ~pointIterator() {}
      
      /**
       * @brief Operation applied on given point/pixel.
       */
      virtual void processPoint(const girl::point &) = 0;

    };
    
    /**
     * @brief Base class of function object used to iterate on 
     * face boundaries or segments.
     *
     */
    class GIRL_API directionIterator
    {
    public:
      /**
       * @brief Destructor.
       */
      virtual ~directionIterator() {}
      
      /**
       * @brief Operation applied on first point of segment.
       */
      virtual void processFirstPoint(const girl::point &) {}

      /**
       * @brief Operation applied on given direction of segment.
       */
      virtual void processDirection(girl::direction) {}

      /**
       * @brief Operation applied on last point of segment.
       */
      virtual void processLastPoint(const girl::point &) {}
    };
    
    
    /**
     * @brief Allows to iterate on pixels of region @a r.
     *
     * @a processPoint() of @a fct will be called exactly once 
     * on each pixel of domain of region @a r.
     * @a processLastPoint() of @a fct will not be called.
     *
     * This function allow to iterate on domain of a region 
     * with a limited number of allocations,
     * and so decent performances.
     *
     * @warning @a process function must not modify toger.
     *
     * @sa regionDomain.
     */
    void iterateOnRegionDomain(girl::region r,
			       girl::toger::pointIterator &fct) const;
    
    /**
     * @brief Allows to iterate on boundary of face @a f.
     *
     * This function allow to iterate on a face boundary
     * with a limited number of allocations,
     * and so decent performances.
     *
     * @a processFirstPoint() of @a fct will be called on first point 
     * of boundary.
     * @a processDirection() of @a fct will be called on each direction 
     * of contour of face @a f.
     * @a processLastPoint() of @a fct will be called on last point 
     * of boundary (which is the same as the first for a closed contour).
     *
     * @warning @a process*() function of @a fct must not modify toger.
     *
     * @sa regionBoundary, regionOuterBoundary.
     */
    void iterateOnFaceBoundary(girl::face f,
			       girl::toger::directionIterator &fct) const;


    /**
     * @brief Allows to iterate on segment of dart @a d.
     *
     * This function allow to iterate on a dart
     * with a limited number of allocations,
     * and so decent performances.
     *
     * @a processFirstPoint() of @a fct will be called on first point 
     * of segment of dart @a d.
     * @a processDirection() of @a fct will be called on each direction 
     * of segment of dart @a d.
     * @a processLastPoint() of @a fct will be called on last point 
     * of segment of dart @a d.
     * @warning last point of segment may also be the first point 
     * of other darts in Sigma*.
     *
     * @warning @a process*() function of @a fct must not modify toger.
     *
     * @sa segment, segmentLength.
     */
    void iterateOnSegment(girl::dart d,
			  girl::toger::directionIterator &fct) const;


    //high-level topological marking functions
  
    /**
     * @brief Inverts mark of region @a r's boundary, in representation, 
     * with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     * Region boundary is symmetrically marked.
     * 
     * @sa unmarkRegionBoundary.
     * @sa xMarkOuterBoundary.
     */
    void xMarkRegionBoundary(girl::region r, girl::toger::markFlags m);

    /**
     * @brief Unmarks region @a r's boundary, in representation, with 
     * mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag
     * Region boundary is symmetrically unmarked.
     * @sa unmarkOuterBoundary.
     */
    void unmarkRegionBoundary(girl::region r, girl::toger::markFlags m);

    /**
     * @brief Inverts mark of region @a r's outer boundary only, 
     * in representation, with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     * Region boundary is symmetrically marked.
     * @sa markRegionBoundary.
     */
    void xMarkOuterBoundary(girl::region r, girl::toger::markFlags m);

    /**
     * @brief Unmarks region @a r's outer boundary only, in representation, 
     * with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     * Region boundary is symmetrically unmarked.
     * @sa unmarkRegionBoundary.
     */
    void unmarkOuterBoundary(girl::region r, girl::toger::markFlags m);



    /**
     * @brief Tells if any dart of region @a r is marked with mark @a m.
     *
     * @a m is an OR'ed combination of markFlag.
     * @return true if at least one dart of region @a r is marked by one 
     * of the mark of @a m.
     *
     * @warning consider boundaries symmetrically marked.
     */
    bool isRegionMarked(girl::region r, girl::toger::markFlags m) const;


    /**
     * @brief Tells if any dart of outer boundary of region @a r is marked 
     * with mark @a m.
     *
     * @a m is an OR'ed combination of markFlag.
     * @return true if at least one dart of outer boundary of region @a r is 
     * marked by one of the mark of @a m.
     *
     * @warning consider boundary symmetrically marked.
     */
    bool isRegionOuterBoundaryMarked(girl::region r, 
				     girl::toger::markFlags m) const;
  

    /**
     * @brief Tells if face @a f is marked with mark @a m.
     *
     * @a m is an OR'ed combination of markFlag.
     * @return true if at least one dart of face @a f is marked by one of 
     * the mark of @a m.
     *
     * @warning consider face symmetrically marked.
     * @warning do not take any eventual child of face @a f into account.
     *
     * @sa isDartMarked.
     */
    bool isFaceMarked(girl::face f, girl::toger::markFlags m) const;

    /**
     * @brief Marks all darts, in representation, with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     */
    void markAllDarts(girl::toger::markFlags m);
  
    //low-level topological marking functions
  
    /**
     * @brief Unmarks all darts, in representation, with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     */
    void unmarkAllDarts(girl::toger::markFlags m);
  
    /**
     * @brief Checks if dart @a d is marked with mark @a f, in representation.
     *
     * @a m is an OR'ed combination of markFlag.
     * @return true if dart @a d is marked by at least one mark of @a m.
     */
    bool isDartMarked(girl::dart d, girl::toger::markFlags m) const;

    /**
     * @brief Marks regions @a r1 and @a r2 common boundary, in 
     * representation, with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     * 
     * @warning Only darts of @a r1 are marked. This operation is not 
     * symmetrical.
     * 
     */
    void markCommonBoundary(girl::region r1, girl::region r2, 
			    girl::toger::markFlags m);

    /**
     * @brief Unmarks regions @a r1 and @a r2 common boundary, in 
     * representation, with mark @a f. 
     *
     * @a m is an OR'ed combination of markFlag.
     * 
     * @warning Only darts of @a r1 are marked. This operation is not 
     * symmetrical.
     * 
     */
    void unmarkCommonBoundary(girl::region r1, girl::region r2, 
			      girl::toger::markFlags m);
  

    /**
     * @brief Save representation to file @a f.
     * For debug purpose only.
     */
    void saveBitmap(const std::string &f) const;
  
  
  
    /**
     * @brief Check overall representation consistency.
     * Will print found errors, if any, to stderr. 
     *
     * @warning this function can be costly.
     */
    bool checkConsistency() const;


    /**
     * @brief Shrink representation to use the less memory possible.
     *
     * It may rename some regions.
     *
     * @sa observer.
     */
    void shrink();


    //io functions
  
 
  
    /**
     * @brief Write toger with writer @a w.
     *
     * flags is OR'ed combination of girl::io::writeFlag.
     * - @a NO_ENDIANNESS_CARE indicates that no care has to be taken regarding endianness. 
     *   It is the case when writing to memory or with a compressor taking care of endianness itself.
     *   It allows to have faster outputs.
     * - @a NO_LABELS_MARKS_CARE indicates that write has not to ensure that 
     *    when reloaded labels and marks of regions/darts will all be the same.
     *   It allows to have smaller outputs.
     * - @a DEFAULT is to take care of endianness and labels.
     *
     * @return true if write was ok, else false.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags flags = girl::io::DEFAULT) const; 


    /**
     * @brief Read toger with reader @a r.
     *
     * flags is OR'ed combination of girl::io::writeFlag.
     * If NO_ENDIANNESS_CARE was used for writing, it must be used for reading !
     *
     * @return true if read was ok, else false.
     *
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags flags = girl::io::DEFAULT); 
  

    
    /**
     * @brief Swap two toger.
     *
     * Assign content of this to @a t and content of @a t to this.
     */
    void swap(girl::toger &t);

  
  
    //edition functions
  

    /**
     * @brief Tells if borders of pixel @p are swapable in geometric 
     * representation of toger.
     */
    bool swapable(const girl::point &p) const;


    /**
     * @brief Swap pixel borders in geometric representation of toger.
     *
     * @warning pixel must swapable.
     * @sa swapable.
     */
    void swap(const girl::point &p);


    /**
     * @brief Zoom the representation to size @a newWidth by @a newHeight.
     *
     * Topology stays the same. Geometry is zoomed using euclidean paths.
     *
     * @warning @a newWidth must be superior to current width.
     * @warning @a newHeight must be superior to current height.
     *
     * @warning as toplogy is unchanged, no method of observers is called.
     */
    void zoom(girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight);


    /**
     * @brief Crop the representation to the set of points in 
     * the box defined by @a p1 and @a p2.
     *
     * @warning We must have 0<=p1.x()<=p2.x()<width() 
     * and 0<=p1.y()<=p2.y()<height().
     *
     * For each added observer o:
     * o.updateRegion() will be called once with updated regions.
     * o.newRegion() will be called once with newly created regions.
     * o.deleteRegion() will then be called with the deleted region.
     */
    void crop(const girl::point &p1, const girl::point &p2);


  private:

    void copy();

  private:

    girl::shared_ptr<girl::togerImpl> m_pimpl;

  };

}

GIRL_DECLARE_OPERATORS_FOR_FLAGS(girl::toger::neighbouringModes)
GIRL_DECLARE_OPERATORS_FOR_FLAGS(girl::toger::markFlags)


#endif // GIRLTOGER_HPP
