/*
  Region based topological and geometrical image representation
  Copyright (C) 2004-2011 Gabriel de Dietrich, Achille Braquelaire, Jean-Philippe Domenger, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlToger.hpp"
#include "girlTogerImpl.hpp"

using namespace girl;


toger::toger(POINT_TYPE pWidth, POINT_TYPE pHeight)
  : m_pimpl(new togerImpl(pWidth, pHeight))
{

}

POINT_TYPE
toger::width() const
{
  return m_pimpl->width();
}

POINT_TYPE
toger::height() const
{
  return m_pimpl->height();
}

void
toger::copy()
{
  assert(m_pimpl.use_count() >= 1);
  m_pimpl.reset(new togerImpl(*m_pimpl));
  assert(m_pimpl.use_count() == 1);
}

void
toger::addObserver(girl::toger::observer *o)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->addObserver(o);
}

void
toger::removeObserver(girl::toger::observer *o)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->removeObserver(o);
}

void
toger::removeAllObservers()
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->removeAllObservers();
}

void
toger::getObservers(std::deque<girl::toger::observer *> &observers) const
{
  m_pimpl->getObservers(observers);
}

void
toger::addObservers(const std::deque<girl::toger::observer *> &observers)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->addObservers(observers);
}


bool
toger::belongsTo(const girl::point &p, girl::region r) const
{
  return m_pimpl->belongsTo(p, r);
}


girl::region
toger::findRegion(const girl::point &p) const
{
  return m_pimpl->findRegion(p);
}

void
toger::findRegions(const girl::point &p1, const girl::point &p2, girl::regionDeque &regions) const
{
  return m_pimpl->findRegions(p1, p2, regions);
}

void
toger::findRegions(const girl::point &p1, const girl::point &p2, girl::regionUSet &regions) const
{
  return m_pimpl->findRegions(p1, p2, regions);
}

void
toger::findRegions(const girl::path &path,
		   girl::regionDeque &regions) const
{
  return m_pimpl->findRegions(path, regions);
}

girl::dart
toger::findDart(const girl::point &p) const
{
  return m_pimpl->findDart(p);
}


girl::point
toger::anyPoint(girl::region r) const
{
  return m_pimpl->anyPoint(r);
}

bool
toger::inSameRegion(const girl::point &p, girl::direction dir) const
{
  return m_pimpl->inSameRegion(p, dir);
}


bool
toger::areNeighbours(girl::region r1, girl::region r2, girl::toger::neighbouringModes mode, girl::toger::markFlags f) const
{
  return m_pimpl->areNeighbours(r1, r2, mode, f);
}


void
toger::allRegions(girl::regionDeque &regs) const
{
  m_pimpl->allRegions(regs);
}

size_t
toger::numRegions() const
{
  return m_pimpl->numRegions();
}



void
toger::innerRegions(girl::region r, girl::regionDeque &regs) const
{
  m_pimpl->innerRegions(r, regs);
}




void
toger::iterateOnRegionDomain(girl::region r,
			     girl::toger::pointIterator &fct) const
{
  m_pimpl->iterateOnRegionDomain(r, fct);
}

void
toger::iterateOnFaceBoundary(girl::face f,
			     girl::toger::directionIterator &fct) const
{
  m_pimpl->iterateOnFaceBoundary(f, fct);
}

void 
toger::iterateOnSegment(girl::dart d,
			girl::toger::directionIterator &fct) const
{
  m_pimpl->iterateOnSegment(d, fct);
}



girl::region
toger::anyRegionNeighbour(girl::region r, girl::toger::neighbouringModes mode, girl::toger::markFlags mark) const
{
  return m_pimpl->anyRegionNeighbour(r, mode, mark);
}

void
toger::regionNeighbourhood(girl::region r, girl::toger::neighbouringModes mode, girl::toger::markFlags mark, girl::regionDeque &regs) const
{
  m_pimpl->regionNeighbourhood(r, mode, mark, regs);
}

girl::region
toger::mergeRegions(girl::region r1, girl::region r2)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->mergeRegions(r1, r2);
}

void
toger::splitRegion(girl::region r,
		   girl::toger::sameRegion &same_r,
		   girl::regionDeque &regs)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->splitRegion(r, same_r, regs);
}

bool
toger::insertPaths(const girl::pathDeque &paths,
		   girl::regionDeque &newRegions,
		   girl::regionDeque &updatedRegions)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->insertPaths(paths, newRegions, updatedRegions);
}

bool
toger::insertPath(const girl::path &path,
		  girl::regionDeque &newRegions,
		  girl::regionDeque &updatedRegions)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->insertPath(path, newRegions, updatedRegions);
  
}

bool
toger::insertContours(const girl::contourDeque &contours,
		      girl::regionDeque &newRegions,
		      girl::regionDeque &updatedRegions)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->insertContours(contours, newRegions, updatedRegions);
}

bool
toger::insertContour(const girl::contour &contour,
		     girl::regionDeque &newRegions,
		     girl::regionDeque &updatedRegions)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->insertContour(contour, newRegions, updatedRegions);
}



void
toger::regionDomain(girl::region r, std::vector<girl::point> &scan) const //vector???
{
  m_pimpl->regionDomain(r, scan);
}


void
toger::regionOuterBoundary(girl::region r, girl::contour &c) const
{
  m_pimpl->regionOuterBoundary(r, c);
}



void
toger::regionBoundary(girl::region r, girl::contourDeque &q) const
{
  m_pimpl->regionBoundary(r, q);
}


void
toger::regionsCommonBoundary(girl::region r1, girl::region r2, girl::pathDeque &s) const
{
  m_pimpl->regionsCommonBoundary(r1, r2, s);
}

void 
toger::segment(girl::dart d, girl::path &p) const
{
  m_pimpl->segment(d, p);
}

size_t
toger::segmentLength(girl::dart d) const
{
  return m_pimpl->segmentLength(d);
}

bool
toger::isRegionMarked(girl::region r, girl::toger::markFlags m) const
{
  return m_pimpl->isRegionMarked(r, m);
}


bool
toger::isRegionOuterBoundaryMarked(girl::region r, girl::toger::markFlags m) const
{
  return m_pimpl->isRegionOuterBoundaryMarked(r, m);
}


bool
toger::isFaceMarked(girl::face f, girl::toger::markFlags m) const
{
  return m_pimpl->isFaceMarked(f, m);
}


void 
toger::xMarkRegionBoundary(girl::region r, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->xMarkRegionBoundary(r, f);
}

void
toger::unmarkRegionBoundary(girl::region r, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->unmarkRegionBoundary(r, f);
}

void
toger::xMarkOuterBoundary(girl::region r, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->xMarkOuterBoundary(r, f);
}

void
toger::unmarkOuterBoundary(girl::region r, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->unmarkOuterBoundary(r, f);
}

void
toger::markCommonBoundary(girl::region r1, girl::region r2, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->markCommonBoundary(r1, r2, f);
}

void
toger::unmarkCommonBoundary(girl::region r1, girl::region r2, girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->unmarkCommonBoundary(r1, r2, f);
}

void
toger::markAllDarts(girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->markAllDarts(f);
}

void
toger::unmarkAllDarts(girl::toger::markFlags f)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->unmarkAllDarts(f);
}

bool
toger::isDartMarked(girl::dart d, girl::toger::markFlags f) const
{
  return m_pimpl->isDartMarked(d, f);
}


void
toger::saveBitmap(const std::string &filename) const
{
  return m_pimpl->saveBitmap(filename);
}

bool
toger::checkConsistency() const
{
  return m_pimpl->checkConsistency();
}

bool
toger::isRegionValid(girl::region r) const
{
  return m_pimpl->isRegionValid(r);
}

bool
toger::isFaceValid(girl::face f) const
{
  return m_pimpl->isFaceValid(f);
}

girl::dart
toger::alpha(girl::dart d) const
{
  return m_pimpl->alpha(d);
}


girl::dart
toger::sigma(girl::dart d) const
{
  return m_pimpl->sigma(d);
}


girl::dart
toger::phi(girl::dart d) const
{
  return m_pimpl->phi(d);
}


girl::face
toger::lambda(girl::dart d) const
{
  return m_pimpl->lambda(d);
}


girl::dart
toger::beta(girl::face f) const
{
  return m_pimpl->beta(f);
}

bool 
toger::isFaceInfinite(girl::face f) const
{
  return m_pimpl->isFaceInfinite(f);
}

girl::face
toger::parent(girl::face fi) const
{
  return m_pimpl->parent(fi);
}


girl::face
toger::firstChild(girl::face f) const
{
  return m_pimpl->firstChild(f);
}


girl::face
toger::nextSibling(girl::face fi) const
{
  return m_pimpl->nextSibling(fi);
}

girl::dart
toger::getDart(const girl::point &pt, girl::direction dir) const
{
  return m_pimpl->getDart(pt, dir);
}



void
toger::shrink()
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->shrink();
}


bool
toger::write(girl::io::Writer &w, girl::io::writeFlags flags) const
{
  return m_pimpl->write(w, flags);
}

bool
toger::read(girl::io::Reader &r, girl::io::writeFlags flags)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->read(r, flags);
}

void
toger::swap(girl::toger &t)
{
  std::swap(m_pimpl, t.m_pimpl);
}


bool
toger::swapable(const girl::point &p) const
{
  return m_pimpl->swapable(p);
}


void
toger::swap(const girl::point &p)
{
  if (m_pimpl.use_count() > 1)
    copy();
  m_pimpl->swap(p);
}

void
toger::zoom(girl::POINT_TYPE newWidth, girl::POINT_TYPE newHeight)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->zoom(newWidth, newHeight);
}

void
toger::crop(const girl::point &p1, const girl::point &p2)
{
  if (m_pimpl.use_count() > 1)
    copy();
  return m_pimpl->crop(p1, p2);
}



