/*
  Timer class.
  Copyright (C) 2004-2011 Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlTimer.hpp
 * @brief Timer class.
 */

#ifndef GIRL_TIMER_HPP
#define GIRL_TIMER_HPP


#if defined (_WIN32)
#define WINDOWS_LEAN_AND_MEAN
#include <windows.h>
#include <ctime>
#include <cassert>
#elif defined (__linux) || defined (__APPLE__)
#include <sys/time.h>
#else
#include <time.h>
#endif


namespace girl {

  class Timer
  {
  public:
    /**
     * @brief Constructor.
     */
    Timer()
      : 
#ifdef _WIN32
      m_frequency(),
      m_startTime(),
#endif
      m_lastStop(0),
      m_started(false)
    {
#ifdef _WIN32
      if
	( ! QueryPerformanceFrequency(&m_frequency))
	{
	  assert(false);
	}
#endif

    }

    /**
     * @brief Start timer.
     *
     * Do nothing if already started.
     * 
     */
    void start()
    {
      if (m_started == false) {

	m_started = true;
#ifdef _WIN32
	QueryPerformanceCounter(&m_startTime);
#else
	gettimeofday(&m_startTime, 0);
#endif
      }
    }

    
    /**
     * @brief Reset and start timer.
     *
     * Do nothing if already started.
     * 
     */
    void restart()
    {
      m_lastStop = 0;
      m_started = true;

#ifdef _WIN32
      QueryPerformanceCounter(&m_startTime);
#else
      gettimeofday(&m_startTime, 0);
#endif
    }


    /**
     * @brief Stop timer and return accumulated time in milliseconds since previous call to restart.
     *
     * Successive calls to stop will return the same value;
     *
     * @warning if start()/stop() is called again, elapsed time will be accumulated.
     */
    size_t stop()
    {
      m_lastStop = getElapsedTime();
      m_started = false;
      return m_lastStop;
    }


    /**
     * @brief Give elapsed time in milliseconds since previous call to start.
     */
    size_t getElapsedTime() const
    {
      if (m_started) {

#ifdef _WIN32
	LARGE_INTEGER lCurrentTime;
	QueryPerformanceCounter(&lCurrentTime);
	return m_lastStop + static_cast<size_t>( (lCurrentTime.QuadPart - m_startTime.QuadPart) / (m_frequency.QuadPart * 0.001) ); //B : works ???
#else
	struct timeval lCurrentTime;
	gettimeofday(&lCurrentTime, 0);
	return m_lastStop + static_cast<size_t>((lCurrentTime.tv_sec-m_startTime.tv_sec)*1000 + (lCurrentTime.tv_usec-m_startTime.tv_usec)*.001);  
#endif

      }
      else {
	return m_lastStop;

      }

    }


  protected:

#ifdef _WIN32
    LARGE_INTEGER m_frequency;
    LARGE_INTEGER m_startTime;
#else
    struct timeval m_startTime;
#endif
  
    size_t m_lastStop;
    
    bool m_started;

  protected:

  };

}


#endif /* ! GIRL_TIMER_HPP */
