/*
  Internal structure used to store scanline information
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlScanlineData.hpp
 * @brief Internal structure used to store scanline information.
 */

#ifndef GIRLSCANLINEDATA_HPP
#define GIRLSCANLINEDATA_HPP

#include <deque>
#include <vector>

#include "girlPoint.hpp"
#include "girlDirection.hpp"


namespace girl {

  class scanlineData
  {
  public:

    /**
     * @brief Constructor.
     *
     * @param width   image width.
     * @param height  image height.
     */
    scanlineData(POINT_TYPE width, POINT_TYPE height);

    
    /**
     * @brief Initializes scanline data.
     *
     * Will restrict its operations to the inside of the given contour,
     * defined by a starting point @a startPt and a deque @a dq of directions.
     *
     * Do not empty deque @a dq !
     */
    void init(const girl::point &startPt, const std::deque<girl::direction> &dq);
    
    /**
     * @brief Add a contour, defined by a starting point @a startPt and a 
     * deque @a dq of directions, to scanline data.
     *
     * @warning will empty deque @a dq !
     */
    void add(const girl::point &startPt, std::deque<girl::direction> &dq);
    
    /**
     * @brief Give start point of the first point pair.
     */
    girl::point firstPairStart() const;
    
    /**
     * @brief Give end point of the first point pair
     * and move to the next first point pair
     */
    girl::point firstPairEndAndRemove();
    
    
  protected:

    typedef std::deque<girl::point> DequeType;

    void reallocateLines(size_t nbLines);
    
    void insertSortedArray(const std::vector<girl::point> &pts);

    scanlineData::DequeType *getLine(POINT_TYPE y);

    const scanlineData::DequeType *getLine(POINT_TYPE y) const;

  protected:

    std::vector<DequeType> m_lines;
    POINT_TYPE m_yMin; //minimum y coordinate of initial contour
    POINT_TYPE m_yMax; //maximum y coordinate of initial contour
    POINT_TYPE m_xMax; //maximum x coordinate of initial contour
    POINT_TYPE m_yHead; //current y coordinate
    POINT_TYPE m_prevYHead; //previous current y coordinate
    bool m_firstInsertionDone; 

  };

}

#endif /* ! GIRLSCANLINEDATA_HPP */
