/*
  Internal structure used to store scanline information
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlScanlineData.hpp"

#include <cassert>
#include <algorithm>

#include "girlCommon.hpp"


using namespace girl;

scanlineData::scanlineData(POINT_TYPE /* width */ GIRL_unused, 
			   POINT_TYPE height GIRL_unused)
  : m_lines(), 
    m_yMin(0), 
    m_yMax(-1), 
    m_xMax(-1), 
    m_yHead(0), 
    m_prevYHead(0), 
    m_firstInsertionDone(false)
{
  
}

#ifndef NDEBUG
static 
bool
P_checkContourIsClosed(const girl::point &startPt, 
		       const std::deque<girl::direction> &dq)
{
  const girl::point lStartPt = startPt;
  girl::point pt = lStartPt;

  std::deque<girl::direction>::const_iterator it = dq.begin();
  const std::deque<girl::direction>::const_iterator itEnd = dq.end();
  for ( ; it != itEnd; ++it)
    pt.move(*it);

  return pt == lStartPt;
}
#endif

void 
scanlineData::init(const girl::point &startPt, 
		   const std::deque<girl::direction> &dq)
{
  assert(P_checkContourIsClosed(startPt, dq));
  assert(! dq.empty());

  std::deque<girl::direction>::const_iterator it = dq.begin();
  assert(it != dq.end());

  const girl::point lStartPt = startPt;
  girl::point cp = lStartPt;
  const girl::direction dirf = *it;
  cp.move(dirf);
  ++it;

  //- compute ymin, ymax, & xmax

  POINT_TYPE ymin = cp.y();
  POINT_TYPE ymax = cp.y();
  POINT_TYPE xmax = cp.x();

  const std::deque<girl::direction>::const_iterator itEnd = dq.end();
  for ( ; it != itEnd; ++it) {
    const girl::direction dir = *it;
    cp.move(dir);
    
    if (cp.y() < ymin)
      ymin = cp.y();
    else if (cp.y() > ymax)
      ymax = cp.y();
    
    if (cp.x() > xmax)
      xmax = cp.x();
  }

  assert(ymin<=ymax);

  m_yMin = ymin;
  m_yMax = ymax;
  m_xMax = xmax;

  const size_t nbLines = ymax - ymin + 1;
  reallocateLines(nbLines);
  m_prevYHead = m_yHead = m_yMin;
  m_firstInsertionDone = false;

}

void 
scanlineData::reallocateLines(size_t nbLines)
{
  std::vector<DequeType> lines(nbLines);
  m_lines.swap(lines);
}

struct pointCmpPred
{
  bool operator()(const girl::point &pt1, const girl::point &pt2) const
  {
    if (pt1.y() == pt2.y())
      return pt1.x() < pt2.x();
    return pt1.y() < pt2.y();
  }
};


void 
scanlineData::add(const girl::point &startPt, std::deque<girl::direction> &dq)
{
  assert(P_checkContourIsClosed(startPt, dq));
  assert(! dq.empty());

  const size_t nbDir = dq.size();
  std::vector<girl::point> pts;
  pts.reserve(nbDir);
  
  const girl::point lStartPt = startPt;
  girl::point pt = lStartPt;

  do {
    const girl::direction d = dq.front();
    dq.pop_front();
    pt.move(d);

    if (d == UP || d == DOWN) {
      girl::point p = pt;
      if (d == UP)
	p.setY(pt.y()+1);
      pts.push_back(p);
    }
  } 
  while (! dq.empty());

  assert(pts.size() % 2 == 0);

  std::sort(pts.begin(), pts.end(), pointCmpPred());

  insertSortedArray(pts);

  m_yHead = pts[0].y();
  m_prevYHead = m_yHead;
}

scanlineData::DequeType *
scanlineData::getLine(POINT_TYPE y)
{
  assert(y>=m_yMin);
  assert(y<=m_yMax);

  return &(m_lines[y - m_yMin]);
}

const scanlineData::DequeType *
scanlineData::getLine(POINT_TYPE y) const
{
  assert(y>=m_yMin);
  assert(y<=m_yMax);

  return &(m_lines[y - m_yMin]);
}

void 
scanlineData::insertSortedArray(const std::vector<girl::point> &pts)
{
  assert(pts.size()%2==0);
  assert(pts.size()>0);

  assert(pts[pts.size()-1].y()<=m_yMax);	


  POINT_TYPE cur_y = m_prevYHead;
  DequeType *d = getLine(cur_y);  
  //impl: for d, we must use a pointer and not a reference, 
  //as it will be modified.
 
  assert(d);
  DequeType::iterator it;
  
  it = d->begin();
  
  if (m_firstInsertionDone == false)
    m_yHead = pts[0].y();

  const size_t nbp = pts.size();
  for (size_t i=0; i<nbp; ) {

    assert(pts[i].y() == pts[i+1].y());
    assert(pts[i].y() >= m_yMin);
    assert(pts[i+1].y() >= m_yMin);
    assert(pts[i].y() <= m_yMax);
    assert(pts[i+1].y() <= m_yMax);
    assert(pts[i].y() >= m_prevYHead);
    
    if (pts[i].y() > cur_y) {
      cur_y = pts[i].y();
      d = getLine(cur_y);
      assert(d);
      it = d->begin();
    }

    assert(d);


    if (it == d->end()) {
      while (i<nbp) {
	if (pts[i].y() == cur_y) {
	  assert(i+1<nbp);
	  assert(d);
	  d->push_back(girl::point(pts[i].x(), pts[i+1].x()));
	  it = d->end();
	}
	else
	  break;
	i+=2;
      }
    }
    else {
      /*
	girl::point pt = *it;
	while (it->x() < pts[i].x()) {
	++it;
	if (it == d->end())
	break;
	pt = *it;
	}
      */
      while (it != d->end() && it->x() < pts[i].x())
	++it;

      if (it == d->end()) {
	while (i<nbp) {
	  if (pts[i].y() == cur_y) {
	    assert(i+1<nbp);
	    assert(d);
	    d->push_back(girl::point(pts[i].x(), pts[i+1].x()));
	    it = d->end();
	  }
	  else
	    break;
	  i+=2;
	}
      }
      else {
	assert(it != d->end());
	assert(i+1<nbp);
	assert(d);
	assert(it->x() > pts[i].x());
	assert(it->x() > pts[i+1].x());
	it = d->insert(it, girl::point(pts[i].x(), pts[i+1].x()));
	++it;
	i+=2;
      }
    }

  }

  m_firstInsertionDone = true;
 
}

girl::point 
scanlineData::firstPairStart() const
{
  if (m_yHead <= m_yMax && m_firstInsertionDone == true) {
    const scanlineData::DequeType *d = getLine(m_yHead);
    assert(! d->empty());
    return girl::point(d->front().x(), m_yHead);
  }
  return girl::point(m_xMax, m_yMax);
}

girl::point 
scanlineData::firstPairEndAndRemove()
{
  assert(m_yHead <= m_yMax);

  DequeType *d = getLine(m_yHead);
  assert(! d->empty());
  
  girl::point res(d->front().y(), m_yHead);
  d->pop_front();
  m_prevYHead = m_yHead;

  if (d->empty()) {
    for (m_yHead = m_yHead+1; m_yHead<=m_yMax; ++m_yHead) {
      d = getLine(m_yHead);
      if (! d->empty())
	break;
    }
  }
  
  return res;
}

