/*
  Hash map for points.
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlPointHashMap.hpp
 * @brief Hash map class for points.
 */

#ifndef GIRL_POINTHASHMAP_HPP
#define GIRL_POINTHASHMAP_HPP


#include "girlPoint.hpp"
#include "girlHashMap.hpp"


namespace girl {

  /**
   * @brief Hash functor for points.
   */
  class ptHasher
    : public std::unary_function<point, std::size_t>
  {
    /*
      static const size_t bucket_size = 4;
      static const size_t min_buckets = 8;
      std::less<T> lt;
      P h;
      //hash<T> h;

      size_t operator( )(T const & s) const
      {
      return h(s);
      }
      bool operator( )(T const & a,
      T const & b) const
      {
      return lt(a, b);
      }
 
    */

  public:
    explicit ptHasher(girl::POINT_TYPE pHeight = 1)
      : m_height(pHeight)
    {}
  
    ptHasher(const ptHasher &o)
      : m_height(o.m_height)
    {}

    inline 
    std::size_t 
    operator()(const girl::point &p) const
    {
      return static_cast<size_t>( p.x()*m_height+p.y() );
    }

#ifdef _MSC_VER
    static const size_t bucket_size = 4;
    static const size_t min_buckets = 8;
      
    bool operator( )(const girl::point& a,
		     const girl::point& b) const
    {
      return (a.y() != b.y()) ? a.y() < b.y() : a.x() < b.x();	
    }
      
#endif


  protected:
    girl::POINT_TYPE m_height;
  };
    


  /**
   * @brief Hash map for points.
   *
   * Allows to hash points randomly distributed on a rectangular surface.
   * It considers that points coordinates are on a width*height surface.
   */
  template <typename T>
  class pointHashMap : public girl::hashMap<girl::point, T, ptHasher>
  {
  public:
    typedef typename girl::hashMap<girl::point, T, ptHasher>::size_type size_type;

    pointHashMap(POINT_TYPE height = 1,
		 size_type n = 10)
      : girl::hashMap<girl::point, T, ptHasher>(n, ptHasher(height))
    {}
    
  };

  


}

#endif /* ! GIRL_POINTHASHMAP_HPP */
