/*
  2D Point class.
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlPoint.hpp
 * @brief 2D Point class.
 */

#ifndef GIRLPOINT_HPP
#define GIRLPOINT_HPP

//#include <girl/girlCommon.h>

#include "girlIO.hpp"
#include "girlDirection.hpp"
#include "girlEndian.hpp"

namespace girl {

  /**
   * @brief Generic 2D point class.
   */
  template <typename T>
  class TPoint
  {
  public:

    /**
     * @brief Constructor.
     */
    TPoint(T px=0, T py=0)
      : m_x(px), m_y(py)
    {}

    /**
     * @brief Gets x coordinate of point.
     */
    inline
    T x() const
    {
      return m_x;
    }

    /**
     * @brief Gets y coordinate of point.
     */
    inline
    T y() const
    {
      return m_y;
    }

    /**
     * @brief Sets both coordinates of point.
     */
    void set(T px, T py)
    {
      m_x = px;
      m_y = py;
    }

    /**
     * @brief Sets x coordinate of point to value @a px.
     */
    void setX(T px)
    {
      m_x = px;
    }

    /**
     * @brief Sets y coordinate of point to value @a py.
     */
    void setY(T py)
    {
      m_y = py;
    }

    /**
     * @brief operator==
     */
    bool operator==(const girl::TPoint<T> &pt) const
    {
      return m_x == pt.m_x && m_y == pt.m_y;
    }

    /**
     * @brief operator!=
     */
    bool operator!=(const girl::TPoint<T> &pt) const
    {
      return m_x != pt.m_x || m_y != pt.m_y;
    }

    /**
     * @brief Add @a pt to this.
     */
    girl::TPoint<T> &operator+=(const girl::TPoint<T> &pt)
    {
      m_x += pt.m_x;
      m_y += pt.m_y;
      return *this;
    }

    /**
     * @brief Subtract @a pt to this.
     */
    girl::TPoint<T> &operator-=(const girl::TPoint<T> &pt)
    {
      m_x -= pt.m_x;
      m_y -= pt.m_y;
      return *this;
    }

  protected:
    /**
     * @brief X coordinate.
     */
    T m_x;

    /**
     * @brief Y coordinate.
     */
    T m_y;

  };


  typedef short POINT_TYPE;

  /**
   * @class point girlPoint.hpp girl/girlPoint.hpp
   *
   * @brief 2D point class.
   */
  class GIRL_API point : public TPoint<POINT_TYPE>
  {
  public:

    /**
     * @brief Constructor.
     */
    point(POINT_TYPE px=0, POINT_TYPE py=0)
      : TPoint<POINT_TYPE>(px, py)
    {}

    
    /**
     * @brief Move this point in direction @a d.
     */
    void move(girl::direction d)
    {
      switch (d)
	{
	case RIGHT:
	  setX(static_cast<POINT_TYPE>(x()+1));
	  break;
	case UP:
	  setY(static_cast<POINT_TYPE>(y()-1));
	  break;
	case LEFT:
	  setX(static_cast<POINT_TYPE>(x()-1));
	  break;
	case DOWN:
	  setY(static_cast<POINT_TYPE>(y()+1));
	  break;
	case INVALID:
	default:
	  assert(false);
	  break;
	}
    }

    /**
     * @brief Give direction from this to point @a pt.
     *
     * @warning consider only 4-connexity.
     */
    girl::direction directionTo(const girl::point &pt) const
    {
      if (pt.m_x == m_x+1 && pt.m_y == m_y)
	return RIGHT;
      else if (pt.m_x == m_x && pt.m_y == m_y-1)
	return UP;
      else if (pt.m_x == m_x-1 && pt.m_y == m_y)
	return LEFT;
      else if (pt.m_x == m_x && pt.m_y == m_y+1)
	return DOWN;

      return INVALID;
    }

    /**
     * @brief Write point @a p with function @a fct
     *
     * @return true if write was ok, else false.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags /* flags */) const
    {
      //Don't care of endianness flag : 
      // it should be faster to always swap than to test if we have to...
      //TODO : bench it...

      assert(sizeof(girl::POINT_TYPE) == 2);
      girl::POINT_TYPE tx = GIRL_swapLE16(m_x);
      girl::POINT_TYPE ty = GIRL_swapLE16(m_y);
      if (w.write(&tx, sizeof(tx)) != sizeof(tx))
	return false;
      if (w.write(&ty, sizeof(ty)) != sizeof(ty))
	return false;
      return true;
    }
    
    /**
     * @brief Read point @a p with function @a fct
     *
     * @a user_data allows to pass a context to the function.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags /* flags */)
    {
      assert(sizeof(girl::POINT_TYPE) == 2);	
      if (r.read(&(m_x), sizeof(m_x)) != sizeof(m_x))
	return false;
      if (r.read(&(m_y), sizeof(m_y)) != sizeof(m_y))
	return false;
      m_x = GIRL_swapLE16(m_x);
      m_y = GIRL_swapLE16(m_y);
      return true;
    }
    
  };


}




#endif /* !GIRLPOINT_HPP */
