/*
  Pending edge finder.
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef GIRLPENDINGEDGEFINDER_HPP
#define GIRLPENDINGEDGEFINDER_HPP

#include <algorithm>
#include <utility> //pair
#include <list>
#include <deque>
#include <iostream>

//#include "girlToger.hpp"
#include "girlHashMap.hpp"
#include "girlHashFunc.hpp"

namespace girl {

  struct edge : public std::pair<girl::face, girl::face>
  {
    edge(girl::face a, girl::face b)
      : std::pair<girl::face, girl::face>(a, b)
    {
      if (b < a) {
	first = b;
	second = a;
      }
    }
  };


  class pendingEdgeFinder
  {
  public:

    pendingEdgeFinder()
      : m_face2vertexProxy()
    {}

    /**
     * @return true if a pending edge is found, false if not.
     */
    bool find(const std::deque<edge> &edgeList)
    {
    
      std::deque<edge>::const_iterator it = edgeList.begin();
      const std::deque<edge>::const_iterator itEnd = edgeList.end();
      for ( ; it!=itEnd; ++it)
	addEdge(*it);

      assert(checkSingleLink());

      bool result = mergeCycles();

      clean();
      assert(m_face2vertexProxy.empty());

      return !result;
    }

  protected:

    struct vertex;

    struct vertexProxy
    {
      vertexProxy()
	: m_v(NULL), m_f(0)
      {}

      vertex *m_v;
      girl::face m_f;
    };

    typedef std::list<vertexProxy*> LinksType;

    struct vertex
    {
      vertex()
	: m_proxies(), m_links()
      {}

      LinksType m_proxies;
      LinksType m_links;
    };

    typedef girl::hashMap<girl::face, vertexProxy *> Face2VertexProxy;
    Face2VertexProxy m_face2vertexProxy;
  

    /**
     * Find the first vertex proxy in list @a l1 of vertex v1,
     * for which the pointed to vertex is @a v2
     * 
     * @return an iterator pointing on found vertex proxy,
     * or on @a l1.end()
     */
    LinksType::iterator 
    find(LinksType &l1, vertex *v2)
    {
      LinksType::iterator it = l1.begin();
      const LinksType::iterator itEnd = l1.end();
      for ( ; it != itEnd; ++it)
	if ((*it)->m_v == v2)
	  return it;
      return l1.end();
    }

    /**
     * Add edge @a e.
     *
     * we merge double edges as we go.
     */
    void
    addEdge(edge e)
    {
      const girl::face f1 = e.first;
      const girl::face f2 = e.second;

      if (f1 == f2) //handle loops
	return;

      Face2VertexProxy::iterator itF1 = m_face2vertexProxy.find(f1);
      if (itF1 != m_face2vertexProxy.end()) {
	//f1 already referenced as a face.
	vertexProxy *vp1 = itF1->second;
	assert(vp1);
	vertex *v1 = vp1->m_v;
	assert(v1);
      
	Face2VertexProxy::iterator itF2 = m_face2vertexProxy.find(f2);
	if (itF2 != m_face2vertexProxy.end()) {
	  //f2 already referenced as a face.
	  vertexProxy *vp2 = itF2->second;
	  assert(vp2);
	  vertex *v2 = vp2->m_v;
	  assert(v2);
	
	  if (v1 != v2) {
	    //both f1 & f2 are refenreced and are not merged.
	    LinksType &l1 = v1->m_links;
	    LinksType &l2 = v2->m_links;
	    LinksType::iterator itL1 = find(l1, v2);
	    if (itL1 != l1.end()) {
	      // f1 already linked to f2 (by *itL1)

	      assert(find(l2, v1) != l2.end());
	      merge(vp1, itL1);
	    }
	    else {
	      //first edge between f1 & f2

	      assert(find(l2, v1) == l2.end());
	      l1.push_back(vp2);
	      l2.push_back(vp1);
	    }
	  }
	  //else v1 == v2 : f1 & f2 have already been merged : nothing to do.
      
	}
	else {
	  // f1 already referenced as a face, but not f2
	  // (so, it is also the first edge between f1 & f2)
	  vertexProxy *vp2 = new vertexProxy;
	  vp2->m_f = f2;
	  vp2->m_v = new vertex;
	  vp2->m_v->m_proxies.push_back(vp2);
	  m_face2vertexProxy[f2] = vp2;

	  vp2->m_v->m_links.push_back(vp1);
	  vp1->m_v->m_links.push_back(vp2);
	}
      }
      else {
	// f1 is not referenced as a face.

	Face2VertexProxy::iterator itF2 = m_face2vertexProxy.find(f2);
	if (itF2 != m_face2vertexProxy.end()) {
	  //f2 already referenced as a face.
	  // (so, it is also the first edge between f1 & f2)
	  vertexProxy *vp1 = new vertexProxy;
	  vp1->m_f = f1;
	  vp1->m_v = new vertex;
	  vp1->m_v->m_proxies.push_back(vp1);
	  m_face2vertexProxy[f1] = vp1;
	
	  vertexProxy *vp2 = itF2->second;
	  assert(vp2);

	  vp2->m_v->m_links.push_back(vp1);
	  vp1->m_v->m_links.push_back(vp2);
	}
	else {
	  // both f1 & f2 are not yet referenced as a face.
	  vertexProxy *vp1 = new vertexProxy;
	  vp1->m_f = f1;
	  vp1->m_v = new vertex;
	  vp1->m_v->m_proxies.push_back(vp1);
	  m_face2vertexProxy[f1] = vp1;
	
	  vertexProxy *vp2 = new vertexProxy;
	  vp2->m_f = f2;
	  vp2->m_v = new vertex;
	  vp2->m_v->m_proxies.push_back(vp2);
	  m_face2vertexProxy[f2] = vp2;

	  vp2->m_v->m_links.push_back(vp1);
	  vp1->m_v->m_links.push_back(vp2);
	}
      }	
    }


    struct removePredicate
    {
      explicit removePredicate(vertex *v2)
	: m_v2(v2)
      {}

      inline
      bool operator()(vertexProxy *vp1) const
      {
	return vp1->m_v == m_v2;
      }

      vertex *m_v2;
    };

    /**
     * Remove all the vertex proxy in list @a l1 of vertex v1,
     * for which the pointed to vertex is @a v2
     * 
     */
    void
    remove(LinksType &l1, vertex *v2)
    {
      l1.remove_if(removePredicate(v2));
    }

    void
    transferProxies(vertex *v1, vertex *v2)
    {
      LinksType &lp1 = v1->m_proxies;
      LinksType &lp2 = v2->m_proxies;
      LinksType::iterator itLP2 = lp2.begin();
      const LinksType::iterator itLP2End = lp2.end();
      for ( ; itLP2 != itLP2End; ++itLP2) {
	vertexProxy *vp2 = *itLP2;
	assert(vp2->m_v == v2);
	vp2->m_v = v1;
	assert(std::find(lp1.begin(), lp1.end(), vp2) == lp1.end());
	lp1.push_back(vp2);
      }
    }  

    /**
     *
     * itL1 is an iterator on @a vp1->m_v->m_links,
     * and we want to merge @a vp1 and @a *itL1
     */
    void
    merge(vertexProxy *vp1, LinksType::iterator &itL1)
    {
      vertexProxy *vp2 = *itL1;
      assert(vp2);
      vertex *v1 = vp1->m_v;
      assert(v1);
      vertex *v2 = vp2->m_v;
      assert(v2);

      if (v1 == v2) {
	std::cerr<<"!!! merge two vertexProxies pointing on the same vertex !!!"<<std::endl;
      }

      transferProxies(v1, v2);
    
      LinksType &l1 = v1->m_links;
    
      assert(find(l1, v2) == l1.end());
    
      l1.erase(itL1);

      assert(find(l1, v2) == l1.end());
    
      if (v1 == v2) {
	std::cerr<<"!!!!!!!!!!!!!!!!  merge two vertexProxies pointing on the same vertex !!!"<<std::endl;
	return;
      }

      LinksType &l2 = v2->m_links;

      while (! l2.empty()) {
	LinksType::iterator itL2 = l2.begin();
	vertexProxy *vp3 = *itL2;
	assert(vp3);
	assert(vp3->m_v != v2);

	l2.erase(itL2);
	if (vp3->m_v != v1) {
	  LinksType::iterator itL1b = find(l1, vp3->m_v);
	  if (itL1b != l1.end()) {
	    // vp3->m_v both referenced by v1 & v2 

	    merge(vp1, itL1b);

	  }
	  else {
	    l1.push_back(vp3);
	  }
	}
      }
    
      assert(l2.empty());
      delete v2;
    }

    bool 
    checkSingleLink()
    {
      Face2VertexProxy::iterator it = m_face2vertexProxy.begin();
      const Face2VertexProxy::iterator itEnd = m_face2vertexProxy.end();
      for ( ; it!=itEnd; ++it) {
	vertexProxy *vp1 = it->second;
	assert(vp1);
	vertex *v1 = vp1->m_v;
	assert(v1);
	LinksType &l1 = v1->m_links;
	LinksType::iterator itL1 = l1.begin();
	const LinksType::iterator itL1End = l1.end();
	for ( ; itL1 != itL1End ; ++itL1) {
	  vertexProxy *vp2 = *itL1;
	  assert(vp2);
	  vertex *v2 = vp2->m_v;
	  assert(v2);
	  LinksType::iterator itL1n = itL1;
	  ++itL1n;
	  for ( ; itL1n != itL1End ; ++itL1n) {
	    vertexProxy *vp3 = *itL1n;
	    assert(vp3);
	    vertex *v3 = vp3->m_v;
	    assert(v3);
	    if (v2 == v3) 
	      return false;
	  }
	}
      }
      return true;
    }
    
    /**
     * Traverse and merge all cycles of graph
     * 
     * When a cycle from a vertex has been completely traversed
     * - its list of links is empty
     * - face/vertexProxy may be removed from map.
     */
    bool
    mergeCycles()
    {
      while (! m_face2vertexProxy.empty()) {
	Face2VertexProxy::iterator it = m_face2vertexProxy.begin();
	vertexProxy *vp1 = it->second;
	bool result = mergeFaceCycle(vp1);
	if (result) {

	  assert(vp1);
	  vertex *v1 = vp1->m_v;
	  assert(v1);
	  assert(vp1->m_v->m_links.empty());

	  LinksType::iterator itLP1 = v1->m_proxies.begin();
	  const LinksType::iterator itLP1End = v1->m_proxies.end();
	  for ( ; itLP1 != itLP1End ; ++itLP1) {
	    vertexProxy *vp = (*itLP1);
	    assert(vp->m_v == v1);
	    vp->m_v = NULL;
	    girl::face f = vp->m_f;
	    Face2VertexProxy::iterator itF = m_face2vertexProxy.find(f);
	    assert(itF != m_face2vertexProxy.end());
	    assert(itF->second == vp);
	    delete vp;
	    itF->second = NULL;
	    m_face2vertexProxy.erase(itF);
	  }
	  delete v1;

	}
	else {
	  //pending edge found !
	  return false;
	}
      }

      assert(m_face2vertexProxy.empty());
      return true;
    }
  
    void
    clean()
    {
      while (! m_face2vertexProxy.empty()) {
	Face2VertexProxy::iterator it = m_face2vertexProxy.begin();
	vertexProxy *vp1 = it->second;
	assert(vp1);
	vertex *v1 = vp1->m_v;
	assert(v1);

	LinksType::iterator itLP1 = v1->m_proxies.begin();
	const LinksType::iterator itLP1End = v1->m_proxies.end();
	for ( ; itLP1 != itLP1End ; ++itLP1) {
	  vertexProxy *vp = (*itLP1);

	  assert(vp->m_v == v1);
	  vp->m_v = NULL;
	  girl::face f = vp->m_f;
	  
	  Face2VertexProxy::iterator itF = m_face2vertexProxy.find(f);
	  assert(itF != m_face2vertexProxy.end());
	  assert(itF->second == vp);
	  delete vp;
	  itF->second = NULL;
	  m_face2vertexProxy.erase(itF);
	}
	
	delete v1;
      }
      assert(m_face2vertexProxy.empty());
    }


    struct sameVertexPredicate
    {
      bool operator()(const vertexProxy *vp1, const vertexProxy *vp2)
      {
	assert(vp1);
	assert(vp2);
	return vp1->m_v == vp2->m_v;
      }
    };

    bool
    mergeFaceCycle(vertexProxy *vp1)
    {
      assert(vp1);
      vertex *v1 = vp1->m_v;
      assert(v1);

      vertexProxy *vpStart = vp1;
      vertex *vStart = v1;

      while (! vStart->m_links.empty()) {
      
	LinksType cycle;
	cycle.push_back(vp1);
	size_t cycleSize = 1;
	v1 = vStart;
      
	vertexProxy *vp2 = *(v1->m_links.begin());
	vertex *v2 = vp2->m_v;
	assert(v2);
	assert(v1 != v2);

	do {

	  assert(v1 != v2);
	  LinksType &l2 = v2->m_links;
	  if (l2.size() == 1) {
	    assert((*l2.begin())->m_v == v1);
	    //pending edge found !!!
	    return false;
	  }

	  assert(! l2.empty()); 
	  assert(l2.size() > 1);
	  assert(find(l2, v1) != l2.end());
	  LinksType::iterator itL2 = l2.begin();
	  if ((*itL2)->m_v == v1)
	    ++itL2;
	  assert((*itL2)->m_v != v1);
	  assert((*itL2)->m_v != v2);

	  cycle.push_back(vp2);
	  ++cycleSize;

	  vertexProxy *vp3 = *itL2;
	  assert(vp3);
	  vertex *v3 = vp3->m_v;
	  assert(v3);
	  assert(v3 != v2);
	  assert(v3 != v1);

	  LinksType::iterator itC = find(cycle, v3);
	  if ( itC != cycle.end() ) {
	    //we found a cycle

	    //we will merge all vp of the cycle.
	    vertexProxy *vp = *itC;
	    vertex *v = vp->m_v;
	    for (LinksType::iterator it = ++itC;
		 it != cycle.end(); 
		 ) {
	      LinksType::iterator itLv = find(v->m_links, (*it)->m_v);
	      if (itLv != v->m_links.end()) {
		merge(vp, itLv);
	      }
	      it = cycle.erase(it);
	      --cycleSize;
	      assert(cycle.size() == cycleSize);

	    }
	    
	    //with successive merges, we could have vertexPointers pointing to same vertex in list
	    //so we remove equivalent elements
	    //(impl: we do not use std::list::unique as we also compute cycleSize during traversing)
	    assert(! cycle.empty());
	    cycleSize = 1;
	    const LinksType::iterator itCbEnd = cycle.end();
	    LinksType::iterator itCb = cycle.begin();
	    LinksType::iterator itCbn = itCb;
	    while (++itCbn != itCbEnd) {
	      if ((*itCb)->m_v == (*itCbn)->m_v)
		cycle.erase(itCbn);
	      else {
		itCb = itCbn;
		++cycleSize;
	      }
	      itCbn = itCb;
	    }
	    assert(cycleSize == cycle.size());

	    if (cycleSize > 1) {	    
	      LinksType::reverse_iterator itCc = cycle.rbegin();
	      vp3 = *itCc;
	      v3 = vp3->m_v;
	      ++itCc; //B??????????????
	      v2 = (*itCc)->m_v;
	      assert(v3 != v2);
	    }

	  }

	  v1 = v2;
	  v2 = v3;
	  vp2 = vp3;

	} while (cycleSize > 1);

	vp1 = vpStart;
	vStart = vpStart->m_v;
	v1 = vStart;

      }
    
      return true;
    }
  
  };


}

#endif /* ! GIRLPENDINGEDGEFINDER_HPP */
