/*
  Freeman code based paths
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlPath.hpp
 * @brief Freeman code based path.
 */

#ifndef GIRLPATH_HPP
#define GIRLPATH_HPP

#include <girl/girlFreemanCodePath.hpp>
#include <girl/girlContour.hpp>

namespace girl {

  /**
   * @class path girlPath.hpp girl/girlPath.hpp
   *
   * @brief Discrete path class.
   */
  class GIRL_API path : public freemanCodePath
  {

  public:
    
    /**
     * @brief Default constructor.
     *
     * Constructs a path of length 0 starting and ending at @a startPt. 
     */
    inline explicit path(const girl::point &startPt = girl::point(0, 0));
    
    /**
     * @brief Constructor
     *
     * Construct a path from a starting point @a startPt
     * and a deque of directions @a dirs.
     */
    path(const girl::point &startPt,
	 const std::deque<girl::direction> &dirs);
    
    /**
     * @brief Constructor
     *
     * Construct a path from a starting point @a startPt,
     * ending at @a endPt, and with directions @a dirs.
     *
     * @warning endPt must be the right end point, 
     * starting from startPt and following directions of @a dirs,
     * else results are undefined.
     */
    inline path(const girl::point &startPt, const girl::point &endPt, 
		const std::deque<girl::direction> &dirs);
    

    /**
     * @brief Constructor
     *
     * Construct a path from a contour.
     */
    inline explicit path(const girl::contour &c);

    /**
     * @brief Set path from a starting point @a startPt
     * and a deque of directions @a dirs.
     */
    void set(const girl::point &startPt,
	     const std::deque<girl::direction> &dirs);

    /**
     * @brief Set path from a starting point @a startPt
     * ending at @a endPt, and with directions @a dirs.
     *
     * @warning endPt must be the right end point, 
     * starting from startPt and following directions of @a dirs,
     * else results are undefined.
     */
    inline void set(const girl::point &startPt, const girl::point &endPt,
		    const std::deque<girl::direction> &dirs);

    

    /**
     * @brief Checks if path @a self is closed,
     * that is start point and end point are equal.
     */
    inline bool isClosed() const;
    
    
    /**
     * @brief Gives starting point of path @a self.
     */
    inline girl::point startPt() const;
    
    
    /**
     * @brief Gives end point of path @a self.
     */
    inline girl::point endPt() const;
        
    /**
     * @brief Path length, i.e. number of directions.
     *
     * O(1).
     */
    inline size_t length() const;
    
    /**
     * @brief Tells if the freemanCodePath is valid.
     * that is two successive directions are not in opposite way.
     */
    inline bool isValid() const;

    /**
     * @brief equality operator
     */
    inline bool operator==(const path &o);

    /**
     * @brief inequality operator
     */
    inline bool operator!=(const path &o);

    /**
     * @brief Inverts orientation of freemanCodePath.
     * 
     * For example, a freemanCodePath with four directions 
     * Right, Up, Left, and Down, will be changed to  
     * Up, Right, Down, and Left.
     *
     * start and end points are swapped.
     */
    void invertOrientation();

    /**
     * @brief Adds a direction @a d to the end of path.
     * 
     * End point is modified accordingly.
     */
    void add(girl::direction d);

    /**
     * @brief Appends direction of path @a p.
     *
     */
    void append(const girl::path &p);


    /**
     * @brief Moves starting point to new point @a newStartPt.
     *
     * End point is modified accordingly.
     */
    void move(const girl::point &newStartPt);

    /**
     * @brief Crop path to keep only @a length directions after 
     * the @a startIndex first directions.
     *
     * We must have startIndex+length<=length().
     */
    void crop(size_t startIndex, size_t length);


    /**
     * @brief Swap two paths.
     *
     * Assign content of this to @a p and content of @a p to this.
     */
    inline void swap(path &p);
    

  private:

    /**
     * @brief Constructor.
     *
     * Allocates a path corresponding to given directions.
     *
     * @a start and @a end are respectively first and end point of path.
     * @a packedDirections is an array of @a nbDirections directions 
     * with 4 directions in each char. 
     * @a packedDirections will be copied. It must be of actual size (nbDirections+(4-1))/4
     *
     * End point could be computed from @a start and @a packedDirections,
     * but it is passed to avoid this computation.
     * 
     */
    path(const girl::point &start, 
	 const girl::point &end, 
	 const std::vector<char> &packedDirections, 
	 size_t nbDirections);

    void set_aux(const std::deque<girl::direction> &dirs);
      
  private:

    girl::point m_end;

  };



  //inline function --------------------------------------------------

  
  path::path(const girl::point &pStartPt)
    : freemanCodePath(pStartPt), m_end(pStartPt)
  {
    assert(isValid());
  }
  
  path::path(const girl::point &pStartPt, const girl::point &pEndPt,
	     const std::deque<girl::direction> &dirs)
    : freemanCodePath(pStartPt, dirs), m_end(pEndPt)
  {
    assert(m_end == freemanCodePath::endPt());
  }

  path::path(const girl::contour &c)
    : freemanCodePath(c), m_end(m_start)
  {

  }

  
  void
  path::set(const girl::point &pStartPt, const girl::point &pEndPt,
	    const std::deque<girl::direction> &dirs)
  {
    freemanCodePath::set(pStartPt, dirs);
    m_end = pEndPt;

    assert(isValid());
  }

  bool 
  path::isClosed() const
  {
    return m_start == m_end;
  }
    
  girl::point 
  path::endPt() const
  {
    return m_end;
  }

  girl::point 
  path::startPt() const
  {
    return freemanCodePath::startPt();
  }
  
  size_t 
  path::length() const
  {
    return freemanCodePath::length();
  }

  bool
  path::isValid() const
  {
    return freemanCodePath::isValid() && (m_end == freemanCodePath::endPt());
  }

  bool
  path::operator==(const path &o)
  {
    return freemanCodePath::operator==(o) && m_end == o.m_end;
  }
  
  bool
  path::operator!=(const path &o)
  {
    return !operator==(o);
  }


  void
  path::swap(path &p)
  {
    freemanCodePath::swap(p);
    std::swap(m_end, p.m_end);
  }


  inline
  void
  swap(path &p1, path &p2)
  {
    p1.swap(p2);
  }
  
}

namespace std {

  /**
   * @brief Swap two paths.
   *
   * Assign content of @a p1 to @a p2 and content of @a p2 to @a p1.
   */
  template <>
  inline 
  void
  swap(girl::path &p1, girl::path &p2)
  {
    p1.swap(p2);
  }  

}



#endif /*  ! GIRLPATH_HPP */
