/*
  Freeman code based paths
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlPath.hpp"

#include <cassert>

using namespace girl;

path::path(const girl::point &pStart, 
	   const girl::point &pEnd, 
	   const std::vector<char> &packedDirections, 
	   size_t nbDirections)
  : freemanCodePath(pStart, packedDirections, nbDirections), m_end(pEnd)
{
  assert(m_end == freemanCodePath::endPt());
}

//impl: could be removed if freemanCodePath::set_aux was virtual.
path::path(const girl::point &pStartPt, 
	   const std::deque<girl::direction> &dirs)
  : freemanCodePath(pStartPt, dirs), m_end(pStartPt)
{
  set_aux(dirs);
  assert(isValid());
}
    
//impl: could be removed if freemanCodePath::set_aux was virtual.
void
path::set(const girl::point &pStartPt,
	  const std::deque<girl::direction> &dirs) 
{
  const size_t s = dirs.size();
  m_path.clear();
  m_path.resize((s+3)/4, 0);
  m_start = pStartPt;
  m_length = s;
  set_aux(dirs);
  assert(isValid());
}

//impl: some code duplication with girlFreemanCodePath
//      but we don't want to compute endPt in girlFreemanCodePath 
//      and we want to avoid a second pass on dirs to compute endPt here.
void
path::set_aux(const std::deque<girl::direction> &dirs) 
{
  assert(m_path.size() == ((dirs.size()+3)/4));

  girl::point pt = startPt();
  std::deque<girl::direction>::const_iterator it = dirs.begin();
  const std::deque<girl::direction>::const_iterator itEnd = dirs.end();
  char *curB = &m_path[0];
  int mod = 0;
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    pt.move(d);
    *curB |= static_cast<char>(d << mod);
    mod += 2;
#if 1
    if (mod == 8) {
      mod = 0;
      ++curB;
    }
#else
    mod &= 7;
    curB += !mod;
#endif
  }
  m_end = pt;
}

void
path::invertOrientation()
{
  assert(isValid());
  assert(freemanCodePath::isValid());
  assert(m_end == freemanCodePath::endPt());
  freemanCodePath::invertOrientation();
  std::swap(m_start, m_end);

  assert(freemanCodePath::isValid());
  assert(m_end == freemanCodePath::endPt());
  assert(isValid());
}

void 
path::add(girl::direction d)
{
  assert(d != girl::INVALID);
  assert((m_length==0) || (*rbegin() != d.opposite()));

  if ((m_length&3) == 0)
    m_path.push_back(0);

  const size_t i = ((m_length & 3) << 1); //shift in char
  m_path[m_length >> 2] |= (d << i);
  ++m_length;
  m_end.move(d);
}

void
path::move(const girl::point &newStartPt)
{
  girl::point r(m_end.x()-m_start.x(), m_end.y()-m_start.y());
  m_start = newStartPt;
  m_end.set(newStartPt.x()+r.x(), newStartPt.y()+r.y());
  assert(isValid());
}

void
path::append(const girl::path &p)
{
  //TODO:OPTIM: could be faster 
  // (we could compute m_char final size, we could compute endPt only once, ...) 
  //warning: take into account that both *this & p can be empty.

  path::const_dir_iterator it = p.begin();
  const path::const_dir_iterator itEnd = p.end();
  for ( ; it!=itEnd; ++it) {
    add(*it);
  }
  
}

void
path::crop(size_t startIndex, size_t newLength)
{
  assert(startIndex+newLength <= length());
  assert(isValid());

  if (startIndex ==0) {
    //just a crop of the end of path
    // m_end must be updated
    
    girl::point lEndPt;
    if (newLength > length()/2) {
      //update m_end from the end

      lEndPt = m_end;
      path::const_reverse_dir_iterator it = rbegin();
      const size_t lEnd = length()-newLength;
      for (size_t i=0; i<lEnd; ++i, ++it) {
	lEndPt.move((*it).opposite());
      }
    }
    else {
      //update m_end from the start

      lEndPt = m_start;
      path::const_dir_iterator it = begin();
      for (size_t i=0; i<newLength; ++i, ++it) {
	lEndPt.move(*it);
      }
    }
    m_end = lEndPt;
    m_path.resize((newLength+3)/4, 0);
    m_length = newLength;
    
    //Warning:  unused bits must be set to 0 (to be able to do path::add())
    if (m_path.size() > 0) 
    {
      const size_t j = m_length - (m_length/4)*4;
      assert(j<4);
      char c2 = 0;
      for (size_t k=0; k<2*j; ++k) {
	c2 |= (1<<k);
      }
      m_path[m_length/4] &= c2;
    }

  }
  else {

    path::const_dir_iterator it = begin();

    girl::point lStartPt = m_start;
    for (size_t i=0; i<startIndex; ++i, ++it) {
      lStartPt.move(*it);
    }
      
    std::deque<girl::direction> dirs;
    girl::point lEndPt = lStartPt;
    for (size_t i=0; i<newLength; ++i, ++it) {
      const girl::direction d = *it;
      dirs.push_back(d);
      lEndPt.move(d);
    }

    *this = path(lStartPt, lEndPt, dirs);
 
    //TODO: OPTIM
    //we could be more low-level
    //For example, we could do memmove on m_path if and only if startIndex % 4 == 0

  }
 
  assert(length() == newLength);
  assert(isValid());
}
