/*
  Combinatorial map based topological region representation
  Copyright (C) 1992, 1996, 2004-2010 Achille Braquelaire, Jean-Philippe Domenger, Luc Brun, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlMap.hpp
 * @brief Combinatorial map based topological region representation.
 */

#ifndef GIRLMAP_HPP
#define GIRLMAP_HPP

#include <vector>
#include <map>

#include "girlToger.hpp" //for girl::dart
#include "girlGeoDart.hpp"
#include "girlHashMap.hpp"

#include "girlIO.hpp"


namespace girl {

  class map
  {

  public:
    /**
     * @brief Constructor.
     */
    map();

    /**
     * @brief Copy constructor.
     */
    map(const map &m);

    /**
     * @brief Assignment operator.
     */
    map &operator=(const map &m);


    /**
     * @brief Sets topology to geometry relation.
     *
     * Sets geometrical dart @a u for dart @a d, in map.
     */
    inline void setGe(girl::dart d, const girl::geoDart &u);

    /**
     * @brief Get geometry information.
     *
     * Get geometrical dart @a u associated with dart @a d in map.
     * @warning @a d must be a valid dart
     */
    inline void getGe(girl::dart d, girl::geoDart &u) const;

    /**
     * @brief Gives image of a dart @a d by alpha.
     */
    inline girl::dart alpha(girl::dart d) const;

    /**
     * @brief Gives image by sigma of dart @a d.
     */
    inline girl::dart getSigma(girl::dart d) const;

    /**
     * @brief Gives image by sigma-1 of dart @a d.
     */
    girl::dart getSigma_1(girl::dart d) const;

    /**
     * @brief Gives the sigma cycle of dart @a d, in *(@a cycle) 
     * and returns its length.
     * 
     * A sigma cycle has at most 4 darts.
     * @a d is the first dart of the returned cycle, 
     * followed by sigma(d) if any, followed by sigma(sigma(d)),  ...
     */
    int getSigmaCycle(girl::dart d, girl::dart cycle[4]) const;

    /**
     * @brief Gives the length of the sigma cycle of dart @a d.
     * @sa getSigmaCycle
     */
    int getSigmaCycleCardinality(girl::dart d) const;

    /**
     * @brief Gives image by phi of dart @a d.
     */
    inline girl::dart getPhi(girl::dart d) const;

    /**
     * @brief Gives image by phi_1 of dart @a d.
     */
    inline girl::dart getPhi_1(girl::dart d) const;

    /**
     * @brief Gives face corresponding to dart @a d in map.
     */
    inline girl::face getLambda(girl::dart d) const;

    
    /**
     * @brief Sets label of dart @a d to @a f.
     *
     * @warning do not check that d is a dart of face @a label.
     * @warning Does not ensure map consistency!
     */
    inline void setLambda(girl::dart d, girl::face label);

    /**
     * @brief Sets canonical dart to @a d for face @a label in map.
     *
     * @warning do not check that d is a dart of face @a label.
     */
    inline void setLambda_1(girl::face label, girl::dart d);

    /**
     * @brief Gives canonical dart corresponding to face @a label in map.
     */
    inline girl::dart getLambda_1(girl::face label) const;

    /**
     * @brief Changes face @a oldLabel to @a label, following 
     * @a old_label's phi cycle.
     * 
     * (canonical dart of face @a oldLabel becomes danonical dart 
     * of face @a label).
     */
    void mergeLambda(girl::face label, girl::face oldLabel);

    /**
     * @brief Sets a new label on phi cycle of dart @a d by lambda.
     */
    girl::face changeLambdaCycle(girl::dart d);

    /**
     * @brief Sets a specified label @a f on phi cycle of dart @a d by lambda.
     */
    void setLambdaCycle(girl::dart d, girl::face f);

    /**
     * @brief Inserts a new dart between @a d and the following dart in 
     * its phi cycle.
     *
     * @warning The new map is not minimal!
     * Inserted dart is labelled like @a d.
     * @return the new dart.
     */
    girl::dart splitEdge(girl::dart d);

    /**
     * @brief Inserts a dangling edge.
     * 
     * The new, returned dart nd will follow d in sigma cycle 
     *  that is sigma(d)==nd
     * 
     * @warning The new dart is not labelled (with lambda).
     * @warning Map becomes inconsistent. You should know what you are 
     * doing, so use with care.
     */
    girl::dart insertDart(girl::dart d);

    girl::dart insertSingleDart();

    /**
     * @brief Clears labels of @a d and alpha(@a d).,
     * that is lambda(d) = lambda(alpha(d)) = 0
     * 
     * @warning Does not ensure lambda-1 consistency!
     */
    inline void clearEdgeFaces(girl::dart d);

    /**
     * @brief Inserts a new segment between darts @a neg and @a pos.
     * 
     * @return the new positive dart of the segment.
     * If returned dart is nd, nd follows @a neg in its sigma cycle, 
     * and alpha(nd) follows @a pos in its sigma cycle.
     *
     * @warning The new dart is not labelled (with lambda).
     * @warning The new map is not minimal!
     */
    girl::dart insertSegment(girl::dart neg, girl::dart pos);

    /**
     * @brief Inserts a new region tangent to the node of dart @a d.
     *
     * @warning given dart @a d must belong to an artificial node 
     * (that is sigma(d)==alpha(d) & sigma(alpha(d))==d)
     * If returned dart is nd, we have :
     *  sigma(d) == alpha(nd), sigma(nd) == d, sigma(d) ==  sigma(d)
     *
     * @warning The new dart is not labelled (with lambda).
     */
    girl::dart insertTangent(girl::dart d);

    /**
     * @brief Join two nodes to make a tangent region.
     *
     * Dart @a d2 will be the canonical dart of the new tangent region.
     * @warning Darts @a d1 and @a d2 must belong to the same face by lambda.
     * Will make a node with a sigma cycle {d2, sigma(d2), d1, sigma(d1)}
     * @return label of phi*(d2).
     */
    girl::face joinNodes(girl::dart d1, girl::dart d2);

    /**
     * @brief Inserts a sigma cycle (d, alpha(d)) in the map,
     * that is inserts a new region with an artificial node and returns 
     * its canonical dart.
     *
     * The dart is associated with a new label by lambda.
     *
     * @sa insertCycle.
     */
    girl::dart insertMap();

    /**
     * @brief Inserts a sigma cycle (d, alpha(d)) in the map,
     * that is inserts a new region with an artificial node and returns 
     * its canonical dart.
     *
     * The dart is NOT associated with a new label by lambda.
     *
     * @sa insertMap.
     */
    girl::dart insertCycle();


    /**
     * @brief Removes the edge (@a d, alpha(@a d)) from the map.
     *
     * If @a d or alpha(@a d) was the canonical dart of its face, 
     * it is replaced 
     * (by its follower in Phi cycle).
     * If d and alpha(@a d) were not on an artificial node, their 
     * sigma cycle is updated.
     * After removal, darts @a d and alpha(@a d) are no longer in use.
     */
    void removeEdge(girl::dart d);

    /**
     * @brief Removes the sigma cycle (@a d, alpha(@a d)) from the map,
     * that is removes a region with an artificial node, with dart @a d.
     *
     * @warning does not free label (by lambda) of faces of darts @a d 
     * and alpha(@a d).
     */
    void removeMap(girl::dart d);

    /**
     * @brief Checks if the dart @a d has been deleted.
     *
     * @return true if dart has been deleted, false if it is still in use.
     */
    bool dartIsDeleted(girl::dart d) const;

    /**
     * @brief Extends one edge from @a d1 to @a d2.
     * Dart @a d1 is kept.
     *
     * May change canonical dart of regions of @a d2 or/and alpha(@a d2).
     * @warning dart @a d1 must have a sigma cycle of length 2,
     * @a d1 must be different than @a d2,
     * and there may be a phi path between @a d1 and @a d2 where each 
     * sigma cycle is of length 2.  
     */
    void extendEdge(girl::dart d1, girl::dart d2);

    /**
     * @brief Simplifies a loop keeping dart @a d.
     *
     * Simplifies, starting from @a d, a loop of artificial nodes,
     * and keeps a simple loop with only one artificial node, of dart @a d.
     * @warning @a d must an artificial node and there must be a phi cycle
     * where each sigma cycle is of length 2. 
     */
    void simplifyLoop(girl::dart d);

    /**
     * @brief Removes a node tangent region.
     *
     * @warning @a d and alpha(@a d) must be consecutive in sigma*(d),
     * that is sigma(d)==alpha(d) or sigma(alpha(d))==d.
     */
    void removeTangent(girl::dart d);

    /**
     * @brief Removes an unnecessary artificial node.
     *
     * @warning @a d and alpha(@a d) must be consecutive in sigma*(d),
     * that is sigma(d)==alpha(d) or sigma(alpha(d))==d.
     * But d must not be the only dart of Phi cycle : Phi(d)!=d
     */
    void removeUnnecessaryArtificialNode(girl::dart d);

    /**
     * @brief Tells if node at dart @a d is artificial
     *
     * A node is artificial if phi(d) == d && alpha(d) == sigma(d).
     */
    inline bool nodeIsArtificial(girl::dart d) const;

    /**
     * @brief Checks if a face @a f is marked in map.
     */
    bool faceIsMarked(girl::face f) const;

    /**
     * @brief Marks a face @a f in map.
     */
    void faceMark(girl::face f) const;

    /**
     * @brief Unmarks a face @a f in map.
     */
    void faceUnmark(girl::face f) const;

    /**
     * @brief Gives the next valid face, starting from a given face @a f.
     *
     * Starting from face 0, it allows to traverse all the existing faces.
     */
    girl::face getNextExistingFace(girl::face f) const;

    /*!
     * The functions below only manage the inclusion tree. 
     * Not the faces per se.
     */

#define GIRL_FINITE_FACE 0

    //! Basic inclusion tree management.

    /**
     * @brief Checks if a face @a f is infinite.
     */
    inline bool faceIsInfinite(girl::face f) const;

    /**
     * @brief Gives mother of a face @a f.
     */
    inline girl::face faceGetMother(girl::face f) const;

    /**
     * @brief Gives first daugther of a face @a f.
     *
     * Allows to traverse all daugthers of a face, with 
     * @a map_faceGetNextSister.
     */
    inline girl::face faceGetFirstDaughter(girl::face f) const;

    /**
     * @brief Gives next sister of face @a f.
     */
    inline girl::face faceGetNextSister(girl::face f) const;

    /**
     * @brief Removes all daughters of face @a f.
     *
     * @sa faceRemoveInfiniteFace.
     */
    void faceRemoveAllDaughters(girl::face f);//B should it be region ?

    /**
     * @brief Removes face @a f from its mother's daughters.
     */
    void faceRemoveInfiniteFace(girl::face f);

    /**
     * @brief Adds face @a d as daughter of face @a m.
     *
     * If face @a m had N daughters, she now has N+1 daughters.
     *
     * @warning Consider that that d is not already in a daughter 
     * list of an other face. 
     */
    void faceAddDaughter(girl::face m, girl::face d);


    /**
     * @brief Removes face @a f from the inclusion tree.
     *
     * Removes label by lambda.
     * In case of an infinite face, face is remove from its mother's daughters.
     */
    void faceRemove(girl::face f);

    /**
     * @brief Free face @a f.
     *
     * @warning Do not check consistency. (Mother must be 0)
     */
    inline void faceFree(girl::face f);

    /**
     * @brief Moves the daughters of \a src to \a dest.
     *
     * Current @a src daughters are reparented and added to @a dst daughters 
     * or its mother if @a dst is infinite). 
     * @a src has no daughter anymore.
     */
    void faceMoveDaughters(girl::face src, girl::face dest);

    /**
     * @brief Renames a finite face in both the map and inclusion tree.
     *
     * The face @a new should not be already in use. The face @a old 
     * will be freed.
     */
    //void faceRename(girl::face oldFace, girl::face newFace);


    /**
     * @brief Marks a dart @a d of map with mark @a m.
     */
    void markDart(girl::dart d, girl::toger::markFlags m);

    /**
     * @brief Inverts mark of dart @a d of map with mark @a m.
     */
    void xMarkDart(girl::dart d, girl::toger::markFlags m);

    /**
     * @brief Unmarks a dart @a d of map with mark @a m.
     */
    void unmarkDart(girl::dart d, girl::toger::markFlags m);

    /**
     * @brief Marks all darts of map with mark @a m.
     */
    void markAllDarts(girl::toger::markFlags m);

    /**
     * @brief Marks all darts of map with mark @a m.
     */
    void unmarkAllDarts(girl::toger::markFlags m);

    /**
     * @brief Checks if a dart @a d is marked in map, 
     * according to mark @a m.
     *
     * @return true if at least one bit of mark @a is set in mark of dart @a d.
     */
    bool dartIsMarked(girl::dart d, girl::toger::markFlags m) const;

    /**
     * @brief Gives mark of a dart @a d in map.
     */
    girl::toger::markFlags getDartMark(girl::dart d) const;

    /**
     * @brief Check overall map consistency.
     * Will print found errors, if any, to stderr. 
     */
    bool checkConsistency() const;

    /**
     * @brief Check dart @a d validity, in map,
     * that is dart really is in use. 
     */
    bool checkDart(girl::dart d) const;

    /**
     * @brief Check face @a f validity, in map,
     * that is face really is in use. 
     */
    bool checkFace(girl::face f) const;

    
    /**
     * @brief Check face @a f is a region.
     */
    bool checkRegion(girl::face f) const;

    /**
     * @brief Check taht face @a f is no more in use,  in map.
     */
    bool checkFaceIsFree(girl::face f) const;

    /**
     * Check consistency of use between a dart @a d and its face (by lambda),
     * in map.
     */
    bool checkDartFaceUse(girl::dart d) const;

    /**
     * Check if a dart @a d is not a dangling edge,
     * that is dart not in use and lambda(d) != lambda(alpha(d)),
     * in map.
     */
    bool checkDartIsNotDanglingEdge(girl::dart d) const;


    /**
     * Check consistency of beta and lambda for a face @a f,
     * that is lambda(beta(f)) == f, 
     * in map.
     */
    bool checkFaceBetaLambda(girl::face f) const;

    /**
     * Check consistency of dart cycle of a dart @a d, in map,
     * that is that for all dart di of phi*(d), sigma_1(phi(di) == alpha(di).
     */
    bool checkDartPhiCycle(girl::dart d) const;


    /**
     * Check consistency of dart cycle of a dart @a d, in map, 
     * according to lambda
     * that is that all darts of phi*(d) have the same face by lambda,
     * in girlmap.
     */
    bool checkDartPhiCycleLambda(girl::dart d) const;

    /**
     * Check face @a f phi cycle consistency.
     * Equivalent to :
     * @a checkFaceBetaLambda, 
     * @a checkDartPhiCycle,
     * @a checkDartPhiCycleLambda.
     */
    bool checkFacePhiCycle(girl::face f) const;

    /**
     * Check ascendance of a face @a f, in map,
     * that is either f has no mother (it's a finite face) 
     * or f has a mother (so f is infinite) and
     * then this mother (a finite face) has not itself a mother  
     */
    bool checkInfiniteFaceAncestry(girl::face f) const;

    /**
     * Check for a given face @a f, in map, 
     * if it is infinite and its canonical dart is d0 (d0=beta(f)),
     * then lambda(alpha(d0)) is finite.
     * @sa girlmap_checkFaceNeighbourhood 
     * @sa checkDartPhiCycle
     */
    bool checkInfiniteFaceCanonicalNeighbour(girl::face f) const;

    /**
     * Check for a given face @a f, in map, 
     * if it is infinite and its canonical dart is d0 (d0=beta(f)),
     * then for all d of phi*(alpha(d0)), lambda(d) is finite.
     * @sa checkDartPhiCycle 
     * @sa checkFaceCanonicalNeighbour
     */
    bool checkInfiniteFaceNeighbourhood(girl::face f) const;

    /**
     * Check for a given face @a f, in map,
     * if f is infinite then its mother must have f as daugther. 
     */
    bool checkInfiniteFaceDaughterOfMother(girl::face f) const;

    /**
     * Check several conditions on a face @a f, in map, if f is infinite 
     * @sa checkInfiniteFaceAncestry, 
     * @sa checkInfiniteFaceNeighbourhood
     * @sa checkInfiniteFaceLineage
     * @sa checkInfiniteFaceDaughterOfMother
     */
    bool checkInfiniteFaceConsistency(girl::face f) const;

    /**
     * Check for a given face @a f, of map,
     * that if f is finite and has daughters, 
     * then each of its daugthers has f as mother.
     */
    bool checkFaceLineage(girl::face f) const;


    /**
     * @brief crop map.
     *
     * @param ip1 : interpixel point of top left corner of crop box
     * @param regionsToRemove : outer regions that will be removed.
     * @param dartsNewOuterContour : darts of new outer contour with their corresponding previous dart in phi*.
     * @param nodesToRemove : useless nodes removed.
     * @param movedGeoDarts : moved geo darts.
     */
    void crop(const girl::point &ip1,
	      const girl::regionDeque &regionsToRemove,
	      const girl::hashMap<girl::dart, girl::dart> &dartsNewOuterContour,
	      std::deque<girl::point> &nodesToRemove,
	      std::map<girl::dart, girl::geoDart> &movedGeoDarts);
    

    /**
     * @brief Shrink memory.
     */
    void shrink(girl::hashMap<girl::face, girl::face> &renamedFaces, 
		girl::hashMap<girl::dart, girl::dart> &renamedDarts);


    /**
     * @brief Write map with writer @a w.
     *
     * @return true if write was ok, else false.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags flags) const;

    /**
     * @brief Read map with reader @a r.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags flags);
    
    /**
     * @brief Swap two map.
     *
     * Assign content of this to @a m and content of @a m to this.
     */
    void swap(girl::map &m);


  protected:

    static const unsigned int USED;
    static const unsigned int FREE;

    static const unsigned int DART_PRIVATE_MARK_SHIFT;
    static const unsigned int DART_PRIVATE_MARK_MASK;

    static const unsigned int DART_PUBLIC_MARK_SHIFT;
    static const unsigned int DART_PUBLIC_MARK_MASK;

    static const unsigned int FACE_PRIVATE_MARK_SHIFT;
    static const unsigned int FACE_PRIVATE_MARK_MASK;

    static const unsigned int FACE_PUBLIC_MARK_SHIFT;
    static const unsigned int FACE_PUBLIC_MARK_MASK;

    static const unsigned int DIR_SHIFT;
    static const unsigned int DIR_MASK;

    inline unsigned int getDartPrivateMark(girl::dart d) const;
    
    inline void setDartPrivateMark(girl::dart d, unsigned int val);

    inline unsigned int getFacePrivateMark(girl::face f) const;

    inline void setFacePrivateMark(girl::face f, unsigned int val);

    inline bool isFaceUsed(girl::face f) const;

    inline bool isFaceFree(girl::face f) const;

    inline bool isDartUsed(girl::dart d) const;

    inline bool isDartFree(girl::dart d) const;

    inline void setFaceUsed(girl::face f);

    inline void setFaceFree(girl::face f);

    inline void setDartUsed(girl::dart d);
    
    inline void setDartFree(girl::dart d);

    inline girl::dart sigma(girl::dart d) const;
    inline girl::dart &sigma(girl::dart d);
    inline girl::dart phi(girl::dart d) const;
    inline girl::dart &phi(girl::dart d);
    inline girl::face lambda(girl::dart d) const;
    inline girl::face &lambda(girl::dart d);
    inline girl::dart beta(girl::face f) const;
    inline girl::dart &beta(girl::face f);

    inline girl::direction getGeoDartDirection(girl::dart d) const;
    inline void setGeoDartDirection(girl::dart d, girl::direction dir);

    inline void setGeoDart(girl::dart dDst, girl::dart dSrc);

    inline void resetDartMarks(girl::dart d);

    bool checkDartBounds(girl::dart d) const;
    
    bool checkFaceBounds(girl::face f) const;
    
    bool checkDartValidity(girl::dart d) const;
    
    bool checkFaceValidity(girl::face f) const;
    
    bool checkFacePrevLink(girl::face f) const;
      
    bool checkFreeFaces() const;

    void setLambdaCycle_aux(girl::face label, girl::dart d);

    void reallocMap();

    girl::dart allocDart();

    void freeDart(girl::dart d);

    void freeDartPath(girl::dart d1, girl::dart d2);

    void reallocFace();

    girl::face allocLambda(girl::dart d);

    void freeLambda(girl::face label);

    bool isFacePubliclyMarked(girl::face f) const;

    void markPubliclyFace(girl::face f) const;

    void unmarkPubliclyFace(girl::face f) const;

    size_t nbFreeFaces() const;
    size_t nbFreeDarts() const;


  protected:

    //Warning: if these struct are changed, 
    // write/read methods must also be changed

    struct dartCell
    {
      //do not store a geoDart, to be able to mix direction & mark.
      girl::point pt;
      int mark_dir;
      girl::dart phi;
      girl::face lambda;

      dartCell()
	: pt(0, 0), mark_dir(0), phi(0), lambda(0)
      {
      }

    };
    
    struct faceCell
    {
      girl::dart d; //canonical dart
      girl::face mother, daughter_prev, next;

      faceCell()
	: d(0), mother(0), daughter_prev(0), next(0)
      {
      }

    };
    
    
    mutable std::vector<dartCell> m_mapV;
    std::vector<faceCell> m_faceV;
    mutable dartCell *m_map;
    faceCell *m_face;
    unsigned int m_mapDim;
    unsigned int m_faceDim;


  };


  //--inline functions --------------

  inline 
  girl::dart 
  map::alpha(girl::dart d) const
  {
    return -d;
  }

  inline 
  girl::dart
  map::sigma(girl::dart d) const
  {
    return m_map[alpha(d)].phi;
  }
  
  inline 
  girl::dart &
  map::sigma(girl::dart d)
  {
    return m_map[alpha(d)].phi;
  }
  
  inline 
  girl::dart 
  map::phi(girl::dart d) const
  {
    return m_map[d].phi;
  }
  
  inline 
  girl::dart &
  map::phi(girl::dart d)
  {
    return m_map[d].phi;
  }
  
  inline 
  girl::face 
  map::lambda(girl::dart d) const
  {
    return m_map[d].lambda;
  }
  
  inline 
  girl::face &
  map::lambda(girl::dart d)
  {
    return m_map[d].lambda;
  }
  
  inline 
  girl::dart 
  map::beta(girl::face f) const
  {
    return m_face[f].d;
  }
  
  inline 
  girl::dart &
  map::beta(girl::face f)
  {
    return m_face[f].d;
  }

  inline
  bool
  map::nodeIsArtificial(girl::dart d) const
  {
    return d == phi(d) && alpha(d) == sigma(d);
    //return sigma(sigma(d))==d; //not the same : does not garantee that the node is the lonly one on the face !?
    //OPTIM:impl: which one is faster ?
    //may only have one lookup in the first one in case of a non artificial node... 
  }


  inline 
  bool 
  map::faceIsInfinite(girl::face f) const
  {
    assert(checkFaceValidity(f));
    return m_face[f].mother != GIRL_FINITE_FACE;
  }

  inline 
  girl::face 
  map::faceGetMother(girl::face f) const
  {
    assert(checkFaceValidity(f));
    assert(isFaceUsed(f));
    return m_face[f].mother;
  }

  inline 
  girl::face 
  map::faceGetFirstDaughter(girl::face f) const
  {
    assert(checkFace(f));
    return m_face[f].daughter_prev;
  }

  inline 
  girl::face 
  map::faceGetNextSister(girl::face f) const
  {
    assert(checkFace(f));
    return m_face[f].next;
  }
 
  inline
  void
  map::faceFree(girl::face f)
  {
    freeLambda(f);
  }


  inline 
  void 
  map::setGe(girl::dart d, const girl::geoDart &u)
  {
    m_map[d].pt.set(u.x(), u.y());
    setGeoDartDirection(d, u.dir());
  }

  inline 
  void 
  map::getGe(girl::dart d, girl::geoDart &u) const
  {
    assert(checkDartValidity(d));
    u.set(m_map[d].pt.x(), m_map[d].pt.y(), getGeoDartDirection(d));
  }

  inline 
  girl::dart 
  map::getSigma(girl::dart d) const
  {
    assert(checkDart(d));
    return sigma(d);
  }

  inline 
  girl::dart 
  map::getPhi(girl::dart d) const
  {
    assert(checkDart(d));
    return phi(d);
  }

  inline 
  girl::dart 
  map::getPhi_1(girl::dart d) const
  {
    assert(checkDart(d));
    return alpha(getSigma_1(d));
  }

  inline 
  girl::face 
  map::getLambda(girl::dart d) const
  {
    assert(checkDart(d));
    return lambda(d);
  }

  inline 
  void 
  map::setLambda(girl::dart d, girl::face label)
  {
    assert(checkDart(d));
    assert(checkFace(label));
    lambda(d) = label;
  }

  inline 
  void 
  map::setLambda_1(girl::face label, girl::dart d)
  {
    assert(checkFace(label));
    beta(label) = d;
  }

  inline 
  girl::dart 
  map::getLambda_1(girl::face label) const
  {
    assert(checkFace(label));
    return beta(label);
  }

  inline 
  void 
  map::clearEdgeFaces(girl::dart d)
  {
    lambda(d) = lambda(alpha(d)) = 0;
  }


  inline 
  unsigned int
  map::getDartPrivateMark(girl::dart d) const
  {
    return (m_map[d].mark_dir >> DART_PRIVATE_MARK_SHIFT)&DART_PRIVATE_MARK_MASK;
  }

  inline 
  void
  map::setDartPrivateMark(girl::dart d, unsigned int val)
  {
    //m_map[d].mark_dir |= ((val & DART_PRIVATE_MARK_MASK) << DART_PRIVATE_MARK_SHIFT);

    m_map[d].mark_dir = ((m_map[d].mark_dir & ~(DART_PRIVATE_MARK_MASK << DART_PRIVATE_MARK_SHIFT))
			 | ((val & DART_PRIVATE_MARK_MASK) << DART_PRIVATE_MARK_SHIFT));
    

    assert(getDartPrivateMark(d) == val);
  }

  inline 
  unsigned int
  map::getFacePrivateMark(girl::face f) const
  {
    assert(checkFaceBounds(f));
    const girl::dart d = beta(f);
    assert(checkDartBounds(d));
    return (m_map[d].mark_dir >> FACE_PRIVATE_MARK_SHIFT)&FACE_PRIVATE_MARK_MASK;
  }

  inline 
  void
  map::setFacePrivateMark(girl::face f, unsigned int val)
  {
    assert(checkFaceBounds(f));
   
    const girl::dart d = beta(f);
    assert(checkDartBounds(d));
 
    m_map[d].mark_dir = ((m_map[d].mark_dir & ~(FACE_PRIVATE_MARK_MASK << FACE_PRIVATE_MARK_SHIFT))
		      | ((val & FACE_PRIVATE_MARK_MASK) << FACE_PRIVATE_MARK_SHIFT));

    assert(getFacePrivateMark(f) == val);
  }

  inline 
  bool
  map::isFaceUsed(girl::face f) const
  {
    assert(checkFaceBounds(f));
    return getFacePrivateMark(f) == USED;
  }

  inline 
  bool
  map::isFaceFree(girl::face f) const
  {
    assert(checkFaceBounds(f));
    return !isFaceUsed(f);
  }

  inline 
  bool
  map::isDartUsed(girl::dart d) const
  {
    assert(checkDartBounds(d));
    return getDartPrivateMark(d) == USED;
  }

  inline 
  bool
  map::isDartFree(girl::dart d) const
  {
    assert(checkDartBounds(d));
    return !isDartUsed(d);
  }

  inline 
  void
  map::setFaceUsed(girl::face f)
  {
    assert(checkFaceBounds(f));
    setFacePrivateMark(f, USED);
    assert(isFaceUsed(f));
  }

  inline 
  void
  map::setFaceFree(girl::face f)
  {
    assert(checkFaceBounds(f));
    unmarkPubliclyFace(f);
    setFacePrivateMark(f, FREE);
    assert(isFaceFree(f));
  }

  inline 
  void
  map::setDartUsed(girl::dart d)
  {
    assert(checkDartBounds(d));
    setDartPrivateMark(d, USED);
    assert(isDartUsed(d));
  }

  inline 
  void
  map::setDartFree(girl::dart d)
  {
    assert(checkDartBounds(d));
    unmarkDart(d, girl::toger::MARK_ANY);
    setDartPrivateMark(d, FREE);
    assert(isDartFree(d));
  }

  inline
  girl::direction 
  map::getGeoDartDirection(girl::dart d) const
  {
    assert(checkDartBounds(d));
    return girl::direction(static_cast<girl::eDirection>((m_map[d].mark_dir >> DIR_SHIFT)&DIR_MASK));
  }

  inline
  void 
  map::setGeoDartDirection(girl::dart d, girl::direction dir)
  {
    assert(checkDartBounds(d));
    m_map[d].mark_dir = ((m_map[d].mark_dir & ~(DIR_MASK << DIR_SHIFT))
			 | ((static_cast<int>(dir) & DIR_MASK) << DIR_SHIFT));

    assert(getGeoDartDirection(d) == dir);
  }

  inline 
  void 
  map::setGeoDart(girl::dart dDst, girl::dart dSrc)
  {
    assert(checkDartBounds(dSrc));
    assert(checkDartBounds(dDst));
    
    m_map[dDst].pt = m_map[dSrc].pt;
    setGeoDartDirection(dDst, getGeoDartDirection(dSrc));

    assert(getGeoDartDirection(dDst) == getGeoDartDirection(dSrc));
  }

  inline
  void
  map::resetDartMarks(girl::dart d)
  {
    assert(checkDartBounds(d));
 
    m_map[d].mark_dir &= ~((DART_PRIVATE_MARK_MASK<<DART_PRIVATE_MARK_SHIFT)
			   |(DART_PUBLIC_MARK_MASK<<DART_PUBLIC_MARK_SHIFT)
			   |(FACE_PRIVATE_MARK_MASK<<FACE_PRIVATE_MARK_SHIFT)
			   |(FACE_PUBLIC_MARK_MASK<<FACE_PUBLIC_MARK_SHIFT));

  }


  inline
  void
  swap(map &m1, map &m2)
  {
    m1.swap(m2);
  }

}

namespace std {

  template <>
  inline
  void
  swap(girl::map &m1, girl::map &m2)
  {
    m1.swap(m2);
  }
  
}

#endif /* !GIRLMAP_HPP */

    
