/*
  Combinatorial map based topological region representation
  Copyright (C) 1992, 1996, 2004-2010 Achille Braquelaire, Jean-Philippe Domenger, Luc Brun, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlMap.hpp"

#include <iostream>
#include <cassert>
#include <cstdlib> //abs
#include <cstring> //memcpy


#include <climits> //INT_MAX, INT_MIN

#include "girlEndian.hpp"
#include "girlHashFunc.hpp"
#include "girlUnorderedSet.hpp"


using namespace girl;


static const int GIRLMAP_DART_ALLOC = 2000;
static const int GIRLMAP_FACE_ALLOC = 1000;


const unsigned int girl::map::USED=0;
const unsigned int girl::map::FREE=1;

const unsigned int girl::map::DART_PRIVATE_MARK_SHIFT=3;
const unsigned int girl::map::DART_PRIVATE_MARK_MASK=0x1;

const unsigned int girl::map::DART_PUBLIC_MARK_SHIFT=8;
const unsigned int girl::map::DART_PUBLIC_MARK_MASK=0xff;

const unsigned int girl::map::FACE_PRIVATE_MARK_SHIFT=4;
const unsigned int girl::map::FACE_PRIVATE_MARK_MASK=0x1;

const unsigned int girl::map::FACE_PUBLIC_MARK_SHIFT=16;
const unsigned int girl::map::FACE_PUBLIC_MARK_MASK=0xff;

const unsigned int girl::map::DIR_SHIFT=0;
const unsigned int girl::map::DIR_MASK=0x3;


/**
 * darts are in [-N; +N]
 * face are in [-1; +M]
 *
 * free darts are linked from dart 0, sur phi.
 * free faces are linked from face 0
 *
 * At construction,
 * - dart 1 & -1 are used for external infinite face
 *   positive darts (0, 2, 3....N) are linked by phi (phi(i) == i+1)
 * - face -1 = external infinite face, has for canonical dart -1 
 *   face 1 = internal finite face, has for canonical dart 1 
 *   free faces are linked from 0 (0, 2, 3,...M) by next
 *
 *  face[i].daughter_prev stores daughter for finite face (mother != 0) 
 *  and previous daughter (sibling) for infinite faces (mother == 0)
 *  Daughters do not form a loop (firstDaugther.daughter_prev = 0 & lastDaughter.next = 0)
 */


#define MAP_INVARIANT_COND			\
  (m_map != 0)					\
  && (m_face != 0)				\
  && (m_mapV.size() == 2*m_mapDim+1)		\
  && (m_faceV.size() == m_faceDim+1)		\
  && (m_mapV.size() > 0)			\
  && (m_faceV.size() > 0)			\
  && (&m_mapV[0] + m_mapDim == m_map)		\
  && (&m_faceV[1] == m_face)



map::map()
  : m_mapV(), m_faceV(), m_map(0), m_face(0), 
    m_mapDim(GIRLMAP_DART_ALLOC), m_faceDim(GIRLMAP_FACE_ALLOC)
{
  const int size = GIRLMAP_DART_ALLOC * 2 + 1;
  
  //- m_map initialization
  m_mapV.resize(size);
  m_map = &m_mapV[0] + GIRLMAP_DART_ALLOC;
  phi(1) = 1;
  phi(alpha(1)) = alpha(1);
  lambda(1) = 1;
  lambda(alpha(1)) = -1;
  setDartUsed(1);
  setDartUsed(alpha(1));

  //  init internal free-list
  phi(0) = 2;
  setDartFree(0);
  for (int i = 2; i < GIRLMAP_DART_ALLOC - 1; ++i) 
    {
      phi(i) = i+1;
      setDartFree(alpha(i));
      setDartFree(i);
    }
  setDartFree(alpha(GIRLMAP_DART_ALLOC-1));
  setDartFree(GIRLMAP_DART_ALLOC-1);
  
  //- m_face initialization
  m_faceV.resize(GIRLMAP_FACE_ALLOC + 1);
  m_face = &m_faceV[1];
  beta(1) = 1;
  m_face[1].mother = GIRL_FINITE_FACE;
  setFaceUsed(1);
  
  beta(-1) = -1;
  m_face[-1].mother = -1; //FIXME
  setFaceUsed(-1);

  //  init internal free-list
  m_face[0].next = 2;
  setFaceFree(0);
  for (size_t i = 2; i < m_faceDim - 1; ++i) 
    {
      m_face[i].next = i+1;
      setFaceFree( static_cast<girl::face>(i) ); //B:useless ! we mark always the same dart !!!
    }
  m_face[m_faceDim - 1].next = 0;
  setFaceFree( static_cast<girl::face>(m_faceDim - 1) );

  assert(MAP_INVARIANT_COND);
}

map::map(const map &m)
  : m_mapV(m.m_mapV),
    m_faceV(m.m_faceV),
    m_map(&m_mapV[0] + m.m_mapDim),
    m_face(&m_faceV[1]),
    m_mapDim(m.m_mapDim),
    m_faceDim(m.m_faceDim)
{
  assert(MAP_INVARIANT_COND);
}


map &
map::operator=(const map &m)
{
  if (this != &m) {
    m_mapV = m.m_mapV;
    m_faceV = m.m_faceV;
    m_map = &m_mapV[0] + m.m_mapDim;
    m_face = &m_faceV[1];
    m_mapDim = m.m_mapDim;
    m_faceDim = m.m_faceDim;
  }
  assert(MAP_INVARIANT_COND);
  return *this;
}


bool
map::checkDartBounds(girl::dart d) const
{
  return (d<static_cast<int>(m_mapDim) && d>-static_cast<int>(m_mapDim));
}

bool
map::checkFaceBounds(girl::face f) const
{
  return (f>=-1 && f<static_cast<int>(m_faceDim));
}

bool
map::checkDartValidity(girl::dart d) const
{
  return (d!=0 && checkDartBounds(d));
}

bool
map::checkFaceValidity(girl::face f) const
{
  return (f!=0 && checkFaceBounds(f));
}

bool
map::checkFacePrevLink(girl::face f) const
{
  //a face has its prev==0 iff it is not used or 
  //its mother is 0 or it is the first daughter of its mother

  if (f == -1)
    return true;

  if (m_face[f].daughter_prev == 0)
    return (!isFaceUsed(f) || m_face[f].mother == 0 || m_face[m_face[f].mother].daughter_prev==f);
  
  if (isFaceUsed(f) && m_face[f].mother != 0 && m_face[m_face[f].mother].daughter_prev != f)
    return m_face[f].daughter_prev != 0;

  return true;
}



girl::dart 
map::getSigma_1(girl::dart d) const
{
  assert(checkDart(d));
  girl::dart dt = d;

#ifndef NDEBUG
  int i=4;
  while (sigma(dt) != d && i!=0) {
    dt = sigma(dt);
    --i;
  }
  assert(i!=0);
#else
  while (sigma(dt) != d)
    dt = sigma(dt);
#endif
  return dt;
}

int 
map::getSigmaCycle(girl::dart d, girl::dart cycle[4]) const
{
  assert(checkDart(d));

  cycle[0] = cycle[1] = cycle[2] = cycle[3] = 0;

  int n = 1;
  girl::dart *c = cycle;

  *c = d;
  girl::dart dt;
  while ((dt = sigma(*c)) != d)
    {
      *++c = dt;
      ++n;
    }

  return n;
}

int 
map::getSigmaCycleCardinality(girl::dart d) const
{
  assert(checkDart(d));

  int n = 1;
  girl::dart sd = d;
  while ((sd = sigma(sd)) != d)
    ++n;

  return n;
}


/*
    
 * @brief Gives the phi cycle from dart @a d, in *(@a cycle) and returns its length.
 * 
 * @warning it's client's responsibility to free the returned cycle.
    

 size_t
 map::getPhiCycle(girl::dart d, girl::dart **cycle) const
 {
 assert(checkDart(b));
 std::deque<girl::dart> s;

 girl::dart current = b;
 s.push_back(current);
 while ((current = phi(current)) != b)
 s.push_back(current);

 const size_t n = s.size();
 girl::dart *c = *cycle = (girl::dart*)malloc(n * sizeof(girl::dart));

 for (size_t i=0; i<n; ++i) {
 *c = s.popBack();
 ++c;
 }

 return n;
 }
*/


void
map::setLambdaCycle_aux(girl::face label, girl::dart d)
{
  girl::dart current = d;
  do
    {
      lambda(current) = label;
      current = phi(current);
    }
  while (current != d); 
}

void
map::reallocMap()
{
  assert(MAP_INVARIANT_COND);

  const size_t old_dim = m_mapDim;
  const size_t old_size = m_mapDim * 2 + 1;
  size_t new_dim = (INT_MAX-1)/2;
  assert(old_dim<=new_dim);
  if (old_size < new_dim)
    new_dim = old_dim * 2;
  assert(old_dim<new_dim);
  const size_t new_size = new_dim * 2 + 1;

  const size_t diff = static_cast<size_t>(new_dim - old_dim);
  std::vector<dartCell> mapV(new_size);

  //TODO: use std::copy !?
  memcpy(&mapV[0] + diff, &m_mapV[0], old_size * sizeof(map::dartCell));
  
  m_mapV.swap(mapV);
  m_mapDim = new_dim;
  m_map = &m_mapV[0] + m_mapDim;

  //- update internal free list
  assert(m_mapDim > 0);
  const size_t mapDim_1 = m_mapDim - 1;
  
  m_map[mapDim_1].phi = m_map[0].phi;
  setDartFree( static_cast<girl::dart>(mapDim_1) );
  setDartFree(alpha( static_cast<girl::dart>(mapDim_1) ) );
  m_map[0].phi = m_mapDim - diff;
  for (size_t i = m_mapDim - diff; i < mapDim_1; ++i)
    {
      phi(i) = i+1;
      setDartFree( static_cast<girl::dart>(i) );
      setDartFree(alpha(static_cast<girl::dart>(i)));
      //lambda(i) = lambda(alpha(i)) = 0;
    }

  assert(MAP_INVARIANT_COND);
}

girl::dart
map::allocDart()
{
  if (m_map[0].phi == 0)
    reallocMap();

  const girl::dart d = m_map[0].phi;
  m_map[0].phi = m_map[d].phi;

  //B self->map[d].lambda = 0;
  m_map[d].phi = m_map[alpha(d)].phi = 0; 
  m_map[d].lambda = m_map[alpha(d)].lambda = 0;
  resetDartMarks(d);
  resetDartMarks(alpha(d));
  setDartUsed(d);
  setDartUsed(alpha(d));

  assert(d);
  assert(checkDart(d));
  assert(checkDart(alpha(d)));

  return abs(d);
}

void
map::freeDart(girl::dart d)
{
  m_map[d].phi = m_map[0].phi;
  m_map[0].phi = d;
  m_map[alpha(d)].lambda = m_map[d].lambda = 0;
  resetDartMarks(d);
  resetDartMarks(alpha(d));  
  setDartFree(d);
  setDartFree(alpha(d));  
}

/**
 * @brief free dart path.
 * Darts from @a b1 to @a b2 must be a phi path where each sigma cycle (except @a b1) is of length 2.
 * Both darts @a b1 and @a b2 are freed.
 */
void
map::freeDartPath(girl::dart d1, girl::dart d2)
{
  girl::dart d = d1;

  while (d != d2)
    {
      resetDartMarks(d);
      resetDartMarks(alpha(d));
      setDartFree(d);
      setDartFree(alpha(d));
      lambda(alpha(d)) = lambda(d) = 0;
      d = phi(d);
    }
  resetDartMarks(d2);
  resetDartMarks(alpha(d2));
  setDartFree(d2);
  setDartFree(alpha(d2));
  lambda(alpha(d2)) = lambda(d2) = 0;
  
  m_map[d2].phi = m_map[0].phi;
  m_map[0].phi = d1;
}

void
map::reallocFace()
{
  assert(MAP_INVARIANT_COND);

  const size_t oldDim = m_faceDim;
  const size_t newDim = m_faceDim*2;
  assert(newDim>oldDim);
 
  //TODO:clean
  //std::vector<faceCell> faceV(newDim);
  //memcpy(&faceV[0], &m_faceV[0], (oldDim+1)*sizeof(faceCell));
  m_faceV.resize(newDim+1);
  
  m_face = &m_faceV[1];
  m_faceDim = newDim;

  //- update internal free list
  m_face[0].next = oldDim;
  const size_t faceDim_1 = m_faceDim - 1;
  for (size_t i = oldDim; i < faceDim_1; ++i)
    {
      m_face[i].next = i+1;
      beta(i) = 0;
      setFaceFree(i); //useless as it will always mark dart 0 !!!
    }
  m_face[faceDim_1].next = 0;
  setFaceFree(faceDim_1);

  assert(MAP_INVARIANT_COND);
}

girl::face
map::allocLambda(girl::dart d)
{
  if (m_face[0].next == 0)
    reallocFace();

  const girl::face newLabel = m_face[0].next;
  assert(newLabel);
  assert(newLabel < static_cast<int>(m_faceDim));
  m_face[0].next = m_face[newLabel].next;

  beta(newLabel) = d;
  m_face[newLabel].mother = GIRL_FINITE_FACE;
  m_face[newLabel].daughter_prev =
    m_face[newLabel].next = 0;
  setFaceUsed(newLabel);

  assert(checkFace(newLabel));

  return newLabel;
}

void
map::freeLambda(girl::face label)
{
  m_face[label].next = m_face[0].next;
  m_face[0].next = label;
  beta(label) = 0; 
  //setFaceFree(label);
}

void 
map::mergeLambda(girl::face label, girl::face oldLabel)
{
  assert(checkFace(oldLabel));

  const girl::dart oldDart = beta(oldLabel);

  setLambdaCycle_aux(label, oldDart);

  beta(label) = oldDart;
  setFaceUsed(label);
}

girl::face 
map::changeLambdaCycle(girl::dart d)
{
  assert(checkDart(d));
  const girl::face newLabel = allocLambda(d);

  setLambdaCycle_aux(newLabel, d);

  return newLabel;
}

void 
map::setLambdaCycle(girl::dart d, girl::face f)
{
  assert(checkDart(d));
  assert(checkFace(f));
  beta(f) = d;
  setLambdaCycle_aux(f, d);
  //setFaceUsed(f);
}

girl::dart 
map::splitEdge(girl::dart d)
{
  const girl::dart s_1 = getSigma_1(alpha(d));

  const girl::dart nd = allocDart();

  sigma(s_1) = alpha(nd);
  sigma(alpha(nd)) = sigma(alpha(d));

  sigma(nd) = alpha(d);
  sigma(alpha(d)) = nd;

  lambda(nd) = lambda(d);
  lambda(alpha(nd)) = lambda(alpha(d));

  markDart(nd, getDartMark(d));
  markDart(alpha(nd), getDartMark(alpha(d)));


  assert(isDartUsed(nd));

  return nd;  
}


girl::dart
map::insertSingleDart()
{
  const girl::dart nd = allocDart();

  sigma(nd) = nd;
  sigma(alpha(nd)) = alpha(nd);

  return nd;
}


girl::dart 
map::insertDart(girl::dart d)
{
  assert(d); 
  //we can not have d==0, because sigma(0) is actually next free dart !

  const girl::dart nd = allocDart();
  //const girl::face f = lambda(d);

  sigma(alpha(nd)) = alpha(nd);
  sigma(nd) = sigma(d);
  sigma(d) = nd;

  //lambda(d) = lambda(alpha(d)) = f;

  return nd;
}

girl::dart 
map::insertSegment(girl::dart neg, girl::dart pos)
{
  assert(neg);
  assert(pos);

  //nd is labelled the same as alpha(neg), and alpha(nd) is labelled the same as alpha(pos).

  //const girl::face label1 = lambda(alpha(neg));
  //const girl::face label2 = lambda(alpha(pos));

  const girl::dart d = allocDart();

  sigma(alpha(d)) = sigma(pos);
  sigma(pos) = alpha(d);

  sigma(d) = sigma(neg);
  sigma(neg) = d;

  //lambda(d) = label1;
  //lambda(alpha(d)) = label2;

  /*   if (label1) */
  /*     beta(label1) = d; */
  /*   if (label2) */
  /*     beta(label2) = alpha(d); */

  return d;
}

girl::dart 
map::insertTangent(girl::dart d)
{
  assert(d);

  const girl::dart nd = allocDart();

  const girl::dart sd = sigma(d);
  sigma(d) = alpha(nd);
  sigma(alpha(nd)) = nd;
  sigma(nd) = sd;

  //  m_map[nd].lambda = allocLambda(nd);
  //  m_map[alpha(nd)].lambda = m_map[alpha(d)].lambda;

  return nd;
}

girl::face 
map::joinNodes(girl::dart d1, girl::dart d2)
{
  const girl::dart sd1 = sigma(d1);
  const girl::dart sd2 = sigma(d2);
  const girl::face f1 = lambda(d1);

  assert(f1 == lambda(d2));
  
  sigma(sd2) = d1;
  sigma(sd1) = d2;

  beta(f1) = d1;
  setFaceUsed(f1);

  return changeLambdaCycle(d2);
}

girl::dart 
map::insertMap()
{
  const girl::dart d = allocDart();

  sigma(d) = alpha(d);
  sigma(alpha(d)) = d;

  lambda(d) = allocLambda(d);
  lambda(alpha(d)) = allocLambda(alpha(d));

  return d;
}

girl::dart 
map::insertCycle()
{
  const girl::dart d = allocDart();

  sigma(d) = alpha(d);
  sigma(alpha(d)) = d;

  return d;
}



void 
map::removeEdge(girl::dart d)
{
  const girl::face label1 = lambda(d);
  const girl::face label2 = lambda(alpha(d));

  d = abs(d); //B ???

  lambda(d) = lambda(alpha(d)) = 0;

  if (beta(label1) == d) {
    beta(label1) = phi(d);
    setFaceUsed(label1);
  }

  if (beta(label2) == alpha(d)) {
    beta(label2) = phi(alpha(d));
    setFaceUsed(label2);
  }

  if (phi(d) != d)
    {
      girl::dart temp;
      temp = getSigma_1(d);
      sigma(temp) = sigma(d);

      temp = getSigma_1(alpha(d));
      sigma(temp) = sigma(alpha(d));
    }

  freeDart(d);
}

void 
map::removeMap(girl::dart d)
{
  assert(sigma(d)==alpha(d) && sigma(alpha(d))==d);

  //d = abs(d);

  freeDart(d);
}

void
map::removeUnnecessaryArtificialNode(girl::dart d)
{
  assert(sigma(sigma(d))==d);
  assert(phi(d) != d);

  const girl::dart d1 = sigma(d);
  const girl::dart s_1 = getSigma_1(alpha(d));
  const girl::dart sad = sigma(alpha(d));
  
  sigma(s_1) = d1;
  sigma(d1) = sad;
  
  freeDart(d);
}


bool 
map::dartIsDeleted(girl::dart d) const
{
  return isDartFree(d);
}

void 
map::extendEdge(girl::dart d1, girl::dart d2)
{
  assert(sigma(sigma(d1)) == d1);
  assert(d2 != d1);

  /*B.
    We could have -d1> <d1 . -d3> <d3 . -d4> <d4 . -d2> <d2 ... 
    where sigma*(d2) node is not artificial, and sigma*(d1) is artificial.

    See section 6.2.3 of Luc Brun's Phd Thesis (with b=-d1, b'=-d2)
  */

  const girl::dart sm = getSigma_1(d2);
  const girl::dart s = sigma(d2);
  const girl::face l1 = lambda(d2);
  const girl::face l2 = lambda(alpha(d2));

  setGeoDart(d1, d2);

  freeDartPath(d2, alpha(sigma(d1)));

  sigma(sm) = d1;
  sigma(d1) = s;
  beta(l1) = d1;
  setFaceUsed(l1);
  beta(l2) = alpha(d1);
  setFaceUsed(l2);
}

void 
map::simplifyLoop(girl::dart d)
{
  assert(sigma(sigma(d)) == d);//artifial node

  const girl::face l1 = lambda(d);
  const girl::face l2 = lambda(alpha(d));
  const girl::dart fd = phi(d);
  const girl::dart sd = sigma(d);

  sigma(d) = alpha(d);
  sigma(alpha(d)) = d;
  setGeoDart(alpha(d), sd);

  freeDartPath(fd, alpha(sd));

  beta(l1) = d;
  setFaceUsed(l1);
  beta(l2) = alpha(d);
  setFaceUsed(l2);
}

void 
map::removeTangent(girl::dart d)
{
  girl::dart d1 = sigma(alpha(d));
  girl::dart d2 = sigma(d);

  assert((d1 == d) || (d2 == alpha(d)));

  if (d1 == d)
    {
      d1 = sigma(d2);
      //girl::face f = m_map[d].lambda;
      beta(lambda(alpha(d))) = d2;
      setFaceUsed(lambda(alpha(d)));
    }
  else // d2 == alpha(d)
    {
      d2 = sigma(d1);
      //girl::face f = m_map[alpha(d)].lambda;
      beta(lambda(d)) = d1;
      setFaceUsed(lambda(d));
    }

  sigma(d2) = d1;
  sigma(d1) = d2;

  freeDart(abs(d));
  //  freeLambda(f); 
}

bool
map::isFacePubliclyMarked(girl::face f) const
{
  return ((m_map[beta(f)].mark_dir >> FACE_PUBLIC_MARK_SHIFT) & FACE_PUBLIC_MARK_MASK) != 0;
}

void
map::markPubliclyFace(girl::face f) const //impl:must be const for toger
{
  m_map[beta(f)].mark_dir |= ((1 & FACE_PUBLIC_MARK_MASK)<<FACE_PUBLIC_MARK_SHIFT);
}

void
map::unmarkPubliclyFace(girl::face f) const //impl:must be const for toger
{
  m_map[beta(f)].mark_dir &= ~((1 & FACE_PUBLIC_MARK_MASK)<<FACE_PUBLIC_MARK_SHIFT);
}



bool 
map::faceIsMarked(girl::face f) const
{
  return isFacePubliclyMarked(f);
}

void 
map::faceMark(girl::face f) const //impl: must be const for toger
{
  assert(checkFace(f));
  markPubliclyFace(f);
  assert(checkFace(f));
}

void 
map::faceUnmark(girl::face f) const  //impl: must be const for toger
{
  assert(checkFace(f));
  unmarkPubliclyFace(f);
  assert(checkFace(f));
}

girl::face 
map::getNextExistingFace(girl::face f) const
{
  while ((++f < static_cast<int>(m_faceDim)) && (isFaceFree(f)))
    ;
  
  if (f < static_cast<int>(m_faceDim))
    return f;
  return 0;

}


void 
map::faceRemoveAllDaughters(girl::face f)
{
  assert(f != 0);

  girl::face fi = m_face[f].daughter_prev;
  assert(m_face[fi].mother == f);
  m_face[f].daughter_prev = 0;
  while (fi)
    {
      const girl::face next = m_face[fi].next;
      m_face[fi].mother =
	m_face[fi].next = 
	m_face[fi].daughter_prev = 0;
      fi = next;
    }
}

void 
map::faceRemoveInfiniteFace(girl::face f)
{
  assert(isFaceUsed(f));

  const girl::face mf = m_face[f].mother;
  //assert(mf != 0); //No ! because can be called from 
  // girl::togerImpl::seekKnownInnerContourS where infinite face has no mother
  
  if (m_face[f].daughter_prev == 0) 
    //equivalent to (m_face[mf].daughter_prev == f), i.e., f is first daughter
    m_face[mf].daughter_prev = m_face[f].next;
  else
    m_face[m_face[f].daughter_prev].next = 
      m_face[f].next;
  
  if (m_face[f].next)
    m_face[m_face[f].next].daughter_prev = 
      m_face[f].daughter_prev;
  
  m_face[f].mother = 
    m_face[f].next = 
    m_face[f].daughter_prev = 0;
}

void 
map::faceAddDaughter(girl::face m, girl::face d)
{
  assert(m != d);
#ifndef NDEBUG
  girl::face cd = m_face[m].daughter_prev;
  while (cd != 0) {
    assert(cd != d);
    cd = m_face[cd].next;
  }
#endif

  m_face[d].mother = m;
  m_face[d].next = m_face[m].daughter_prev;
  m_face[m_face[m].daughter_prev].daughter_prev = d;
  m_face[m].daughter_prev = d;
  assert(m_face[d].daughter_prev == 0);
}

void 
map::faceRemove(girl::face f)
{
  assert(checkFaceValidity(f));
  assert(isFaceUsed(f));

  if (m_face[f].mother)
    faceRemoveInfiniteFace(f);

  freeLambda(f);

  assert(isFaceFree(f));
}

void 
map::faceMoveDaughters(girl::face src, girl::face dest)
{
  assert(checkFaceValidity(src));
  assert(checkFaceValidity(dest));
  assert(isFaceUsed(src));
  assert(isFaceUsed(dest));
  assert(m_face[src].mother == GIRL_FINITE_FACE); //src is a finite face

  const girl::face md = m_face[dest].mother;
  
  if (md != GIRL_FINITE_FACE) //dest is infinite
    dest = md;
  
  girl::face f = m_face[src].daughter_prev;
  if (f != 0)
    {
      girl::face last;
      while (f)
	{
	  assert(isFaceUsed(f));

	  m_face[f].mother = dest;
	  last = f;
	  f = m_face[f].next;
	}

      f = m_face[src].daughter_prev;
      m_face[last].next = m_face[dest].daughter_prev;
      if (m_face[dest].daughter_prev)
	m_face[m_face[dest].daughter_prev].daughter_prev = last;
      m_face[dest].daughter_prev = f;
      m_face[src].daughter_prev = 0;
      assert(m_face[m_face[dest].daughter_prev].daughter_prev == 0);
    } 
  
}

/*
  void 
  map::faceRename(girl::face oldFace, girl::face newFace) //B oldFace: should be region ?
  {
  assert(m_face[oldFace].mother == 0);
  assert(checkFaceValidity(newFace));
  assert(isFaceFree(newFace));

  // extract newFace from the free list
  girl::face fn = 0;
  while (m_face[fn].next != newFace)
  {
  fn = m_face[fn].next;
  assert(fn);
  }
  m_face[fn].next = m_face[newFace].next;
  //setFaceUsed(newFace);
  m_face[newFace].next = 0;

  faceMoveDaughters(oldFace, newFace);
  
  const girl::dart d = beta(oldFace);

  setLambdaCycle_aux(newFace, d);

  assert(getLambda(d) == newFace);
  assert(checkDartPhiCycleLambda(d));

  beta(newFace) = d;
  setFaceUsed(newFace);

  freeLambda(oldFace);

  assert(checkFace(newFace));
  }
*/

void 
map::markDart(girl::dart d, girl::toger::markFlags m)
{
  m_map[d].mark_dir |= ((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT);
}

void 
map::xMarkDart(girl::dart d, girl::toger::markFlags m)
{
  m_map[d].mark_dir ^= ((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT);
}

void 
map::unmarkDart(girl::dart d, girl::toger::markFlags m)
{
  m_map[d].mark_dir &= ~((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT);
}

void 
map::markAllDarts(girl::toger::markFlags m)
{
  for (size_t i = 0; i < m_mapDim; ++i)
    {
      if (isDartUsed(static_cast<girl::dart>(i)) == true)
	{
	  markDart(static_cast<girl::dart>(i), m);


	  /**/
	  assert(isDartUsed(alpha(static_cast<girl::dart>(i))));

	  markDart(alpha(static_cast<girl::dart>(i)), m);

	}
      /*
      if (isDartUsed(alpha(static_cast<girl::dart>(i))) == true)
	{
	  markDart(alpha(static_cast<girl::dart>(i)), m);
	}
      */
    }  
}

void 
map::unmarkAllDarts(girl::toger::markFlags m)
{
  for (size_t i = 0; i < m_mapDim; ++i)
    {
      if (isDartUsed(static_cast<girl::dart>(i)) == true)
	{
	  unmarkDart(static_cast<girl::dart>(i), m);

	  /**/
	  assert(isDartUsed(alpha(static_cast<girl::dart>(i))));

	  unmarkDart(alpha(static_cast<girl::dart>(i)), m);
	}
      /*
      if (isDartUsed(alpha(static_cast<girl::dart>(i))) == true)
	{
	  unmarkDart(alpha(static_cast<girl::dart>(i)), m);
	}
      */
    }  

}

bool 
map::dartIsMarked(girl::dart d, girl::toger::markFlags m) const
{
  
  //check that at least one bit at 1 in m is at 1 in m_map[d].mark
  return (m_map[d].mark_dir & ((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT)) != 0;
  //does not work with MARK_NONE == 0

  //return (m_map[d].mark & ((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT))^((m & DART_PUBLIC_MARK_MASK) << DART_PUBLIC_MARK_SHIFT); //all bits of mark are set
}

girl::toger::markFlags 
map::getDartMark(girl::dart d) const
{
  return static_cast<girl::toger::markFlag>((m_map[d].mark_dir >> DART_PUBLIC_MARK_SHIFT) & DART_PUBLIC_MARK_MASK);
}



bool 
map::checkConsistency() const
{
  assert(MAP_INVARIANT_COND);

  bool consistent = true;

  for (size_t i = 0; i < m_mapDim; ++i)
    {
      if (isDartUsed(static_cast<girl::dart>(i)) == true && isFaceUsed(lambda(static_cast<girl::dart>(i))) == false)
	{
	  consistent = false;
	  std::cerr<<"dart "<<static_cast<girl::dart>(i)<<" and its face "<<lambda(static_cast<girl::dart>(i))<<" are not both in use or free\n";
	  if (isDartUsed(static_cast<girl::dart>(i)))
	    std::cerr<<"  dart is used and face is free\n";
	  else
	    std::cerr<<"  dart is free and face is used\n";

	}

      if (isDartUsed(alpha(static_cast<girl::dart>(i))) == true && isFaceUsed(lambda(alpha(static_cast<girl::dart>(i)))) == false)
	{
	  consistent = false;
	  std::cerr<<"girl::dart "<<alpha(static_cast<girl::dart>(i))<<" and its face "<<lambda(alpha(static_cast<girl::dart>(i)))<<" are not both in use or free\n";
	  if (isDartUsed(alpha(static_cast<girl::dart>(i))))
	    std::cerr<<"  dart is used and face is free\n";
	  else
	    std::cerr<<"  dart is free and face is used\n";
	}

      if (isDartUsed(static_cast<girl::dart>(i)) == true &&
	  lambda(static_cast<girl::dart>(i)) == lambda(alpha(static_cast<girl::dart>(i))))
	{
	  consistent = false;
	  std::cerr<<"dangling edge at dart "<<static_cast<girl::dart>(i)<<"\n";
	}

      if (isDartUsed(static_cast<girl::dart>(i)) == true && getSigma_1(phi(static_cast<girl::dart>(i))) != alpha(static_cast<girl::dart>(i)))
	{
	  consistent = false;
	  std::cerr<<"sigma_1(phi("<<static_cast<girl::dart>(i)<<") is "<<getSigma_1(phi(static_cast<girl::dart>(i)))<<" and should be "<<alpha(static_cast<girl::dart>(i))<<"\n";

	  return consistent; //DEBUG
	}
    }

  for (size_t i = 0; i < m_faceDim; ++i)
    {
      if (isFaceUsed(static_cast<girl::face>(i)) == true && isDartUsed(beta(static_cast<girl::face>(int(i)))) == false)
	{
	  consistent = false;
	  std::cerr<<"face "<<static_cast<girl::face>(i)<<" and its canonical girl::dart "<<beta(static_cast<girl::face>(i))<<" are not both in use or free\n";
	}


      if (isFaceFree(static_cast<girl::face>(i)) == true && isFaceUsed(m_face[static_cast<girl::face>(i)].next))
	{
	  consistent = false;
	  std::cerr<<"face "<<static_cast<girl::dart>(i)<<" is not in use but its next ("<<m_face[static_cast<girl::dart>(i)].next<<") is in use !\n";
	} 
      
      
      if (isFaceFree(static_cast<girl::face>(i)))
	continue;
      
      if (!checkFaceBetaLambda(static_cast<girl::face>(i)))
	{
	  consistent = false;
	  std::cerr<<"face "<<static_cast<girl::face>(i)<<" has beta/lambda inconsistency : beta("<<static_cast<girl::face>(i)<<")="<<beta(static_cast<girl::face>(i))<<" lambda(beta("<<static_cast<girl::face>(i)<<")="<<lambda(beta(static_cast<girl::face>(i)))<<"!="<<static_cast<girl::face>(i)<<" \n";
	}

      int d = beta(static_cast<girl::face>(i));
      do
	{
	  if (lambda(d) != static_cast<girl::face>(i))
	    {
	      consistent = false;
	      std::cerr<<"face "<<static_cast<girl::face>(i)<<" has inconsistent phi cycle at dart "<<d<<" : lambda("<<d<<")="<<lambda(d)<<"!="<<static_cast<girl::face>(i)<<"\n";
	    }
	  d = phi(d);
	}
      while (d != beta(static_cast<girl::face>(i)));

      int m = m_face[static_cast<girl::face>(i)].mother;

      if (!checkInfiniteFaceAncestry(static_cast<girl::face>(i)))
	{
	  consistent = false;
	  std::cerr<<"face "<<static_cast<girl::face>(i)<<" is transfinite\n";
	}

      if (!checkInfiniteFaceCanonicalNeighbour(static_cast<girl::face>(i)))
	{
	  consistent = false;
	  std::cerr<<"face "<<static_cast<girl::face>(i)<<" is infinite and has infinite neighbour\n";
	}

      if (!m)
	{
	  girl::face f = m_face[static_cast<girl::face>(i)].daughter_prev;
	  while (f)
	    {
	      if (m_face[f].mother != static_cast<girl::face>(i))
		{
		  consistent = false;
		  std::cerr<<"face "<<static_cast<girl::face>(i)<<" has parent inconsistency with "<<f<<"\n";
		}
	      f = m_face[f].next;
	    }
	}
    }

  size_t n;

  n = 0;
  girl::face f = 0;
  girl::face pf = 0;
  do {
    if (! isFaceFree(f)) {
      std::cerr<<"face "<<f<<" in free list but not in free state\n";
      consistent = false;
    }
    pf = f;
    f = m_face[f].next;
    if (! checkFaceBounds(f)) {
      std::cerr<<"face "<<pf<<" has invalid 'next' ("<<f<<")"<<std::endl;
      consistent = false;
      break;
    }
     
    ++n;
    if (n > m_faceDim+1) {
      std::cerr<<"two many faces"<<std::endl;
      consistent = false;
      break;
    }
    if (f == pf && f != 0) {
      std::cerr<<"face "<<f<<" is linked to itself in free list !"<<std::endl;
      consistent = false;
      break;
    }
  }
  while (f != 0);


  typedef girl::unorderedSet<girl::dart> DartHash;
  DartHash hd;

  n=0;
  girl::dart d = 0;
  girl::dart pd = 0;
  do {
    if (! isDartFree(d)) {
      std::cerr<<"dart "<<d<<" in free list but not in free state\n";
      consistent = false;
    }

    if (hd.find(d) != hd.end()) {
      std::cerr<<"dart d "<<d<<" several times in free list. There must be a loop !"<<std::endl;
      consistent = false;
    }
    if (hd.find(alpha(d)) != hd.end()) {
      std::cerr<<"dart d "<<d<<" and alpha(d) "<<alpha(d)<<" are both in free list !"<<std::endl;
      consistent = false;
    }
    hd.insert(d);

    pd = d;
    d = m_map[d].phi;
    if (! checkDartBounds(d)) {
      std::cerr<<"dart "<<pd<<" has invalid 'phi' ("<<d<<")"<<std::endl;
      consistent = false;
      break;
    }

    ++n;
    if (n > 2*m_mapDim+1) {
      std::cerr<<"too many darts"<<std::endl;
      consistent = false;
      break;
    }
    if (d == pd && d != 0) {
      std::cerr<<"dart "<<d<<" is linked to itself in free list !"<<std::endl;
      consistent = false;
      break;
    }
  }
  while (d != 0);
  

  assert(MAP_INVARIANT_COND);

  return consistent;
}

bool 
map::checkDart(girl::dart d) const
{
  return (checkDartValidity(d) && isDartUsed(d));
}

bool 
map::checkFace(girl::face f) const
{
  return (checkFaceValidity(f) && isFaceUsed(f) && checkFacePrevLink(f));
}

bool
map::checkRegion(girl::face f) const 
{
  return checkFace(f) && m_face[f].mother ==0;
}


bool
map::checkFreeFaces() const
{
  girl::face f = 0;
  do {
    if (! isFaceFree(f))
      return false;
    f = m_face[f].next;
  } while (f != 0);
  return true;
}

bool 
map::checkFaceIsFree(girl::face f) const
{
  if (isFaceUsed(f))
    return false;

  return checkFreeFaces();
}

bool 
map::checkDartFaceUse(girl::dart d) const
{
  return isDartUsed(d) == isFaceUsed(lambda(d));
}

bool 
map::checkDartIsNotDanglingEdge(girl::dart d) const
{
  return (isDartUsed(d) == true && lambda(d) != lambda(alpha(d)));
}


bool 
map::checkFaceBetaLambda(girl::face f) const
{
  return lambda(beta(f)) == f;
}

bool 
map::checkDartPhiCycle(girl::dart d) const
{
  bool consistent = true;
  girl::dart cd = d;
  do
    {
      if (!checkDart(cd))
	{
	  std::cerr<<"girl::dart "<<cd <<" is invalid\n";
	  consistent = false;
	  break;
	}
      girl::dart nd = phi(cd);
      if (!checkDart(nd))
	{
	  std::cerr<<"girl::dart "<<nd<<" is invalid2\n";
	  consistent = false;
	  break;
	}
      if (getSigma_1(nd) != alpha(cd))
	{
	  std::cerr<<"false continuity: getSigma_1("<<nd<<")="<<getSigma_1(nd)<<"  != alpha("<<cd<<")="<<alpha(cd)<<"\n";
	  consistent = false;
	  break;
	}	  
      cd = nd;
    }
  while (cd != d);
  return consistent;  
}

bool
map::checkDartPhiCycleLambda(girl::dart d) const
{
  if (!checkDart(d))
    return false;
  bool consistent = true;
  const girl::face f = lambda(d);
  girl::dart cd = d;
  do
    {
      cd = phi(cd);
      if (!checkDart(cd))
	{
	  consistent = false;
	  break;
	}
      if (lambda(cd) != f)
	{
	  consistent = false;
	  break;
	}
    }
  while (cd != d);
  return consistent;
}

bool 
map::checkFacePhiCycle(girl::face f) const
{
  return checkFaceBetaLambda(f)
    && checkDartPhiCycle(beta(f))
    && checkDartPhiCycleLambda(beta(f));
}

bool 
map::checkInfiniteFaceAncestry(girl::face f) const
{
  return (!m_face[f].mother || !m_face[m_face[f].mother].mother);
}

bool 
map::checkInfiniteFaceCanonicalNeighbour(girl::face f) const
{
  return (!m_face[f].mother || !(m_face[lambda(alpha(beta(f)))].mother));
}

bool 
map::checkInfiniteFaceNeighbourhood(girl::face f) const
{
  return (!m_face[f].mother || 
	  (!(m_face[lambda(alpha(beta(f)))].mother) 
	   && checkDartPhiCycle(alpha(beta(f))) ) );  
}


bool 
map::checkInfiniteFaceDaughterOfMother(girl::face f) const
{
  bool consistent = true;
  const girl::face m = m_face[f].mother;
  if (!m)
    return consistent;

  consistent = false;
  girl::face fi = m_face[m].daughter_prev;
  while (fi)
    {
      if (fi == f) {
	consistent = true;
	break;
      }
      fi = m_face[fi].next;
    }
  return consistent;  
}

bool 
map::checkInfiniteFaceConsistency(girl::face f) const
{
  return checkInfiniteFaceAncestry(f) 
    && checkInfiniteFaceNeighbourhood(f)
    && checkInfiniteFaceDaughterOfMother(f);
}

bool 
map::checkFaceLineage(girl::face f) const
{
  bool consistent = true;
  if (!m_face[f].mother || !m_face[f].daughter_prev)
    return consistent;

  //finite girl::face with daughters
  girl::face fi = m_face[f].daughter_prev;
  while (fi)
    {
      if (m_face[fi].mother != f)
	{
	  consistent = false;
	  break;
	}
      fi = m_face[fi].next;
    }
  return consistent;
}



size_t
map::nbFreeFaces() const
{
  size_t nb = 0;
  girl::face f = 0;
  while ((f = m_face[f].next) != 0)
    ++nb;

  /*
  //REM: it seems more efficient than this:

  size_t nb2 = 0;
  for (size_t i = 1; i < m_faceDim; ++i) 
  if (isFaceFree(i))
  ++nb2;
  
  */

  return nb;
}


size_t
map::nbFreeDarts() const
{
  /*
    size_t nb = 0;
    girl::dart d = 0;
    while ((d = m_map[d].phi) != 0)
    ++nb;
  */

  size_t nb = 0;
  girl::dart d = 0;
  while ((d = m_map[d].phi) != 0) {
    if (nb > 2*m_mapDim) {
      std::cerr<<"Error: map::nbFreeDarts() !"<<std::endl;
      std::cerr<<"nb="<<nb<<" m_mapDim="<<m_mapDim<<std::endl;
      size_t nb2 = 0;
      for (size_t i = 1; i < m_mapDim; ++i) 
	if (isDartFree(i))
	  ++nb2;
      std::cerr<<"true number: "<<nb2<<std::endl;

      assert(false);
      exit(EXIT_FAILURE);
    }
    ++nb;
  }
  
  

  /*
  //REM: it seems more efficient than this:

  size_t nb2 = 0;
  for (size_t i = 1; i < m_mapDim; ++i) 
  if (isDartFree(i))
  ++nb2;
  
  */

  return nb;
}

#ifndef NDEBUG
namespace {
template <class T>
bool
P_shrink_postCondition(const girl::hashMap<T, T> &renamedElts)
{
  typedef girl::hashMap<T, T> EltsMap;

  //- check that (newId < oldId)
  {
    typename EltsMap::const_iterator it = renamedElts.begin();
    const typename  EltsMap::const_iterator itEnd = renamedElts.end();
    for ( ; it != itEnd; ++it) {
      const T oldId = it->first;
      const T newId = it->second;
      if (oldId <= newId) {
	std::cerr<<"oldId="<<oldId<<" should be > to newId="<<newId<<std::endl;
	return false;
      }
    }
  }
  
  /*
  it = renamedElts.begin();
  if (it == itEnd) 
    return true;
  typename EltsMap::const_iterator itN = it;
  ++itN;
  if (itN ==itEnd) 
    return true;
  for ( ; itN != itEnd; ++it, ++itN) {
    //id must be sorted in increasing order
    if (itN->second <= it->second) {
      std::cerr<<"next="<<itN->second<<" should be > to previous="<<it->second<<std::endl;      
      return false;
    }
  }
  */

  //check that all newIds are different
  {
    if (renamedElts.size() > 1) {
      typename EltsMap::const_iterator it = renamedElts.begin();
      typename EltsMap::const_iterator itj = renamedElts.begin();
      const typename EltsMap::const_iterator itEnd = renamedElts.end();
      //const typename EltsMap::const_iterator itEnd_1 = --(renamedElts.end());
      typename EltsMap::const_iterator itEnd_1 = itEnd;
      for (it = renamedElts.begin(); it != itEnd ; ) {
	++it;
	if (it != itEnd)
	  itEnd_1 = it;
      }
      

      for (it = renamedElts.begin(); it != itEnd_1 ; ++it) {
	const T newId1 = it->second;
	itj = it;
	++itj;
	for ( ; itj != itEnd; ++itj) {
	  const T newId2 = itj->second;
	  if (newId1 == newId2) {
	    std::cerr<<"oldId="<<it->first<<" && oldId="<<itj->first<<" both renamed to same newId="<<newId1<<std::endl;
	    return false;
	  }
	}
      }
    }

  }

  return true;
}
} //end anonymous namespace
#endif



void
map::crop(const girl::point &ip1,
	  const girl::regionDeque &regionsToRemove,
	  const girl::hashMap<girl::dart, girl::dart> &dartsNewOuterContour,
	  std::deque<girl::point> &nodesToRemove,
	  std::map<girl::dart, girl::geoDart> &movedGeoDarts)
{
  assert(checkConsistency());

  typedef std::map<girl::dart, girl::geoDart> D2GM;
  typedef girl::hashMap<girl::dart, girl::dart> D2DH;


  const D2DH::const_iterator itEnd = dartsNewOuterContour.end();



  girl::regionDeque::const_iterator itR = regionsToRemove.begin();
  const girl::regionDeque::const_iterator itREnd = regionsToRemove.end();
  for ( ; itR != itREnd; ++itR) {

    girl::region r = *itR;

    const girl::dart cd = beta(r);


    girl::dart d = cd;
    do {

      D2DH::const_iterator itd = dartsNewOuterContour.find(d);
      D2DH::const_iterator itad = dartsNewOuterContour.find(alpha(d));

      girl::dart nd = phi(d);

      if (itd == itEnd && itad == itEnd) {
	//dart d and -d are not on new contour
	// we can remove it.


	girl::face fd = lambda(d);
	if (isFaceUsed(fd) && faceIsInfinite(fd) && fd != -1) {
	  freeLambda(fd);
	}
	girl::face fad = lambda(alpha(d));
	if (isFaceUsed(fad) && faceIsInfinite(fad) && fad != -1) {
	  freeLambda(fad);
	}

	if (isDartUsed(d)) {
	  freeDart(d);
	}
      }
      else {
	//d and -d are on new contour
	//we do not remove them and update the phi cycle and lambda

	//We do not update here because we are not sure that we pass here for all outer contour darts 
	// In particular, if new outer contour is a infinite face, we never pass here.

	/*
	D2DH::const_iterator it = itd;
	if (itad != itEnd)
	  it = itad;

	phi(it->second) = it->first;
	lambda(it->second) = -1;
	lambda(it->first) = -1;
	
	std::cerr<<" node at ("<<m_map[it->first].pt.x()<<"; "<<m_map[it->first].pt.y()<<") ";
	m_map[it->first].pt -= girl::point(ip1.x()+1, ip1.y()+1);
	std::cerr<<" transferred to ("<<m_map[it->first].pt.x()<<"; "<<m_map[it->first].pt.y()<<") "<<std::endl;
	*/
      }

      d = nd;

    } 
    while (d != cd);

    assert(isFaceUsed(r));

    freeLambda(r);

  }

  assert(! dartsNewOuterContour.empty());

  D2DH::const_iterator it = dartsNewOuterContour.begin();

  girl::dart d1 = it->first;
  girl::face f1 = lambda(d1);
  if (isFaceUsed(f1) && faceIsInfinite(f1) && f1 != -1) {
    freeLambda(f1);
  }
  beta(-1) = d1;

  for ( ; it != itEnd; ++it) {

    phi(it->second) = it->first;
    //lambda(it->second) = -1;
    lambda(it->first) = -1;
    
    /*
    NON !!! IL FAUT TRANSLATER TOUTES LES POSITIONS, DE TOUS LES DARTS!!!

    std::cerr<<" node at ("<<m_map[it->first].pt.x()<<"; "<<m_map[it->first].pt.y()<<") ";
    m_map[it->first].pt -= girl::point(ip1.x()+1, ip1.y()+1);
    std::cerr<<" transferred to ("<<m_map[it->first].pt.x()<<"; "<<m_map[it->first].pt.y()<<") "<<std::endl;

    std::cerr<<" node at ("<<m_map[alpha(it->first)].pt.x()<<"; "<<m_map[alpha(it->first)].pt.y()<<") ";
    m_map[alpha(it->first)].pt -= girl::point(ip1.x()+1, ip1.y()+1);
    std::cerr<<" transferred to ("<<m_map[alpha(it->first)].pt.x()<<"; "<<m_map[alpha(it->first)].pt.y()<<") "<<std::endl;
    */
  }

  /*
  FAIRE TRANSLATION ICI (LES NOEUDS A SUPPRIMER DOIVENT DEJA ETRE TRANSLATés) !?
   -> ca serait mieux de translater quand on a supprimer les darts inutiles !
    Dans ce cas, dans togerImpl::crop() on doit faire le ge2to::crop et bitmap::crop après le map::crop et la suppression des noeuds et deplacments des geoDarts !!!
  */

  for (it = dartsNewOuterContour.begin() ; it != itEnd; ++it) {

    //must be done in two pass, because we need that sigma cycle be correct for previous and next dart in phi cycle of d.

    const girl::dart d = it->first; 

    if (sigma(sigma(d))==d) {//node of cardinality 2

      if (phi(d) != d) {  
	//other dart in phi cycle
	//=> this node is useless

	//we ensure that faces linked to d and alpha(d) have correct cardinal darts.
	assert(phi(d) != 0);
	assert(phi(alpha(d)) != 0);
	assert(lambda(d) != 0);
	assert(lambda(alpha(d)) != 0);      
	
	beta(lambda(d)) = phi(d);
	beta(lambda(alpha(d))) = phi(alpha(d));
	
	nodesToRemove.push_back(m_map[d].pt);
		
	const girl::dart sd = sigma(d);
	girl::geoDart u;
	getGe(alpha(d), u);
	

	removeUnnecessaryArtificialNode(d); //does not change geoDart
	
	D2GM::iterator itDG = movedGeoDarts.find(d);
	if (itDG != movedGeoDarts.end()) {
	  movedGeoDarts.erase(itDG);
	}
	itDG = movedGeoDarts.find(alpha(d));
	if (itDG != movedGeoDarts.end()) {
	  movedGeoDarts.erase(itDG);
	}

	setGe(sd, u);
	
	movedGeoDarts[sd] = u;
      } 
      else { 
	//this is he only artificial node of phi cycle
	
	//ensure that alpha(d) have the same position than d and a correct direction
	//m_map[alpha(d)].pt = m_map[d].pt;
	//setGeoDartDirection(alpha(d), getGeoDartDirection(d).previous());


      }

    }
    else {}
    
  }

  
  const girl::point p1(ip1.x()+1, ip1.y()+1);
  for (size_t i = 0; i < m_mapDim; ++i) {
    const girl::dart d = static_cast<girl::dart>(i);

    if (isDartUsed(d) == true) {

      m_map[d].pt -= p1;

      assert(isDartUsed(alpha(d)));
      
      m_map[alpha(d)].pt -= p1;

    }
  }
  





  assert(MAP_INVARIANT_COND);

  /*
  //DEBUG !!!
  if (! checkConsistency())
    std::cerr<<"map::crop checkConsistency failed !!! "<<std::endl;
  */

  assert(checkConsistency());
}

void
map::shrink(girl::hashMap<girl::face, girl::face> &renamedFaces, 
	    girl::hashMap<girl::dart, girl::dart> &renamedDarts)
{
  //TODO: use a second map ????

  typedef girl::hashMap<girl::face, girl::face> FacesMap;
  typedef girl::hashMap<girl::dart, girl::dart> DartsMap;

  const size_t nFreeFaces = nbFreeFaces();

  std::vector<faceCell> new_faceV;
  faceCell *new_face = NULL;
  size_t face_dim = 0;

  //  if (nFreeFaces > 0) { //???

  const size_t nbUsedFaces = m_faceDim - nFreeFaces;

  face_dim = nbUsedFaces;
  const size_t new_sizeF = face_dim + 1;
  assert(new_sizeF>=2);
  new_faceV.resize(new_sizeF);
  new_face = &new_faceV[1];
  new_faceV[0] = m_faceV[0];
  new_faceV[1] = m_faceV[1];
  new_face[0].next = 0; 
    
  size_t i = 1;
  size_t j = 1;
  while (i < face_dim) {
    
    assert(j<m_faceDim);
    while (isFaceFree(j)) {
      ++j;
      assert(j<m_faceDim);
    }
     
    assert(isFaceUsed(j));
     
    new_face[i] = m_face[j];
    if (i != j) {
      renamedFaces[j] = i;
    }

    ++i;
    ++j;
  }
  
  assert(P_shrink_postCondition<girl::face>(renamedFaces));
  //  }

  const size_t nFreeDarts = nbFreeDarts();

  std::vector<dartCell> new_mapV;
  dartCell *new_map = NULL;
  size_t map_dim = 0;

  //  if (nFreeDarts > 0) { //???

  const size_t nbUsedDarts = m_mapDim - nFreeDarts;

  map_dim = nbUsedDarts;
  const size_t new_sizeM = 2*map_dim + 1;
  new_mapV.resize(new_sizeM);
  new_map = &new_mapV[0] + map_dim;
    
  new_map[0] = m_map[0];
  new_map[0].phi = 0;

  i = 1;
  j = 1;
  while (i < map_dim) {
    
    assert(j<m_mapDim);
    while (isDartFree(j)) {
      ++j;
      assert(j<m_mapDim);
    }
    
    assert(isDartUsed(j));

    new_map[i] = m_map[j];
    new_map[alpha(i)] = m_map[alpha(j)];
    if (i != j) {
      renamedDarts[j] = i; //keep only renamed positive darts
    }

    ++i;
    ++j;
  }
  assert(P_shrink_postCondition<girl::dart>(renamedDarts));
    
  //  }


  //#ifndef NDEBUG
#if 0
  std::cout<<"map::shrink(): darts: "<<nFreeDarts<<"free/"<<m_mapDim<<"="<<nFreeDarts/(float)m_mapDim*100.0<<"% - faces: "<<nFreeFaces<<"free/"<<m_faceDim<<"="<<nFreeFaces/(float)m_faceDim*100.0<<"%"<<std::endl;
#endif

  if (! renamedDarts.empty() || ! renamedFaces.empty() ) {

    const DartsMap::iterator itDEnd = renamedDarts.end();
    DartsMap::iterator itD;
 
    const FacesMap::iterator itFEnd = renamedFaces.end();
    FacesMap::iterator itF;
 


    for (size_t k=1; k<map_dim; ++k) {
      const girl::dart od = new_map[k].phi;
      itD = renamedDarts.find(abs(od));
      if (itD != itDEnd) {
	new_map[k].phi = od > 0 ? itD->second : alpha(itD->second);
      }

      const girl::face of = new_map[k].lambda;
      itF = renamedFaces.find(of);
      if (itF != itFEnd)
	new_map[k].lambda = itF->second;


      const girl::dart oad = new_map[alpha(k)].phi;
      itD = renamedDarts.find(abs(oad));
      if (itD != itDEnd) {
	new_map[alpha(k)].phi = oad > 0 ? itD->second : alpha(itD->second);
      }

      const girl::face oaf = new_map[alpha(k)].lambda;
      itF = renamedFaces.find(oaf);
      if (itF != itFEnd)
	new_map[alpha(k)].lambda = itF->second;

    }


    const girl::dart od_1 = new_face[-1].d;
    itD = renamedDarts.find(abs(od_1));
    if (itD != itDEnd)
      new_face[-1].d = od_1 > 0 ? itD->second : alpha(itD->second);

    for (size_t k=1; k<face_dim; ++k) {
      const girl::face om = new_face[k].mother;
      itF = renamedFaces.find(om);
      if (itF != itFEnd)
	new_face[k].mother = itF->second;
      
      const girl::face odp = new_face[k].daughter_prev;
      itF = renamedFaces.find(odp);
      if (itF != itFEnd)
	new_face[k].daughter_prev = itF->second;
      
      const girl::face on = new_face[k].next;
      itF = renamedFaces.find(on);
      if (itF != itFEnd)
	new_face[k].next = itF->second;
	
      const girl::dart od = new_face[k].d;
      itD = renamedDarts.find(abs(od));
      if (itD != itDEnd)
	new_face[k].d = od > 0 ? itD->second : alpha(itD->second);
    }

  }

  m_mapV.swap(new_mapV);
  assert(m_mapV.size() > 0);
  m_map = new_map;
  m_mapDim = map_dim;
  

  m_faceV.swap(new_faceV);
  assert(m_faceV.size() > 1);
  m_face = new_face;
  m_faceDim = face_dim;
    
  assert(m_map[0].phi == 0);
  assert(m_face[0].next == 0);

  assert(MAP_INVARIANT_COND);
  assert(checkConsistency());
}



bool
map::write(girl::io::Writer &w, girl::io::writeFlags flags) const
{
  //all is saved in little endian

  if ((flags & girl::io::NO_ENDIANNESS_CARE) || GIRL_BYTEORDER == GIRL_LIL_ENDIAN)
    {
      assert(sizeof(m_mapDim) == sizeof(int)); //must be 32 bits
      assert(sizeof(m_faceDim) == sizeof(int));
      
      assert(sizeof(dartCell) == 16); 
      assert(sizeof(faceCell) == 16);

      if (w.write(&(m_mapDim), sizeof(m_mapDim)) != sizeof(m_mapDim))
	return false;
      const size_t mapTotalSize = (2*m_mapDim + 1)*sizeof(dartCell);
      if (w.write(&m_mapV[0], mapTotalSize) != mapTotalSize)
	return false;
      
      if (w.write(&(m_faceDim), sizeof(m_faceDim)) != sizeof(m_faceDim))
	return false;
      const size_t faceTotalSize = (m_faceDim + 1)*sizeof(faceCell);
      if (w.write(&m_faceV[0], faceTotalSize) != faceTotalSize)
	return false;

    }
  else
    {
      assert((flags & girl::io::NO_ENDIANNESS_CARE) == 0);
      assert(GIRL_BYTEORDER == GIRL_BIG_ENDIAN);
      { //block to free temporary vector asap
	assert(sizeof(girl::POINT_TYPE) == 2);
	unsigned int mapDim = GIRL_swapLE32(m_mapDim);
	if (w.write(&(mapDim), sizeof(mapDim)) != sizeof(mapDim))
	  return false;

	const size_t mapTotalSize = (2*m_mapDim + 1);
	std::vector<dartCell> new_mapV = m_mapV; //temporary vector
	for (size_t i=0; i<mapTotalSize; ++i) {
	  dartCell &dc = new_mapV[i];
	  dc.pt.setX(GIRL_swapLE16(dc.pt.x()));
	  dc.pt.setY(GIRL_swapLE16(dc.pt.y()));
	  dc.mark_dir = GIRL_swapLE32(dc.mark_dir);
	  dc.phi = GIRL_swapLE32(dc.phi);
	  dc.lambda = GIRL_swapLE32(dc.lambda);
	}
	if (w.write(&m_mapV[0], mapTotalSize) != mapTotalSize)
	  return false;
      }
      { //block to free temporary vector asap
	unsigned int faceDim = GIRL_swapLE32(m_mapDim);
	if (w.write(&(faceDim), sizeof(faceDim)) != sizeof(faceDim))
	  return false;

	const size_t new_sizeF = (m_faceDim + 1);
	std::vector<faceCell> new_faceV(new_sizeF); //temporary vector
	for (size_t i=0; i<new_sizeF; ++i) {
	  faceCell &fc = new_faceV[i];
	  fc.d = GIRL_swapLE32(fc.d);
	  fc.mother = GIRL_swapLE32(fc.mother);
	  fc.daughter_prev = GIRL_swapLE32(fc.daughter_prev);
	  fc.next = GIRL_swapLE32(fc.next);
	}
      }	
      
    }
	
  return true;
}

bool
map::read(girl::io::Reader &r, girl::io::writeFlags flags)
{
  //all is saved in little endian

  unsigned int map_dim;
  std::vector<dartCell> new_mapV;

  unsigned int face_dim;
  std::vector<faceCell> new_faceV;

  if (r.read(&map_dim, sizeof(map_dim)) != sizeof(map_dim))
    return false;
  map_dim = GIRL_swapLE32(map_dim);
  const size_t new_sizeM = 2*map_dim + 1;
  const size_t newMapTotalSize = new_sizeM*sizeof(dartCell);
  new_mapV.resize(new_sizeM);
  if (r.read(&new_mapV[0], newMapTotalSize) != newMapTotalSize)
    return false;
      
  if (r.read(&face_dim, sizeof(face_dim)) != sizeof(face_dim))
    return false;
  face_dim = GIRL_swapLE32(face_dim);
  const size_t new_sizeF = face_dim + 1;
  const size_t newFaceTotalSize = new_sizeF*sizeof(faceCell);
  new_faceV.resize(new_sizeF);
  if (r.read(&new_faceV[0], newFaceTotalSize) != newFaceTotalSize)
    return false;

  if ((flags & girl::io::NO_ENDIANNESS_CARE) == 0 && GIRL_BYTEORDER == GIRL_BIG_ENDIAN) {
    
    assert(sizeof(girl::POINT_TYPE) == 2);
      
    for (size_t i=0; i<new_sizeM; ++i) {
      dartCell &dc = new_mapV[i];
      dc.pt.setX(GIRL_swapLE16(dc.pt.x()));
      dc.pt.setY(GIRL_swapLE16(dc.pt.y()));
      dc.mark_dir = GIRL_swapLE32(dc.mark_dir);
      dc.phi = GIRL_swapLE32(dc.phi);
      dc.lambda = GIRL_swapLE32(dc.lambda);
    }
   
    for (size_t i=0; i<new_sizeF; ++i) {
      faceCell &fc = new_faceV[i];
      fc.d = GIRL_swapLE32(fc.d);
      fc.mother = GIRL_swapLE32(fc.mother);
      fc.daughter_prev = GIRL_swapLE32(fc.daughter_prev);
      fc.next = GIRL_swapLE32(fc.next);
    }

  }
  
  m_mapV.swap(new_mapV);
  assert(m_mapV.size() > 0);
  m_map = &m_mapV[0] + map_dim;
  m_mapDim = map_dim;
  
  
  m_faceV.swap(new_faceV);
  assert(m_faceV.size() > 1);
  m_face = &m_faceV[1];
  m_faceDim = face_dim;
  
  assert(MAP_INVARIANT_COND);
  
  return true;
}

void
map::swap(map &m)
{
  std::swap(m_mapV, m.m_mapV);
  std::swap(m_faceV, m.m_faceV);
  std::swap(m_map, m.m_map);
  std::swap(m_face, m.m_face);
  std::swap(m_mapDim, m.m_mapDim);
  std::swap(m_faceDim, m.m_faceDim);
}
