/*
  Input/Output prototypes and constants definitions.
  Copyright (C) 2004-2009  Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlIO.hpp
 * @brief Input/Output prototypes and constants definitions.
 */

#ifndef GIRL_IO_HPP
#define GIRL_IO_HPP

#include <cassert>
#include <cstddef> //for size_t

#include <girl/girlCommon.hpp>
#include <girl/girlFlags.hpp>


namespace girl {

  namespace io {

    /**
     * @brief write flags for IO.
     * - @a NO_ENDIANNESS_CARE indicates that no care has to be taken regarding endianness. 
     *   It is the case when writing to memory or with a compressor taking care of endianness itself.
     *   It allows to have faster outputs.
     * - @a NO_LABELS_MARKS_CARE indicates that write has not to ensure that 
     *    when reloaded labels and marks of regions/darts will all be the same.
     *   It allows to have smaller outputs.
     * - @a DEFAULT is to take care of endianness and labels.
     */
    typedef enum writeFlag {
      DEFAULT=0,
      NO_ENDIANNESS_CARE = 2,
      NO_LABELS_MARKS_CARE = 4
    } writeFlag;
    GIRL_DECLARE_FLAGS(writeFlags, writeFlag)


    /**
     * @brief Base class for writing toger.
     */
    class Writer
    {
    public:
      /**
       * @brief Destructor.
       */
      virtual ~Writer() {}

      /**
       * @brief Write function type,
       * will write @a size bytes of data obtaining them from the location given by @a p.
       *
       * @return number of bytes successfully written.
       * If this number is different than @a size, there was an error.
       */
      virtual size_t write(const void *p, size_t size) = 0;

      
      /**
       * @brief Tell that writes are over.
       * It will flush write calls.
       * @return true if everything was ok, else false.
       */
      virtual bool end() = 0;
    };


    /**
     * @brief Base class for reading toger.
     */
    class Reader 
    {
    public:
      /**
       * @brief Destructor.
       */
      virtual ~Reader() {}

      /**
       * @brief Read function type,
       * will read @a size bytes of data and store them at the location given by @a p.
       *
       * @return number of bytes successfully read.
       * If this number is different than @a size, there was an error.
       */
      virtual size_t read(void *p, size_t size) = 0;
    };



#define GIRL_IO_LITTLE_ENDIAN 0
#define GIRL_IO_BIG_ENDIAN 1

    /**
     * @brief Gives machine endianness.
     */
    static inline int machineEndianness(void)
    {
      short s = 0x0102;
      char *p = reinterpret_cast<char *>(&s);
      if (p[0] == 0x02) // Lowest address contains the least significant byte
	return GIRL_IO_LITTLE_ENDIAN;
      else
	return GIRL_IO_BIG_ENDIAN;
    }

    
    /**
     * @brief write a char.
     *
     * @return 1 if write was ok, else 0.
     */
    static inline int writeChar(char v, girl::io::Writer &w)
    {
      return w.write(&v, sizeof(char));
    }

    /**
     * @brief write an int taking care of endianness.
     *
     * @return 1 if write was ok, else 0.
     */
    static inline int writeInt(int v, girl::io::Writer &w)
    {
      assert(sizeof(int) == 4);
      unsigned char b;
      int k = 0;
      size_t s = 0;
      for (k=0; k<32; k+=8) {
	b = static_cast<unsigned char>(v>>k);
	s += w.write(&b, 1);
      }
      if (s != 4)
	return 0;
      return 1;
    }

    /**
     * @brief write an int taking care of endianness.
     *
     * @return 1 if write was ok, else 0.
     */
    static inline int writeUInt(unsigned int v, girl::io::Writer &w)
    {
      return writeInt(static_cast<int>(v), w);
    }



    /**
     * @brief read a char.
     */
    static inline int readChar(char *v, girl::io::Reader &r)
    {
      char c;
      if (r.read(&c, sizeof(char)) != sizeof(char))
	return 0;
      *v = c;
      return 1;
    }

    /**
     * @brief read an int taking care of endianness.
     *
     * @return 1 if read was ok, else 0.
     */
    static inline int readInt(int *v, girl::io::Reader &r)
    {
      assert(sizeof(int) == 4);
      unsigned char b;
      *v = 0;
      int k = 0;
      for (k=0; k<32; k+=8) {
	if (r.read(&b, sizeof(unsigned char)) != sizeof(unsigned char))
	  return 0;
	*v |= (static_cast<unsigned int>(b)<<k);
      }  
  
      return 1;
    }


    /**
     * @brief read an int taking care of endianness.
     *
     * @return 1 if read was ok, else 0.
     */
    static inline int readUInt(unsigned int *v, girl::io::Reader &r)
    {
      assert(sizeof(unsigned int) == 4);
      unsigned char b;
      *v = 0;
      int k = 0;
      for (k=0; k<32; k+=8) {
	if (r.read(&b, sizeof(unsigned char)) != sizeof(unsigned char))
	  return 0;
	*v |= (static_cast<unsigned int>(b)<<k);
      }  
  
      return 1;
    }


  }

}

GIRL_DECLARE_OPERATORS_FOR_FLAGS(girl::io::writeFlags)


#endif //GIRL_IO_HPP
