/*
  Geometrical dart class
  Copyright (C) 2005-2009 Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlGeoDart.hpp
 * @brief Geometrical Dart class.
 */

#ifndef GIRLGEODART_HPP
#define GIRLGEODART_HPP

#include "girlPoint.hpp"
#include "girlDirection.hpp"
#include "girlIO.hpp"


namespace girl {


  /**
   * @brief Geometrical dart class.
   *
   * It stores a point and a direction.
   */
  class geoDart
  {
  public:
    /**
     * @brief Constructor.
     */
    geoDart(POINT_TYPE px=0, POINT_TYPE py=0, girl::direction d=INVALID)
      : m_p(px, py), m_d(d)
    {}

    /**
     * @brief Constructor.
     */
    geoDart(const girl::point &pt, girl::direction d)
      : m_p(pt), m_d(d)
    {}

    /** 
     * @brief Get point of geometrical dart.
     */
    girl::point point() const
    {
      return m_p;
    }

    /**
     * @brief Gets x coordinate of point of geometrical dart.
     */
    inline
    POINT_TYPE x() const
    {
      return m_p.x();
    }

    /**
     * @brief Gets y coordinate of point of geometrical dart.
     */
    inline
    POINT_TYPE y() const
    {
      return m_p.y();
    }

    /**
     * @brief Gets direction of geometrical dart.
     */
    inline
    girl::direction dir() const
    {
      return m_d;
    }
    
    /**
     * @brief Sets x coordinate of geometrical dart to value @a px.
     */
    void setX(POINT_TYPE px)
    {
      m_p.setX(px);
    }

    /**
     * @brief Sets y coordinate of geometrical dart to value @a py.
     */
    void setY(POINT_TYPE py)
    {
      m_p.setY(py);
    }

    /**
     * @brief Sets direction of geometrical dart to @a d.
     */
    void setDir(girl::direction d)
    {
      m_d = d;
    }

    /**
     * @brief Sets point of geometrical dart to @a p.
     */
    void setPoint(const girl::point &p)
    {
      m_p = p;
    }

    /**
     * @brief Sets point coordinates and direction.
     */
    void set(POINT_TYPE px, POINT_TYPE py, girl::direction d)
    {
      m_p = girl::point(px, py);
      m_d = d;
    }

    /**
     * @brief Sets point and direction.
     */
    void set(const girl::point &p, girl::direction d)
    {
      m_p = p;
      m_d = d;
    }

    /**
     * @brief equality operator.
     */
    bool operator==(const girl::geoDart &u) const
    {
      return m_p == u.m_p && m_d == u.m_d;
    }

    /**
     * @brief inequality operator.
     */
    bool operator!=(const girl::geoDart &u) const
    {
      return m_p != u.m_p || m_d != u.m_d;
    }

   /**
     * @brief Write geometrical dart with function @a fct
     *
     * @a user_data allows to pass a context to the function.
     */
    void write(girl::io::Writer &w, girl::io::writeFlags flags) const
    {
      m_p.write(w, flags);
      m_d.write(w, flags);
    }

    /**
     * @brief Read geometrical dart with function @a fct
     *
     * @a user_data allows to pass a context to the function.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags flags)
    {
      if (! m_p.read(r, flags))
	return 0;
      if (! m_d.read(r, flags))
	return 0;
      return 1;
    }

  protected:
    girl::point m_p;
    girl::direction m_d;

  };

}


#endif /* !GIRLGEODART_HPP */
