/*
  Geometrical to topological dart map
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlGe2to.hpp
 * @brief Geometrical to topological dart map.
 */

#ifndef GIRLGE2TO_HPP
#define GIRLGE2TO_HPP

#include <map>

#include "girlToger.hpp" //for dart
#include "girlIO.hpp"
#include "girlPointHashMap.hpp"


namespace girl {

  class geoDart;

  class ge2to
  {
  public:
    /**
     * @brief Constructor.
     *
     * @param width   image width.
     * @param height  image height.
     */
    ge2to(girl::POINT_TYPE width, girl::POINT_TYPE height);

    /**
     * @brief Gets dart corresponding to geometrical dart @a u in map @a self.
     *
     * @return found dart or 0.
     */
    girl::dart getDart(const girl::geoDart &u) const;

    /**
     * @brief Sets dart @a d corresponding to geometrical dart @a u in map @a self.
     */
    void setDart(const girl::geoDart &u, girl::dart d);

    /**
     * @brief Remove dart corresponding to geometrical dart @a u from map @a self.
     *
     * @warning it is client responsibility to delete node if it has no more darts.
     */
    girl::dart clearDart(const girl::geoDart &u);

    /**
     * @brief Remove all darts starting from point @a p from map @a self.
     *
     * At most 4 darts (one in each direction) is removed.
     */
    void deleteNode(const girl::point &p);


    /**
     * @brief Replace direction associated with geometrical dart @a u with new direction @a newDir.
     *
     * @warning if there is a dart at (@a u->pt, @a newDir), it will be lost.
     * @warning there must be a node at @a u->pt.
     *
     * @return moved dart.
     */
    girl::dart replaceDirection(const girl::geoDart &u, girl::direction newDir);

    /**
     * @brief transfer all darts at point @a oldPt to point @a newPt.
     *
     * If point @a oldPt is not a node, nothing is done, else it will be deleted.
     * If point @a newPt is not already a node, it will be added.
     * @warning if there are darts associated with point @a newPt, they will be lost.
     */
    void transferDarts(const girl::point &oldPt, const girl::point &newPt);


    /**
     * @brief Get number of darts starting from point in four directions.
     */
    int getNbDarts(const girl::point &pt) const;


    /**
     * @brief Get darts starting from point in four directions.
     * 
     * @a darts is indexed with directions,
     * that is darts[UP] has dart at point @pt in UP direction (or 0 if there is none)
     */
    void getDarts(const girl::point &pt, girl::dart darts[4]) const;

    /**
     * @brief Set darts starting from point in four directions.
     * 
     * @a darts is indexed with directions,
     * that is darts[UP] has dart at point @pt in UP direction (or 0 if there is none)
     */
    void setDarts(const girl::point &pt, girl::dart darts[4]);


    /*
     * @brief Crop ge2to to the set of points in 
     * the box defined by inter-pixel points @a p1 and @a p2.
     *     
     * @warning We must have p1.x()<p2.x()
     * and p1.y()<p2.y().
     *
     * All points p outside box defined by @a p1 & @a p2, 
     * that is p.x() < p1.x() || p.y() < p1.y() || p.x() > p2.x() || p.y() > p2.y()
     * will be removed.
     * Ge2to will be cropped to new width and height.
     */
    void crop(const girl::point &p1, const girl::point &p2);


    /**
     * @brief Rename darts in ge2to.
     *
     * @param dartsMap map old dart label to new dart label.
     * Only positive darts are present in @a dartsMap.
     */
    void shrink(const girl::hashMap<girl::dart, girl::dart> &dartsMap);

    /**
     * @brief Write ge2to with writer @a w.
     *
     * @return true if write was ok, else false.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags flags) const;

    /**
     * @brief Read ge2to with reader @a r.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags flags);

    /**
     * @brief Swap two ge2to.
     *
     * Assign content of this to @a g and content of @a g to this.
     */
    void swap(girl::ge2to &g);
    
    /**
     * @brief Check overall ge2to consistency.
     * Will print found errors, if any, to stderr. 
     */
    bool checkConsistency() const;

  protected:

    struct dart4
    {

      dart4(size_t i=0, girl::dart d=0)
      {
	m_d[0] = m_d[1] = m_d[2] = m_d[3] = 0;

	m_d[i] = d;
      }

      dart4(girl::dart d0, girl::dart d1, girl::dart d2, girl::dart d3)
      {
	m_d[0] = d0;
	m_d[1] = d1;
	m_d[2] = d2;
	m_d[3] = d3;
      }

      girl::dart operator[](size_t i) const
      {
	return m_d[i];
      }

      girl::dart &operator[](size_t i)
      {
	return m_d[i];
      }

      dart4 &operator=(const dart4 &d4)
      {
	m_d[0] = d4.m_d[0];
	m_d[1] = d4.m_d[1];
	m_d[2] = d4.m_d[2];
	m_d[3] = d4.m_d[3];
	return *this;
      }
      
      girl::dart m_d[4];
      
    };


    typedef girl::pointHashMap<girl::ge2to::dart4> HashMapType;

    HashMapType m_hashMap;

    girl::POINT_TYPE m_height;
    

  };


  inline
  void
  swap(ge2to &g1, ge2to &g2)
  {
    g1.swap(g2);
  }

}

namespace std {

  template <>
  inline
  void
  swap(girl::ge2to &g1, girl::ge2to &g2)
  {
    g1.swap(g2);
  }
  
}


#endif /* ! GIRLGE2TO_HPP */
