/*
  Geometrical to topological dart map
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlGe2to.hpp"

#include <cstdlib>
#include "girlGeoDart.hpp"
#include "girlEndian.hpp"
#include "girlHashFunc.hpp"
#include "girlUnorderedSet.hpp"

#include <iostream> //DEBUG

using namespace girl;


ge2to::ge2to(girl::POINT_TYPE /*width*/, girl::POINT_TYPE height)
  : m_hashMap(height), m_height(height)
{
  //impl: we keep "height" to be able to instantiate the correct ptHasher after a write/read...


#if defined(GIRL_HASH_MAP_USES_GOOGLE_SPARSE_HASH_MAP) || defined(HASH_MAP_USES_GOOGLE_SPARSE_HASH_MAP) || defined(GIRL_HASH_MAP_USES_GOOGLE_DENSE_HASH_MAP) || defined(HASH_MAP_USES_GOOGLE_DENSE_HASH_MAP)
#if defined(GIRL_HASH_MAP_USES_GOOGLE_DENSE_HASH_MAP) || defined(HASH_MAP_USES_GOOGLE_DENSE_HASH_MAP) 
  m_hashMap.set_empty_key(ptHasher(height)(girl::point(-3, -3)));
#endif
  m_hashMap.set_deleted_key(ptHasher(height)(girl::point(-2, -2)));
#endif

  assert(checkConsistency());
}

girl::dart 
ge2to::getDart(const girl::geoDart &u) const
{
  const HashMapType::const_iterator it = m_hashMap.find(u.point());
  if (it != m_hashMap.end())
    return it->second[u.dir()];
  else
    return 0;
}

void 
ge2to::setDart(const girl::geoDart &u, girl::dart d)
{
  const HashMapType::iterator it = m_hashMap.find(u.point());
  if (it != m_hashMap.end())
    it->second[u.dir()] = d;
  else {
    m_hashMap[u.point()] = ge2to::dart4(static_cast<size_t>(u.dir()), d);
  }
}

girl::dart 
ge2to::clearDart(const girl::geoDart &u)
{
  const HashMapType::iterator it = m_hashMap.find(u.point());
  if (it != m_hashMap.end()) {
    const girl::dart d = it->second[u.dir()];
    it->second[u.dir()] = 0;
    return d;
  }
  return 0;
}

void 
ge2to::deleteNode(const girl::point &p)
{
  m_hashMap.erase(p);
}


girl::dart 
ge2to::replaceDirection(const girl::geoDart &u, girl::direction newDir)
{
  const HashMapType::iterator it = m_hashMap.find(u.point());
  if (it != m_hashMap.end()) {
    const girl::direction oldDir = u.dir();
    const girl::dart d = it->second[oldDir];
    it->second[oldDir] = 0;
    it->second[newDir] = d;
    return d;
  }
  return 0;
}

void 
ge2to::transferDarts(const girl::point &oldPt, const girl::point &newPt)
{
  const HashMapType::iterator it = m_hashMap.find(oldPt);
  if (it != m_hashMap.end()) {
    dart4 &oldDarts = it->second;
    const HashMapType::iterator it2 = m_hashMap.find(newPt);
    if (it2 != m_hashMap.end())
      it2->second = oldDarts;
    else 
      m_hashMap[newPt] = oldDarts;
    m_hashMap.erase(it);
  }
}

int
ge2to::getNbDarts(const girl::point &pt) const
{
  int nb = 0;
  const HashMapType::const_iterator it = m_hashMap.find(pt);
  if (it != m_hashMap.end()) {
    const dart4 &ds = it->second;
    nb += (ds[0] != 0);
    nb += (ds[1] != 0);
    nb += (ds[2] != 0);
    nb += (ds[3] != 0);
  }
  assert(nb>=0 && nb<=4);
  return nb;
}

void 
ge2to::getDarts(const girl::point &pt, girl::dart darts[4]) const
{
  const HashMapType::const_iterator it = m_hashMap.find(pt);
  if (it != m_hashMap.end()) {
    const dart4 &ds = it->second;
    darts[0] = ds[0]; 
    darts[1] = ds[1]; 
    darts[2] = ds[2]; 
    darts[3] = ds[3];
  }
  else {
    darts[0] = 0;
    darts[1] = 0;
    darts[2] = 0;
    darts[3] = 0;
  }
}

void 
ge2to::setDarts(const girl::point &pt, girl::dart darts[4])
{
  const HashMapType::iterator it = m_hashMap.find(pt);
  if (it != m_hashMap.end()) {
    dart4 &ds = it->second;
    ds[0] = darts[0];
    ds[1] = darts[1];
    ds[2] = darts[2];
    ds[3] = darts[3];
  }
  else {
    m_hashMap[pt] = dart4(darts[0], darts[1], darts[2], darts[3]);
  }
}

void
ge2to::crop(const girl::point &p1, const girl::point &p2)
{
  assert(p1.x() < p2.x());
  assert(p1.y() < p2.y());


  girl::POINT_TYPE newHeight = p2.y() - p1.y(); //not const, for swap

  HashMapType newHashMap(newHeight);

  HashMapType::iterator it = m_hashMap.begin();
  const HashMapType::iterator itEnd = m_hashMap.end();
  for ( ; it != itEnd; ++it) {
    girl::point p = it->first; //copy
    if (p.x() >= p1.x() && p.y() >= p1.y() && p.x() <= p2.x() && p.y() <= p2.y()) {
      //point inside kept rectangle

      //remove darts in outer directions on borders
      if (p.x() == p1.x()) 
	it->second[girl::LEFT] = 0;
      else if (p.x() == p2.x())
	it->second[girl::RIGHT] = 0;
      if (p.y() == p1.y()) 
	it->second[girl::UP] = 0;
      else if (p.y() == p2.y())
	it->second[girl::DOWN] = 0;
      
      //translate point
      p.set(p.x()-p1.x()-1, p.y()-p1.y()-1); //inter-pixels!

      //insert in new hash map   [allocation]
      newHashMap[p] = it->second;
    }
  }

  std::swap(m_hashMap, newHashMap);
  std::swap(m_height, newHeight);

  //REM:Implementation: we cannot do it directly on m_hashMap
  // Indeed, we cannot modify the hash key without 
  // (it->first.set(x, y) is not possible)

  assert(checkConsistency());

}


void
ge2to::shrink(const girl::hashMap<girl::dart, girl::dart> &dartsMap)
{
  typedef girl::hashMap<girl::dart, girl::dart> DartsMap;

  if (dartsMap.size() > 0) {

    DartsMap::const_iterator itD;
    const DartsMap::const_iterator itDEnd = dartsMap.end();
    
    HashMapType::iterator it = m_hashMap.begin();
    const HashMapType::iterator itEnd = m_hashMap.end();
    for ( ; it != itEnd; ++it) {
      dart4 &ds = it->second;
      
      for (int i=0; i<4; ++i) {
	
	const girl::dart di = ds[i];
	itD = dartsMap.find(abs(di));
	if (itD != itDEnd) {
	  assert(itD->second > 0);
	  const girl::dart ndi = di > 0 ? itD->second : girl::dart(-(itD->second));
	  assert((ndi!=0) | (di==0));
	  ds[i] = ndi;
	}
	
      }
      
    }
  }

  assert(checkConsistency());
}


bool
ge2to::write(girl::io::Writer &w, girl::io::writeFlags flags) const
{
  //all is saved in little endian

  girl::POINT_TYPE height = m_height;

  size_t size = m_hashMap.size();

  assert(m_hashMap.size() < static_cast<size_t>(static_cast<unsigned int>(-1)));
  unsigned int sz = size;

  //temporary vector to minimize number of w.write calls.
  typedef std::pair<girl::point, girl::ge2to::dart4> HashCell;
  std::vector<HashCell> vec(size);
  
  HashMapType::const_iterator it = m_hashMap.begin();
  const HashMapType::const_iterator itEnd = m_hashMap.end();

  if ((flags & girl::io::NO_ENDIANNESS_CARE) || GIRL_BYTEORDER == GIRL_LIL_ENDIAN) {

    for (size_t i=0 ; it != itEnd; ++it, ++i) {
      HashCell &c = vec[i];
      c.first = it->first;
      c.second = it->second;
    }

  }
  else {
    assert((flags & girl::io::NO_ENDIANNESS_CARE) == 0);
    assert(GIRL_BYTEORDER == GIRL_BIG_ENDIAN);
    assert(sizeof(girl::POINT_TYPE) == 2);

    for (size_t i=0 ; it != itEnd; ++it, ++i) {
      HashCell &c = vec[i];
      c.first = it->first;
      c.second = it->second;
      
      c.first.setX(GIRL_swapLE16(c.first.x()));
      c.first.setY(GIRL_swapLE16(c.first.y()));
      
      c.second[0] = GIRL_swapLE32(c.second[0]);
      c.second[1] = GIRL_swapLE32(c.second[1]);
      c.second[2] = GIRL_swapLE32(c.second[2]);
      c.second[3] = GIRL_swapLE32(c.second[3]);
    }

    sz = GIRL_swapLE32(sz);
    height = GIRL_swapLE16(height);
  }

  if (w.write(&height, sizeof(height)) != sizeof(height))
    return false;
  if (w.write(&sz, sizeof(sz)) != sizeof(sz))
    return false;
  const size_t sizeV = size*sizeof(HashCell);
  if (w.write(&vec[0], sizeV) != sizeV)
    return false;

  return true;
}


bool
ge2to::read(girl::io::Reader &r, girl::io::writeFlags flags)
{
  //all is saved in little endian

  girl::POINT_TYPE height;
  if (r.read(&height, sizeof(height)) != sizeof(height))
    return false;
  height = GIRL_swapLE16(height);
  assert(height>=0);
  
  unsigned int size;
  if (r.read(&size, sizeof(size)) != sizeof(size))
    return false;
  size = GIRL_swapLE32(size);
  
  //temporary vector to minimize number of r.read calls.
  typedef std::pair<girl::point, girl::ge2to::dart4> HashCell;
  std::vector<HashCell> vec(size);
  const size_t sizeV = size*sizeof(HashCell);
  if (r.read(&vec[0], sizeV) != sizeV)
    return false;


  m_height = height;
  m_hashMap = HashMapType(height);
  
  if ((flags & girl::io::NO_ENDIANNESS_CARE) || GIRL_BYTEORDER == GIRL_LIL_ENDIAN) {

    for (size_t i=0; i<size; ++i) {
      HashCell &c = vec[i];
      m_hashMap[c.first] = c.second;
    }
    
  }
  else {

    assert((flags & girl::io::NO_ENDIANNESS_CARE) == 0);
    assert(GIRL_BYTEORDER == GIRL_BIG_ENDIAN);
    assert(sizeof(girl::POINT_TYPE) == 2);
    
    for (size_t i=0; i<size; ++i) {
      HashCell &c = vec[i];
      
      c.first.setX(GIRL_swapLE16(c.first.x()));
      c.first.setY(GIRL_swapLE16(c.first.y()));
      
      c.second[0] = GIRL_swapLE32(c.second[0]);
      c.second[1] = GIRL_swapLE32(c.second[1]);
      c.second[2] = GIRL_swapLE32(c.second[2]);
      c.second[3] = GIRL_swapLE32(c.second[3]);

      m_hashMap[c.first] = c.second;
    }

  }

  return true;
}

void
ge2to::swap(girl::ge2to &g)
{
  std::swap(m_hashMap, g.m_hashMap);
  std::swap(m_height, g.m_height);
}


bool
ge2to::checkConsistency() const
{
  //For now just check that there are no duplicates among darts

  bool consistent = true;
  if (m_hashMap.size() == 0)
    return true; //? only valid for constructors. after that, ge2to should never be empty !?


#if 0

  //REM: we cannot code it like this as it takes too much memory space 
  // for relatively big togers (chessboard 1000x1000) !!!

  typedef std::deque<girl::point> PointDq;

  typedef girl::hashMap<girl::dart, PointDq> DartPresenceH;
  DartPresenceH dH(3*m_hashMap.size());

  //sizeof(PointDq) is ~80 by default ! It is too big.

  HashMapType::const_iterator itH = m_hashMap.begin();
  const HashMapType::const_iterator itHEnd = m_hashMap.end();
  for ( ; itH != itHEnd; ++itH) {
    const dart4 &ds = itH->second;
    
    for (int i=0; i<4; ++i) {
      girl::dart d = ds[i];
      if (d != 0) {
	
	const DartPresenceH::iterator it = dH.find(d);
	if (it != dH.end()) { //dart already present for another point
	  consistent = false;
	  it->second.push_back(itH->first);
	  std::cerr<<"dart "<<d<<" present several time in ge2to at: ";
	  for (PointDq::const_iterator itD=it->second.begin();
	       itD != it->second.end();
	       ++itD)
	    std::cerr<<"("<<itD->x()<<"; "<<itD->y()<<") ";
	  std::cerr<<std::endl;
	}
	else {
	  PointDq dq;
	  dq.push_back(itH->first);
	  dH[d] = dq;
	}

      }
      
    }

  }

#elif 1

  //We use a hashmap to store points 
  //If a point is already there, we traverse the hashmap again to find and print where it also is.

  typedef girl::unorderedSet<girl::dart> DartPresenceH;
  DartPresenceH dH(3*m_hashMap.size());

  HashMapType::const_iterator itH = m_hashMap.begin();
  const HashMapType::const_iterator itHEnd = m_hashMap.end();
  for ( ; itH != itHEnd; ++itH) {
    const dart4 &ds = itH->second;
    
    for (int i=0; i<4; ++i) {
      const girl::dart d = ds[i];
      if (d != 0) {
	
	const DartPresenceH::iterator it = dH.find(d);
	if (it != dH.end()) { //dart already present for another point
	  consistent = false;
	  std::cerr<<"dart "<<d<<" present several time in ge2to at ("<<itH->first.x()<<"; "<<itH->first.y()<<")";
	  HashMapType::const_iterator itH2 = m_hashMap.begin();
	  for ( ; itH2 != itHEnd; ++itH2) {
	    if (itH2 != itH) {
	      const dart4 &ds2 = itH2->second;
	      for (int j=0; j<4; ++j) {
		const girl::dart d2 = ds2[j];
		if (d2 == d) {
		  std::cerr<<" and ("<<itH2->first.x()<<"; "<<itH2->first.y()<<")";
		  break; //only this level
		}
	      }
	    }
	  }
	}
	else {
	  dH.insert(d);
	}
	
      }
      
    }
    
  }
  

#else
  //very long !
  //O(N2) !!!

  HashMapType::const_iterator itH = m_hashMap.begin();
  const HashMapType::const_iterator itHEnd = m_hashMap.end();
  for ( ; itH != itHEnd; ++itH) {

    const dart4 &ds = itH->second;
    int s=0; //index of first non null dart (for this point)
    for ( ; s<4; ++s) {
      if (ds[s] != 0)
	break;
    }
    if (s==4) { //no valid dart (for this point) !
      consistent = false;
      std::cerr<<"No valid dart in ge2to at : ("<<itH->first.x()<<"; "<<itH->first.y()<<")"<<std::endl;      continue;
      continue;
    }
    //first check duplicates among darts of current point.
    const girl::dart d = ds[s];
    for (int j=s+1; j<4; ++s) {
      const girl::dart d2 = ds[j];
      if (d2 == d) {
	consistent = false;
	std::cerr<<"dart "<<d<<" present several time in ge2to at : ("<<itH->first.x()<<"; "<<itH->first.y()<<")"<<std::endl;
      }
    }
    if (consistent == false)
      continue;

    //then check duplicates among darts of other points.
    HashMapType::const_iterator itHNext = itH;
    ++itHNext;
    if (itHNext == itHEnd)
      break;
    for ( ; itHNext != itHEnd; ++itHNext) {
      const dart4 &ds2 = itHNext->second;
      for (int i=s; i<4; ++i) {
	const girl::dart d = ds[i];
	if (d != 0) {
	  for (int j=0; j<4; ++j) {
	    const girl::dart d2 = ds2[j];
	    if (d == d2) {
	      consistent = false;
	      std::cerr<<"dart "<<d<<" present several time in ge2to at: ("<<itH->first.x()<<"; "<<itH->first.y()<<") and ("<<itHNext->first.x()<<"; "<<itHNext->first.y()<<")"<<std::endl;
	      //return consistent;
	    }
	  }
	}
      }
    }
	    
  }  


#endif
  
  return consistent;
}

