/*
  Freeman code based paths
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlFreemanCodePath.hpp
 * @brief Freeman code based path.
 * This class is not to be used directly. Use its derivative, girl::path & girl::contour instead.
 */

#ifndef GIRLFREEMANCODEPATH_HPP
#define GIRLFREEMANCODEPATH_HPP

#include <vector>
#include <deque>

#include <girl/girlCommon.hpp>
#include <girl/girlPoint.hpp>
#include <girl/girlDirection.hpp>

namespace girl {

  /**
   * @brief Freeman Code Path class.
   *
   * It encodes a 4-connexe path with a freeman code,
   * that is a staring point and directions.
   *
   * This class serves for code factorization,
   * and should not be used directly.
   * Use girl::contour or girl::path instead.
   */
  class GIRL_API freemanCodePath
  {
  protected:

    /**
     * @brief Default constructor.
     *
     * Constructs a freemanCodePath of length 0 starting and ending at @a startPt. 
     */
    inline explicit freemanCodePath(const girl::point &startPt = girl::point(0, 0));
    
    /**
     * @brief Constructor
     *
     * Construct a freemanCodePath from a starting point @a startPt
     * and a deque of directions @a dirs.
     */
    freemanCodePath(const girl::point &startPt,
		    const std::deque<girl::direction> &dirs);

  public:

    /**
     * @brief Set freemanCodePath from a starting point @a startPt
     * and a deque of directions @a dirs.
     */
    void set(const girl::point &startPt,
	     const std::deque<girl::direction> &dirs);

    /**
     * @brief Gives starting point of freemanCodePath @a self.
     */
    inline girl::point startPt() const;
    
    
    /**
     * @brief Gives end point of freemanCodePath @a self.
     *
     * @warning this function is not virtual.
     */
    girl::point endPt() const;
        
    /**
     * @brief FreemanCodePath length, i.e. number of directions.
     *
     * O(1).
     */
    inline size_t length() const;
    

    /**
     * @brief Accessor.
     *
     * @warning @a i must be in [0; length()[.
     *
     * O(1)
     * but it is preferable to use a const_dir_iterator when possible.
     */
    inline girl::direction operator[](size_t i) const;


    /**
     * @brief Moves starting point to new point @a newStartPt.
     *
     * End point is modified accordingly.
     */
    inline void move(const girl::point &newStartPt);


    /**
     * @brief Inverts orientation of freemanCodePath.
     * 
     * For example, a freemanCodePath with four directions Right, Up, Left, and Down,
     * will be changed to  Up, Right, Down, and Left.
     *
     * @warning this function is not virtual.
     */
    void invertOrientation();

    /**
     * @brief Tells if the freemanCodePath is closed,
     * i.e., end point equals start point.
     *
     * @return true if freemanCodePath is empty.
     *
     * @warning this function is not virtual.
     */
    bool isClosed() const;


    /**
     * @brief Tells if the freemanCodePath is valid.
     * that is two successive directions are not in opposite way.
     *
     * @warning this function is not virtual.
     */
    bool isValid() const;

    /**
     * @brief Tells if the freemanCodePath is both valid and closed.
     * 
     * @sa isValid. 
     * @sa isClosed.
     */
    bool isValidAndClosed() const;

    /**
     * @brief Tells if point @a pt belongs to the freemanCodePath.
     *
     * @warning this function is not virtual.
     */
    bool belongs(const girl::point &pt) const;


    /**
     * @brief equality operator
     */
    inline bool operator==(const freemanCodePath &o);

    /**
     * @brief inequality operator
     */
    inline bool operator!=(const freemanCodePath &o);


    class GIRL_API const_dir_iterator
    {
    public:
      /**
       * @brief Default constructor.
       */
      const_dir_iterator()
	: m_c(0), m_i(0)
      {}


      const_dir_iterator &operator=(const const_dir_iterator &o)
      {
	m_c = o.m_c;
	m_i = o.m_i;
	return *this;
      }

      bool operator==(const const_dir_iterator &o) const
      {
	return m_i == o.m_i && m_c == o.m_c;
      }
      
      bool operator!=(const const_dir_iterator &o) const
      {
	return m_i != o.m_i || m_c != o.m_c;
      }
      
      const_dir_iterator &operator++()
      {
	//m_i is in {0;1;2;3}
	m_i = (m_i+1)&3; //m_i = i%4;
	m_c += !m_i;   //if m_i==0 (after increment) then ++m_c
	return *this;
      }
      
      const_dir_iterator &operator--()
      {
	//m_i is in {0;1;2;3}
	m_c -= !m_i; //if m_i==0 then --m_c
	m_i = (m_i+3)&3;     //if m_i==0 then m_i=3, else --m_i
	return *this;
      }
      
      const_dir_iterator operator++(int) //post increment
      {
	const_dir_iterator tmp(*this);
	++(*this);
	return tmp; 
      }

      const_dir_iterator operator--(int) //post decrement
      {
	const_dir_iterator tmp(*this);
	--(*this);
	return tmp;
      }

      girl::direction operator*() const
      {
	return static_cast<eDirection>((*m_c >>(m_i<<1))&3);
      }

      friend const_dir_iterator operator+(const const_dir_iterator &it, size_t ind)
      {
	const_dir_iterator tmp(it);
	if (ind > 0) {
	  const int s = tmp.m_i + ind;
	  const int d = s/4; 
	  const int r = s - d*4;
	  tmp.m_i = r;
	  tmp.m_c += d;
	}
	return tmp;
      }
      

    private:
      const girl::direction *operator->();
      //impl: cannot provide this operator as directions are not actually stored as directions.


    protected:
      const_dir_iterator(const char *c, int i = 0)
	: m_c(c), m_i(i)
      {}

      friend class girl::freemanCodePath;


    protected:
      const char *m_c;
      int m_i;  //i is in {0;1;2;3}

      //impl: idea is to use cache locality 
      //  (so to have m_c points to the current char, and not the vector start.
      //      and to avoid branches in iterators...
    }; 


    class GIRL_API const_reverse_dir_iterator
    {
    public:
      /**
       * @brief Default constructor.
       */
      const_reverse_dir_iterator()
	: m_c(0), m_i(0)
      {}


      const_reverse_dir_iterator &operator=(const const_reverse_dir_iterator &o)
      {
	m_c = o.m_c;
	m_i = o.m_i;
	return *this;
      }

      bool operator==(const const_reverse_dir_iterator &o) const
      {
	return m_i == o.m_i && m_c == o.m_c;
      }
      
      bool operator!=(const const_reverse_dir_iterator &o) const
      {
	return m_i != o.m_i || m_c != o.m_c;
      }
      
      const_reverse_dir_iterator &operator++()
      {
	//m_i is in {0;1;2;3}
	m_c -= !m_i; //if m_i==0 then --m_c
	m_i = (m_i+3)&3;     //if m_i==0 then m_i=3, else --m_i
	return *this;
      }
      
      const_reverse_dir_iterator &operator--()
      {
	//m_i is in {0;1;2;3}
	m_i = (m_i+1)&3; //m_i = (i+1)%4;
	m_c += !m_i;   //if m_i==0 (after increment) then ++m_c
	return *this;
      }
      
      const_reverse_dir_iterator operator++(int) //post increment
      {
	const_reverse_dir_iterator tmp(*this);
	++(*this);
	return tmp; 
      }

      const_reverse_dir_iterator operator--(int) //post decrement
      {
	const_reverse_dir_iterator tmp(*this);
	--(*this);
	return tmp;
      }

      girl::direction operator*() const
      {
	return static_cast<eDirection>((*m_c >>(m_i<<1))&3);
      }

    private:
      const girl::direction *operator->();
      //impl: cannot provide this operator as directions are not actually stored as directions.

    protected:
      const_reverse_dir_iterator(const char *c, int i = 0)
	: m_c(c), m_i(i)
      {}

      friend class girl::freemanCodePath;

    protected:
      const char *m_c;
      int m_i;  //i is in {0;1;2;3}

      //impl: idea is to use cache locality 
      //  (so to have m_c points to the current char, and not the vector start.
      //      and to avoid branches in iterators...
    }; 


    /**
     * @return an iterator on first direction of freemanCodePath.
     */
    inline const_dir_iterator begin() const
    {
      return const_dir_iterator(&m_path[0]);
    }

    /**
     * @return an iterator past the last direction of freemanCodePath.
     */
    inline const_dir_iterator end() const
    {
      return ++const_dir_iterator(&m_path[m_path.size()-1], (m_length-1)&3);
    }
   
    
    /**
     * @return an iterator on last direction of freemanCodePath.
     *
     * Allow to traverse directions of a path in reverse order.
     */
    inline const_reverse_dir_iterator rbegin() const
    {
      return const_reverse_dir_iterator(&m_path[m_path.size()-1], (m_length-1)&3);
    }

    /**
     * @return an iterator past/before the first direction of freemanCodePath.
     *
     * Allow to traverse directions of a path in reverse order.
     */
    inline const_reverse_dir_iterator rend() const
    {
      return ++const_reverse_dir_iterator(&m_path[0], 0);
    }
    
    /**
     * @brief Swap two freemanCodePath.
     *
     * Assign content of this to @a fcp and content of @a fcp to this.
     */
    void swap(freemanCodePath &fcp);

  protected:

     /**
     * @brief Constructor.
     *
     * Constructs a freemanCodePath corresponding to given directions.
     *
     * @a start is first point of path.
     * @a packedDirections is an array of @a nbDirections directions 
     * with 4 directions in each char. 
     * @a packedDirections will be copied. It must be of actual size (nbDirections+(4-1))/4
     *
     */
    inline freemanCodePath(const girl::point &start, 
			   const std::vector<char> &packedDirections, 
			   size_t nbDirections);
   

    void set_aux(const std::deque<girl::direction> &dirs);

  protected:

    ~freemanCodePath() {}

  protected:

    std::vector<char> m_path; //4 steps per byte, 2 bits per step.
    girl::point m_start;
    size_t m_length;
    //impl: m_length is different of m_path.size()
    //      actually m_length/4 <= m_path.size()
  };



  //inline function --------------------------------------------------

  
  freemanCodePath::freemanCodePath(const girl::point &pStartPt)
    : m_path(), m_start(pStartPt), m_length(0)
  {
    assert(isValid());
  }

  freemanCodePath::freemanCodePath(const girl::point &pStartPt, 
				   const std::vector<char> &packedDirections, 
				   size_t nbDirections)
    : m_path(packedDirections), m_start(pStartPt), m_length(nbDirections)
  {
      assert(isValid());
  }

  girl::point 
  freemanCodePath::startPt() const
  {
    return m_start;
  }
  
  size_t 
  freemanCodePath::length() const
  {
    return m_length;
  }

  girl::direction
  freemanCodePath::operator[](size_t i) const
  {
    assert(i < length());

    return static_cast<eDirection>((m_path[i/4] >>((i%4)<<1))&3);   
  }

  void
  freemanCodePath::move(const girl::point &newStartPt)
  {
    m_start = newStartPt;
  }


  bool
  freemanCodePath::operator==(const freemanCodePath &o)
  {
    return m_start == o.m_start && m_length == o.m_length && m_path == o.m_path;
  }

  bool
  freemanCodePath::operator!=(const freemanCodePath &o)
  {
    return !operator==(o);
  }

  inline
  void
  swap(freemanCodePath &f1, freemanCodePath &f2)
  {
    f1.swap(f2);
  }

}

namespace std {

  template <>
  inline
  void
  swap(girl::freemanCodePath &f1, girl::freemanCodePath &f2)
  {
    f1.swap(f2);
  }

}
    
#endif /* ! GIRLFREEMANCODEPATH_HPP */
