/*
  Freeman code based paths
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlFreemanCodePath.hpp"


using namespace girl;

static inline
size_t
compressedSize(size_t nbDirs)
{
  return (nbDirs+3)/4;
}


freemanCodePath::freemanCodePath(const girl::point &pStartPt,
				 const std::deque<girl::direction> &dirs)
  : m_path(compressedSize(dirs.size())), 
    m_start(pStartPt), 
    m_length(dirs.size())
{
  set_aux(dirs);
  assert(isValid());
}

void
freemanCodePath::set(const girl::point &pStartPt,
		     const std::deque<girl::direction> &dirs)
{
  const size_t s = dirs.size();
  m_path.clear();
  m_path.resize(compressedSize(s));
  m_start = pStartPt;
  m_length = s;
  set_aux(dirs);
  assert(isValid());
}

void
freemanCodePath::set_aux(const std::deque<girl::direction> &dirs)
{
  assert(m_path.size() == compressedSize(dirs.size()));

  std::deque<girl::direction>::const_iterator it = dirs.begin();
  const std::deque<girl::direction>::const_iterator itEnd = dirs.end();
  char *curB = &m_path[0];
  int mod = 0;
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    *curB |= static_cast<char>(d << mod);
    mod += 2;
#if 0
    if (mod == 8) {
      mod = 0;
      ++curB;
    }
#else
    mod &= 7;
    curB += !mod;
#endif
  }  
}


void
freemanCodePath::invertOrientation()
{
  if (length() == 0)
    return;

  std::vector<char> path(m_path.size());
  path[m_path.size()-1] = 0;


  int mod = 0;
  char *curB = &path[0];
  freemanCodePath::const_reverse_dir_iterator it = rbegin();
  const freemanCodePath::const_reverse_dir_iterator itREnd = rend();
  for ( ; it != itREnd; ++it) {
    const girl::direction d = (*it).opposite();
    *curB |= static_cast<char>(d << mod);
    mod += 2;    
    mod &= 7;
    curB += !mod;
  }

  /*
  int mod = 0;
  char *curB = &path[0];
  for (size_t i=m_length-1; i!=(size_t)-1; --i) {
    
    const girl::direction d = girl::direction((eDirection)((m_path[i >> 2] >> ((i & 3) << 1)) & 3)).opposite();
    *curB |= (char)(d << mod);
    mod += 2;
#if 0
    if (mod == 8) {
      mod = 0;
      ++curB;
    }
#else
    mod &= 7;
    curB += !mod;
#endif
  }
  */

  std::swap(path, m_path);

}

girl::point
freemanCodePath::endPt() const
{
  if (length() == 0)
    return m_start;

  girl::point pt = startPt();
  freemanCodePath::const_dir_iterator it = begin();
  const freemanCodePath::const_dir_iterator itEnd = end();
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    pt.move(d);
  }
  return pt;
}

bool 
freemanCodePath::isClosed() const
{
  return endPt() == startPt();
}

bool
freemanCodePath::isValid() const
{
  if (length() == 0)
    return true; 

  freemanCodePath::const_dir_iterator it = begin();
  const freemanCodePath::const_dir_iterator itEnd = end();
  girl::direction prevd = *it;
  ++it;
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    if (d == prevd.opposite())
      return false;
    prevd = d;
  }
  return true;
}

bool
freemanCodePath::isValidAndClosed() const
{
  if (length() == 0)
    return true; 

  girl::point pt = startPt();
  freemanCodePath::const_dir_iterator it = begin();
  const freemanCodePath::const_dir_iterator itEnd = end();
  girl::direction prevd = *it;
  pt.move(prevd);
  ++it;
  for ( ; it != itEnd; ++it) {
    const girl::direction d = *it;
    if (d == prevd.opposite())
      return false;
    pt.move(d);
    prevd = d;
  }
  return pt == startPt();  
}



bool
freemanCodePath::belongs(const girl::point &pt) const
{
  if (length() == 0)
    return false; 

  girl::point cpt = startPt();
  if (cpt == pt)
    return true;
  
  freemanCodePath::const_dir_iterator it = begin();
  const freemanCodePath::const_dir_iterator itEnd = end();
  for ( ; it != itEnd; ++it) {
    cpt.move(*it);
    if (cpt == pt)
      return true;
  }

  return false;
}

void
freemanCodePath::swap(freemanCodePath &fcp)
{
  std::swap(m_path, fcp.m_path);
  std::swap(m_start, fcp.m_start);
  std::swap(m_length, fcp.m_length);
}
