/*
  Endianness functions borrowed from SDL_endian.h
  Copyright (C) 1997-2006 Sam Lantinga
  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlEndian.hpp
 * @brief Endianness functions borrowed from SDL_endian.h.
 */

#ifndef GIRL_ENDIAN_HPP
#define GIRL_ENDIAN_HPP

//Boris
#if defined(_MSC_VER) && (_MSC_VER < 1600)
 //stdint.h only available from Visual C++ 2010, i.e., _MSC_VER >= 1600
#include <girl/pstdint.h>
#else
#include <stdint.h>
#endif

//Boris
#ifdef _MSC_VER
  #define INLINE __inline 
#else
  #define INLINE __inline__
#endif

/* The two types of endianness */
#define GIRL_LIL_ENDIAN	1234
#define GIRL_BIG_ENDIAN	4321

#ifndef GIRL_BYTEORDER	/* Not defined in GIRL_config.h? */
#ifdef __linux__
#include <endian.h>
#define GIRL_BYTEORDER  __BYTE_ORDER
#else /* __linux __ */
#if defined(__hppa__) || \
    defined(__m68k__) || defined(mc68000) || defined(_M_M68K) || \
    (defined(__MIPS__) && defined(__MISPEB__)) || \
    defined(__ppc__) || defined(__POWERPC__) || defined(_M_PPC) || \
    defined(__sparc__)
#define GIRL_BYTEORDER	GIRL_BIG_ENDIAN
#else
#define GIRL_BYTEORDER	GIRL_LIL_ENDIAN
#endif
#endif /* __linux __ */
#endif /* !GIRL_BYTEORDER */


/* Set up for C function definitions, even when using C++ */
#ifdef __cplusplus
extern "C" {
#endif

/* Use inline functions for compilers that support them, and static
   functions for those that do not.  Because these functions become
   static for compilers that do not support inline functions, this
   header should only be included in files that actually use them.
*/
#if defined(__GNUC__) && defined(__i386__) && \
   !(__GNUC__ == 2 && __GNUC_MINOR__ <= 95 /* broken gcc version */)
static INLINE uint16_t GIRL_swap16(uint16_t x)
{
	__asm__("xchgb %b0,%h0" : "=q" (x) :  "0" (x));
	return x;
}
#elif defined(__GNUC__) && defined(__x86_64__)
static INLINE uint16_t GIRL_swap16(uint16_t x)
{
	__asm__("xchgb %b0,%h0" : "=Q" (x) :  "0" (x));
	return x;
}
#elif defined(__GNUC__) && (defined(__powerpc__) || defined(__ppc__))
static INLINE uint16_t GIRL_swap16(uint16_t x)
{
	uint16_t result;

	__asm__("rlwimi %0,%2,8,16,23" : "=&r" (result) : "0" (x >> 8), "r" (x));
	return result;
}
#elif defined(__GNUC__) && (defined(__M68000__) || defined(__M68020__))
static INLINE uint16_t GIRL_swap16(uint16_t x)
{
	__asm__("rorw #8,%0" : "=d" (x) :  "0" (x) : "cc");
	return x;
}
#else
static INLINE uint16_t GIRL_swap16(uint16_t x) {
	return((x<<8)|(x>>8));
}
#endif

#if defined(__GNUC__) && defined(__i386__) && \
   !(__GNUC__ == 2 && __GNUC_MINOR__ <= 95 /* broken gcc version */)
static INLINE uint32_t GIRL_swap32(uint32_t x)
{
	__asm__("bswap %0" : "=r" (x) : "0" (x));
	return x;
}
#elif defined(__GNUC__) && defined(__x86_64__)
static INLINE uint32_t GIRL_swap32(uint32_t x)
{
	__asm__("bswapl %0" : "=r" (x) : "0" (x));
	return x;
}
#elif defined(__GNUC__) && (defined(__powerpc__) || defined(__ppc__))
static INLINE uint32_t GIRL_swap32(uint32_t x)
{
	uint32_t result;

	__asm__("rlwimi %0,%2,24,16,23" : "=&r" (result) : "0" (x>>24), "r" (x));
	__asm__("rlwimi %0,%2,8,8,15"   : "=&r" (result) : "0" (result),    "r" (x));
	__asm__("rlwimi %0,%2,24,0,7"   : "=&r" (result) : "0" (result),    "r" (x));
	return result;
}
#elif defined(__GNUC__) && (defined(__M68000__) || defined(__M68020__))
static INLINE uint32_t GIRL_swap32(uint32_t x)
{
	__asm__("rorw #8,%0\n\tswap %0\n\trorw #8,%0" : "=d" (x) :  "0" (x) : "cc");
	return x;
}
#else
static INLINE uint32_t GIRL_swap32(uint32_t x) {
	return((x<<24)|((x<<8)&0x00FF0000)|((x>>8)&0x0000FF00)|(x>>24));
}
#endif

#ifdef GIRL_HAS_64BIT_TYPE
#if defined(__GNUC__) && defined(__i386__) && \
   !(__GNUC__ == 2 && __GNUC_MINOR__ <= 95 /* broken gcc version */)
static INLINE Uint64 GIRL_swap64(Uint64 x)
{
	union { 
		struct { uint32_t a, b; } s;
		Uint64 u;
	} v;
	v.u = x;
	__asm__("bswapl %0 ; bswapl %1 ; xchgl %0,%1" 
	        : "=r" (v.s.a), "=r" (v.s.b) 
	        : "0" (v.s.a), "1" (v.s.b)); 
	return v.u;
}
#elif defined(__GNUC__) && defined(__x86_64__)
static INLINE Uint64 GIRL_swap64(Uint64 x)
{
	__asm__("bswapq %0" : "=r" (x) : "0" (x));
	return x;
}
#else
static INLINE Uint64 GIRL_swap64(Uint64 x)
{
	uint32_t hi, lo;

	/* Separate into high and low 32-bit values and swap them */
	lo = (uint32_t)(x&0xFFFFFFFF);
	x >>= 32;
	hi = (uint32_t)(x&0xFFFFFFFF);
	x = GIRL_swap32(lo);
	x <<= 32;
	x |= GIRL_swap32(hi);
	return(x);
}
#endif
#else
/* This is mainly to keep compilers from complaining in GIRL code.
   If there is no real 64-bit datatype, then compilers will complain about
   the fake 64-bit datatype that GIRL provides when it compiles user code.
*/
#define GIRL_swap64(X)	(X)
#endif /* GIRL_HAS_64BIT_TYPE */


/* Byteswap item from the specified endianness to the native endianness */
#if GIRL_BYTEORDER == GIRL_LIL_ENDIAN
#define GIRL_swapLE16(X)	(X)
#define GIRL_swapLE32(X)	(X)
#define GIRL_swapLE64(X)	(X)
#define GIRL_swapBE16(X)	GIRL_swap16(X)
#define GIRL_swapBE32(X)	GIRL_swap32(X)
#define GIRL_swapBE64(X)	GIRL_swap64(X)
#else
#define GIRL_swapLE16(X)	GIRL_swap16(X)
#define GIRL_swapLE32(X)	GIRL_swap32(X)
#define GIRL_swapLE64(X)	GIRL_swap64(X)
#define GIRL_swapBE16(X)	(X)
#define GIRL_swapBE32(X)	(X)
#define GIRL_swapBE64(X)	(X)
#endif

/* Ends C function definitions when using C++ */
#ifdef __cplusplus
}
#endif



#endif /* ! GIRL_ENDIAN_HPP */
