/*
  4-connexes direction.
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlDirection.hpp
 * @brief 4-connexe direction.
 */

#ifndef GIRLDIRECTION_HPP
#define GIRLDIRECTION_HPP


//#include <girl/girlCommon.h>

#include "girlIO.hpp"

namespace girl {

  /**
   * @brief enum of directions.
   */
  typedef enum eDirection 
    { 
      RIGHT = 0, 
      UP = 1, 
      LEFT = 2, 
      DOWN = 3,
      INVALID = -2
    } eDirection;


  /**
   * @class direction girlDirection.hpp girl/girlDirection.hpp
   *
   * @brief 4-connexe direction class.
   */
  class GIRL_API direction
  {
  public:
    /**
     * @brief Constructor.
     */
    //constructor is intentionnaly not explicit
    direction(eDirection d = INVALID)
      : m_d(static_cast<char>(d))
    {}

    /**
     * @brief int conversion operator.
     */
    inline operator int () const
    {
      return static_cast<int>(m_d);
    }


    /**
     * @brief Gives the opposite direction.
     *
     * @warning direction @a d must not be INVALID
     */
    inline girl::direction opposite() const
    {
      assert(static_cast<eDirection>(m_d)!=INVALID);
      return static_cast<eDirection>(static_cast<int>(m_d)^2);
    }
    
    /**
     * @brief Gives previous direction of @a d in positive order.
     *
     * @warning direction @a d must not be INVALID
     */
    inline girl::direction previous() const
    {
      assert(static_cast<eDirection>(m_d)!=INVALID);
      //currently CCW
      return static_cast<eDirection>((static_cast<int>(m_d)+1)&3);
    }
    
    /**
     * @brief Gives next direction of @a d in positive order. 
     *
     * @warning direction @a d must not be INVALID
     */
    inline girl::direction next() const
    {
      assert(static_cast<eDirection>(m_d)!=INVALID);
      //currently CW
      return static_cast<eDirection>((static_cast<int>(m_d)-1)&3);
    }

    /**
     * @brief Tells if direction is vertical, i.e., UP or DOWN.
     */
    inline bool isVertical() const
    {
      assert(static_cast<eDirection>(m_d)!=INVALID);
      
      return ((static_cast<int>(m_d)&1) == 1);
    }

    /**
     * @brief Tells if direction is vertical, i.e., RIGHT or LEFT.
     */
    inline bool isHorizontal() const
    {
      assert(static_cast<eDirection>(m_d)!=INVALID);
      
      return ((static_cast<int>(m_d)&1) == 0);
    }


    /**
     * @brief Write direction @a d with function @a fct
     *
     * @a user_data allows to pass a context to the function.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags /* flags */) const
    {
      if (w.write(&m_d, sizeof(m_d)) != sizeof(m_d))
	return false;
      return true;
    }
    
    /**
     * @brief Read direction @a d with function @a fct
     *
     * @a user_data allows to pass a context to the function.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags /* flags */)
    {
      if (r.read(&m_d, sizeof(m_d)) != sizeof(m_d))
	return false;
      return true;
    }
    
  

  private:
    //girl::eDirection m_d;
    char m_d;
  };
 

}



#endif /* !GIRLDIRECTION_HPP */
