/*
  Freeman code based clockwise oriented closed contours
  Copyright (C) 2004-2009 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlContour.hpp
 * @brief Freeman code based oriented closed contours.
 */

#ifndef GIRLCONTOUR_HPP
#define GIRLCONTOUR_HPP

#include <girl/girlFreemanCodePath.hpp>


namespace girl {

  /**
   * @class contour girlContour.hpp girl/girlContour.hpp
   *
   * @brief Closed contour class.
   *
   * start and end points are the same.
   */
  class GIRL_API contour : public freemanCodePath
  {
  public:
    
    /**
     * @brief Default constructor.
     *
     * Creates an empty contour.
     */
    inline explicit contour(const girl::point &startPt = girl::point(0, 0));

    /**
     * @brief Constructor
     *
     * Construct a freemanCodePath from a starting point @a startPt
     * and a deque of directions @a dirs.
     *
     * @warning directions must describe a closed contour.
     */
    inline contour(const girl::point &startPt,
		   const std::deque<girl::direction> &dirs);

    
    /**
     * @brief Set freemanCodePath from a starting point @a startPt
     * and a deque of directions @a dirs.
     *
     * @warning directions must describe a closed contour.
     */    
    inline void set(const girl::point &startPt,
		    const std::deque<girl::direction> &dirs);
    

    /**
     * @brief Gives starting point of path @a self.
     */
    inline girl::point startPt() const;
    
    /**
     * @brief Gives end point of path @a self.
     *
     * As a contour is closed, it is the same as startPt().
     */
    inline girl::point endPt() const;

    /**
     * @brief Path length, i.e. number of directions.
     *
     * O(1).
     */
    inline size_t length() const;
    
    /**
     * @brief Checks if contour @a self is positively oriented.
     *
     * @return true if contour is positively oriented, false if not.
     * @return true for empty freemanCodePath.
     */
    bool isPositivelyOriented() const;

    /**
     * @brief Inverts orientation of contour.
     * 
     * For example, a contour with four directions Right, Up, Left, and Down,
     * will be changed to  Up, Right, Down, and Left.
     */
    inline void invertOrientation();


    /**
     * @brief Swap two contour.
     *
     * Assign content of this to @a c and content of @a c to this.
     */
    inline void swap(contour &c);

  };

  //inline function --------------------------------------------------

  
  contour::contour(const girl::point &pStartPt)
    : freemanCodePath(pStartPt)
  {
    assert(isValidAndClosed());
  }

  contour::contour(const girl::point &pStartPt,
		   const std::deque<girl::direction> &dirs)
    : freemanCodePath(pStartPt, dirs)
  {
    assert(freemanCodePath::isValidAndClosed());
  }

  void
  contour::set(const girl::point &pStartPt,
	       const std::deque<girl::direction> &dirs)
  {
    freemanCodePath::set(pStartPt, dirs);
    assert(freemanCodePath::isValidAndClosed());
  }

  
  girl::point 
  contour::startPt() const
  {
    return freemanCodePath::startPt();
  }
  
  girl::point 
  contour::endPt() const
  {
    return freemanCodePath::startPt();
  }

  size_t 
  contour::length() const
  {
    return freemanCodePath::length();
  }

  void
  contour::invertOrientation()
  {
    freemanCodePath::invertOrientation();
  }

  void
  contour::swap(contour &p)
  {
    freemanCodePath::swap(p);
  }

  inline
  void
  swap(contour &p1, contour &p2)
  {
    p1.swap(p2);
  }

}

namespace std {

  /**
   * @brief Swap two contours.
   *
   * Assign content of @a c1 to @a c2 and content of @a c2 to @a c1.
   */
  template <>
  inline 
  void
  swap(girl::contour &c1, girl::contour &c2)
  {
    c1.swap(c2);
  }  

}


#endif /* ! GIRLCONTOUR_HPP */
