/*
  Bitmap based geometrical region representation
  Copyright (C) 1992, 1996, 2004-2010 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


/**
 * @file girlBitmap.hpp
 * @brief Bitmap based geometrical region representation.
 */

#ifndef GIRLBITMAP_HPP
#define GIRLBITMAP_HPP

#include <vector>
#include <string>

#include <girl/girlCommon.hpp>
#include <girl/girlPoint.hpp>
#include <girl/girlDirection.hpp>
#include <girl/girlGeoDart.hpp>

namespace girl {

  /**
   * @brief Bitmap class, allows to store geomtric information of a segmentation. 
   *
   */
  class GIRL_API bitmap 
  {

  public:

    /**
     * @brief Constructor.
     *
     * Allocates a new bitmap for the discrete geometric map (DGM).
     * Created bitmap will have a border on its outside boundary,
     * and an artificial node on its outside boundary.
     *
     * @param width   original image width.
     * @param height  original image height.
     *
     * @warning both @a width and @a height must be strictly superior to 0.
     */
    bitmap(POINT_TYPE width=1, POINT_TYPE height=1);


    /**
     * @brief Return bitmap width.
     */
    POINT_TYPE width() const;

    /**
     * @brief Return bitmap height.
     */
    POINT_TYPE height() const;


    /**
     * @brief Reset bitmap : clear it, reset border and add an artificial node.
     *
     * Erases the whole bitmap and reset a border on its outside boundary,
     * and create an artificial node on its border.
     */
    void reset();


    /**
     * @brief Clear the bitmap.
     *
     * It will remove all borders and nodes in the bitmap.
     *
     * @sa clear.
     */
    void clear();

    /**
     * @brief Set a border on the outside border of the bitmap.
     *
     * @sa clear.
     */
    void setBorder();

    /**
     * @brief Set/unset an artificial node on the outside border of the bitmap.
     *
     * @sa clear.
     */
    void setBorderArtificialNode(bool onoff = true);



     /**
     * @brief Checks if given position @a pt is a node in bitmap.
     */
    inline bool isNode(const girl::point &pt) const;

    /**
     * @brief Sets a node at given position @a pt.
     */
    inline void setNode(const girl::point &pt);

    /**
     * @brief Unsets a node at given position @a pt.
     */
    inline void unsetNode(const girl::point &pt);

    /**
     * @brief Clears a segment from the bitmap @a self, starting from geometrical dart @a u.
     * 
     * Does not clear the nodes.
     */
    void clearSegment(const girl::geoDart &u);

    
    /**
     * @brief Sounds the bitmap from left to right starting at point @a pt until 
     * a contour point is found, then sets the point @a contour_pt.
     *
     * @warning point @a pt must be a valid point, in bitmap coordinate system.
     * @warning @a contour_pt is set to (INVALID; INVALID) if no contour point is found.
     */ 
    void sound(const girl::point &pt, girl::point &contour_pt) const;


    /**
     * @brief Tracks a contour until a node is found.
     *
     * The starting point @a pt should be a contour point, certaintly the result of a @a sounding.
     * Returns the geometric dart @a g associated to the region which contour was tracked.
     */
    void contourTracking(const girl::point &pt, girl::geoDart &g) const;



    /**
     * @brief Checks overall bitmap @a self consistency.
     * Will print found errors, if any, to stderr. 
     * @sa check_nodes
     */
    bool checkConsistency() const; 

    /**
     * @brief Checks that outer border of bitmap is marked.
     */
    bool checkBorder() const;

    /**
     * @brief Checks bitmap nodes consistency.
     * Check for each point of the bitmap its node consistency: @a check_node_consistency 
     * @warning this function can be costly.
     */
    bool checkNodes() const;

    /**
     * @brief Checks that no point in bitmap is marked.
     */
    bool checkNoMark() const;

    /**
     * @brief Write bitmap @a self with writer @a w.
     *
     * @return true if write was ok, else false.
     */
    bool write(girl::io::Writer &w, girl::io::writeFlags flags) const;

    /**
     * @brief Read bitmap @a self with reader @a r.
     *
     * @return true if read was ok, else false.
     */
    bool read(girl::io::Reader &r, girl::io::writeFlags flags);

    /**
     *@brief Tells if borders of pixel @a pt are swapable.
     */
    bool swapable(const girl::point &pt) const;

    /**
     * @brief Swap pixel border at position @a pt..
     *
     * return an int with 18 valid bits :
     * - 16 bits for 4 pixel corner (in this order [(x-1;y)|(x-1;y-1)|((x;y-1)|(x;y)] 
     *   - 4 bits (DLUR : Down, Left, Up, Right) for each corner (DR, UP, UL, DL).
     *   - a bit is set if the pt is a node and has a path in this direction.
     * - 2 bits for the number of nodes, i.e. (result>>16) is the number of nodes.
     * So will return 0 if swapped pt has not touched any node.
     * 
     * @warning pixel must swapable.
     * @sa swapable.
     */
    int swap(const girl::point &pt);


    /**
     * @brief Saves bitmap to file @a filename.
     */
    void save(const std::string &filename) const;


    /**
     * @brief Swap two bitmaps.
     *
     * Assign content of this to @a btm and content of @a btm to this.
     *
     * O(1).
     */
    void swap(girl::bitmap &btm);

  public:
    
    /**
     * @brief Const iterator on a bitmap.
     *
     * It allows to traverse and interrogate a bitmap.
     */
    class GIRL_API const_iterator 
    {
    public:
      
      /**
       * @brief Constructor.
       */
      inline const_iterator(const girl::bitmap &b, const girl::point &pt);
      
      /**
       * @brief Copy Constructor.
       */
      inline const_iterator(const const_iterator &it);

      /**
       * @brief Checks if current position is a node.
       */
      inline bool isNode() const;

      /**
       * @brief Checks if current position is marked
       */
      inline bool isMarked() const;

      /**
       * @brief Checks if current position has a neighbour in given direction @a d.
       */
      inline bool hasNeighbourInDirection(girl::direction d) const;

      /**
       * @brief Checks if current position has a neighbour in direction RIGHT.
       */
      inline bool hasRightNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction UP.
       */
      inline bool hasUpNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction DOWN.
       */
      inline bool hasDownNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction LEFT.
       */
      inline bool hasLeftNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction RIGHT or UP.
       */
      inline bool hasRightOrUpNeighbour() const;

      /**
       * @brief Checks if current point is a contour point,
       * i.e., has at least two neighbours !
       */
      inline bool isContourPoint() const;

      /**
       * @brief Checks if current point is on outside border of bitmap,
       * i.e., has getPoisition().x() == -1 or width()-1 
       *     or getPoisition().y() == -1 or height()-1
       */
      inline bool isBorderPoint() const;

       /**
       * @brief Gives cardinality of the current position,
       * that is its number of neighbours (belongs to [0, 4]).
       */
      inline int getPointCardinality() const;

      /**
       * @brief Checks node consistency.
       */
      inline bool checkNodeConsistency() const;

      /**
       * @brief Gives the outcoming direction, given an incoming direction @a prev to the current point. 
       */
      inline girl::direction getNextDirection(girl::direction prev) const;

      /**
       * @brief Gets current position in @a pt.
       */
      inline const girl::point &getPosition() const;

      /**
       * @brief Moves current position in given direction @a move.
       */
      inline void moveInDirection(girl::direction move);

      /**
       * @brief Moves current position in RIGHT direction.
       */
      inline void moveRight();
      
      /**
       * @brief Moves current position in UP direction.
       */
      inline void moveUp();
      
      /**
       * @brief Moves current position in LEFT direction.
       */
      inline void moveLeft();
      
      /**
       * @brief Moves current position in DOWN direction.
       */
      inline void moveDown();
      
      /**
       * @brief Moves current position to point @a pt.
       */
      inline void moveTo(const girl::point &pt);

      /**
       * @brief Moves current position in RIGHT direction.
       */
      inline void moveRightTillUpNeighbour();
      
    protected:
      
      inline void moveRight_aux();
      
    private :
      girl::bitmap::const_iterator &operator=(const const_iterator &it);



    protected:
      
      const char *m_word; //pointer to current pixel (but as there is 2 pixels per char, we need to apply shift to choose the good 4 bits)
      girl::point m_currentPoint;
      int m_shift;
      const girl::bitmap &m_b;

    };

    /**
     * @brief Iterator on a bitmap.
     *
     * It allows to traverse, interrogate and modify a bitmap.
     */
    class GIRL_API iterator 
    {
    public:

      /**
       * @brief Constructor.
       */
      inline iterator(girl::bitmap &b, const girl::point &pt);

      /**
       * @brief Copy constructor.
       */
      inline iterator(const girl::bitmap::iterator &it);

      /**
       * @brief Assignment operator.
       */
      inline bitmap::iterator &operator=(const girl::bitmap::iterator &it);
 
      /**
       * @brief Checks if current position is a node.
       */
      inline bool isNode() const;

      /**
       * @brief Sets a node at current position.
       */
      inline void setNode() const;

      /**
       * @brief Unsets a node at current position.
       */
      inline void unsetNode() const;

      /**
       * @brief Checks if current position is marked
       */
      inline bool isMarked() const;

      /**
       * @brief Sets current position as marked
       */
      inline void mark() const;

      /**
       * @brief Sets current position as not marked
       */
      inline void unmark() const;

      /**
       * @brief Checks if current position has a neighbour in direction RIGHT.
       */      
      inline bool hasRightNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction UP.
       */      
      inline bool hasUpNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction DOWN.
       */      
      inline bool hasDownNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in direction LEFT.
       */      
      inline bool hasLeftNeighbour() const;

      /**
       * @brief Checks if current position has a neighbour in given direction @a d.
       */
      inline bool hasNeighbourInDirection(girl::direction d) const;

      /**
       * @brief Gives cardinality of the current position,
       * that is its number of neighbours (belongs to [0, 4]).
       */
      inline int getPointCardinality() const;

      /**
       * @brief Checks if current point is a contour point,
       * i.e., has at least two neighbours !
       */
      inline bool isContourPoint() const;

      /**
       * @brief Checks if current point is on outside border of bitmap,
       * i.e., has getPoisition().x() == -1 or width()-1 
       *     or getPoisition().y() == -1 or height()-1
       */
      inline bool isBorderPoint() const;


      /**
       * @brief Checks node consistency.
       */
      inline bool checkNodeConsistency() const;

      /**
       * @brief Gives the outcoming direction, given an incoming direction @a prev to the current point. 
       */
      inline girl::direction getNextDirection(girl::direction prev) const;

      /**
       * @brief Switches on or off neighbour of current position, in direction @a next_move, according to @a on.
       *
       * @warning DANGEROUS! Does not ensure consistence!
       */
      inline void writeNeighbourInDirection(girl::direction next_move, bool on) const;

      /**
       * @brief Gets current position in @a pt.
       */
      inline const girl::point &getPosition() const;

      /**
       * @brief Moves current position in given direction @a move.
       */
      inline void moveInDirection(girl::direction move);

      /**
       * @brief Moves current position in RIGHT direction.
       */
      inline void moveRight();
      
      /**
       * @brief Moves current position in UP direction.
       */
      inline void moveUp();
      
      /**
       * @brief Moves current position in LEFT direction.
       */
      inline void moveLeft();
      
      /**
       * @brief Moves current position in DOWN direction.
       */
      inline void moveDown();
      
    protected:

      friend class girl::bitmap;
      
      inline void writeRightNeighbour(bool on) const;

      inline void writeUpNeighbour(bool on) const;

      inline void writeLeftNeighbour(bool on) const;

      inline void writeDownNeighbour(bool on) const;

    protected:

      char *m_word; //pointer to current pixel (but as there is 2 pixels per char, we need to apply shift to choose the good 4 bits)
      girl::point m_currentPoint;
      int m_shift;
      girl::bitmap &m_b;
      
    };

    /**
     * @return a const iterator at point @a pt.
     */
    inline const_iterator getIterator(const girl::point &pt) const;

    /**
     * @return an iterator at point @a pt.
     */
    inline iterator getIterator(const girl::point &pt);

    /**
     * @brief change node at point @a pt.
     *
     * @param on true for setting node, false for unsetting.
     */
    void makeNode(const girl::point &pt, bool on);


  protected:


    enum
    {
      UNLIGHT = 0,
      RIGHT_NEIGHBOUR = 1,
      UP_NEIGHBOUR = 2,
      NODE_CONST = 4,
      MARK_CONST = 8
    };

    enum 
    {
      ODD_SHIFT = 0,
      EVEN_SHIFT = 4
    };

    static int getShift(const girl::point &pt)
    {    
      return (pt.x() & 1) ? ODD_SHIFT : EVEN_SHIFT;
    }

    inline char *getPtr(const girl::point &pt);

    inline const char *getPtr(const girl::point &pt) const;
    
    inline int getMask(const girl::point &pt) const;

    
    bitmap(POINT_TYPE width, POINT_TYPE height, bool);

    friend class togerImpl; //for crop !

    void crop(const girl::point &p1, const girl::point &p2);


  protected:

    std::vector<char> m_bitm;
    POINT_TYPE m_lineSize;
    POINT_TYPE m_dimX;
    POINT_TYPE m_dimY;
    POINT_TYPE m_xMax;
    POINT_TYPE m_yMax;
    

  };


  //--------- inline functions -----------------------


  inline 
  void 
  bitmap::setNode(const girl::point &pt)
  {
    makeNode(pt, true);
  }
  
  inline 
  void 
  bitmap::unsetNode(const girl::point &pt)
  {
    makeNode(pt, false);
  }
  
  inline
  char *
  bitmap::getPtr(const girl::point &pt)
  {
    assert(pt.x()>=-1 && pt.x()<=m_xMax
	   && pt.y()>=-1 && pt.y()<=m_yMax);
    
    return &m_bitm[0] + (m_lineSize*(pt.y() + 1) + ((pt.x() + 1) >> 1));
  }
  
  inline
  const char *
  bitmap::getPtr(const girl::point &pt) const
  {
    assert(pt.x()>=-1 && pt.x()<=m_xMax
	   && pt.y()>=-1 && pt.y()<=m_yMax);
    
    return &m_bitm[0] + (m_lineSize*(pt.y() + 1) + ((pt.x() + 1) >> 1));
  }
  
  inline
  int
  bitmap::getMask(const girl::point &pt) const
  {
    return ((*getPtr(pt) >> getShift(pt)) & 0x0f);
  }
    
  inline
  bool 
  bitmap::isNode(const girl::point &pt) const
  {
    return (getMask(pt) & NODE_CONST) != 0;
  }

  bitmap::const_iterator 
  bitmap::getIterator(const girl::point &pt) const
  {
    return bitmap::const_iterator(*this, pt);
  }

  bitmap::iterator 
  bitmap::getIterator(const girl::point &pt)
  {
    return bitmap::iterator(*this, pt);
  }





  bitmap::const_iterator::const_iterator(const girl::bitmap &b, const girl::point &pt)
    : m_word(b.getPtr(pt)), m_currentPoint(pt), m_shift(bitmap::getShift(pt)), m_b(b)
  {
    //TODO: throw for invalid point !??
  }
 
  bitmap::const_iterator::const_iterator(const const_iterator &it)
    :  m_word(it.m_word), m_currentPoint(it.m_currentPoint), m_shift(it.m_shift), m_b(it.m_b)
  {
  }


  bool 
  bitmap::const_iterator::isNode() const
  {
    return ((*m_word>>m_shift)&NODE_CONST) != 0;
  }

  bool 
  bitmap::const_iterator::isMarked() const
  {
    return ((*m_word>>m_shift)&MARK_CONST) != 0;
  }


  bool 
  bitmap::const_iterator::hasRightNeighbour() const
  {
    return (*m_word>>m_shift)&RIGHT_NEIGHBOUR;
  }

  bool 
  bitmap::const_iterator::hasRightOrUpNeighbour() const
  {
    return ((*m_word>>m_shift)&(UP_NEIGHBOUR|RIGHT_NEIGHBOUR)) != 0;
  }

  bool 
  bitmap::const_iterator::hasDownNeighbour() const
  {
    if (m_currentPoint.y() >= m_b.m_yMax)
      return false;
    return m_b.getIterator(girl::point(m_currentPoint.x(), m_currentPoint.y()+1)).hasUpNeighbour();
  }

  bool 
  bitmap::const_iterator::hasLeftNeighbour() const
  {
    if (m_currentPoint.x() <= -1)
      return false;
    return m_b.getIterator(girl::point(m_currentPoint.x()-1, m_currentPoint.y())).hasRightNeighbour();
  }


  bool 
  bitmap::const_iterator::hasNeighbourInDirection(girl::direction d) const
  {
    switch (d) {
    case RIGHT:
      return hasRightNeighbour();
    case UP:
      return hasUpNeighbour();
    case LEFT:
      return hasLeftNeighbour();
    case DOWN:
      return hasDownNeighbour();
    case INVALID:
    default:
      return false;
    }
  }
  
  bool
  bitmap::const_iterator::hasUpNeighbour() const
  {
    return ((*m_word>>m_shift)&UP_NEIGHBOUR) != 0;
  }

  girl::direction 
  bitmap::const_iterator::getNextDirection(girl::direction prev) const
  {
    assert(prev != INVALID);

    girl::direction nextDir = prev.opposite();

    for (int i=0; i<4; ++i) {
      nextDir = nextDir.next();
      if (hasNeighbourInDirection(nextDir))
	return nextDir;
    }
    return INVALID;
  }

  void
  bitmap::const_iterator::moveRight()
  {
    assert(m_currentPoint.x() < m_b.m_xMax);

    m_currentPoint.setX(m_currentPoint.x()+1);
    if (m_shift == ODD_SHIFT)
      m_shift = EVEN_SHIFT;
    else {
      ++m_word;
      m_shift = ODD_SHIFT;
    }  
  }

  void
  bitmap::const_iterator::moveLeft()
  {
    assert(m_currentPoint.x() > -1);

    m_currentPoint.setX(m_currentPoint.x()-1);
    if (m_shift == ODD_SHIFT) {
      --m_word;
      m_shift = EVEN_SHIFT;
    }
    else
      m_shift = ODD_SHIFT;
  }

  void
  bitmap::const_iterator::moveUp()
  {
    assert(m_currentPoint.y() > -1);

    m_currentPoint.setY(m_currentPoint.y()-1);
    m_word -= m_b.m_lineSize;
  }

  void
  bitmap::const_iterator::moveDown()
  {
    assert(m_currentPoint.y() < m_b.m_yMax);

    m_currentPoint.setY(m_currentPoint.y()+1);
    m_word += m_b.m_lineSize;
  }


  void 
  bitmap::const_iterator::moveRightTillUpNeighbour()
  {
#if 0

    do 
      moveRight();
    while (! hasUpNeighbour());

#else
    //move word by word whenever possible

    if (m_shift == ODD_SHIFT) {
      m_shift = EVEN_SHIFT;
      m_currentPoint.setX(m_currentPoint.x()+1);
      if (hasUpNeighbour())
	return;
    }
    if (m_shift == EVEN_SHIFT) {
      m_shift = ODD_SHIFT;
      ++m_word;
      m_currentPoint.setX(m_currentPoint.x()+1);
    }
    
    static const char UP2 = ((UP_NEIGHBOUR<<ODD_SHIFT) | (UP_NEIGHBOUR<<EVEN_SHIFT));
    while (! (*m_word & UP2)) {
      ++m_word;
      m_currentPoint.setX(m_currentPoint.x()+2);
    }
    while (! hasUpNeighbour()) {
      moveRight();
    }

#endif
  }

  void
  bitmap::const_iterator::moveInDirection(girl::direction move)
  {
    switch (move)
      {
      case RIGHT:
	moveRight();
	break;
      case UP:
	moveUp();
	break;
      case LEFT:
	moveLeft();
	break;
      case DOWN:
	moveDown();
	break;
      case INVALID:
      default:
	assert(false);
	break;
      }  
  }

  bool 
  bitmap::const_iterator::isContourPoint() const
  {
    return (getPointCardinality() >= 2);
  }

  bool 
  bitmap::const_iterator::isBorderPoint() const
  {
    return (m_currentPoint.x() == -1 || m_currentPoint.x() == m_b.m_xMax
	    || m_currentPoint.y() == -1 || m_currentPoint.y() == m_b.m_yMax);
  }

  const girl::point &
  bitmap::const_iterator::getPosition() const
  {
    return m_currentPoint;
  }




  bitmap::iterator::iterator(girl::bitmap &b, const girl::point &pt)
    : m_word(b.getPtr(pt)), m_currentPoint(pt), m_shift(bitmap::getShift(pt)), m_b(b)
  {
  }

  bitmap::iterator::iterator(const girl::bitmap::iterator &it)
    : m_word(it.m_word), m_currentPoint(it.m_currentPoint), m_shift(it.m_shift), m_b(it.m_b)
  {
  }

  bitmap::iterator &
  bitmap::iterator::operator=(const girl::bitmap::iterator &it)
  {
    m_word = it.m_word;
    m_currentPoint = it.m_currentPoint;
    m_shift = it.m_shift;
    m_b = it.m_b;
    return *this;
  }


  const girl::point &
  bitmap::iterator::getPosition() const
  {
    return m_currentPoint;
  }
  
  bool
  bitmap::iterator::isNode() const
  {
    return ((*m_word>>m_shift)&NODE_CONST) != 0;
  }

  void
  bitmap::iterator::setNode() const
  {
    *m_word |= (NODE_CONST<<m_shift);
  }

  void
  bitmap::iterator::unsetNode() const
  {
    *m_word &= ~(NODE_CONST<<m_shift);
  }

  bool 
  bitmap::iterator::isMarked() const
  {
    return ((*m_word>>m_shift)&MARK_CONST) != 0;
  }

  void 
  bitmap::iterator::mark() const
  {
    *m_word |= (MARK_CONST<<m_shift);
  }

  void 
  bitmap::iterator::unmark() const
  {
    *m_word &= ~(MARK_CONST<<m_shift);
  }

  bool 
  bitmap::iterator::isContourPoint() const
  {
    return (getPointCardinality() >= 2);
  }

  bool 
  bitmap::iterator::isBorderPoint() const
  {
    return (m_currentPoint.x() == -1 || m_currentPoint.x() == m_b.m_xMax
	    || m_currentPoint.y() == -1 || m_currentPoint.y() == m_b.m_yMax);
  }

  bool
  bitmap::iterator::hasUpNeighbour() const
  {
    return ((*m_word>>m_shift)&UP_NEIGHBOUR) != 0;
  }

  bool 
  bitmap::iterator::hasRightNeighbour() const
  {
    return (*m_word>>m_shift)&RIGHT_NEIGHBOUR;
  }

  bool 
  bitmap::iterator::hasDownNeighbour() const
  {
    if (m_currentPoint.y() >= m_b.m_yMax)
      return false;
    return m_b.getIterator(girl::point(m_currentPoint.x(), m_currentPoint.y()+1)).hasUpNeighbour();
  }

  bool 
  bitmap::iterator::hasLeftNeighbour() const
  {
    if (m_currentPoint.x() <= -1)
      return false;
    return m_b.getIterator(girl::point(m_currentPoint.x()-1, m_currentPoint.y())).hasRightNeighbour();
  }

  int 
  bitmap::const_iterator::getPointCardinality() const
  {
    int cardinality = 0;
    cardinality += hasUpNeighbour();
    cardinality += hasDownNeighbour();
    cardinality += hasLeftNeighbour();
    cardinality += hasRightNeighbour();
    return cardinality;
  }

  bool
  bitmap::const_iterator::checkNodeConsistency() const
  {
    const int c = getPointCardinality();
    if ((c>=3 && !isNode()) || (isNode() && c<2) || (c==1)) 
      return false;
    return true;
  }

  void
  bitmap::const_iterator::moveTo(const girl::point &pt)
  {
    assert(pt.x()>=-1 && pt.x()<=m_b.m_xMax
	   && pt.y()>=-1 && pt.y()<=m_b.m_yMax);

    m_word = m_b.getPtr(pt);
    m_currentPoint = pt;
    m_shift = bitmap::getShift(pt);
  }


  bool
  bitmap::iterator::hasNeighbourInDirection(girl::direction d) const
  {
    switch (d) {
    case RIGHT:
      return hasRightNeighbour();
    case UP:
      return hasUpNeighbour();
    case LEFT:
      return hasLeftNeighbour();
    case DOWN:
      return hasDownNeighbour();
    case INVALID:
    default:
      return false;
    }
  }


  int
  bitmap::iterator::getPointCardinality() const
  {
    return const_iterator(m_b, m_currentPoint).getPointCardinality();
  }

  bool
  bitmap::iterator::checkNodeConsistency() const
  {
    return const_iterator(m_b, m_currentPoint).checkNodeConsistency();
  }

  girl::direction
  bitmap::iterator::getNextDirection(girl::direction prev) const
  {
    return const_iterator(m_b, m_currentPoint).getNextDirection(prev);
  }

  void 
  bitmap::iterator::writeNeighbourInDirection(girl::direction next_move, bool on) const
  {
    switch (next_move)
      {
      case RIGHT:
	writeRightNeighbour(on);
	break;
      case UP:
	writeUpNeighbour(on);
	break;
      case LEFT:
	writeLeftNeighbour(on);
	break;
      case DOWN:
	writeDownNeighbour(on);
	break;
      case INVALID:
      default:
	break;
      }
  }

  void
  bitmap::iterator::writeRightNeighbour(bool on) const
  {
    if (m_currentPoint.x() > m_b.m_xMax)
      return;
    if (on)
      *m_word |= (RIGHT_NEIGHBOUR<<m_shift);
    else 
      *m_word &= ~(RIGHT_NEIGHBOUR<<m_shift);
  }

  void
  bitmap::iterator::writeUpNeighbour(bool on) const
  {
    if (m_currentPoint.y() < -1)
      return;
    if (on)
      *m_word |= (UP_NEIGHBOUR<<m_shift);
    else
      *m_word &= ~(UP_NEIGHBOUR<<m_shift);
  }


  void
  bitmap::iterator::writeLeftNeighbour(bool on) const
  {
    //writing left neighbour of (x;y)
    // is writing right neighbour of (x-1;y)

    if (m_currentPoint.x() <= -1)
      return;
    if (m_shift == ODD_SHIFT) { //odd point
      if (on)
	*(m_word - 1) |= RIGHT_NEIGHBOUR << 4;
      else
	*(m_word - 1) &= ~(RIGHT_NEIGHBOUR << 4);
    }
    else {
      if (on)
	*m_word |= RIGHT_NEIGHBOUR;
      else
	*m_word &= ~RIGHT_NEIGHBOUR;
    }    
  }

  void
  bitmap::iterator::writeDownNeighbour(bool on) const
  {
    //writing down neighbour of (x;y)
    // is writing up neighbour of (x;y+1)
    if (m_currentPoint.y() >= m_b.m_yMax)
      return;

    if (on)
      *(m_word + m_b.m_lineSize) |= (UP_NEIGHBOUR<<m_shift);
    else
      *(m_word + m_b.m_lineSize) &= ~(UP_NEIGHBOUR<<m_shift);
  }

  void
  bitmap::iterator::moveInDirection(girl::direction move)
  {
    //code duplication with const_iterator

    switch (move)
      {
      case RIGHT:
	moveRight();
	break;
      case UP:
	moveUp();
	break;
      case LEFT:
	moveLeft();
	break;
      case DOWN:
	moveDown();
	break;
      case INVALID:
      default:
	break;
      }    
  }

  void
  bitmap::iterator::moveRight()
  {
    //code duplication with const_iterator

    if (m_currentPoint.x() == m_b.m_xMax)
      return;

    m_currentPoint.setX(m_currentPoint.x()+1);
    if (m_shift == ODD_SHIFT)
      m_shift = EVEN_SHIFT;
    else {
      ++m_word;
      m_shift = ODD_SHIFT;
    }
  }

  void
  bitmap::iterator::moveLeft()
  {
    //code duplication with const_iterator

    if (m_currentPoint.x() == -1)
      return;
    m_currentPoint.setX(m_currentPoint.x()-1);
    if (m_shift == ODD_SHIFT) {
      --m_word;
      m_shift = EVEN_SHIFT;
    }
    else
      m_shift = ODD_SHIFT;
  }

  void
  bitmap::iterator::moveUp()
  {
    //code duplication with const_iterator

    if (m_currentPoint.y() == -1)
      return;
    m_currentPoint.setY(m_currentPoint.y()-1);
    m_word -= m_b.m_lineSize;
  }

  void
  bitmap::iterator::moveDown()
  {
    //code duplication with const_iterator

    if (m_currentPoint.y() == m_b.m_yMax)
      return;
    m_currentPoint.setY(m_currentPoint.y()+1);
    m_word += m_b.m_lineSize;
  }

  inline
  void
  swap(bitmap &b1, bitmap &b2)
  {
    b1.swap(b2);
  }

}

namespace std {

  /**
   * @brief Swap two bitmaps.
   *
   * Assign content of @a b1 to @a b2 and content of @a b2 to @ b1.
   *
   * O(1).
   */
  template <>
  inline 
  void
  swap(girl::bitmap &b1, girl::bitmap &b2)
  {
    b1.swap(b2);
  }  

}


#endif /* GIRLBITMAP_HPP */
