/*
  Bitmap based geometrical region representation
  Copyright (C) 1992, 1996, 2004-2010 Achille Braquelaire, Jean-Philippe Domenger, Gabriel de Dietrich, Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "girlBitmap.hpp"

#include <cassert>
#include <cstring> //memset
#include <iostream>
#include <fstream>
#include <vector>
#include "girlAssert.hpp"
#include "girlEndian.hpp"
#include "girlToger.hpp"

using namespace girl;


/*
  #include <stdint.h> //for UINTPTR_MAX   ??? limits ???
*/

/**
 * Pixels are coded on 4 bits. Thus, 2 pixels per char.
 * If x is odd, we have [(x+1)|(x)].
 * If x is even, we have [(x)|(x+1)].
 * A line of girl::bitmap.m_bitm starts like this :
 * [(0)|(-1)][(2)|(1)][(4)|(3)][(6)|(5)]...
 */

/*
 * girl::bitmap::iterator.m_word points to the char of the current position,
 * girl::bitmap::iterator.m_shift is the shift to reach the right pixel in the char.
 * girl::bitmap::iterator.m_shift is 0 for x odd, 4 for x even.
 */

/*
enum
  {
    UNLIGHT = 0,
    RIGHT_NEIGHBOUR = 1,
    UP_NEIGHBOUR = 2,
    NODE_CONST = 4,
    MARK_CONST = 8
  };

static const int ODD_SHIFT = 0;
static const int EVEN_SHIFT = 4;
*/

// #define IS_NODE(val)     ((val) & NODE_CONST)
// #define IS_UNLIGHT(val)  ((val) == UNLIGHT)
// #define IS_INFINITE(val) ((val) == INFINITE)

// /* #define HAS_RIGHT_NEIGHBOUR(val) ( ((val) & RIGHT_NEIGHBOUR) != 0) */
// /* #define HAS_UP_NEIGHBOUR(val)    ( ((val) & UP_NEIGHBOUR)    != 0) */
// #define HAS_RIGHT_NEIGHBOUR(val) ( ((val) & RIGHT_NEIGHBOUR))
// #define HAS_UP_NEIGHBOUR(val)    ( ((val) & UP_NEIGHBOUR))


girl::bitmap::bitmap(POINT_TYPE dimx, POINT_TYPE dimy)
  : m_bitm(), 
    m_lineSize((dimx+2)/2),
    m_dimX(dimx+1), m_dimY(dimy+1),
    m_xMax(dimx-1), m_yMax(dimy-1)    
{
  assert(dimx > 0);
  assert(dimy > 0);

  reset();
}

/*
 Constructor used internally only [third parameter not used]
 Does not initialize bitmap data, only dimensions.
 */
girl::bitmap::bitmap(POINT_TYPE dimx, POINT_TYPE dimy, bool /* unused */)
  : m_bitm(), 
    m_lineSize((dimx+2)/2),
    m_dimX(dimx+1), m_dimY(dimy+1),
    m_xMax(dimx-1), m_yMax(dimy-1)    
{
  assert(dimx > 0);
  assert(dimy > 0);
  
  clear(); //resize
}


girl::POINT_TYPE
girl::bitmap::width() const
{
  return m_dimX-1;
}

girl::POINT_TYPE
girl::bitmap::height() const
{
  return m_dimY-1;
}


void
girl::bitmap::clear()
{
  //TODO: check memset to 0 is done !?

  m_bitm.resize(0);
  m_bitm.resize(m_lineSize*m_dimY, 0);
}

/**
 * @brief Clears the bitmap, sets the image border path and add an artifical node.
 */
void
girl::bitmap::reset()
{
  clear();

  setBorder();

  setBorderArtificialNode();
}

void
girl::bitmap::setBorder()
{
  char *b = &m_bitm[0];
  const char c_R_R  = (RIGHT_NEIGHBOUR << 4) | RIGHT_NEIGHBOUR;

  //Upper and lower row: 
  // all pixels have a right neighbour, except on the last column.
  //Left and right columns:
  // all pixels have an up neighbour, except on the first row.

  
  //Upper and lower row: set byte per byte, and 
  //         will correct last column afterwards.

  const size_t s = (m_lineSize - 1) * sizeof(char); //size of one row except last char.


  //- upper row
  memset(b, c_R_R, s); 

  //- upper right corner
  if ((m_dimX & 1) == 0) 
    //m_dimX is even, width is odd : upper right corner must be corrected
    b[m_lineSize-1] = RIGHT_NEIGHBOUR;
  
  
  //- left and right columns
  if (m_dimY > 2) { //height >= 2 i.e there are left and right columns.
    const char c_lU = UP_NEIGHBOUR;
    char c_rU = UP_NEIGHBOUR << 4;
    if (m_dimX & 1) 
      //m_dimX is odd, width is even : UP for right column must be shifted
      c_rU = UP_NEIGHBOUR;
    
    char *lb = &b[m_lineSize]; //on first column
    *lb = c_lU;
    lb += m_lineSize-1;
    for (int i = 2; i < m_dimY - 1; ++i) { 
      *lb |= c_rU; // "|=" instead of "=" for the case m_lineSize==1
      lb[1] = c_lU;
      lb += m_lineSize;
    }
    *lb |= c_rU; // "|=" instead of "=" for the case m_lineSize==1
  }

  //- lower-left corner, lower row, lower right corner 
  if (m_lineSize > 1) {
    //- lower-left corner
    b[m_lineSize * (m_dimY - 1)] = (RIGHT_NEIGHBOUR << 4) | UP_NEIGHBOUR | RIGHT_NEIGHBOUR;
    //- lower row (except first char)
    memset(b + m_lineSize * (m_dimY - 1) + 1, c_R_R, s-1);
    //- lower right corner 
    const int lower_right_corner_index = m_lineSize * m_dimY - 1;
    if (m_dimX & 1)
      b[lower_right_corner_index] = UP_NEIGHBOUR;
    else
      b[lower_right_corner_index] = 
	(UP_NEIGHBOUR << 4) | RIGHT_NEIGHBOUR;
  }
  else {
    //- lower-left corner and lower right corner 
    b[m_lineSize * (m_dimY - 1)] = (UP_NEIGHBOUR << 4) | UP_NEIGHBOUR | RIGHT_NEIGHBOUR;
  }

  assert(checkBorder());
}


void
girl::bitmap::setBorderArtificialNode(bool onoff)
{
  //Set fictitious node at arbitrary position
  //girl::point p(-1, -1);
  girl::point p(m_xMax, -1);
  makeNode(p, onoff);
}



/**
 * @brief Sets or unsets a node according to @a on at position @a pt.
 */
void 
girl::bitmap::makeNode(const girl::point &pt, bool on)
{
  assert(pt.x()>=-1 && pt.x()<=m_xMax
	 && pt.y()>=-1 && pt.y()<=m_yMax);

  const int val = NODE_CONST << getShift(pt);
  char *word = getPtr(pt);

  if (on)
    *word |= val;
  else
    *word &= ~val;
}


void 
bitmap::clearSegment(const girl::geoDart &u)
{
  assert(u.point().x()>=-1 && u.point().x()<=m_xMax
	 && u.point().y()>=-1 && u.point().y()<=m_yMax);

  bitmap::iterator it = getIterator(u.point());
  girl::direction od = u.dir();
  it.moveInDirection(od);

  while (! it.isNode()) {
    const girl::direction d = it.getNextDirection(od);
    it.writeNeighbourInDirection(od.opposite(), false);
    it.moveInDirection(d);
    od = d;
  }
  it.writeNeighbourInDirection(od.opposite(), false);
}


void 
bitmap::sound(const girl::point &pt,
	      girl::point &contour_pt) const
{
  assert(pt.x()>=-1 && pt.x()<=m_xMax
	 && pt.y()>=-1 && pt.y()<=m_yMax);

  bitmap::const_iterator it = getIterator(pt);
  if (! it.hasDownNeighbour() && ! it.hasLeftNeighbour())
    while (! it.hasRightOrUpNeighbour())
      it.moveRight();
  contour_pt = it.getPosition();
}

/*
  We track the contour keeping the region on the left until a node is found.
  Then we revert the direction and we obtain the right geometrical dart with
  \a m and \a node_p. (Now the region is on the right as it should be).
*/

void 
bitmap::contourTracking(const girl::point &pt, girl::geoDart &g) const
{
  assert(pt.x()>=-1 && pt.x()<=m_xMax
	 && pt.y()>=-1 && pt.y()<=m_yMax);
  
  bitmap::const_iterator it = getIterator(pt);
  assert(it.isContourPoint());

  //first initialize new_mov with a direction 
  // that could be the direction used to come to the current point
  // (tracking a contour)

  girl::direction new_mov = INVALID;
  if (it.hasLeftNeighbour())
    new_mov = RIGHT;
  else {
    assert(it.hasRightNeighbour() || it.hasUpNeighbour());
    //As it is a contour point, it has at least two neighbours.
    //As it has no left neighbour, it necessarily has a right or up neighbour
    
    if (it.hasDownNeighbour())
      new_mov = UP;
    else
      new_mov = LEFT; //??? 
  }

  assert(new_mov != INVALID);
  while (! it.isNode()) {
    new_mov = it.getNextDirection(new_mov);
    assert(new_mov != INVALID);
    it.moveInDirection(new_mov);
  }
  
  const girl::point npt = it.getPosition();
  g.set(npt.x(), npt.y(), new_mov.opposite());
}

bool 
bitmap::checkConsistency() const
{
  bool consistent = true;

  //check points and nodes cardinality
  for (int y=-1; y <= m_yMax ; ++y)
    for (int x=-1; x <= m_xMax ; ++x)
      {
	girl::bitmap::const_iterator it = getIterator(girl::point(x, y));
	const int c = it.getPointCardinality();
	if (c>=3 && !it.isNode())
	  {
	    std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  point("<<x<<"; "<<y<<") has a cardinality of "<<c<<" and is not a node!\n";
	    if (it.hasRightNeighbour())
	      std::cerr<<"RIGHT ";
	    if (it.hasUpNeighbour())
	      std::cerr<<"UP ";
	    if (it.hasLeftNeighbour())
	      std::cerr<<"LEFT ";
	    if (it.hasDownNeighbour())
	      std::cerr<<"DOWN ";
	    std::cerr<<std::endl;

	    consistent = false;
	  }
	else if (it.isNode() && c<2)
	  {
	    std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  point("<<x<<"; "<<y<<") is a node and has a cardinality of "<<c<<"!\n";
	    if (it.hasRightNeighbour())
	      std::cerr<<"RIGHT ";
	    if (it.hasUpNeighbour())
	      std::cerr<<"UP ";
	    if (it.hasLeftNeighbour())
	      std::cerr<<"LEFT ";
	    if (it.hasDownNeighbour())
	      std::cerr<<"DOWN ";
	    std::cerr<<std::endl;

	    consistent = false;
	  }
      }
  
  //check that border is marked

  //Up border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, -1));
    for (int x=-1; x < m_xMax ; ++x) {
      if (! it.hasRightNeighbour()) {
	std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  border point("<<x<<"; "<<-1<<") has no right neighbour !"<<std::endl;
	consistent = false;
      }
      it.moveRight();
    }
  }

  //Left border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, -1));
    for (int y=-1; y < m_yMax ; ++y) {
      if (! it.hasDownNeighbour()) {
	std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  border point("<<-1<<"; "<<y<<") has no down neighbour !"<<std::endl;
	consistent = false;
      }
      it.moveDown();
    }
  }

  //Right border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(m_xMax, -1));
    for (int y=-1; y < m_yMax ; ++y) {
      if (! it.hasDownNeighbour()) {
	std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  border point("<<m_xMax<<"; "<<y<<") has no down neighbour !"<<std::endl;
	consistent = false;
      }
      it.moveDown();
    }
  }

  //Down border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, m_yMax));
    for (int x=-1; x < m_xMax ; ++x) {
      if (! it.hasRightNeighbour()) {
	std::cerr<<"girl::bitmap("<<width()<<", "<<height()<<")  border point("<<x<<"; "<<m_yMax<<") has no right neighbour !"<<std::endl;
	consistent = false;
      }
      it.moveRight();
    }
  }



  return consistent;
}


bool
bitmap::checkBorder() const
{

  //Up border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, -1));
    for (int x=-1; x < m_xMax ; ++x) {
      if (! it.hasRightNeighbour()) {
	std::cerr<<"Up border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should have a right neighbour !"<<std::endl;
	return false;
      }
      if (it.hasUpNeighbour()) {
	std::cerr<<"Up border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a up neighbour !"<<std::endl;
	return false;
      }
      it.moveRight();
    }

    if (it.hasUpNeighbour()) {
      std::cerr<<"Up border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a up neighbour !"<<std::endl;
      return false;
    }
    //right neighbour will be tested in "right border" test.
  }

  //Left border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, -1));
    for (int y=-1; y < m_yMax ; ++y) {
      if (! it.hasDownNeighbour()) {
	std::cerr<<"Left border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should have a down neighbour !"<<std::endl;
	return false;
      }
      if (it.hasLeftNeighbour()) {
	std::cerr<<"Left border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a left neighbour !"<<std::endl;
	return false;
      }
      it.moveDown();
    }

    if (it.hasLeftNeighbour()) {
      std::cerr<<"Left border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a left neighbour !"<<std::endl;
      return false;
    }
 
  }

  //Right border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(m_xMax, -1));
    for (int y=-1; y < m_yMax ; ++y) {
      if (! it.hasDownNeighbour()) {
	return false;
      }
      if (it.hasRightNeighbour()) {
	std::cerr<<"Right border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a right neighbour !"<<std::endl;
	return false;
      }
      it.moveDown();
    }

    if (it.hasRightNeighbour()) {
      std::cerr<<"Right border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a right neighbour !"<<std::endl;
      return false;
    }
    
  }

  //Down border
  {
    girl::bitmap::const_iterator it = getIterator(girl::point(-1, m_yMax));
    for (int x=-1; x < m_xMax ; ++x) {
      if (! it.hasRightNeighbour()) {
	return false;
      }
      if (it.hasDownNeighbour()) {
	std::cerr<<"Down border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a down neighbour !"<<std::endl;
	return false;
      }
       it.moveRight();
    }
    if (it.hasDownNeighbour()) {
      std::cerr<<"Down border: ("<<it.getPosition().x()<<"; "<<it.getPosition().y()<<") should not have a down neighbour !"<<std::endl;
      return false;
    }
  }

  return true;
}

bool 
bitmap::checkNodes() const
{
  for (int y=-1; y <= m_yMax ; ++y)
    for (int x=-1; x <= m_xMax ; ++x) {
      girl::bitmap::const_iterator it = getIterator(girl::point(x, y));
      if (! it.checkNodeConsistency()) {
	return false;
      }
    }
  return true;
}

bool 
bitmap::checkNoMark() const
{
  for (int y=-1; y <= m_yMax ; ++y)
    for (int x=-1; x <= m_xMax ; ++x) {
      girl::bitmap::const_iterator it = getIterator(girl::point(x, y));
      if (it.isMarked()) {
	return false;
      }
    }
  return true;
}


bool
bitmap::write(girl::io::Writer &w, girl::io::writeFlags /* flags */) const
{
  //all is saved in little endian

  assert(sizeof(girl::POINT_TYPE) == 2);
  POINT_TYPE dimX = GIRL_swapLE16(m_dimX);
  POINT_TYPE dimY = GIRL_swapLE16(m_dimY);
  POINT_TYPE lineSize = GIRL_swapLE16(m_lineSize);
  if (w.write(&(dimX), sizeof(dimX)) != sizeof(dimX))
    return false;
  if (w.write(&(dimY), sizeof(dimY)) != sizeof(dimY))
    return false;
  if (w.write(&(lineSize), sizeof(lineSize)) != sizeof(lineSize))
    return false;
  
  const size_t bitmapSize = m_lineSize * m_dimY * sizeof(char);

  if (w.write(&m_bitm[0], bitmapSize) != bitmapSize)
    return false;

  return true;
}

bool
bitmap::read(girl::io::Reader &r, girl::io::writeFlags /* flags */)
{
  //all is saved in little endian

  //const size_t oldBitmapSize = m_lineSize * m_dimY * sizeof(char);
  
  POINT_TYPE lineSize;
  POINT_TYPE dimX;
  POINT_TYPE dimY;
  POINT_TYPE xMax;
  POINT_TYPE yMax;

  if (r.read(&(dimX), sizeof(dimX)) != sizeof(dimX))
    return false;
  if (r.read(&(dimY), sizeof(dimY)) != sizeof(dimY))
    return false;
  if (r.read(&(lineSize), sizeof(lineSize)) != sizeof(lineSize))
    return false;
  assert(sizeof(girl::POINT_TYPE) == 2);
  dimX = GIRL_swapLE16(dimX);
  dimY = GIRL_swapLE16(dimY);
  lineSize = GIRL_swapLE16(lineSize);
  assert(lineSize>-1);
  assert(dimX>0);
  assert(dimY>0);
  
  xMax = dimX-2;
  yMax = dimY-2;
  assert(xMax>-1);
  assert(yMax>-1);
  
  const size_t newBitmapSize = lineSize * dimY;
  std::vector<char> bitm(newBitmapSize);
  
  assert(sizeof(char)==1);
  if (r.read(&bitm[0], newBitmapSize) != newBitmapSize)
    return false;

  m_bitm.swap(bitm);
  m_lineSize = lineSize;
  m_dimX = dimX;
  m_dimY = dimY;
  m_xMax = xMax;
  m_yMax = yMax;
  
  return true;
}

bool 
bitmap::swapable(const girl::point &pt) const
{
  if (pt.x() < 0 || pt.y() < 0 ||
      pt.x() > m_xMax || pt.y() > m_yMax)
    return false;

  assert(sizeof(int)>=4 && sizeof(char)==1);
 

  //- First, we get all borders information in local variables 


  //As each point of the bitmap store only UP and RIGHT,
  // for pixel (x, y), 
  //we have to consider 3 lines (y-1, y, y+1)
  //for 3 positions (x-2, x-1, x)

  bool xm2_ym1_R = false;
  bool xm2_y_R = false;
  if (pt.x() > 0) {
    xm2_ym1_R = getIterator(point(pt.x()-2, pt.y()-1)).hasRightNeighbour();
    xm2_y_R = getIterator(point(pt.x()-2, pt.y())).hasRightNeighbour();
  }

  bool xm1_yp1_U = false;
  bool x_yp1_U = false;
  if (pt.y() < m_yMax) {
    xm1_yp1_U = getIterator(point(pt.x()-1, pt.y()+1)).hasUpNeighbour();
    x_yp1_U = getIterator(point(pt.x(), pt.y()+1)).hasUpNeighbour();
  }

  const bitmap::const_iterator it_xm1_ym1 = getIterator(point(pt.x()-1, pt.y()-1));
  const bool xm1_ym1_U = it_xm1_ym1.hasUpNeighbour();
  const bool xm1_ym1_R = it_xm1_ym1.hasRightNeighbour();
  const bool xm1_ym1_N = it_xm1_ym1.isNode();
  const bitmap::const_iterator it_xm1_y   = getIterator(point(pt.x()-1, pt.y()));
  const bool xm1_y_U   = it_xm1_y.hasUpNeighbour();
  const bool xm1_y_R   = it_xm1_y.hasRightNeighbour();
  const bool xm1_y_N   = it_xm1_y.isNode();
  const bitmap::const_iterator it_x_ym1   = getIterator(point(pt.x(), pt.y()-1));
  const bool x_ym1_U   = it_x_ym1.hasUpNeighbour();
  const bool x_ym1_R   = it_x_ym1.hasRightNeighbour();
  const bool x_ym1_N   = it_x_ym1.isNode();
  const bitmap::const_iterator it_x_y     = getIterator(point(pt.x(), pt.y()));
  const bool x_y_U     = it_x_y.hasUpNeighbour();
  const bool x_y_R     = it_x_y.hasRightNeighbour();
  const bool x_y_N     = it_x_y.isNode();


  //- Check obvious cases : no borders, 4 borders, 2 non consecutives borders around pixel.

  const int borders = (static_cast<int>(x_y_U)
		       | (static_cast<int>(xm1_ym1_R) << 1)
		       | (static_cast<int>(xm1_y_U)   << 2)
		       | (static_cast<int>(xm1_y_R )  << 3));
  if (borders == 0
      || borders == 15 //4 borders
      || borders == 5 || borders == 10 //2 non consecutives borders
      ) {
#ifdef VERBOSE
    printf("  no borders (%d), 4 borders (%d), or 2 non-consecutives borders (%d)\n", (borders == 0), (borders == 15), (borders == 5 || borders == 10));
#endif
    return false;
  }

  //- Check other cases at each corner
  
  int inCard; //cardinality of bitmap point regarding pixel borders only
  int totalCard; //total cardinality of bitmap point
  
  //(x; y)
  inCard = xm1_y_R + x_y_U;
  totalCard = inCard + x_y_R + x_yp1_U;
  if (inCard == 0 && totalCard != 0) { //point not linked to a border with a non null cardinality
#ifdef VERBOSE
    std::cout<<"  isolated point\n";
#endif
    return false; 
  }
  if (x_y_N) {
    if (inCard == 2 && totalCard >= 3) { //node with cardinality >= 3 with 2 assured by pixel borders
#ifdef VERBOSE
      std::cout<<"  node with cardinality 3 not movable\n";
#endif
      return false;
    }
#ifdef VERBOSE
    else {
      std::cout<<" arbitrary node\n";
    }
#endif
  }
  
  //(x; y-1)
  inCard = xm1_ym1_R + x_y_U;
  totalCard = inCard + x_ym1_R + x_ym1_U; 
  if (inCard == 0 && totalCard != 0) { //point not linked to a border with a non null cardinality
#ifdef VERBOSE
    std::cout<<"  isolated point\n";
#endif
    return false; 
  }
  if (x_ym1_N) {
    if (inCard == 2 && totalCard >= 3) { //node with cardinality >= 3 with 2 assured by pixel borders
#ifdef VERBOSE
      std::cout<<"  node with cardinality 3 not movable\n";
#endif
      return false;
    }
#ifdef VERBOSE
    else {
      std::cout<<" arbitrary node\n";
    }
#endif
  }
  
  //(x-1; y-1)
  inCard = xm1_ym1_R + xm1_y_U;
  totalCard = inCard + xm2_ym1_R + xm1_ym1_U; 
  if (inCard == 0 && totalCard != 0) { //point not linked to a border with a non null cardinality
#ifdef VERBOSE
    std::cout<<"  isolated point\n";
#endif
    return false; 
  }
  if (xm1_ym1_N) {
    if (inCard == 2 && totalCard >= 3) { //node with cardinality >= 3 with 2 assured by pixel borders
#ifdef VERBOSE
      std::cout<<"  node with cardinality 3 not movable\n";
#endif
      return false;
    }
#ifdef VERBOSE
    else {
      std::cout<<" arbitrary node\n";
    }
#endif
  }
  
  //(x-1; y)
  inCard = xm1_y_U + xm1_y_R;
  totalCard = inCard + xm2_y_R + xm1_yp1_U;
  if (inCard == 0 && totalCard != 0) { //point not linked to a border with a non null cardinality
#ifdef VERBOSE
    std::cout<<"  isolated point\n";
#endif
    return false; 
  }
  if (xm1_y_N) {
    if (inCard == 2 && totalCard >= 3) { //node with cardinality >= 3 with 2 assured by pixel borders
#ifdef VERBOSE
      std::cout<<"  node with cardinality 3 not movable\n";
#endif
      return false;
    }
#ifdef VERBOSE
    else {
      std::cout<<" arbitrary node\n";
    }
#endif
  }
  
  
  return true;
}

int 
bitmap::swap(const girl::point &pt)
{
  assert(swapable(pt));

#ifdef VERBOSE
  std::cout<<" swap point (%d; %d)\n", p.x(), p.y();
#endif

  bool xm2_ym1_R = false;
  bool xm2_y_R = false;
  if (pt.x() > 0) {
    xm2_ym1_R = getIterator(point(pt.x()-2, pt.y()-1)).hasRightNeighbour();
    xm2_y_R = getIterator(point(pt.x()-2, pt.y())).hasRightNeighbour();
  }
  bool xm1_yp1_U = false;
  bool x_yp1_U = false;
  if (pt.y() < m_yMax) {
    xm1_yp1_U = getIterator(point(pt.x()-1, pt.y()+1)).hasUpNeighbour();
    x_yp1_U = getIterator(point(pt.x(), pt.y()+1)).hasUpNeighbour();
  }


  const bitmap::iterator it_xm1_ym1 = getIterator(point(pt.x()-1, pt.y()-1));
  bool xm1_ym1_U = it_xm1_ym1.hasUpNeighbour();
  bool xm1_ym1_R = it_xm1_ym1.hasRightNeighbour();
  bool xm1_ym1_N = it_xm1_ym1.isNode();
  const bitmap::iterator it_xm1_y   = getIterator(point(pt.x()-1, pt.y()));
  bool xm1_y_U   = it_xm1_y.hasUpNeighbour();
  bool xm1_y_R   = it_xm1_y.hasRightNeighbour();
  bool xm1_y_N   = it_xm1_y.isNode();
  const bitmap::iterator it_x_ym1   = getIterator(point(pt.x(), pt.y()-1));
  bool x_ym1_U   = it_x_ym1.hasUpNeighbour();
  bool x_ym1_R   = it_x_ym1.hasRightNeighbour();
  bool x_ym1_N   = it_x_ym1.isNode();
  const bitmap::iterator it_x_y     = getIterator(point(pt.x(), pt.y()));
  bool x_y_U     = it_x_y.hasUpNeighbour();
  bool x_y_R     = it_x_y.hasRightNeighbour();
  bool x_y_N     = it_x_y.isNode();



#ifdef VERBOSE
  std::cout<<"girl::bitmap_swap before : "<<x_y_U<<" "<< xm1_ym1_R<<" "<< xm1_y_U<<" "<< xm1_y_R<<"\n";

  std::cout<<"("<<p.x()<<"; "<<p.y()<<"): "<<x_y_R<<" "<<  x_y_U<<" "<<  xm1_y_R<<" "<<  x_yp1_U<<"\n";

  std::cout<<"("<<p.x()<<"; "<<p.y()-1<<x_ym1_R<<" "<<x_ym1_U<<" "<<xm1_ym1_R<<" "<<x_y_U<<"\n";
  std::cout<<"("<<p.x()-1<<"; "<<p.y()-1<<" "<<xm1_ym1_R<<" "<<xm1_ym1_U<<" "<<xm2_ym1_R<<" "<<xm1_y_U<<"\n";
  std::cout<<"("<<p.x()-1<<"; "<<p.y()<<" "<<xm1_y_R<<" "<<xm1_y_U<<" "<<xm2_y_R<<" "<<xm1_yp1_U<<"\n";
  
  std::cout<<" xm1_ym1_R="<<xm1_ym1_R<<" xm1_y_R="<<xm1_y_R<<" xm1_y_U="<<xm1_y_U<<" x_yp1_U="<<x_yp1_U<<"\n";

#endif


  const int nbNodes = xm1_ym1_N + x_ym1_N + xm1_y_N + x_y_N;
  assert(nbNodes<3);

#ifdef VERBOSE
  std::cout<<"nbNodes="<<nbNodes<<"\n";
  if (nbNodes != 0)
    std::cout<<"x_y_N="<<x_y_N<<" x_ym1_N="<<x_ym1_N <<" xm1_ym1_N="<<xm1_ym1_N <<" xm1_y_N="<<xm1_y_N <<"\n";
#endif
  
  
  


  //we only move arbitrary nodes, with their two darts on pixel borders
  if (nbNodes==1) {

    if (xm1_ym1_N && xm1_ym1_R && xm1_y_U) { //(x-1; y-1) -> (x; y)    
      it_xm1_ym1.unsetNode();
      it_x_y.setNode();

      xm1_ym1_N = 0;
      assert(x_y_N == 0);
      x_y_N   = 1;
    }
    else if (x_ym1_N && xm1_ym1_R && x_y_U) { //(x; y-1) -> (x-1; y)
      
      it_x_ym1.unsetNode();
      it_xm1_y.setNode();

      x_ym1_N = 0;
      assert(xm1_y_N == 0);
      xm1_y_N   = 1;
    }
    else if (x_y_N && xm1_y_R && x_y_U) { //(x;y) -> (x-1;y-1)

      it_x_y.unsetNode();
      it_xm1_ym1.setNode();

      x_y_N   = 0;
      assert(xm1_ym1_N == 0);
      xm1_ym1_N = 1;
    }
    else if (xm1_y_N && xm1_y_U && xm1_y_R) { //(x-1;y) -> (x;y-1)

      it_xm1_y.unsetNode();
      it_x_ym1.setNode();

      xm1_y_N = 0;
      assert(x_ym1_N == 0);
      x_ym1_N = 1;
    }

  }

  it_xm1_ym1.writeRightNeighbour(!xm1_ym1_R);
  it_xm1_y.writeRightNeighbour(!xm1_y_R);
  it_xm1_y.writeUpNeighbour(!xm1_y_U);
  it_x_y.writeUpNeighbour(!x_y_U);

  xm1_ym1_R = !xm1_ym1_R;
  xm1_y_R = !xm1_y_R;
  xm1_y_U = !xm1_y_U;
  x_y_U = !x_y_U;


#ifdef VERBOSE
  std::cout<<"after: xm1_ym1_R="<<xm1_ym1_R <<", xm1_y_R="<<xm1_y_R <<", xm1_y_U="<<xm1_y_U <<", x_yp1_U="<<x_yp1_U <<"\n";
  if (nbNodes != 0)
    std::cout<<"x_y_N="<<x_y_N <<" x_ym1_N="<<x_ym1_N <<" xm1_ym1_N="<< xm1_ym1_N<<" xm1_y_N="<<xm1_y_N <<"\n";
#endif


  int res = 0;

  if (x_y_N) {
    res |= ( static_cast<int>(x_y_R) 
	     | (static_cast<int>(x_y_U) << 1)
	     | (static_cast<int>(xm1_y_R) << 2)
	     | (static_cast<int>(x_yp1_U) << 3)
	     );
  }
  if (x_ym1_N) {
    res |= (static_cast<int>(x_ym1_R)
	    | (static_cast<int>(x_ym1_U) << 1)
	    | (static_cast<int>(xm1_ym1_R) << 2)
	    | (static_cast<int>(x_y_U) << 3)
	    ) << 4;
  }
  if (xm1_ym1_N) {
    res |= (static_cast<int>(xm1_ym1_R)
	    | (static_cast<int>(xm1_ym1_U) << 1)
	    | (static_cast<int>(xm2_ym1_R) << 2)
	    | (static_cast<int>(xm1_y_U) << 3)
	    ) << 8;
  }
  if (xm1_y_N) {
    res |= (static_cast<int>(xm1_y_R)
	    | (static_cast<int>(xm1_y_U) << 1)
	    | (static_cast<int>(xm2_y_R) << 2)
	    | (static_cast<int>(xm1_yp1_U) << 3)
	    ) << 12;
  }
  res |= nbNodes << 16;



#ifdef VERBOSE

  std::cout<<"girl::bitmap::swap after : "<<x_y_U <<" "<<xm1_ym1_R <<" "<<xm1_y_U <<" "<<xm1_y_R <<"\n";
  std::cout<<"("<<p.x()<<"; "<<p.y()<<"): "<<x_y_R<<" "<<  x_y_U<<" "<<  xm1_y_R<<" "<<  x_yp1_U<<"\n";
  
  std::cout<<"("<<p.x()<<"; "<<p.y()-1<<x_ym1_R<<" "<<x_ym1_U<<" "<<xm1_ym1_R<<" "<<x_y_U<<"\n";
  std::cout<<"("<<p.x()-1<<"; "<<p.y()-1<<" "<<xm1_ym1_R<<" "<<xm1_ym1_U<<" "<<xm2_ym1_R<<" "<<xm1_y_U<<"\n";
  std::cout<<"("<<p.x()-1<<"; "<<p.y()<<" "<<xm1_y_R<<" "<<xm1_y_U<<" "<<xm2_y_R<<" "<<xm1_yp1_U<<"\n";
#endif

  //girlAssert1(checkConsistency());
  assert(checkConsistency());

  return res;

}



void 
girl::bitmap::save(const std::string &filename) const
{
  std::ofstream file(filename.c_str());
  if (file) {
    file<<"P5\n# Generated by GIRL\n";
    file<<m_dimX<<" "<<m_dimY<<"\n15\n";

    std::vector<char> ln(m_xMax+2);
    char *line = &ln[1];
    for (int i = -1; i <= m_yMax; ++i) {
      for (int j = -1; j <= m_xMax; ++j) {
	line[j] = static_cast<char>(getMask(girl::point(j, i)));
      }
      file.write(&ln[0], ln.size()); 
    }
  }
}


void
girl::bitmap::swap(girl::bitmap &btm)
{
  std::swap(m_bitm, btm.m_bitm);
  std::swap(m_lineSize, btm.m_lineSize);
  std::swap(m_dimX, btm.m_dimX);
  std::swap(m_dimY, btm.m_dimY);
  std::swap(m_xMax, btm.m_xMax);
  std::swap(m_yMax, btm.m_yMax);
}


/*
 * @brief Crop the bitmap to the set of points in 
 * the box defined by @a p1 and @a p2.
 * But do not ensure that there is a border and at least a node on this border !

 * @warning We must have 0<=p1.x()<=p2.x()<width() 
 * and 0<=p1.y()<=p2.y()<height().
 */
void
bitmap::crop(const girl::point &p1, const girl::point &p2)
{
  assert(p1.x() <= p2.x());
  assert(p1.y() <= p2.y());

  const size_t newWidth = p2.x()-p1.x()+1;
  const size_t newHeight = p2.y()-p1.y()+1;
  if (newWidth==0 || newHeight==0) { //no point
    return;
  }

  bitmap btm2(newWidth, newHeight, false); //private constructor: correct size, but not initialized.
  

  const girl::point ip1(p1.x()-1, p1.y()-1); //transfer to interpixel

  //copy
  char *b = &m_bitm[0];
  char *b2 = &btm2.m_bitm[0];
  
  if ( (ip1.x() & 1) != 0) { //ip1.x() is odd (-1, 1, 3, 5, ...) or p1.x() is even 
    //we can do memcopies...
    
    b += p1.y()*m_lineSize + p1.x()/2;

    for (POINT_TYPE j=0; j<btm2.m_dimY; ++j) {
      memcpy(b2, b, btm2.m_lineSize);
      b += m_lineSize;
      b2 += btm2.m_lineSize;
    }
    
    //if ( (btm2.m_dimX & 1) != 0) //bmt2.m_dimX is odd
    // do we have to set 4 bits of each byte of last coord to zero ???
  }
  else { //ip1.x() is even (0, 2, 4, ...) or p1.x() is odd
    
    //we can not do memcopies, we have to copy 4 bits per 4 bits...

    b += p1.y()*m_lineSize + p1.x()/2;
    
    const char MASK0 = (char)(0x0f);
    const char MASK1 = (char)(0xf0);

    const size_t shift = m_lineSize - btm2.m_lineSize;

    for (POINT_TYPE j=0; j<btm2.m_dimY; ++j) {
    
      for (POINT_TYPE i=0; i<btm2.m_lineSize; ++i) {

	*b2 = ((*b & MASK1)>>4) | ((*(b+1) & MASK0)<<4);

	assert( (((*b2 & MASK1)>>4)&MASK0) == (*(b+1) & MASK0) );
	assert( (((*b2 & MASK0)<<4)&MASK1) == (*(b) & MASK1) );

	++b2;
	++b;
      }
      
      b += shift;
    }

  }

  swap(btm2);


  //TODO:OPTIM: the following code could be optimized
  // by a lower-level access : 
  // we could write yte per byte (instead of half-byte per half-byte like here)

  //Correct Up border (first row)
  {
    girl::bitmap::iterator it = getIterator(girl::point(-1, -1));
    for (int x=-1; x <= m_xMax ; ++x) {
      it.writeUpNeighbour(false);
      it.moveRight();
    }
  }
  //Correct Right border (last column)
  {
    girl::bitmap::iterator it = getIterator(girl::point(m_xMax, -1));
    for (int y=-1; y <= m_yMax ; ++y) {
      it.writeRightNeighbour(false);
      it.moveDown();
    }
  }

  
  //DEBUG
  /*
  {
    if (! checkBorder()) 
      std::cerr<<"bitmap::checkBorder() failed !!!"<<std::endl;

    if (! checkNodes()) 
      std::cerr<<"bitmap::checkNodes() failed !!!"<<std::endl;

    if (! checkConsistency()) 
      std::cerr<<"bitmap::checkConsistency() failed !!!"<<std::endl;

    assert(checkBorder());
    assert(checkNodes());
  }
  */

  assert(checkConsistency());
}


