/*
  Assert with levels
  Copyright (C) 2004-2009 Boris Mansencal  boris.mansencal@labri.fr
 
  This file is part of GIRL.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

/**
 * @file girlAssert.hpp
 *
 * @brief Assert with levels.
 *
 * girlAssert1, girlAssert2, and girlAssert3 are macros that behave like assert 
 * except they are still active in release mode.
 * Besides, they can be activated/deactivated by level:
 *  - NDEBUG1 disables girlAssert1, girlAssert2 and girlAssert3
 *  - NDEBUG2 disables girlAssert2 and girlAssert3
 *  - NDEBUG3 disables girlAssert3
 */

#include <cassert>

#ifndef NDEBUG3             
#define girlAssert3(condition) girlAssert2(condition)
#else
#define girlAssert3(condition) ((void)0)
#endif

#ifndef NDEBUG2
#define girlAssert2(condition) girlAssert1(condition) 
#else
#define girlAssert2(condition) ((void)0)
#endif

#ifndef NDEBUG1

#include <cstdio>
#include <cstdlib>
             
#define girlAssert1(condition) \
  ((condition)?((void)(0)):((void)(fprintf(stderr, "%s:%d Assertion \'%s\' failed.\n",  __FILE__, __LINE__, # condition), \
      (void)(abort()) ))) 
#else
#define girlAssert1(condition) ((void)0)
#endif


