/*
  Euclidean points corresponding to closed contour.
  Copyright (C) 2005-2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef EUCLIDEANPATH_HPP
#define EUCLIDEANPATH_HPP

#include <girl/girlCommon.hpp>

#include <girl/girlContour.hpp>


namespace girl {

  typedef float REAL_TYPE;


  class discreteTangent;


  /**
   * 
   * @brief Euclidean path of region contour.
   *
   * Gives euclidean (real) points corresponding to dicrete points of a (closed) contour.
   */
  class GIRL_API euclideanPath
  {

  public:

#ifdef EP_QUANTIF
    typedef int EP_SHIFT_TYPE;
#else
    typedef girl::REAL_TYPE EP_SHIFT_TYPE;
#endif

    /**
     * @brief Constructor.
     *
     * Constructs an empty path.
     */
    euclideanPath();

    /**
     * @brief Constructor.
     *
     * Constructs an euclidean path from contour @a c.
     */
    explicit euclideanPath(const girl::contour &c);

    /**
     * @brief Set contour @a c as new contour.
     */
    void set(const girl::contour &c);

  
    /**
     * @return path length.
     *
     * O(1).
     */
    size_t length() const
    {
      return m_contour.length();
    }


    /**
     * @brief Get real value corresponding to shift.
     */
    static girl::REAL_TYPE realValue(EP_SHIFT_TYPE shift);


    /**
     * @brief Get euclidean area of euclidean path.
     */
    girl::REAL_TYPE area() const;
  
    /**
     * @brief Get euclidean perimeter of euclidean path.
     */
    girl::REAL_TYPE perimeter() const;



  protected:

    typedef std::pair<EP_SHIFT_TYPE, EP_SHIFT_TYPE> ShiftPair;
    typedef std::vector<ShiftPair> ShiftPairVector;


  public:

    /**
     * @brief Const iterator on an euclidean path.
     *
     *
     */
    class const_iterator
    {
    public:
      const_iterator()
	: m_it(), 
	  m_sp(NULL), 
	  m_pt()
      {

      }

      bool operator==(const const_iterator &o) const
      {
	return m_it == o.m_it && m_sp == o.m_sp && m_pt == o.m_pt;
      }
    
      bool operator!=(const const_iterator &o) const
      {
	return m_it != o.m_it || m_sp != o.m_sp || m_pt != o.m_pt;
      }

      const_iterator &operator++()
      {
	m_pt.move(*m_it);
	++m_it;
	++m_sp;
	return *this;
      }

      const_iterator &operator--()
      {
	--m_it;
	m_pt.move((*m_it).opposite());
	--m_sp;
	return *this;
      }

      /**
       * @brief Get next direction on contour.
       */
      girl::direction getNextDir() const
      {
	return *m_it;
      }
    
      /**
       * @brief Get current discrete point.
       */
      girl::point getPoint() const
      {
	return m_pt;
      }

      /**
       * @brief Get shifts for euclidean point corresponding to current discrete point.
       *
       * @warning in Euclidean Path Coordinate System.
       * 
       * @sa euclideanPath::realValue.
       */
      void getShifts(EP_SHIFT_TYPE &x, EP_SHIFT_TYPE &y) const
      {
	x = m_sp->first;
	y = m_sp->second;
      }

      /**
       * @brief Get euclidean point coordinates corresponding to current discrete point.
       *
       */
      void getEuclideanPoint(girl::REAL_TYPE &x, girl::REAL_TYPE &y) const
      {
	x = static_cast<girl::REAL_TYPE>(m_pt.x()) + euclideanPath::realValue(m_sp->first);
	y = static_cast<girl::REAL_TYPE>(m_pt.y()) - euclideanPath::realValue(m_sp->second);//warning: change of Coordinate System : from euclidean path CS to toger CS !
      }

    protected:
    
      explicit const_iterator(const euclideanPath &ep)
	: m_it(ep.m_contour.begin()),
	  m_sp(&ep.m_shifts[0]),
	  m_pt(ep.m_contour.startPt())
      {
      
      }
      
      const_iterator(const euclideanPath &ep, girl::contour::const_dir_iterator it)
	: m_it(it),
	  m_sp(&ep.m_shifts[0]+ep.m_shifts.size()),
	  m_pt(ep.m_contour.endPt())
      {
      
      }

      friend class euclideanPath;
    
    protected:
    
      girl::contour::const_dir_iterator m_it;
      const euclideanPath::ShiftPair *m_sp;    
      girl::point m_pt;
    };    

    /**
     * @return an iterator on first point of euclideanPath.
     */
    inline const_iterator begin() const
    {
      return const_iterator(*this);
    }

    /**
     * @return an iterator past the last point of euclideanPath.
     */
    inline const_iterator end() const
    {
      return const_iterator(*this, m_contour.end());
    }
  
    /**
     * @brief Compute moments of tangent for whole closed contour @a c.
     */
    static void computeTangentMoments(const girl::contour &c,
				      double &angleNorm_m0,
				      double &angleNorm_m1,
				      double &angleNorm_m2);
    
    /**
     * @brief Compute moments of tangent for path,
     * from closed contour @a c, starting after @a startIndex directions
     * and for @a len directions.
     *
     * @a startIndex must be in [0; c.length()[
     * @a startIndex+len must be in ]0; c.length()]
     */
    static void computeTangentMoments(const girl::contour &c,
				      size_t startIndex, size_t len,
				      double &tangent_m0,
				      double &tangent_m1,
				      double &tangent_m2);

    /**
     * @brief Swap two euclideanPaths.
     *
     * Assign content of this to @a p and content of @a p to this.
     */
    void swap(euclideanPath &p);


  protected:

    static double computeTangent(const discreteTangent &d);


    void computePoint(int ind_pt, const discreteTangent &d);

    void computeShifts();
  
    //void computeShiftsKeepTangentInfo(girlExt::Int2 * tangentcarac);

    girl::POINT_TYPE deltaX(size_t index) const;
    girl::POINT_TYPE deltaY(size_t index) const;

    girl::REAL_TYPE realXShift(int ind_pt) const;

    girl::REAL_TYPE realYShift(int ind_pt) const;


  protected:

    girl::contour m_contour;

    ShiftPairVector m_shifts;
  };

  inline
  void
  swap(euclideanPath &p1, euclideanPath &p2)
  {
    p1.swap(p2);
  }

}

namespace std {

  /**
   * @brief Swap two euclideanPaths.
   *
   * Assign content of @a p1 to @a p2 and content of @a p2 to @a p1.
   */
  template <>
  inline 
  void
  swap(girl::euclideanPath &p1, girl::euclideanPath &p2)
  {
    p1.swap(p2);
  }

}


#endif /* ! EUCLIDEANPATH_HPP */
