/*
  Euclidean points corresponding to closed contour.
  Copyright (C) 2005-2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include <cmath>
#include "euclideanPath.hpp"
#include "discreteTangent.hpp"

#ifndef M_PI
//Visual C++ does not define these constants in <cmath>
#define M_PI 3.14159265358979323846
#endif 

#include <iostream>//DEBUG

using namespace girl;

#ifdef EP_QUANTIF
#define PIXEL_ECH 256
#endif

//#define EP_NON_INCREMENTAL 1
#define EP_INCREMENTAL 1
//#define EP_EXT_NON_INCREMENTAL 1




euclideanPath::euclideanPath()
  : m_contour(), m_shifts() 
{
}

euclideanPath::euclideanPath(const girl::contour &c)
  : m_contour(c), m_shifts(c.length())
{
  computeShifts();
}

void
euclideanPath::set(const girl::contour &c)
{
  m_contour = c;
  m_shifts.resize(c.length());
  computeShifts();
}


void
euclideanPath::computePoint(int ind_pt, const discreteTangent &d)
{
  static const int sym4[4][2] = {{1, 1}, {-1, 1}, {-1, -1}, {1, -1}};

  const int a = d.a();
  const int b = d.b();
  const int mu = d.mu();
  const int quadrant = d.quadrant();
    
  const girl::REAL_TYPE alpha = static_cast<girl::REAL_TYPE>(2*mu + a + b - 1)/static_cast<girl::REAL_TYPE>(2*(a+b));
#ifdef EP_QUANTIF
  const int delt_X = static_cast<int>((sym4[quadrant][0]*alpha+0.5)*PIXEL_ECH);
  const int delt_Y = static_cast<int>((-sym4[quadrant][1]*alpha+0.5)*PIXEL_ECH);
#else
  const girl::REAL_TYPE delt_X = (sym4[quadrant][0]*alpha);
  const girl::REAL_TYPE delt_Y = (-sym4[quadrant][1]*alpha);
#endif

  m_shifts[ind_pt] = ShiftPair(delt_X, delt_Y);

}



/**
 * Compute shifts for euclidean path corresponding to discrete path (contour).
 */
void
euclideanPath::computeShifts()
{
  assert(m_shifts.size() == length());

  discreteTangent localTangent;
  const size_t nbPts = length();
#ifdef EP_NON_INCREMENTAL
  {
    //non incremental computation of tangent
    for (size_t i = 0; i < nbPts; ++i) {
      localTangent.compute(m_contour, i);
      computePoint(i, localTangent);
      //localTangent.reinit();
    }
  }
#elif defined(EP_INCREMENTAL)
  {
    //incremental computation of tangent
    const size_t firstIndex = 0;
    localTangent.compute(m_contour, firstIndex);
    computePoint(firstIndex, localTangent);
    for (size_t i = 1; i < nbPts; ++i) { //start from 1
      localTangent.compute_incr(m_contour, i);
      computePoint(i, localTangent);
    }
  }
#elif defined(EP_EXT_NON_INCREMENTAL)
  {
    //non incremental computation of extended tangent
    for (size_t i = 0; i < nbPts; ++i) {
      localTangent.compute_ext(m_contour, i);
      computePoint(i, localTangent);
      //localTangent.reinit();
    }
  }
#else
  {
    //incremental computation of extended tangent
    const size_t firstIndex = 0;
    localTangent.compute_ext(m_contour, firstIndex);
    computePoint(firstIndex, localTangent);
    for (size_t i = 1; i < nbPts; ++i) { //start from 1
      localTangent.compute_ext_incr(m_contour, i);
      computePoint(i, localTangent);
    }
  }
#endif
}


//Compute tangent from discreteLine @a d.
double
euclideanPath::computeTangent(const discreteTangent &d)
{
  const int a = d.a();
  const int b = d.b();
  const int quadrant = d.quadrant();
  
  double angleTgt = 0.0;
  if (a != 0) {
    angleTgt = atan(b/static_cast<double>(a));
    if (quadrant == 1 || quadrant == 3)
      angleTgt = M_PI - angleTgt;
  }

  return angleTgt;
}

void
euclideanPath::computeTangentMoments(const girl::contour &c,
				     double &tangent_m0,
				     double &tangent_m1,
				     double &tangent_m2)
{
  const size_t startIndex = 0;
  const size_t len = c.length();
  
  computeTangentMoments(c, startIndex, len, tangent_m0, tangent_m1, tangent_m2);
}



void
euclideanPath::computeTangentMoments(const girl::contour &c,
				     size_t startIndex, size_t len,
				     double &tangent_m0,
				     double &tangent_m1,
				     double &tangent_m2)
{
  size_t endIndex = startIndex + len;
  assert(endIndex <= c.length());

  tangent_m0 = 0.0;

  assert(endIndex >= startIndex);
  tangent_m0 += len;
  assert(tangent_m0 >= 0.0);

  discreteTangent localTangent;
#ifdef EP_NON_INCREMENTAL
  {
    //non incremental computation of tangent
    tangent_m1 = 0.0;
    tangent_m2 = 0.0;
    for (size_t i = startIndex; i < endIndex; ++i) {

      localTangent.compute(c, i);

      const double tangent = euclideanPath::computeTangent(localTangent);   
      tangent_m1 += tangent;
      tangent_m2 += tangent*tangent;

      //localTangent.reinit();
    }
  }
#elif defined(EP_INCREMENTAL)
  {
    //incremental computation of tangent
    //const size_t firstIndex = 0;
    localTangent.compute(c, startIndex); //firstIndex); //B: ??????
    const double tangent0 = euclideanPath::computeTangent(localTangent);
    tangent_m1 += tangent0;
    tangent_m2 += tangent0*tangent0;
    for (size_t i = startIndex+1; i < endIndex; ++i) {
      localTangent.compute_incr(c, i);

      const double tangent = euclideanPath::computeTangent(localTangent);
      tangent_m1 += tangent;
      tangent_m2 += tangent*tangent;
    }
  }
#elif defined(EP_EXT_NON_INCREMENTAL)
  {
    //non incremental computation of extended tangent
    tangent_m1 = 0.0;
    tangent_m2 = 0.0;
    for (size_t i = startIndex; i < endIndex; ++i) {

      localTangent.compute_ext(c, i);

      const double tangent = euclideanPath::computeTangent(localTangent);   
      tangent_m1 += tangent;
      tangent_m2 += tangent*tangent;

      //localTangent.reinit();
    }
  }
#else
  {
    //incremental computation of extended tangent
    const size_t firstIndex = 0;
    localTangent.compute_ext(c, firstIndex);
    const double tangent0 = euclideanPath::computeTangent(localTangent);
    tangent_m1 += tangent0;
    tangent_m2 += tangent0*tangent0;
    for (size_t i = startIndex+1; i < endIndex; ++i) {
      localTangent.compute_ext_incr(c, i);

      const double tangent = euclideanPath::computeTangent(localTangent);
      tangent_m1 += tangent;
      tangent_m2 += tangent*tangent;
    }      
  }
#endif

}

/* SA */
/*
  void
  euclideanPath::computeShiftsKeepTangentInfo(girlExt::Int2 * tangentcarac)
  {
  discreteLine localTangent;
  int a = 0;
  int b = 0;

  for (int i = 0; i < m_nbPts; ++i)
  {
  localTangent.computeTangent(m_contour, i);
  computePoint(i, localTangent);
  if (localTangent.giveQuadrant() == 0) {
  a = localTangent.giveA();
  b = localTangent.giveB() - a;
  }
  else if (localTangent.giveQuadrant() == 1) {
  a = localTangent.giveA();
  b = a - localTangent.giveB();
  }
  else if ((localTangent.giveQuadrant() == 2)) {
  a = - localTangent.giveA();
  b = -a - localTangent.giveB();
  }
  else if ((localTangent.giveQuadrant() == 3)) {
  a = - localTangent.giveA();
  b =  localTangent.giveB() + a;
  }
      
  girlExt::Int2 &v = tangentcarac[i];
  v[0] = b;
  v[1] = a;
  localTangent.reinit();
     
  }

  }
*/


/* SA */
/*
  void
  euclideanPath::computeFromContour(const girl::contour &c, girlExt::Int2 ** tangentcarac) //B:TODO: should be a std::vector !!!!
  {
  m_contour = discretePath(c);
  m_nbPts = m_contour.length();
  *tangentcarac = new girlExt::Int2[m_nbPts];
  computeShiftsKeepTangentInfo(*tangentcarac);
  }
*/



/* 
//CALCUL D'UN CHEMIN EUCLIDIEN A PARTIR D'UN SEGMENT 

void
euclideanPath::compute_from_segment(girl::toger &tog, girl::dart brin) {
m_contour.compute_from_segment(tog, brin);
m_nbPts = m_contour.nb_codes() + 1;
calcul_decalages_priv(m_contour);
}
*/



/**
 *  CALCUL DU DECALAGE REEL EN FONCTION DU DECALAGE ENTIER 
 */
girl::REAL_TYPE
euclideanPath::realValue(EP_SHIFT_TYPE shift)
{
#ifdef EP_QUANTIF
  return  static_cast<girl::REAL_TYPE>(shift)/static_cast<girl::REAL_TYPE>(PIXEL_ECH) - 0.5f;
#else
  return shift;
#endif
}

girl::REAL_TYPE
euclideanPath::realXShift(int ind_pt) const
{
  assert(ind_pt < static_cast<int>(length()));
  return realValue(m_shifts[ind_pt].first);
}

girl::REAL_TYPE
euclideanPath::realYShift(int ind_pt) const
{
  assert(ind_pt < static_cast<int>(length()));
  return realValue(m_shifts[ind_pt].second);
}



static 
girl::POINT_TYPE 
delta_x(girl::direction d)
{
  static girl::POINT_TYPE DELTA_X[] = {+1, 0, -1, 0};
  return DELTA_X[static_cast<int>(d)];
}

static 
girl::POINT_TYPE 
delta_y(girl::direction d)
{
  static girl::POINT_TYPE DELTA_Y[] = {0, -1, 0, 1};
  return DELTA_Y[static_cast<int>(d)];
}

girl::POINT_TYPE
euclideanPath::deltaX(size_t ind_code) const
{
  return delta_x(m_contour[ind_code]);
}

girl::POINT_TYPE
euclideanPath::deltaY(size_t ind_code) const
{
  return delta_y(m_contour[ind_code]);
}




/*
 * Compute some features on euclidean path
 */

girl::REAL_TYPE
euclideanPath::area() const
{
  const int nbPts = m_contour.length();

  girl::REAL_TYPE surf = 0;

  const int Xi_moins = 1000; /* pt d'ind 0. pour avoir des coordonnees positives */
  const int Yi_moins = 1000;
  const girl::REAL_TYPE x0 = Xi_moins + realXShift(0);
  girl::REAL_TYPE yi_moins = Yi_moins - realYShift(0);
  const girl::REAL_TYPE y0 = yi_moins;

  int Xi = Xi_moins + deltaX(0); /* pt d'ind 1*/
  int Yi = Yi_moins + deltaY(0);
  girl::REAL_TYPE xi = static_cast<girl::REAL_TYPE>(Xi) + realXShift(1);
  girl::REAL_TYPE yi = static_cast<girl::REAL_TYPE>(Yi) - realYShift(1);
  const girl::REAL_TYPE y1 = yi;

  for (int i = 1; i < nbPts-1; ++i) //starts from 1
    {
      const int Xi_plus = Xi + deltaX(i);
      const int Yi_plus = Yi + deltaY(i);
      const girl::REAL_TYPE xi_plus = static_cast<girl::REAL_TYPE>(Xi_plus) + realXShift(i+1);
      const girl::REAL_TYPE yi_plus = static_cast<girl::REAL_TYPE>(Yi_plus) - realYShift(i+1);
      surf += xi*(yi_plus - yi_moins);

      yi_moins = yi;
      xi = xi_plus;
      yi = yi_plus;
      Xi = Xi_plus;
      Yi = Yi_plus;
    }
  surf += xi*(y0-yi_moins);/*dernier terme de la somme */
  surf += x0*(y1-yi); /*premier terme de la somme */

  surf = fabsf(surf); //B: bug? there was "abs" instead of "fabsf"
  return surf/2.0f;
}

static inline
girl::REAL_TYPE
P_euclideanDistance(girl::REAL_TYPE x1, girl::REAL_TYPE y1, girl::REAL_TYPE x2, girl::REAL_TYPE y2)
{
  return sqrtf((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2));
}


girl::REAL_TYPE
euclideanPath::perimeter() const
{
  const size_t nbPts = m_contour.length();
  if (nbPts == 0)
    return 0;

  girl::REAL_TYPE peri = 0.0f;
  
  girl::REAL_TYPE x1 = realXShift(0);
  girl::REAL_TYPE y1 = - realYShift(0);
  int X1 = 0;
  int Y1 = 0;
  girl::REAL_TYPE x2, y2;

  for (size_t i = 0; i < nbPts - 1; ++i)
    {
      int X2 = X1 + deltaX(i);
      int Y2 = Y1 + deltaY(i);
      x2 = static_cast<girl::REAL_TYPE>(X2) + realXShift(i+1);
      y2 = static_cast<girl::REAL_TYPE>(Y2) - realYShift(i+1);

      peri += P_euclideanDistance(x1, y1, x2, y2); 
      X1 = X2;
      Y1 = Y2;
      x1 = x2;
      y1 = y2;
    }

  x2 = realXShift(0);
  y2 = - realYShift(0);

  peri += P_euclideanDistance(x1, y1, x2, y2); 

  return peri;
}

void
euclideanPath::swap(euclideanPath &p)
{
  std::swap(m_contour, p.m_contour);
  std::swap(m_shifts, p.m_shifts);
}
