/*
  Discrete line
  Copyright (C) 2005 Anne Vialard  anne.vialard@labri.fr
  Copyright (C) 2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef DISCRETE_TANGENT_HPP
#define DISCRETE_TANGENT_HPP

#include <girl/girlContour.hpp>
#include <girl/discreteLine.hpp>


namespace girl {

  class GIRL_API discreteTangent
  {
  public:
    /**
     * @brief Constructor.
     */
    discreteTangent();

    /**
     * Compute discrete tangent for contour @a c at index @a index.
     *
     * Non incremental computation.
     *
     * @warning @a index must be inferior to @a length.
     */
    void compute(const girl::contour &c, size_t index);

    /**
     * Compute incrementally discrete tangent for contour @a c at index @a index.
     *
     * Incremental computation.
     * @warning compute() must have been previously called at least once for this contour.
     * 
     * @warning @a index must be inferior to @a length.
     */
    void compute_incr(const girl::contour &c, size_t index);

    /**
     * Compute extended discrete tangent for contour @a c at index @a index.
     *
     * Non incremental computation.
     *
     * @warning @a index must be inferior to @a length.
     */
    void compute_ext(const girl::contour &c, size_t index);

    /**
     * Compute incrementally extended discrete tangent for contour @a c at index @a index.
     *
     * Incremental computation.
     * @warning compute() must have been previously called at least once for this contour.
     * 
     * @warning @a index must be inferior to @a length.
     */
    void compute_ext_incr(const girl::contour &c, size_t index);


    int a() const;

    int b() const;

    int mu() const;

    int size() const;

    int extension() const;

    girl::direction quadrant() const;

  protected:

    void updateDirections();


  protected:
  
    /* droite 4-c du premier quadrant */
    discreteLine m_d; 

    /* la partie symetrique de la tangente est de taille 2n+1
       (calcul points) ou 2n (calcul aretes) en nb de points */
    int m_n;  

    /* nombre de points de l'extension, le signe indique le
       côté de l'extension*/
    int m_ext; 

    /* directions paire et impaire composant la */
    /* tangente, l'une d'elle peut etre indefinie  */
    girl::direction m_dir0, m_dir1; 

    /* points extremes de la tangente dans le 1er */
    /* quadrant - axes x vers la droite et y vers le haut */
    girl::point m_Pmoins, m_Pplus; 


  };

}

#endif /* ! DISCRETE_TANGENT_HPP */
