/*
  Discrete line
  Copyright (C) 2005 Anne Vialard  anne.vialard@labri.fr
  Copyright (C) 2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "discreteTangent.hpp"

#include <cassert>

using namespace girl;

discreteTangent::discreteTangent()
: m_n(0), m_ext(0)
{

}

int
discreteTangent::a() const
{
  return m_d.a();
}

int
discreteTangent::b() const
{
  return m_d.b();
}

int
discreteTangent::mu() const
{
  return m_d.mu();
}

int
discreteTangent::size() const
{
  return m_n;
}

int
discreteTangent::extension() const
{
  return m_ext;
}

girl::direction
discreteTangent::quadrant() const
{
  if (m_dir0 == girl::INVALID) {
    assert(m_dir1 != girl::INVALID);
    return m_dir1;
  }
  if (m_dir1 == girl::INVALID) {
    assert(m_dir0 != girl::INVALID);
    return m_dir0;
  }

  if (m_dir0 == girl::RIGHT && m_dir1 == girl::DOWN)
    return girl::DOWN;
    
  return m_dir0 <= m_dir1 ? m_dir0 : m_dir1;
}

static
bool
P_compatibleDirections(girl::direction dir, 
		       girl::direction *dir0, 
		       girl::direction *dir1)
{
  if (dir == *dir0 || dir == *dir1)
    return true;

  if (*dir0 == girl::INVALID && (dir == *dir1-1 || dir == (*dir1+1)%4)) {
    *dir0 = dir;
    return true;
  }

  if (*dir1 == girl::INVALID && (dir == *dir0+1 || dir == (*dir0+3)%4)) {
    *dir1 = dir;
    return true;
  }

  return false;
}

static
girl::direction
P_circularAccess(const girl::contour &c, size_t index, int shift)
{
  //B: I don't understand fully.
  // from noyau-4C/code_freeman.c: code_freeman_direction()
  // it seems:
  // @param index is an index of point
  // @param shift is a number of directions from point index to point index+shift (circularly)
  //   and we return the direction before point index+shift if shift is positive
  //                 the direction before point index if shift is negative.


  //B
  //assert(shift != 0); //shift may only be strictly positive or negative
  assert(index < c.length());
  
  const int s = static_cast<int>(index)+shift;
  const int len = static_cast<int>(c.length());

  if (shift > 0)
    return c[(s - 1)%len];
  
  //shift < 0
  if (s >= 0)
    return c[s];

  //s < 0
  return c[(s%len + len)];

}



/* recherche complete de la tangente symétrique autour d'un point par
   ajouts successifs de paires de points.
   Ce calcul est fait dans le 1er quadrant */ 
void
discreteTangent::compute(const girl::contour &c, size_t index)
{
  assert(index < c.length());

  girl::POINT_TYPE x_plus, y_plus, x_moins, y_moins;

  //initializations
  m_ext = 0;
  m_n = 0;
  girl::direction dir = P_circularAccess(c, index, 1);


  if (dir%2 == 0) {
    //null initial slope
    x_plus = 1; y_plus = 0;
    m_dir0 = dir;
    m_dir1 = girl::INVALID;
    m_d.init(false);
  }
  else { 
    //infinite initial slope
    x_plus = 0; y_plus = 1;
    m_dir0 = girl::INVALID;
    m_dir1 = dir;
    m_d.init(true);
  }

  dir = P_circularAccess(c, index, -1);
  P_compatibleDirections(dir, &m_dir0, &m_dir1); // tjrs vrai
  discreteLine d_save;
  if (dir%2 == 0) {
    x_moins = -1; y_moins = 0;
  }
  else {
    x_moins = 0; y_moins = -1;
  }
  bool ok = m_d.addNegativePoint(x_moins, y_moins);

  int i = 2;
  while (ok) {
    d_save = m_d;
    m_Pmoins.set(x_moins, y_moins);
    m_Pplus.set(x_plus, y_plus);

    ++m_n;

    // Add the positive point
    dir = P_circularAccess(c, index, i);
    if (P_compatibleDirections(dir, &m_dir0, &m_dir1))
      {
	x_plus += (dir%2 == 0);
	y_plus += (dir%2 == 1);
	ok = m_d.addPositivePoint(x_plus, y_plus);
      }
    else
      ok = false;

    //Add the negative point
    dir = P_circularAccess(c, index, -i);
    if (ok && 
	P_compatibleDirections(dir, &m_dir0, &m_dir1))
      {
	x_moins -= (dir%2 ==0);
	y_moins -= (dir%2 ==1);
	ok = m_d.addNegativePoint(x_moins, y_moins);
      }
    else 
      ok = false;
      
    ++i;
  }
  
  m_d = d_save;
}


/* maj des directions (pour permettre le changement de quadrant) lors
   de la suppression d'un point */
void
discreteTangent::updateDirections()
{
  if (m_d.b() == 0)
    m_dir0 = girl::INVALID;
  else if (m_d.a() == 0)
    m_dir1 = girl::INVALID;
}

/* calcul incremental de la tangente en un point a partir de la */
/* tangente au point precedent (supposee verifier n>=2) */
void
discreteTangent::compute_incr(const girl::contour &c, size_t index)
{
  assert(index < c.length());

  //B:
  //assert(m_n != 0);
  //assert(m_n != -1);

  girl::POINT_TYPE xplus = m_Pplus.x();
  girl::POINT_TYPE yplus = m_Pplus.y();
  girl::POINT_TYPE xmoins = m_Pmoins.x();
  girl::POINT_TYPE ymoins = m_Pmoins.y();
  
  //deletion of negative point the further away
  m_d.deleteNegativePoint(xmoins, ymoins, xplus-xmoins, yplus-ymoins);
  girl::direction dir = P_circularAccess(c, index, -m_n-1);  
  xmoins += (dir%2 == 0);
  ymoins += (dir%2 == 1);
  updateDirections();

  dir = P_circularAccess(c, index, m_n);
  if (P_compatibleDirections(dir, &m_dir0, &m_dir1) 
      && m_d.addPositivePoint(xplus+(dir%2==0), yplus+(dir%2==1))) { 
    // Add of first positive point OK
    discreteLine d_save = m_d;
    xplus += (dir%2==0);
    yplus += (dir%2==1);
    m_Pmoins.set(xmoins, ymoins);
    m_Pplus.set(xplus, yplus);

    /* Tentative d'ajout d'une paire de points */
    /* Ajout du point positif */
    dir = P_circularAccess(c, index, m_n+1);
    if (P_compatibleDirections(dir, &m_dir0, &m_dir1) &&
	m_d.addPositivePoint(xplus+(dir%2==0), yplus+(dir%2==1))) {
      xplus += (dir%2==0);
      yplus += (dir%2==1);

      /* Ajout du point negatif */
      dir = P_circularAccess(c, index, -m_n-1);
      if (P_compatibleDirections(dir, &m_dir0, &m_dir1) &&
	  m_d.addNegativePoint(xmoins-(dir%2==0), ymoins-(dir%2==1))) {
	xmoins -= (dir%2==0);
	ymoins -= (dir%2==1);
	++m_n;
	m_Pmoins.set(xmoins, ymoins);
	m_Pplus.set(xplus, yplus);
      }
      else { 
	// negative add failed, pair can not be added
	m_d = d_save;
      }
    }
  }
  else { 
    // 2nd deletion negative point
    m_d.deleteNegativePoint(xmoins, ymoins, 
			    xplus-xmoins,
			    yplus-ymoins); 
    dir = P_circularAccess(c, index, -m_n);
    xmoins += (dir%2 == 0);
    ymoins += (dir%2 == 1);
    m_n -= 1;
    m_Pplus.set(xplus, yplus);
    m_Pmoins.set(xmoins, ymoins);
    updateDirections();
  }

  // change frame/CS
  dir = P_circularAccess(c, index, -1);
  if (dir%2 == 1) {
    m_d.translateFrame(0, 1);
    m_Pplus.setY(m_Pplus.y()-1);
    m_Pmoins.setY(m_Pmoins.y()-1);
  }
  else {
    m_d.translateFrame(1, 0);
    m_Pplus.setX(m_Pplus.x()-1);
    m_Pmoins.setX(m_Pmoins.x()-1);
  }
}

/* recherche complete de la tangente ETENDUE autour d'un point par
   ajouts successifs de paires de points - ce calcul est fait dans le
   1er quadrant */ 
void
discreteTangent::compute_ext(const girl::contour &c, size_t index)
{
  assert(index < c.length());

  girl::direction dir;
  girl::POINT_TYPE x_plus, y_plus, x_moins, y_moins;
  bool ext_plus = 0, ext_moins = 0;

  /*initialisations */
  m_ext = 0;
  m_n = 1;
  dir = P_circularAccess(c, index, 1);

  if (dir%2 == 0)
    { /* pente initiale nulle */
      x_plus = 1; y_plus = 0;
      m_dir0 = dir;
      m_dir1 = girl::INVALID;
      m_d.init(false);
    }
  else 
    { /* pente initiale infinie */
      x_plus = 0; y_plus = 1;
      m_dir0 = girl::INVALID;
      m_dir1 = dir;
      m_d.init(true);
    }

  dir = P_circularAccess(c, index, -1);
  P_compatibleDirections(dir, &m_dir0, &m_dir1); // tjrs vrai
  discreteLine d_save;
  if (dir%2 == 0)
    {
      x_moins = -1; y_moins = 0;
    }
  else
    {
      x_moins = 0; y_moins = -1;
    }
  bool ok = m_d.addNegativePoint(x_moins, y_moins);

  int i = 2;
  while (ok)
    {
      d_save = m_d;
      m_Pmoins.set(x_moins, y_moins);
      m_Pplus.set(x_plus, y_plus);

      /* Ajout du point positif */
      dir = P_circularAccess(c, index, i);
      if (P_compatibleDirections(dir, &m_dir0, &m_dir1))
	{
	  x_plus += (dir%2 == 0);
	  y_plus += (dir%2 == 1);
	  ok = m_d.addPositivePoint(x_plus, y_plus);
	}
      else
	{
	  ok = 0;
	  ext_moins = 1; // autorise l'extension côté négatif
	}

      /* Ajout du point negatif */
      dir = P_circularAccess(c, index, -i);
      if (ok)
	{
	  if (P_compatibleDirections(dir, &m_dir0, &m_dir1))
	    {
	      x_moins -= (dir%2 ==0);
	      y_moins -= (dir%2 ==1);
	      ok = m_d.addNegativePoint(x_moins, y_moins);
	      m_n += ok;
	    }
	  else 
	    {
	      ok = 0;
	      ext_plus = 1;
	    }
	}
      ++i;
    }

  if (ext_moins) /* extension côté négatif */
    {
      --i;
      ok = 1;
      while (ok)
	{
	  dir = P_circularAccess(c, index, -i);
	  if (P_compatibleDirections(dir, &m_dir0, &m_dir1))
	    {
	      x_moins -= (dir%2 ==0);
	      y_moins -= (dir%2 ==1);
	      ok = m_d.addNegativePoint(x_moins, y_moins);
	      m_ext -= ok;
	    }
	  else 
	    ok = 0;
	  ++i;
	}
    }
  else if (ext_plus && d_save.a() != 0 && d_save.b() != 0)
    /* extension côté positif */
    /* CP sur a et b du à l'éventuelle tangente horizontale ou verticale */
    {
      m_ext = 1; /* 1 pt positif a déjà été ajouté */
      ok = 1;
      while (ok)
	{
	  dir = P_circularAccess(c, index, i);
	  if (P_compatibleDirections(dir, &m_dir0, &m_dir1))
	    {
	      x_plus += (dir%2 == 0);
	      y_plus += (dir%2 == 1);
	      ok = m_d.addPositivePoint(x_plus, y_plus);
	      m_ext += ok;
	    }
	  else 
	    ok = 0;
	  ++i;
	}
    }
  else /* pas d'extension */
    m_d = d_save;

}


void
discreteTangent::compute_ext_incr(const girl::contour & /* c */, size_t /* index */)
{
  assert(false);
}

