/*
  Discrete line
  Copyright (C) 2005 Anne Vialard  anne.vialard@labri.fr
  Copyright (C) 2009 Boris Mansencal  boris.mansencal@labri.fr

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include "discreteLine.hpp"

using namespace girl;

discreteLine::discreteLine()
  : m_a(0), m_b(1), m_mu(0)
{

}


void
discreteLine::init(bool initialStep)
{
  m_mu = 0;
  m_U.set(0, 0);
  m_L.set(0, 0);

  if (! initialStep) {
    m_a = 0;
    m_b = 1;
    m_Us.set(1, 0);
    m_Ls.set(1, 0);
  }
  else {
    m_a = 1;
    m_b = 0;
    m_Us.set(0, 1);
    m_Ls.set(0, 1);
  }

}

bool
discreteLine::checkAdd(girl::POINT_TYPE x, girl::POINT_TYPE y) const
{
  const girl::POINT_TYPE rest = m_a*x - m_b*y;
  return ((m_mu-1 <= rest) && (rest < m_mu + m_b));
}
    
bool
discreteLine::addPositivePoint(girl::POINT_TYPE x, girl::POINT_TYPE y)
{
  const girl::POINT_TYPE rest = m_a*x - m_b*y;
  if ((m_mu <= rest) && (rest < m_mu + m_a + m_b)) {
    if (rest == m_mu)
      m_Us.set(x, y);
    if (rest == m_mu+m_a+m_b-1)
      m_Ls.set(x, y);
    return true;
  }

  if (rest == m_mu - 1) { //slope increase
    // update leaning points
    m_L = m_Ls;
    m_Us.set(x, y);

    // update characteristics
    m_a = y - m_U.y();
    m_b = x - m_U.x();
    m_mu = m_a*x - m_b*y;
    return true;
  }
  
  if (rest == m_mu + m_a + m_b) { // slope decrease
    //update leaning points
    m_U = m_Us;
    m_Ls.set(x, y);

    //update characteristics
    m_a = y - m_L.y();
    m_b = x - m_L.x();
    m_mu = m_a*(x-1) - m_b*(y+1) + 1;
    return true;
  }
  
  //else new point does not allow to continue discrete line
  return false;
    
}
  
bool
discreteLine::addNegativePoint(girl::POINT_TYPE x, girl::POINT_TYPE y)
{
  girl::POINT_TYPE rest = m_a*x - m_b*y;
  if (m_mu <= rest && rest < m_mu + m_a + m_b) {
    if (rest == m_mu)
      m_U.set(x, y);
    if (rest == m_mu+m_a+m_b-1)
      m_L.set(x, y);
    return true;
  }

  if (rest == m_mu - 1) { // slope decrease
    //update leaning points
    m_Ls = m_L;
    m_U.set(x, y);

    //update characteristics
    m_a = m_Us.y() - y;
    m_b = m_Us.x() - x;
    m_mu = m_a*x - m_b*y;
    return true;
  }

  if (rest == m_mu + m_a + m_b) { //slope increase
    //update leaning points
    m_Us = m_U;
    m_L.set(x, y);

    //update characteristics
    m_a = m_Ls.y() - y;
    m_b = m_Ls.x() - x;
    m_mu = m_a*(x-1) - m_b*(y+1) + 1;
    return true;
  }
  //else new point does not allow to continue discrete line
  return false;
  
}
  
void
discreteLine::deleteNegativePoint(girl::POINT_TYPE x, girl::POINT_TYPE y,
				  girl::POINT_TYPE dx, girl::POINT_TYPE dy)
{
  //TODO: A REVOIR : ADAPTATION EN 4-CONNEXITE */

  if (x == m_U.x() && y == m_U.y()) {
    // M = U, upper leaning point of minimal abscissa

    if (m_Us.x() - x == m_b &&
	m_Us.y() - y == m_a &&
	m_Ls.x() == m_L.x() &&
	m_Ls.y() == m_L.y()) {
      //deleting M increase slope

      //update characteristics
      m_a = m_L.y() - y + 1;
      m_b = m_L.x() - x - 1;
      m_mu = m_a * m_Us.x() - m_b * m_Us.y();
      
      //update leaning points
      if (m_b != 0) {
	//B??? girl::POINT_TYPE or int for k & l ??????????????????????????????
	const girl::POINT_TYPE k = (m_Us.x() - x - 1) / m_b;
	m_U.set(m_Us.x() - k * m_b,
		m_Us.y() - k * m_a);
	const girl::POINT_TYPE l = (dx - 1) / m_b - 1;
	m_Ls.set(m_L.x() + l * m_b,
		 m_L.y() + l * m_a);
      }
      else { //new line is vertical
	m_U = m_L;
	m_Ls = m_Us;
      }
	
    }
    else { //update of U
      m_U.set(x + m_b, y + m_a);
    }
  }

  if (x == m_L.x() && y == m_L.y()) {
    // M = L, lower leaning point of minimal abscissa

    if (m_Ls.x() - x == m_b &&
	m_Ls.y() - y == m_a &&
	m_Us.x() == m_U.x() &&
	m_Us.y() == m_U.y()) {
      //deleting M decrease slope
      
      //update characteristics
      m_a = m_U.y() - y - 1;
      m_b = m_U.x() - x + 1;
      m_mu = m_a * m_U.x() - m_b * m_U.y();
      
      //update leaning points
      if (m_a != 0) {
	//B??? girl::POINT_TYPE or int for k & l ??????????????????????????????
	const girl::POINT_TYPE k = (m_Ls.y() - y - 1) / m_a;
	m_L.set(m_Ls.x() - k * m_b,
		m_Ls.y() - k * m_a);
	const girl::POINT_TYPE l = (dy - 1) / m_a - 1;
	m_Us.set(m_U.x() + l * m_b,
		 m_U.y() + l * m_a);
      }
      else {  //new line is horrizontal
	m_L = m_U;
	m_Us = m_Ls;
      }
    }
    else { // update of L
      m_L.set(x + m_b, y + m_a);
    }
  }
  
}


void
discreteLine::translateFrame(girl::POINT_TYPE dx, girl::POINT_TYPE dy)
{
  m_U.set(m_U.x() - dx, m_U.y() - dy);
  m_Us.set(m_Us.x() - dx, m_Us.y() - dy);
  m_L.set(m_L.x() - dx, m_L.y() - dy);
  m_Ls.set(m_Ls.x() - dx, m_Ls.y() - dy);
  
  m_mu = m_a * m_U.x() - m_b * m_U.y();
}

