#!/bin/sh

#Colors
RED='\e[0;31m'
GREEN='\e[0;32m'
ORANGE='\e[0;33m'
GREY='\e[0;37m'
NORMAL='\e[0;m'

#Globals vars
RESULTDIR="."
TOKENPARSE="Saliency Heat Interpolation"
TOKENOK=0
SALIENCY_THRESHOLD=2000
FIRST_FRAME_WITH_EYES_OPEN=0

echo_err() {
  echo -e $RED $1 $NORMAL
}

echo_succ() {
  echo -e $GREEN $1 $NORMAL
}

echo_warn() {
  echo -e $ORANGE $1 $NORMAL
}

echo_emph() {
  echo -e $GREY $1 $NORMAL
}

usage(){
  echo "Usage: ./stats.sh -file <logs file> <arguments>"
  echo "If no argument given, the script returns immediatly."
  echo "Values for <arguments> :"
  echo "  -split                print values needed to make a curve."
  echo "  -plot                 display the curves created."
  echo "  -cpu			display cpu in the curves' title"
  echo "  -gpu			display gpu in the curves' title"
  echo "  -stats                make stats (average and standard deviation)."
  echo "  -boxplot              display the average and some things interesting."
  echo "  -dir <directory>      store results inside the <directory>, by default it's the current directory."
  echo "  -help, -h             show this help."
}


#----------------------------------------------------------------
# Parse - Compute average and some things interesting...
#----------------------------------------------------------------

parse(){

  VALUES=$(cat $1 | grep $2 | sed -e "s/$2://g")
  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  MOYENNE=0
  MOYENNE_EYES_OPEN=0
  CPT=0
  for j in $VALUES ; do
    if [ $FIRST_FRAME_WITH_EYES_OPEN -eq 0 ]; then
      if [ $j -ge $SALIENCY_THRESHOLD ]; then
        FIRST_FRAME_WITH_EYES_OPEN=$CPT
        MOYENNE_EYES_OPEN=$(($MOYENNE_EYES_OPEN + $j))
      fi
    elif [ $CPT -ge $FIRST_FRAME_WITH_EYES_OPEN ]; then
      MOYENNE_EYES_OPEN=$(($MOYENNE_EYES_OPEN + $j))
    fi
    MOYENNE=$(($MOYENNE + $j))
    CPT=$(($CPT + 1))
  done

  CPT_EYES_OPEN=$(($CPT-$FIRST_FRAME_WITH_EYES_OPEN))
  MOYENNE_EYES_OPEN=$(($MOYENNE_EYES_OPEN / $CPT_EYES_OPEN ))
  MOYENNE=$(($MOYENNE / $CPT ))

  VARIANCE=0
  VARIANCE_EYES_OPEN=0
  CPT_V=0
  for j in $VALUES ; do
    TMP=$(($j - $MOYENNE))
    TMP=$(($TMP * $TMP))
    VARIANCE=$(($VARIANCE + $TMP))
    if [ $CPT_V -ge $FIRST_FRAME_WITH_EYES_OPEN ]; then
      TMP=$(($j - $MOYENNE_EYES_OPEN))
      TMP=$(($TMP * $TMP))
      VARIANCE_EYES_OPEN=$(($VARIANCE_EYES_OPEN + $TMP))
    fi
    CPT_V=$(($CPT_V+1))
  done
  VARIANCE=$(($VARIANCE / ($CPT)))
  VARIANCE_EYES_OPEN=$(($VARIANCE_EYES_OPEN / ($CPT_EYES_OPEN)))

  ECARTTYPE=$(echo "sqrt($VARIANCE)" | bc)
  ECARTTYPE_EYES_OPEN=$(echo "sqrt($VARIANCE_EYES_OPEN)" | bc)

  echo "$2 : µ = $MOYENNE ± $ECARTTYPE µs" >> $FILE;
  echo "$2 : µ = $MOYENNE_EYES_OPEN ± $ECARTTYPE_EYES_OPEN µs" >> $FILE_EYES_OPEN;

  if [ ! -z $CURVE ] && [ ! -z $BOXPLOT ]
  then
#    echo $2"_AV(t)=$MOYENNE, \\" >> $GPFILE;
#    echo $2"_AV(x) title \"Average of $2 \" with lines, \\" >> $GPFILE;
#    echo $2"_MIN(t)=$(( $MOYENNE - $ECARTTYPE)), \\" >> $GPFILE;
#    echo $2"_MIN(x) title \"Average - dev of $2 \" with lines, \\" >> $GPFILE;
#    echo $2"_MAX(t)=$(( $MOYENNE + $ECARTTYPE)), \\" >> $GPFILE;
#    echo $2"_MAX(x) title \"Average + dev of $2 \" with lines, \\" >> $GPFILE;
    echo "Boxplot Coming soon"
  fi

}

#----------------------------------------------------------------
# GenPlot - Create plot file to display curves
#----------------------------------------------------------------

genPlot(){
  UNAME=$(basename $GLOBALFILE | cut -d. -f1)
  NAME=$(echo $UNAME | sed -e "s/_/ /")

  echo "#!/usr/bin/gnuplot

  set term png #enhanced monochrome dashed" > $GPFILE
  

  if [ ! -z CPU ]; then
    echo "set output \"$RESULTDIR/graph_${UNAME}_cpu.png\"
          set title \"${NAME} sur CPU\"" >> $GPFILE
  elif [ ! -z GPU ]; then
    echo "set output \"$RESULTDIR/graph_${UNAME}_gpu.png\"
          set title \"${NAME} sur GPU\"" >> $GPFILE
  else
    echo "set output \"$RESULTDIR/graph_${UNAME}.png\"
          set title \"$NAME\"" >> $GPFILE
  fi
  echo -n "

  set key top left

  set style line 12 lc rgb'#808080' lt 0 lw 1
  set grid back ls 12

  set xlabel \"Time in video (in s)\"
  set tics scale 3
  set xtics 5
  set mxtics 5
  set ylabel \"Time to compute (in µs)\"
  
  plot " >> $GPFILE;

}

#----------------------------------------------------------------
# Makecurve - Print results in separatly files to make curves
#----------------------------------------------------------------

makecurve(){

  VALUES=$(cat $1 | grep $2 | sed -e "s/$2://g")

  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  FILE=$( mktemp -p $RESULTDIR $2.XXXXXX )

  CPT=0
  for j in $VALUES ; do
    echo "$(echo "($CPT / 25)" | bc -l) $j" >> $FILE
    CPT=$(($CPT + 1))
  done

  echo_succ "=> Results stored in $FILE !"

  if [ ! -z $SHOW ]
  then
    echo "\"$FILE\" title \"$2\" with lines, \\" >> $GPFILE;
  fi

  TOKENOK=$(($TOKENOK + 1))
}

#----------------------------------------------------------------
# Makecurvesum - Print sum of each time to compute
#----------------------------------------------------------------

makecurvesum(){

  VALUES=$(cat $1)

  for j in $TOKENPARSE ; do
    VALUES=$(echo $VALUES | sed -e "s/$j://g")
  done

  #Don't compute if there is empty
  if [ -z "$VALUES" ]
  then
    return 0;
  fi

  FILE=$( mktemp -p $RESULTDIR Sum.XXXXXX )

  CPT=0
  TMP=0
  for j in $VALUES ; do
    TMP=$(($TMP + $j))
    CPT=$(($CPT + 1))
    if [ $(($CPT % $2 )) == 0 ]
    then
      echo "$(echo "($(echo "($CPT / 25)" | bc -l) / $2)" | bc -l) $TMP" >> $FILE
      TMP=0
    fi
  done

  echo_succ "=> Results stored in $FILE !"

  if [ ! -z $SHOW ]
  then
    echo "\"$FILE\" title \"Total Time\" with lines, \\" >> $GPFILE;
  fi
  
}

#----------------------------------------------------------------
# Curve - Print results to make curves
#----------------------------------------------------------------

curve(){
  #create plot configuration file
  if [ ! -z $SHOW ]
  then
    genPlot;
  fi

  #parse file with token
  for j in $TOKENPARSE ; do
    makecurve $1 $j;
  done

  #uncomment if you want to display the time spend to compute one frame.
  makecurvesum $1 $TOKENOK;
  
}

#----------------------------------------------------------------
# Main
#----------------------------------------------------------------

#parse parameters
while [[ $# > 0 ]]
do
  key="$1"

  case $key in
    "-help"|"-h")
      usage;
      exit;
    ;;
    "-file")
      GLOBALFILE="$2"
      shift
    ;;
    "-split")
      CURVE="OK"
    ;;
    "-plot")
      SHOW="OK"
    ;;
    "-cpu")
      CPU="OK"
    ;;
    "-gpu")
      GPU="OK"
    ;;
    "-stats")
      STATS="OK"
    ;;
    "-boxplot")
      BOXPLOT="OK"
    ;;
    "-dir")
      RESULTDIR="$2"
      RESULTDIR=$(echo $RESULTDIR | sed -e "s/\/\//\//g")
      if [ -f $RESULTDIR ]
      then
        echo_err "$RESULTDIR is a file... We can't store data inside !"
        exit;
      fi
      if [ ! -e $RESULTDIR ]
      then
        mkdir $RESULTDIR;
      fi
      shift
    ;;
    *)
      echo_err "$key : Parameter unrecognized" 
      usage;
      exit;
    ;;
  esac
shift 
done

GPFILE=$( mktemp -p $RESULTDIR ConfGnuPlot.XXXXXX )

if [ -z $GLOBALFILE ] || [ ! -e $GLOBALFILE ]
then
  echo_err "Logs file is empty or doesn't exist !"
  usage;
  exit;
fi

#if we need to compute curve
if [ ! -z $CURVE ]
then
    curve $GLOBALFILE;
    echo_succ "=> GnuPlot Configuration File created in $GPFILE !"
    gnuplot $GPFILE;
fi

if [ ! -z $STATS ]
then
  FILE=$( mktemp -p $RESULTDIR Stats.XXXXXX )
  FILE_EYES_OPEN=$( mktemp -p $RESULTDIR Stats_eyes_open.XXXXXX )
  #parse file with token
  for j in $TOKENPARSE ; do
    parse $GLOBALFILE $j
  done
  echo_succ " => Stats stored in $FILE !"
  echo_succ " => Stats with eyes open stored in $FILE_EYES_OPEN !"
fi
