// ##################################################
// ###              TOBII_PROJECT_CPP             ###
// ##################################################
#ifndef TOBII_PROJECT_CPP
#define TOBII_PROJECT_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Project.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

namespace Tobii {

	// ##################################################
	// ###               CLASS PROJECT                ###
	// ##################################################

	IOVideoStreamPtr Project::Null( NULL );

	static const QString Video, Saliency, SaliencyHeatMap, SaliencyWeightedMap, Annotation, AnnotationHeatMap, AnnotationWeightedMap;

	const QString Project::Video                 ( "Video"                    );
	const QString Project::Saliency              ( "Saliency"                 );
	const QString Project::SaliencyHeatMap       ( "Saliency (HeatMap)"       );
	const QString Project::SaliencyWeightedMap   ( "Saliency (WeightedMap)"   );
	const QString Project::Annotation            ( "Annotation"               );
	const QString Project::AnnotationHeatMap     ( "Annotation (HeatMap)"     );
	const QString Project::AnnotationWeightedMap ( "Annotation (WeightedMap)" );

	// --------------------------------------------------
	// CONSTRUCTOR - DESTRUCTOR

	Project::Project ( const QString& folder, bool openEmpty, const QStringList& streamNames )
	: _fullpath( folder ), _isOpen( false )
	{
		if ( ! folder.isEmpty() )
			this->open( folder, openEmpty, streamNames );
	}

	Project::Project ( const Project& project )
	: _name( project.name() ), _fullpath( project.fullpath() ), _isOpen( project.isOpen() ), _streams( project.streams() ), _streamNames( project.streamNames() )
	{}

	Project::~Project ()
	{}

	Project& Project::operator= ( const Project& project )
	{
		this->fullpath()	 = project.fullpath();
		this->name()         = project.name();
		this->isOpen()		 = project.isOpen();
		this->streams()		 = project.streams();
		this->streamNames()	 = project.streamNames();
		return (*this);
	}

	// --------------------------------------------------
	// GETTERS - SETTERS

	const QString&			Project::name			() const	{ return this->_name;			}
	QString&				Project::name			()			{ return this->_name;			}
	const QString&			Project::fullpath		() const	{ return this->_fullpath;		}
	QString&				Project::fullpath		()			{ return this->_fullpath;		}
	const VideoStreamsMap&	Project::streams		() const	{ return this->_streams;		}
	VideoStreamsMap&		Project::streams		()			{ return this->_streams;		}
	const QStringList&		Project::streamNames	() const	{ return this->_streamNames;	}
	QStringList&			Project::streamNames	()			{ return this->_streamNames;	}

	const IOVideoStreamPtr& Project::getStream ( const QString& name ) const
	{
		if ( this->streams().find( name ) != this->streams().end() )
			return this->streams().at( name );

		return Project::Null;
	}

	IOVideoStreamPtr& Project::getStream ( const QString& name )
	{
		if ( this->streams().find( name ) != this->streams().end() )
			return this->streams().at( name );
		
		return Project::Null;
	}

	cv::Size Project::resolution () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->resolution();
		return cv::Size(0,0);
	}

	int Project::fps () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->fps();
		return 0;
	}

	int Project::frameCount () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->frameCount();
		return 0;
	}

	double Project::format () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->format();
		return -1;
	}

	Int64 Project::duration () const
	{
		return this->lastTimeStamp() - this->firstTimeStamp();
	}

	Int64 Project::currentTimeStamp () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->currentTimeStamp();
		return -1;
	}

	Int64 Project::currentIndex () const
	{
		if ( this->streams().size() != 0 )
			return this->streams().begin()->second->currentIndex();
		return -1;
	}

	Int64 Project::firstIndex () const
	{
		if ( this->streams().size() == 0 )
			return -1;
		return this->streams().begin()->second->firstIndex();
	}

	Int64 Project::firstTimeStamp () const
	{
		if ( this->streams().size() == 0 )
			return -1;
		return this->streams().begin()->second->firstTimeStamp();
	}

	Int64 Project::lastIndex () const
	{
		if ( this->streams().size() == 0 )
			return -1;
		return this->streams().begin()->second->lastIndex();
	}

	Int64 Project::lastTimeStamp () const
	{
		if ( this->streams().size() == 0 )
			return -1;
		return this->streams().begin()->second->lastTimeStamp();
	}

	void Project::next ()
	{
		for ( auto& streamEntry : this->streams() )
			streamEntry.second->next();
	}
	
	void Project::previous ()
	{
		for ( auto& streamEntry : this->streams() )
			streamEntry.second->previous();
	}

	void Project::setCurrentIndex ( Int64 index )
	{
		for ( auto& stream : this->streams() )
			stream.second->setCurrentIndex( index );
	}

	void Project::setCurrentTimeStamp ( Int64 timestamp )
	{
		for ( auto& stream : this->streams() )
			stream.second->setCurrentTimeStamp( timestamp );
	}

	// --------------------------------------------------
	// OPEN - CLOSE

	void Project::move ( const QString& folder )
	{
		QDir dirFrom( this->fullpath() );
		QDir dirTo( folder );

		if ( ! dirTo.exists() )
			dirTo.mkpath( dirTo.absolutePath() );

		for ( auto& stream : this->streams() )
			stream.second->move( dirTo.absolutePath()+"/"+this->name() );

		dirFrom.removeRecursively();
		this->fullpath() = dirTo.absolutePath();
	}

	void Project::open ( const QString& folder, bool openEmpty, const QStringList& channels )
	{
		// Dir exist ?
		QDir dir( folder );
		if ( ! dir.exists() )
			dir.mkpath( dir.absolutePath() );
		
		this->name() = dir.dirName();

		// Keep track of open streams by name
		this->streamNames() = channels;
		QStringList streamNames = channels;

		for ( const QFileInfo& fileinfo : dir.entryInfoList(QDir::Dirs|QDir::NoDotAndDotDot) )
		{
			int index = streamNames.indexOf( fileinfo.fileName() );

			// If this is a stream we ask for, or if we open any stream
			if ( index != -1 || streamNames.isEmpty() )
			{
				// Open this stream
				this->streams().emplace(
					fileinfo.fileName(),
					std::make_shared< VideoFramesStream >( fileinfo.absoluteFilePath(), openEmpty )
				);

				// If the stream was open, remove it from the list of stream to open
				if ( index != -1 )
					streamNames.removeAt(index);

				// If no stream in particular is asked, add it to the list of opened stream
				if ( channels.isEmpty() )
					this->streamNames() << fileinfo.fileName();
			}
		}

		// Open the rest
		for ( const QString& streamName : streamNames )
		{
			QString folder = dir.absolutePath() + "/" + streamName;
			dir.mkpath( folder );
			this->streams().emplace(
					streamName,
					std::make_shared< VideoFramesStream >( folder )
				);
		}

		this->fullpath() = dir.absolutePath();
		this->isOpen() = true;
	}

	bool	Project::isOpen	() const { return this->_isOpen; }
	bool&	Project::isOpen	() 		 { return this->_isOpen; }

	void Project::close	()
	{
		for ( auto& stream : this->streams() )
			stream.second->close();

		this->isOpen() = false;
	}

	// --------------------------------------------------
	// READ

	FramesMap Project::getNextFrames ()
	{
		FramesMap frames;
		for ( auto& streamEntry : this->streams() )
			frames.emplace(
				streamEntry.first,
				streamEntry.second->getNextFrame()
			);
		return frames;
	}

	FramesMap Project::getPreviousFrames ()
	{
		FramesMap frames;
		for ( auto& streamEntry : this->streams() )
			frames.emplace(
				streamEntry.first,
				streamEntry.second->getPreviousFrame()
			);
		return frames;
	}

	FramesMap Project::getCurrentFrames ()
	{
		FramesMap frames;
		for ( auto& streamEntry : this->streams() )
			frames.emplace(
				streamEntry.first,
				streamEntry.second->getCurrentFrame()
			);
		return frames;
	}

	FramesMap Project::getFramesAt ( Int64 timestamp )
	{
		FramesMap frames;
		for ( auto& streamEntry : this->streams() )
			frames.emplace(
				streamEntry.first,
				streamEntry.second->getFrameAt(timestamp)
			);
		return frames;
	}

	Frame Project::getFrameOf ( const QString& name, Int64 timeStamp ) const
	{
		Frame frame;
		if ( this->getStream(name) )
			frame = this->getStream(name)->getFrameAt( (timeStamp == -1 ? this->getStream(name)->currentTimeStamp() : timeStamp) );
		return frame;
	}

	// --------------------------------------------------
	// WRITE

	void Project::setFramesAt ( Int64 timestamp, const FramesMap& frames )
	{
		// for each pair 'streamName' + 'Frame'
		for ( auto& entry : frames )
		{
			// if a stream with name 'streamName' (entry.first) exists
			if ( this->streams().find(entry.first) != this->streams().end() )
			{
				// Set it's 'Frame'
				this->streams().at(entry.first)->setFrameAt( timestamp, entry.second );
			}
		}
	}

	void Project::setFrameOf ( const QString& name, const Frame& frame, Int64 timeStamp )
	{
		if ( this->getStream(name) )
			this->getStream(name)->setFrameAt( (timeStamp == -1 ? this->getStream(name)->currentTimeStamp() : timeStamp), frame );

	}

	void Project::setCurrentFrames ( const FramesMap& frames )
	{
		// for each pair 'streamName' + 'Frame'
		for ( auto& entry : frames )
		{
			// if a stream with name 'streamName' (entry.first) exists
			if ( this->streams().find(entry.first) != this->streams().end() )
			{
				// Set it's 'Frame'
				this->streams().at(entry.first)->setCurrentFrame( entry.second );
			}
		}
	}

	// --------------------------------------------------
	// ADD
	
	void Project::addFrames ( const FramesMap& frames )
	{
		// When adding the first frame, we have to keep track of the resolution
		cv::Size resolution = this->resolution();

		// for each pair 'streamName' + 'Frame'
		for ( auto& entry : frames )
		{
			// if a stream with name 'streamName' (entry.first) exists
			if ( this->streams().find(entry.first) != this->streams().end() )
			{
				// Copy the resolution of the first frame
				if ( resolution.width == 0 && resolution.height == 0 )
					resolution = cv::Size(entry.second.cols, entry.second.rows);

				// Set it's 'Frame'
				this->streams().at(entry.first)->addFrame( entry.second );
			}
		}

		// For each opened stream
		for ( auto& entry : this->streams() )
		{
			// If no frames has been assign to this stream
			if ( frames.find(entry.first) == frames.end() )
			{
				// Add a blank frame
				entry.second->addFrame( Frame(0,0, cv::Mat(resolution, this->format(), cv::Scalar(0,0,0))) );
			}
		}
	}

	// END CLASS PROJECT
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_PROJECT_CPP
// ##################################################