// ##################################################
// ###             TOBII_RECORDER_CPP             ###
// ##################################################
#ifndef TOBII_RECORDER_CPP
#define TOBII_RECORDER_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Recorder.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###                CLASS RECORDER              ###
	// ##################################################
	
	Recorder::Recorder ( Int64 windowSize, bool keepAllData, bool forceInterpolation, bool forceComputingDistance, QObject* parent )
	: Synchronizer( parent ), _keepAllData( keepAllData ), _forceInterpolation( forceInterpolation ), _forceComputingDistance( forceComputingDistance ), _windowSize( windowSize )
	{
		// Preset configuration, that is overriden bt the parse argument function
		this->setGazeBufferMaxSize( std::numeric_limits<Int64>::max() );
		this->setFrameBufferMaxSize( 1 );
		this->setAccelerationBufferMaxSize( 1 );
		this->setOrientationBufferMaxSize( 1 );
	}

	Recorder::~Recorder () {}

	bool							Recorder::keepAllData				() const	{ return this->_keepAllData;			}
	bool&							Recorder::keepAllData				()			{ return this->_keepAllData;			}
	bool							Recorder::forceInterpolation		() const	{ return this->_forceInterpolation;		}
	bool&							Recorder::forceInterpolation		()			{ return this->_forceInterpolation;		}
	bool							Recorder::forceComputingDistance	() const	{ return this->_forceComputingDistance;	}
	bool&							Recorder::forceComputingDistance	()			{ return this->_forceComputingDistance;	}
	Int64							Recorder::windowSize				() const	{ return this->_windowSize;				}
	Int64&							Recorder::windowSize				()			{ return this->_windowSize;				}
	Int64							Recorder::gazeBufferSize			() const	{ return this->gazeRecordings().bufferSize();			}
	Int64&							Recorder::gazeBufferSize			()			{ return this->gazeRecordings().bufferSize();			}
	Int64							Recorder::frameBufferSize			() const	{ return this->frameRecordings().bufferSize();			}
	Int64&							Recorder::frameBufferSize			()			{ return this->frameRecordings().bufferSize();			}
	Int64							Recorder::accelerationBufferSize	() const	{ return this->accelerationRecordings().bufferSize();	}
	Int64&							Recorder::accelerationBufferSize	()			{ return this->accelerationRecordings().bufferSize();	}
	Int64							Recorder::orientationBufferSize		() const	{ return this->orientationRecordings().bufferSize();	}
	Int64&							Recorder::orientationBufferSize		()			{ return this->orientationRecordings().bufferSize();	}
	const GazeRecordings&			Recorder::gazeRecordings			() const	{ return this->_gazeRecordings;			}
	GazeRecordings&					Recorder::gazeRecordings			()			{ return this->_gazeRecordings;			}
	const AccelerationRecordings&	Recorder::accelerationRecordings	() const	{ return this->_accelerationRecordings;	}
	AccelerationRecordings&			Recorder::accelerationRecordings	()			{ return this->_accelerationRecordings;	}
	const OrientationRecordings&	Recorder::orientationRecordings		() const	{ return this->_orientationRecordings;	}
	OrientationRecordings&			Recorder::orientationRecordings		()			{ return this->_orientationRecordings;	}
	const FrameRecordings&			Recorder::frameRecordings			() const	{ return this->_frameRecordings;		}
	FrameRecordings&				Recorder::frameRecordings			()			{ return this->_frameRecordings;		}

	void Recorder::setKeepAllData ( bool value )
	{
		this->keepAllData() = value;
	}

	void Recorder::setForceInterpolation ( bool value )
	{
		this->forceInterpolation() = value;
	}
	void Recorder::setForceComputingDistance ( bool  value )
	{
		this->forceComputingDistance() = value;
	}

	void Recorder::setWindowSize ( Int64 value )
	{
		this->windowSize() = value;
	}

	void Recorder::setGazeBufferMaxSize ( Int64 value )
	{
		this->gazeBufferSize() = value;
	}

	void Recorder::setFrameBufferMaxSize ( Int64 value )
	{
		this->frameBufferSize() = value ;
	}

	void Recorder::setAccelerationBufferMaxSize ( Int64 value )
	{
		this->accelerationBufferSize() = value;
	}

	void Recorder::setOrientationBufferMaxSize ( Int64 value )
	{
		this->orientationRecordings().setBufferSize( value );
	}

	void Recorder::recordData ( DataPacketPtr dataPacket )
	{
		// If the pointer isnt null
		if ( dataPacket )
		{
			// Check the status:
			if ( (this->keepAllData() == false) && (dataPacket->status() > 0) )
			 	return;

			// Synchronize
			dataPacket = this->synchronizeData( dataPacket );

			// If the synchronizer isn't set, discard the packet
			if ( this->firstFrameTimeStamp() == 0 )
			 	return;


			// Record an accelerometer data
			if ( dataPacket->name() == "ac" )
			{
				// Record it
				this->accelerationRecordings().insert(
					dynamic_cast< const AccelerometerPacket* >( dataPacket.get() )
				);
				
				// Emit it:
				emit this->accelerationRecorded(
					std::make_shared< Acceleration >( this->accelerationRecordings().at(dataPacket->timeStamp()) )
				);
			}

			// Record an orientation data
			else if ( dataPacket->name() == "gy" )
			{
				// Record it
				this->orientationRecordings().insert(
					dynamic_cast< const GyroscopePacket* >( dataPacket.get() )
				);

				// Emit it:
				emit this->orientationRecorded(
					std::make_shared< Orientation >( this->orientationRecordings().at(dataPacket->timeStamp()) )
				);
			}

			// Record a gaze fixation data
			else
			{
				// Record it
				this->gazeRecordings().insert( dataPacket );

				// Emit it
				emit this->gazeRecorded(
					std::make_shared< GazeFixation >( this->gazeRecordings().at(dataPacket->timeStamp()) )
				);
			}
		}
	}

	void Recorder::recordFrame ( FramePtr frame )
	{
		// Record the frame
		this->frameRecordings().insert( *frame );

		// Interpolate the gaze fixation
		GazeFixationPtr ptrFixation = std::make_shared< GazeFixation >(
			Interpolate::live(
				this->gazeRecordings(),
				frame->timeStamp(),
				this->windowSize(),
				this->forceInterpolation(),
				this->forceComputingDistance()
			)
		);

		// Record it as well
		this->gazeRecordings().insert( *ptrFixation );

		// Emit the frame and the interpolated gaze,
		// for a live application to use it
		emit this->frameInterpolated( frame, ptrFixation );
	}

	// END CLASS RECORDER
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_RECORDER_CPP
// ##################################################