// ##################################################
// ###              TOBII_SALIENCY_CPP            ###
// ##################################################
#ifndef TOBII_SALIENCY_CPP
#define TOBII_SALIENCY_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Saliency.hpp"
#include <iostream>
#include <vector>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###             NAMESPACE SALIENCY             ###
	// ##################################################	
	namespace Saliency {

		float sigmaAtResolution ( const cv::Size& resolution )
		{
			return std::max( resolution.width/2.0f*std::tan(alpha * PI / 180.0f)/std::tan(beta * PI / 180.0f), 0.0f );
		}

		float computeSigma ( const GazeFixation& fixation, cv::Size resolution )
		{
			return std::max( sigmaAtResolution(resolution)*A/(fixation.distance() + epsilon), 0.0f );
		}
			
		Frame computeSaliencyMap ( const GazeFixation& fixation, cv::Size resolution )
		{
			// Create zero saliency map
			Frame saliencyMap( fixation.timeStamp(), 0, cv::Mat::zeros(resolution, CV_8UC1) );

			// Check if the fixation is complete
			if ( fixation.isFieldSet( Tobii::GazeFixation::ValidityFlag::POSITION_2D | Tobii::GazeFixation::ValidityFlag::DISTANCE ) )
			{
				// Read frame dimension
				int width  = resolution.width;
				int height = resolution.height;
				int width_2  = std::floor( width/2.0);
				int height_2 = std::floor(height/2.0);

				// Read fixation
				Tobii::Point2 center = fixation.position2D();
				int center_x = std::floor(center.x() * width  );
				int center_y = std::floor(center.y() * height );

				// Compute sigma
				double sigma = computeSigma( fixation, resolution );

				// Compute the saliency
				cv::Mat gaussian1DX, gaussian1DY;

				gaussian1DX = cv::getGaussianKernel( height*3, sigma, CV_32FC1 );
				gaussian1DY = cv::getGaussianKernel( width*3,  sigma, CV_32FC1 );

				gaussian1DX = gaussian1DX.rowRange( height + height_2 - center_y, 2*height + height_2 - center_y );
				gaussian1DY = gaussian1DY.rowRange(  width + width_2  - center_x, 2*width  + width_2  - center_x );

				saliencyMap = gaussian1DX * gaussian1DY.t();

				// Normalize it, scale its dynamic range, and convert to unsigedn 64-bits floats
				double min(0), max(0);
				cv::minMaxLoc( saliencyMap, &min, &max );
				saliencyMap.convertTo( saliencyMap, CV_8UC1, 255/max );
			}

			return saliencyMap;
		}

		Frame computeSaliencyMapGPU ( const GazeFixation& fixation, cv::Size resolution )
		{
			// Create zero saliency map
			Frame saliencyMap( fixation.timeStamp(), 0, cv::Mat::zeros(resolution, CV_8UC1) );

			// Check if the fixation is complete
			if ( fixation.isFieldSet( Tobii::GazeFixation::ValidityFlag::POSITION_2D | Tobii::GazeFixation::ValidityFlag::DISTANCE ) )
			{
				// Read frame dimension
				int width  = resolution.width;
				int height = resolution.height;
				int width_2  = std::floor( width/2.0);
				int height_2 = std::floor(height/2.0);

				// Read fixation
				Tobii::Point2 center = fixation.position2D();
				int center_x = std::floor(center.x() * width  );
				int center_y = std::floor(center.y() * height );

				// Compute sigma
				double sigma = computeSigma( fixation, resolution );

				// Compute the saliency
				cv::Mat gaussian1DX, gaussian1DY;

				gaussian1DX = cv::getGaussianKernel( height*3, sigma, CV_32FC1 );
				gaussian1DY = cv::getGaussianKernel( width*3,  sigma, CV_32FC1 );

				gaussian1DX = gaussian1DX.rowRange( height + height_2 - center_y, 2*height + height_2 - center_y );
				gaussian1DY = gaussian1DY.rowRange(  width + width_2  - center_x, 2*width  + width_2  - center_x );

				// ---------------------------------------------------------------
				// GPU
				static cv::gpu::GpuMat gpuGaussian1DX, gpuGaussian1DY, gpuSaliency;

				// Upload gaussian X, and Y
				gpuGaussian1DX.upload( gaussian1DX );
				gpuGaussian1DY.upload( gaussian1DY );

				// this is equivalent to: gpuSaliency = 1.0 * gpuGaussian1DY * ( gpuGaussian1DY.t() ) + 0.0*cv::gpu::GpuMat()
				// see http://docs.opencv.org/2.4/modules/gpu/doc/operations_on_matrices.html
				cv::gpu::gemm( gpuGaussian1DX, gpuGaussian1DY, 1.0, cv::gpu::GpuMat(), 0.0, gpuSaliency, cv::GEMM_2_T );

				// Normalize it, scale its dynamic range, and convert to unsigedn 64-bits floats
				double min(0), max(0);
				cv::gpu::minMax( gpuSaliency, &min, &max );
				gpuSaliency.convertTo( gpuSaliency, CV_8UC1, 255/max );

				// Dowload back
				gpuSaliency.download( saliencyMap );
			}

			return saliencyMap;
		}

    //***********************************************************
    //    Functions to create LUT, found in OpenCV sources
    //***********************************************************

    static cv::Mat linspace(float x0, float x1, int n)
    {
        cv::Mat pts(n, 1, CV_32FC1);
        float step = (x1-x0)/(n-1);
        for(int i = 0; i < n; i++)
            pts.at<float>(i,0) = x0+i*step;
        return pts;
    }

    static void sortMatrixRowsByIndices(cv::InputArray _src, cv::InputArray _indices, cv::OutputArray _dst)
    {
      if(_indices.getMat().type() != CV_32SC1)
          CV_Error(CV_StsUnsupportedFormat, "cv::sortRowsByIndices only works on integer indices!");
      cv::Mat src = _src.getMat();
      std::vector<int> indices = _indices.getMat();
      _dst.create(src.rows, src.cols, src.type());
      cv::Mat dst = _dst.getMat();
      for(size_t idx = 0; idx < indices.size(); idx++) {
          cv::Mat originalRow = src.row(indices[idx]);
          cv::Mat sortedRow = dst.row((int)idx);
          originalRow.copyTo(sortedRow);
      }
    }

    static cv::Mat sortMatrixRowsByIndices(cv::InputArray src, cv::InputArray indices)
    {
        cv::Mat dst;
        sortMatrixRowsByIndices(src, indices, dst);
        return dst;
    }

    static cv::Mat argsort(cv::InputArray _src, bool ascending=true)
    {
        cv::Mat src = _src.getMat();
        if (src.rows != 1 && src.cols != 1)
            CV_Error(CV_StsBadArg, "cv::argsort only sorts 1D matrices.");
        int flags = CV_SORT_EVERY_ROW+(ascending ? CV_SORT_ASCENDING : CV_SORT_DESCENDING);
        cv::Mat sorted_indices;
        sortIdx(src.reshape(1,1),sorted_indices,flags);
        return sorted_indices;
    }

    template <typename _Tp> static
    cv::Mat interp1_(const cv::Mat& X_, const cv::Mat& Y_, const cv::Mat& XI)
    {
        int n = XI.rows;
        // sort input table
        std::vector<int> sort_indices = argsort(X_);

        cv::Mat X = sortMatrixRowsByIndices(X_,sort_indices);
        cv::Mat Y = sortMatrixRowsByIndices(Y_,sort_indices);
        // interpolated values
        cv::Mat yi = cv::Mat::zeros(XI.size(), XI.type());
        for(int i = 0; i < n; i++) {
            int c = 0;
            int low = 0;
            int high = X.rows - 1;
            // set bounds
            if(XI.at<_Tp>(i,0) < X.at<_Tp>(low, 0))
                high = 1;
            if(XI.at<_Tp>(i,0) > X.at<_Tp>(high, 0))
                low = high - 1;
            // binary search
            while((high-low)>1) {
                c = low + ((high - low) >> 1);
                if(XI.at<_Tp>(i,0) > X.at<_Tp>(c,0)) {
                    low = c;
                } else {
                    high = c;
                }
            }
            // linear interpolation
            yi.at<_Tp>(i,0) += Y.at<_Tp>(low,0)
            + (XI.at<_Tp>(i,0) - X.at<_Tp>(low,0))
            * (Y.at<_Tp>(high,0) - Y.at<_Tp>(low,0))
            / (X.at<_Tp>(high,0) - X.at<_Tp>(low,0));
        }
        return yi;
    }

    static cv::Mat interp1(cv::InputArray _x, cv::InputArray _Y, cv::InputArray _xi)
    {
        // get matrices
        cv::Mat x = _x.getMat();
        cv::Mat Y = _Y.getMat();
        cv::Mat xi = _xi.getMat();
        // check types & alignment
        assert((x.type() == Y.type()) && (Y.type() == xi.type()));
        assert((x.cols == 1) && (x.rows == Y.rows) && (x.cols == Y.cols));
        // call templated interp1
        switch(x.type()) {
            case CV_8SC1: return interp1_<char>(x,Y,xi); break;
            case CV_8UC1: return interp1_<unsigned char>(x,Y,xi); break;
            case CV_16SC1: return interp1_<short>(x,Y,xi); break;
            case CV_16UC1: return interp1_<unsigned short>(x,Y,xi); break;
            case CV_32SC1: return interp1_<int>(x,Y,xi); break;
            case CV_32FC1: return interp1_<float>(x,Y,xi); break;
            case CV_64FC1: return interp1_<double>(x,Y,xi); break;
            default: CV_Error(CV_StsUnsupportedFormat, ""); break;
        }
        return cv::Mat();
    }


   static cv::Mat linear_colormap(cv::InputArray X,
            cv::InputArray r, cv::InputArray g, cv::InputArray b,
            cv::InputArray xi) {
        cv::Mat lut, lut8;
        cv::Mat planes[] = {
                interp1(X, b, xi),
                interp1(X, g, xi),
                interp1(X, r, xi)};
        merge(planes, 3, lut);
        lut.convertTo(lut8, CV_8U, 255.);
        return lut8;
    }

    static cv::Mat getLutColorMapJet() {
      cv::Mat lut;
      cv::Mat X = linspace(0,1,256);
      // define the basemap
      float r[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.00588235294117645,0.02156862745098032,0.03725490196078418,0.05294117647058827,0.06862745098039214,0.084313725490196,0.1000000000000001,0.115686274509804,0.1313725490196078,0.1470588235294117,0.1627450980392156,0.1784313725490196,0.1941176470588235,0.2098039215686274,0.2254901960784315,0.2411764705882353,0.2568627450980392,0.2725490196078431,0.2882352941176469,0.303921568627451,0.3196078431372549,0.3352941176470587,0.3509803921568628,0.3666666666666667,0.3823529411764706,0.3980392156862744,0.4137254901960783,0.4294117647058824,0.4450980392156862,0.4607843137254901,0.4764705882352942,0.4921568627450981,0.5078431372549019,0.5235294117647058,0.5392156862745097,0.5549019607843135,0.5705882352941174,0.5862745098039217,0.6019607843137256,0.6176470588235294,0.6333333333333333,0.6490196078431372,0.664705882352941,0.6803921568627449,0.6960784313725492,0.7117647058823531,0.7274509803921569,0.7431372549019608,0.7588235294117647,0.7745098039215685,0.7901960784313724,0.8058823529411763,0.8215686274509801,0.8372549019607844,0.8529411764705883,0.8686274509803922,0.884313725490196,0.8999999999999999,0.9156862745098038,0.9313725490196076,0.947058823529412,0.9627450980392158,0.9784313725490197,0.9941176470588236,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0.9862745098039216,0.9705882352941178,0.9549019607843139,0.93921568627451,0.9235294117647062,0.9078431372549018,0.892156862745098,0.8764705882352941,0.8607843137254902,0.8450980392156864,0.8294117647058825,0.8137254901960786,0.7980392156862743,0.7823529411764705,0.7666666666666666,0.7509803921568627,0.7352941176470589,0.719607843137255,0.7039215686274511,0.6882352941176473,0.6725490196078434,0.6568627450980391,0.6411764705882352,0.6254901960784314,0.6098039215686275,0.5941176470588236,0.5784313725490198,0.5627450980392159,0.5470588235294116,0.5313725490196077,0.5156862745098039,0.5};
      float g[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0.001960784313725483,0.01764705882352935,0.03333333333333333,0.0490196078431373,0.06470588235294117,0.08039215686274503,0.09607843137254901,0.111764705882353,0.1274509803921569,0.1431372549019607,0.1588235294117647,0.1745098039215687,0.1901960784313725,0.2058823529411764,0.2215686274509804,0.2372549019607844,0.2529411764705882,0.2686274509803921,0.2843137254901961,0.3,0.3156862745098039,0.3313725490196078,0.3470588235294118,0.3627450980392157,0.3784313725490196,0.3941176470588235,0.4098039215686274,0.4254901960784314,0.4411764705882353,0.4568627450980391,0.4725490196078431,0.4882352941176471,0.503921568627451,0.5196078431372548,0.5352941176470587,0.5509803921568628,0.5666666666666667,0.5823529411764705,0.5980392156862746,0.6137254901960785,0.6294117647058823,0.6450980392156862,0.6607843137254901,0.6764705882352942,0.692156862745098,0.7078431372549019,0.723529411764706,0.7392156862745098,0.7549019607843137,0.7705882352941176,0.7862745098039214,0.8019607843137255,0.8176470588235294,0.8333333333333333,0.8490196078431373,0.8647058823529412,0.8803921568627451,0.8960784313725489,0.9117647058823528,0.9274509803921569,0.9431372549019608,0.9588235294117646,0.9745098039215687,0.9901960784313726,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0.9901960784313726,0.9745098039215687,0.9588235294117649,0.943137254901961,0.9274509803921571,0.9117647058823528,0.8960784313725489,0.8803921568627451,0.8647058823529412,0.8490196078431373,0.8333333333333335,0.8176470588235296,0.8019607843137253,0.7862745098039214,0.7705882352941176,0.7549019607843137,0.7392156862745098,0.723529411764706,0.7078431372549021,0.6921568627450982,0.6764705882352944,0.6607843137254901,0.6450980392156862,0.6294117647058823,0.6137254901960785,0.5980392156862746,0.5823529411764707,0.5666666666666669,0.5509803921568626,0.5352941176470587,0.5196078431372548,0.503921568627451,0.4882352941176471,0.4725490196078432,0.4568627450980394,0.4411764705882355,0.4254901960784316,0.4098039215686273,0.3941176470588235,0.3784313725490196,0.3627450980392157,0.3470588235294119,0.331372549019608,0.3156862745098041,0.2999999999999998,0.284313725490196,0.2686274509803921,0.2529411764705882,0.2372549019607844,0.2215686274509805,0.2058823529411766,0.1901960784313728,0.1745098039215689,0.1588235294117646,0.1431372549019607,0.1274509803921569,0.111764705882353,0.09607843137254912,0.08039215686274526,0.06470588235294139,0.04901960784313708,0.03333333333333321,0.01764705882352935,0.001960784313725483,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
      float b[] = {0.5,0.5156862745098039,0.5313725490196078,0.5470588235294118,0.5627450980392157,0.5784313725490196,0.5941176470588235,0.6098039215686275,0.6254901960784314,0.6411764705882352,0.6568627450980392,0.6725490196078432,0.6882352941176471,0.7039215686274509,0.7196078431372549,0.7352941176470589,0.7509803921568627,0.7666666666666666,0.7823529411764706,0.7980392156862746,0.8137254901960784,0.8294117647058823,0.8450980392156863,0.8607843137254902,0.8764705882352941,0.892156862745098,0.907843137254902,0.9235294117647059,0.9392156862745098,0.9549019607843137,0.9705882352941176,0.9862745098039216,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0.9941176470588236,0.9784313725490197,0.9627450980392158,0.9470588235294117,0.9313725490196079,0.915686274509804,0.8999999999999999,0.884313725490196,0.8686274509803922,0.8529411764705883,0.8372549019607844,0.8215686274509804,0.8058823529411765,0.7901960784313726,0.7745098039215685,0.7588235294117647,0.7431372549019608,0.7274509803921569,0.7117647058823531,0.696078431372549,0.6803921568627451,0.6647058823529413,0.6490196078431372,0.6333333333333333,0.6176470588235294,0.6019607843137256,0.5862745098039217,0.5705882352941176,0.5549019607843138,0.5392156862745099,0.5235294117647058,0.5078431372549019,0.4921568627450981,0.4764705882352942,0.4607843137254903,0.4450980392156865,0.4294117647058826,0.4137254901960783,0.3980392156862744,0.3823529411764706,0.3666666666666667,0.3509803921568628,0.335294117647059,0.3196078431372551,0.3039215686274508,0.2882352941176469,0.2725490196078431,0.2568627450980392,0.2411764705882353,0.2254901960784315,0.2098039215686276,0.1941176470588237,0.1784313725490199,0.1627450980392156,0.1470588235294117,0.1313725490196078,0.115686274509804,0.1000000000000001,0.08431372549019622,0.06862745098039236,0.05294117647058805,0.03725490196078418,0.02156862745098032,0.00588235294117645,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
        // now build lookup table
      lut = linear_colormap(X,
                 cv::Mat(256,1, CV_32FC1, r).clone(), // red
                 cv::Mat(256,1, CV_32FC1, g).clone(), // green
                 cv::Mat(256,1, CV_32FC1, b).clone(), // blue
                 linspace(0,1,256));

      return lut;
   }

    //***********************************************************
    //    ComputeHeatMapGPU
    //***********************************************************


	Frame	computeHeatMapGPU ( const Frame& image, const Frame& saliencyMap, double alpha )
	{
		// Compute the heat-map
		Frame heatMap;
		static cv::gpu::GpuMat gpuHeatMap, gpuSrc, gpuImage, gpuIdentity;

		// Create LUT with color cv::COLORMAP_JET
		static Frame lut;
		//If uninitialized because lut is __static__
		if (lut.rows == 0) {
			lut = getLutColorMapJet();
		}

		cv::Mat src = ((cv::InputArray) saliencyMap).getMat();

		cv::cvtColor(src.clone(), src, CV_GRAY2BGR);

		gpuSrc.upload( src );
		gpuImage.upload( image );

		// Apply the ColorMap.
		cv::gpu::LUT(gpuSrc, lut, gpuHeatMap);

		// Mix with the frame
		//heatMap = (alpha * image) + ((1-alpha) * heatMap);

		static cv::gpu::GpuMat gpuChannelsImage[3];
		static cv::gpu::GpuMat gpuChannelsHeatMap[3];

		//Split image in 3 channels
		cv::gpu::split(gpuImage, gpuChannelsImage);
		cv::gpu::split(gpuHeatMap, gpuChannelsHeatMap);

		gpuChannelsImage[0].convertTo(gpuChannelsImage[0], CV_32FC1);
		gpuChannelsImage[1].convertTo(gpuChannelsImage[1], CV_32FC1);
		gpuChannelsImage[2].convertTo(gpuChannelsImage[2], CV_32FC1);

		gpuChannelsHeatMap[0].convertTo(gpuChannelsHeatMap[0], CV_32FC1);
		gpuChannelsHeatMap[1].convertTo(gpuChannelsHeatMap[1], CV_32FC1);
		gpuChannelsHeatMap[2].convertTo(gpuChannelsHeatMap[2], CV_32FC1);

		gpuIdentity.upload( Frame::eye(gpuChannelsImage[0].rows, gpuChannelsImage[0].rows, CV_32FC1) );

		//Apply gemm on each channel
		cv::gpu::gemm(gpuIdentity, gpuChannelsImage[0], alpha, gpuChannelsHeatMap[0], (1.0-alpha), gpuChannelsHeatMap[0]);
		cv::gpu::gemm(gpuIdentity, gpuChannelsImage[1], alpha, gpuChannelsHeatMap[1], (1.0-alpha), gpuChannelsHeatMap[1]);
		cv::gpu::gemm(gpuIdentity, gpuChannelsImage[2], alpha, gpuChannelsHeatMap[2], (1.0-alpha), gpuChannelsHeatMap[2]);

		gpuChannelsHeatMap[0].convertTo(gpuChannelsHeatMap[0], CV_8UC1 );
		gpuChannelsHeatMap[1].convertTo(gpuChannelsHeatMap[1], CV_8UC1 );
		gpuChannelsHeatMap[2].convertTo(gpuChannelsHeatMap[2], CV_8UC1 );

		//merge heatmap
		cv::gpu::merge(gpuChannelsHeatMap, 3, gpuHeatMap);

		gpuHeatMap.download( heatMap );

		heatMap.convertTo( heatMap, CV_8UC3 );
		Frame frameHeatMap(heatMap);
		return frameHeatMap;
	}

	Frame	computeHeatMap ( const Frame& image, const Frame& saliencyMap, double alpha )
	{
		Frame saliency01 = saliencyMap;
		if ( saliency01.type() != CV_8UC1 )
			cv::cvtColor( saliency01, saliency01, CV_BGR2GRAY );

		// Compute the heat-map
		Frame heatMap;

		// Create LUT with color cv::COLORMAP_JET
		static Frame lut;
		//If uninitialized because lut is __static__
		if (lut.rows == 0)
			lut = getLutColorMapJet();

		cv::Mat src = ((cv::InputArray) saliency01).getMat();

		cv::cvtColor(src.clone(), src, CV_GRAY2BGR);
		// Apply the ColorMap.
		cv::LUT(src, lut, heatMap);

		// Mix with the frame
		heatMap = alpha * image + (1-alpha) * heatMap;
		heatMap.convertTo( heatMap, CV_8UC3 );

		return heatMap;
	}

  //   Old version of computeHeatMap....
  /*
		Frame computeHeatMap ( const Frame& frame, const Frame& saliencyMap, double alpha )
		{
			// Compute the heat-map
			Frame heatMap( frame.timeStamp(), 0 );
			cv::applyColorMap( saliencyMap, heatMap, cv::COLORMAP_JET );

			// Mix with the frame
			heatMap = alpha * frame + (1-alpha) * heatMap;
			heatMap.convertTo( heatMap, CV_8UC3 );

			return heatMap;
		}
  */

		Frame computeWeightedMap ( const Frame& frame, const Frame& saliencyMap )
		{
			cv::Mat imageHLS;
			cv::cvtColor( frame, imageHLS, CV_BGR2HLS );

			std::vector< cv::Mat > channelsHLS;
			cv::split( imageHLS, channelsHLS );

			cv::Mat saliency01 = saliencyMap;
			if ( saliency01.type() != CV_8UC1 )
				cv::cvtColor( saliency01, saliency01, CV_BGR2GRAY );

			saliency01.convertTo( saliency01, CV_32FC1, 1.0f/255.0f );
			channelsHLS.at(1).convertTo( channelsHLS.at(1), CV_32FC1 );

			cv::multiply( channelsHLS.at(1), saliency01, channelsHLS.at(1) );

			channelsHLS.at(1).convertTo( channelsHLS.at(1), CV_8UC1 );

			Frame output( frame.timeStamp(), 0 );
			cv::merge( channelsHLS, output );
			cv::cvtColor( output, output, CV_HLS2BGR );

			return output;
		}

		Frame threshold ( const Frame& saliency, double threshold )
		{
			Frame thresholded = saliency;
			if ( thresholded.type() != CV_8UC1 )
			{
				cv::cvtColor( thresholded, thresholded, CV_BGR2GRAY );
				cv::threshold( thresholded, thresholded, threshold, 255, cv::THRESH_BINARY );
				cv::cvtColor( thresholded, thresholded, CV_GRAY2BGR );
			}
			else
				cv::threshold( thresholded, thresholded, threshold, 255, cv::THRESH_BINARY );
			return thresholded;
		}


	} // END CLASS NAMESPACE SALIENCY
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_SALIENCY_CPP
// ##################################################
