// ##################################################
// ###          VIDEOFILESTREAMER_CPP             ###
// ##################################################
#ifndef VIDEOFILESTREAMER_CPP
#define VIDEOFILESTREAMER_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "VideoFileStreamer.hpp"
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###           CLASS VIDEOFILESTREAMER          ###
	// ##################################################

	VideoFileStreamer::VideoFileStreamer ( const cv::Size& resolution, QObject* parent )
	: QThread( parent ), _continueReading( false ), _resolution( resolution )
	{}

	VideoFileStreamer::VideoFileStreamer ( UInt8 width, UInt8 height, QObject* parent )
	: QThread( parent ), _continueReading( false ), _resolution( width, height )
	{}

	VideoFileStreamer::~VideoFileStreamer ()
	{
	 	this->stopReading();
	}

	bool VideoFileStreamer::hasFinished () const
	{
		return (this->continueReading() == false);
	}

	void VideoFileStreamer::run ()
	{
		while ( this->continueReading() )
		{
			// Get the next frame
			FramePtr frame = std::make_shared< Frame >( this->videoStream().getNextFrame() );

			if ( frame->cols>0 && frame->rows>0 )
			{
				if ( this->resolution().area() != 0 )
					cv::resize( *frame, *frame, this->resolution() );

				emit this->frameAvailable( frame );
			}
			else
				this->continueReading() = false;

			QThread::msleep(40);
		}
		return this->quit();
	}

	const InputVideoStream&	VideoFileStreamer::videoStream 		() const	{ return this->_videoStream;		}
	InputVideoStream&		VideoFileStreamer::videoStream 		()			{ return this->_videoStream;		}
	bool					VideoFileStreamer::continueReading	() const	{ return this->_continueReading;	}
	bool&					VideoFileStreamer::continueReading	()			{ return this->_continueReading;	}
	const cv::Size&			VideoFileStreamer::resolution		() const	{ return this->_resolution;			}
	cv::Size&				VideoFileStreamer::resolution		()			{ return this->_resolution;			}

	void VideoFileStreamer::startReading ( const QString& fileName )
	{
		if ( fileName.isEmpty() == false )
		{
			// Open the video file
			this->videoStream().open( fileName );
			if ( ! this->videoStream().isOpen() )
			{
				std::cerr << "Could not open video \"" << fileName.toStdString() << "\" !" << std::endl;
				return;
			}

			// Set the reader flag
			this->continueReading() = true;

			// Launch the thread
			this->start();
		}
	}

	void VideoFileStreamer::stopReading ()
	{
		if ( this->continueReading() == true )
		{
			this->continueReading() = false;
			this->wait();
		}

		if ( this->videoStream().isOpen() )
			this->videoStream().close();
	}

	void VideoFileStreamer::finishReading ()
	{
		this->wait();
	}

	// END CLASS VIDEOFILESTREAMER
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // VIDEOFILESTREAMER_CPP
// ##################################################
