// ##################################################
// ###               FILEDEVICE_CPP               ###
// ##################################################
#ifndef FILEDEVICE_CPP
#define FILEDEVICE_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "FileDevice.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###               CLASS FILEDEVICE             ###
	// ##################################################
	
	FileDevice::FileDevice ( const cv::Size& resolution, QObject* parent )
	: Device( parent ), _dataStreamer( NULL ), _videoStreamer( resolution, NULL )
	{
		connect( &this->dataStreamer(),  SIGNAL( dataAvailable(std::string)), this, SLOT(          parseData(std::string)), Qt::DirectConnection );
		connect( &this->dataStreamer(),  SIGNAL(      finished()           ), this, SLOT( dataReaderFinished()           ), Qt::DirectConnection );
		connect( &this->videoStreamer(), SIGNAL(frameAvailable(FramePtr)   ), this, SLOT(          frameRead(FramePtr)   ), Qt::DirectConnection );
		connect( &this->videoStreamer(), SIGNAL(      finished()           ), this, SLOT(videoReaderFinished()           ), Qt::DirectConnection );
	}

	FileDevice::FileDevice ( UInt8 width, UInt8 height, QObject* parent )
	: Device( parent ), _dataStreamer( NULL ), _videoStreamer( width, height, NULL )
	{
		connect( &this->dataStreamer(),  SIGNAL( dataAvailable(std::string)), this, SLOT(          parseData(std::string)), Qt::DirectConnection );
		connect( &this->dataStreamer(),  SIGNAL(      finished()           ), this, SLOT( dataReaderFinished()           ), Qt::DirectConnection );
		connect( &this->videoStreamer(), SIGNAL(frameAvailable(FramePtr)   ), this, SLOT(          frameRead(FramePtr)   ), Qt::DirectConnection );
		connect( &this->videoStreamer(), SIGNAL(      finished()           ), this, SLOT(videoReaderFinished()           ), Qt::DirectConnection );
	}

	FileDevice::~FileDevice ()
	{
		this->stopReading();
	}

	bool FileDevice::hasFinished () const
	{
		return ( this->dataStreamer().hasFinished() && this->videoStreamer().hasFinished() );
	}

	const cv::Size&	FileDevice::resolution	() const { return this->videoStreamer().resolution();	}
	cv::Size&		FileDevice::resolution	()		 { return this->videoStreamer().resolution();	}

	const DataFileStreamer&		FileDevice::dataStreamer  () const	{ return this->_dataStreamer;	}
	DataFileStreamer&			FileDevice::dataStreamer  ()		{ return this->_dataStreamer;	}
	const VideoFileStreamer&	FileDevice::videoStreamer () const	{ return this->_videoStreamer;	}
	VideoFileStreamer&			FileDevice::videoStreamer ()		{ return this->_videoStreamer;	}

	void FileDevice::startReading ( const QString& dataFileName, const QString& videoFileName )
	{
		this->dataStreamer().startReading( dataFileName );
		this->videoStreamer().startReading( videoFileName );
	}

	void FileDevice::stopReading ()
	{
		this->dataStreamer().stopReading();
		this->videoStreamer().stopReading();
		emit this->finished();
	}

	void FileDevice::finishReading ()
	{
		while ( !this->dataStreamer().hasFinished() || !this->videoStreamer().hasFinished() )
		{
			QThread::msleep(10);
		}
		emit this->finished();
	}

	void FileDevice::frameRead ( FramePtr frame )
	{
		emit this->frameAvailable(frame);
	}

	void FileDevice::videoReaderFinished ()
	{
		if ( this->dataStreamer().hasFinished() )
			emit this->finished();
	}

	void FileDevice::dataReaderFinished  ()
	{
		if ( this->videoStreamer().hasFinished() )
			emit this->finished();
	}

	// END CLASS FILEDEVICE
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // FILEDEVICE_HPP
// ##################################################