// ##################################################
// ###            TOBII_DATAPARSER_CPP            ###
// ##################################################
#ifndef TOBII_DATAPARSER_CPP
#define TOBII_DATAPARSER_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "DataParser.hpp"
#include <iostream>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###              CLASS DATAPARSER              ###
	// ##################################################

	DataParser::DataParser ( QObject* parent )
	: QObject( parent )
	{
		qRegisterMetaType< DataPacketPtr >( "DataPacketPtr" );
	}

	DataParser::~DataParser () {}

	void DataParser::parseData ( const std::string& data ) const
	{
		rapidjson::Document json;
		
		if ( ! json.Parse<0>( data.c_str() ).HasParseError() )
		{
			this->parseDataPacket( json );
			return;
		}
		
		std::cerr << "Parse error, packet= " << data << std::endl;
	}

	DataPacketPtr DataParser::parseDataPacket ( const rapidjson::Document& json ) const
	{
		// Parse-In the time stamp and the status that each packet posses
		Int64 timeStamp(0);
		UInt8 status(0);
		if (( json.HasMember("ts") && json["ts"].IsInt64() )
		&&  ( json.HasMember("s") && json["s"].IsUint() ))
		{
			timeStamp = json["ts"].GetInt64();
			status    = json["s"].GetUint();
		}
		// If one of the fields is not valid:
		else
			return DataPacketPtr();

		// Parse In the specific packet type:
		DataPacketPtr packet;
		
		// Case of a 'pc' packet
		if ( json.HasMember("pc") && json.HasMember("eye") )
			packet = this->parsePupilCenterPacket( timeStamp, status, json );

		// Case of a 'pd' packet
		else if ( json.HasMember("pd") && json.HasMember("eye") )
			packet = this->parsePupilDiameterPacket( timeStamp, status, json );

		// Case of a 'gd' packet
		else if ( json.HasMember("gd") && json.HasMember("eye") )
			packet = this->parseGazeDirectionPacket( timeStamp, status, json );

		// Case of a 'gp' packet
		else if ( json.HasMember("gp") )
			packet = this->parseGaze2DPositionPacket( timeStamp, status, json );

		// Case of a 'gp3' packet
		else if ( json.HasMember("gp3") )
			packet = this->parseGaze3DPositionPacket( timeStamp, status, json );

		// Case of a 'gy' packet
		else if ( json.HasMember("gy") )
			packet = this->parseGyroscopePacket( timeStamp, status, json );

		// Case of a 'ac' packet
		else if ( json.HasMember("ac") )
			packet = this->parseAccelerometerPacket( timeStamp, status, json );
		
		// Case of a 'pts' packet
		else if ( json.HasMember("pts") && json.HasMember("pv") )
			packet = this->parsePTSSyncPacket( timeStamp, status, json );

		// Case of a 'pts' packet
		else if ( json.HasMember("vts") )
			packet = this->parseVTSSyncPacket( timeStamp, status, json );

		// Emit the packet for recording
		emit this->dataAvailable( packet );
		return packet;
	}

	PupilCenterPacketPtr DataParser::parsePupilCenterPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if (( json.HasMember("pc")  && json["pc"].IsArray()   )
		&&  ( json.HasMember("eye") && json["eye"].IsString() ))
		{
			// Which Eye
			std::string eye = json["eye"].GetString();

			// The Point3
			const rapidjson::Value& point = json["pc"];
			if ( point.Size() == 3u && point[0u].IsDouble() && point[1u].IsDouble() && point[2u].IsDouble() )
			{
				return std::make_shared< PupilCenterPacket >(
					timeStamp, statuts,
					Point3( point[0u].GetDouble(), point[1u].GetDouble(), point[2u].GetDouble() ),
					(eye=="left" ? EyeDataPacket::Eye::LEFT : EyeDataPacket::Eye::RIGHT)
				);
			}
		}
		return PupilCenterPacketPtr();
	}

	PupilDiameterPacketPtr DataParser::parsePupilDiameterPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if (( json.HasMember("pd")  && json["pd"].IsDouble()   )
		&&  ( json.HasMember("eye") && json["eye"].IsString() ))
		{
			return std::make_shared< PupilDiameterPacket >(
				timeStamp, statuts,
				 json["pd"].GetDouble(),
				(std::string(json["eye"].GetString())=="left" ? EyeDataPacket::Eye::LEFT : EyeDataPacket::Eye::RIGHT)
			);
		}
		return PupilDiameterPacketPtr();
	}
	
	GazeDirectionPacketPtr DataParser::parseGazeDirectionPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if (( json.HasMember("gd")  && json["gd"].IsArray()   )
		&&  ( json.HasMember("eye") && json["eye"].IsString() ))
		{
			// Which Eye
			std::string eye = json["eye"].GetString();

			// The Vector3
			const rapidjson::Value& direction = json["gd"];
			if ( direction.Size() == 3u && direction[0u].IsDouble() && direction[1u].IsDouble() && direction[2u].IsDouble() )
			{
				return std::make_shared< GazeDirectionPacket >(
					timeStamp, statuts,
					Vector3( direction[0u].GetDouble(), direction[1u].GetDouble(), direction[2u].GetDouble() ),
					(eye=="left" ? EyeDataPacket::Eye::LEFT : EyeDataPacket::Eye::RIGHT)
				);
			}
		}
		return GazeDirectionPacketPtr();
	}
	
	Gaze2DPositionPacketPtr DataParser::parseGaze2DPositionPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if ( json.HasMember("gp") && json["gp"].IsArray() )
		{
			const rapidjson::Value& position = json["gp"];

			if ( position.Size() == 2u && position[0u].IsDouble() && position[1u].IsDouble() )
			{
				return std::make_shared< Gaze2DPositionPacket >(
					timeStamp, statuts,
					Point2( position[0u].GetDouble(), position[1u].GetDouble() )
				);
			}
		}
		return Gaze2DPositionPacketPtr();
	}
	
	Gaze3DPositionPacketPtr DataParser::parseGaze3DPositionPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if ( json.HasMember("gp3") && json["gp3"].IsArray() )
		{
			const rapidjson::Value& position = json["gp3"];

			if ( position.Size() == 3u && position[0u].IsDouble() && position[1u].IsDouble() && position[2u].IsDouble() )
			{
				return std::make_shared< Gaze3DPositionPacket >(
					timeStamp, statuts,
					Point3( position[0u].GetDouble(), position[1u].GetDouble(), position[2u].GetDouble() )
				);
			}
		}
		return Gaze3DPositionPacketPtr();
	}
	
	GyroscopePacketPtr DataParser::parseGyroscopePacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if ( json.HasMember("gy") && json["gy"].IsArray() )
		{
			const rapidjson::Value& gyroscope = json["gy"];

			if ( gyroscope.Size() == 3u && gyroscope[0u].IsDouble() && gyroscope[1u].IsDouble() && gyroscope[2u].IsDouble() )
			{
				return std::make_shared< GyroscopePacket >(
					timeStamp, statuts,
					Vector3( gyroscope[0u].GetDouble(), gyroscope[1u].GetDouble(), gyroscope[2u].GetDouble() )
				);
			}
		}
		return GyroscopePacketPtr();
	}
	
	AccelerometerPacketPtr DataParser::parseAccelerometerPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if ( json.HasMember("ac") && json["ac"].IsArray() )
		{
			const rapidjson::Value& acceleration = json["ac"];

			if ( acceleration.Size() == 3u && acceleration[0u].IsDouble() && acceleration[1u].IsDouble() && acceleration[2u].IsDouble() )
			{
				return std::make_shared< AccelerometerPacket >(
					timeStamp, statuts,
					Vector3( acceleration[0u].GetDouble(), acceleration[1u].GetDouble(), acceleration[2u].GetDouble() )
				);
			}
		}
		return AccelerometerPacketPtr();
	}

	PTSSyncPacketPtr DataParser::parsePTSSyncPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if (( json.HasMember("pts") && json["pts"].IsInt64() )
		&&  ( json.HasMember("pv")  && json["pts"].IsUint()   ))
		{
			return std::make_shared< PTSSyncPacket >(
				timeStamp, statuts,
				json["pts"].GetInt64(), json["pts"].GetUint()
			);
		}
		return PTSSyncPacketPtr();
	}

	VTSSyncPacketPtr DataParser::parseVTSSyncPacket ( Int64 timeStamp, UInt8 statuts, const rapidjson::Document& json ) const
	{
		if ( json.HasMember("vts") && json["vts"].IsInt64() )
		{
			return std::make_shared< VTSSyncPacket >( timeStamp, statuts, json["vts"].GetInt64() );
		}
		return VTSSyncPacketPtr();
	}

	// END CLASS DATAPARSER
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_DATAPARSER_CPP
// ##################################################