// ##################################################
// ###            TOBII_VECTOR3_HPP               ###
// ##################################################
#ifndef TOBII_VECTOR3_HPP
#define TOBII_VECTOR3_HPP

/**
 * \file Vector3.hpp
 * \brief Describes 3 dimentional vectors with floating-point type.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/
 
// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD DEPENDENCES
#include <cmath>
#include <sstream>
#include <string>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###                CLASS VECTOR3               ###
	// ##################################################
	
	/**
	 * \class Vector3
	 * \brief Describes 3 dimentional vectors with floating-point type.
	 *
	 * Describes 3 dimentional vectors with floating-point type.
	 * 
	 **/
	class Vector3
	{
	private:
		float _dx; /*!< Coordinate on the X (1.0f, 0.0f, 0.0f) axis. */
		float _dy; /*!< Coordinate on the Y (0.0f, 1.0f, 0.0f) axis. */
		float _dz; /*!< Coordinate on the Z (0.0f, 0.0f, 1.0f) axis. */

	public:
		/**
		 * \brief Constructor (Default)
		 * \param dx (float) Norm of the vector along the X axis.
		 * \param dy (float) Norm of the vector along the Y axis.
		 * \param dz (float) Norm of the vector along the Z axis.
		 *
		 * Creates and initialize a 3 dimentional vector with floating point coordinates.
		 *
		 **/
		Vector3 ( float dx = 0, float dy = 0, float dz = 0 );

		/**
		 * \brief Constructor (Copy)
		 * \param vector (const Vector3&) The Vector3 to copy.
		 *
		 * Creates and initialize a copy of an object of type Vector3.
		 *
		 **/
		Vector3 ( const Vector3& vector );

		/**
		 * \brief Creates a 3 dimentional vector with coordinates (0.0f, 0.0f, 0.0f).
		 * \return (Vector3) The Vector3 created.
		 *
		 * Creates Creates a 3 dimentional vector with coordinates (0.0f, 0.0f, 0.0f).
		 *
		 **/
		static Vector3 zero ();
		
		/**
		 * \brief Getter (float)
		 * \return (float) The norm of this vector along the X axis.
		 *
		 * Return the norm of this vector along the X axis.
		 *
		 **/
		float dx () const;

		/**
		 * \brief Getter (float)
		 * \return (float) The reference of the norm of this vector along the X axis.
		 *
		 * Returns the reference of the norm of this vector along the X axis, whiwh can later be used to modify it.
		 *
		 **/
		float& dx ();

		/**
		 * \brief Getter (float)
		 * \return (float) The norm of this vector along the Y axis.
		 *
		 * Return the norm of this vector along the Y axis.
		 *
		 **/
		float dy () const;

		/**
		 * \brief Getter (float)
		 * \return (float) The reference of the norm of this vector along the Y axis.
		 *
		 * Returns the reference of the norm of this vector along the Y axis, whiwh can later be used to modify it.
		 *
		 **/
		float& dy ();

		/**
		 * \brief Getter (float)
		 * \return (float) The norm of this vector along the Z axis.
		 *
		 * Return the norm of this vector along the Z axis.
		 *
		 **/
		float dz () const;

		/**
		 * \brief Getter (float)
		 * \return (float) The reference of the norm of this vector along the Z axis.
		 *
		 * Returns the reference of the norm of this vector along the Z axis, whiwh can later be used to modify it.
		 *
		 **/
		float& dz ();

		/**
		 * \brief Get the norm of the Vector3 (float)
		 * \return (float) The L2 norm of this Vector3.
		 *
		 * Return the L2 norm of this Vector3.
		 *
		 **/
		float norm () const;

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of 'this' Vector3
		 *
		 * Creates and return a std::string representation of 'this' Vector3.
		 * The representation is in JSON format:
		 * [dx,dy,dz]
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Assignment operator (const Vector3&)
		 * \param vector (const Vector3&) the Vector3 to copy.
		 * \return (Vector3&) reference to 'this' Vector3 once set.
		 *
		 * Sets 'this' Vector3 coordinates from a given Vector3.
		 *
		 **/
		Vector3& operator= ( const Vector3& vector );
	
	}; // END CLASS VECTOR3
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, const Vector3& vector );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // DATA_VECTOR3_HPP
// ##################################################