// ##################################################
// ###             TOBII_RECORD_HPP               ###
// ##################################################
#ifndef TOBII_RECORD_HPP
#define TOBII_RECORD_HPP

/**
 * \file Record.hpp
 * \brief Describes Record object data type, at the root of all Tobii Glasses 2 Records.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD DEPENDENCES
#include <memory>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"

// ##################################################
// ###               NAMESPACE TOBII              ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
namespace Tobii {

	// ##################################################
	// ###                CLASS RECORD                ###
	// ##################################################
	
	/**
	 * \class Record
	 * \brief Describes Tobii Glasses 2 Record.
	 *
	 * This object represents Record comming from the Tobii Glasses 2 accelerometer with their time-stamp and status.
	 * 
	 **/
	class Record
	{
	private:
		Int64 _timeStamp;	/*!< The time-stamp of the Record.	*/
		UInt8 _status;	 	/*!< The status of the Record		*/

	public:
		/**
		 * \brief Constructor (Default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at whiwh the recording was performed.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 *
		 * Creates and initialize an object of type Record.
		 *
		 **/
		Record ( Int64 timeStamp = 0, UInt8 status = 0 );

		/**
		 * \brief Constructor (Copy)
		 * \param record (const Record&) The Record to copy
		 *
		 * Creates and initialize a copy of Record.
		 *
		 **/
		Record ( const Record& record );

		/**
		 * \brief Destructor (virtual)
		 *
		 * Virtual destructor to allow for safe cast-to-base-class in derived methods
		 **/
		virtual ~Record ();

		/**
		 * \brief Getter (Int64)
		 * \return (Int64) The time stamp of this Record.
		 *
		 * Returns the time stamp of this record.
		 *
		 **/
		Int64 timeStamp () const;
		
		/**
		 * \brief Getter / Setter (UInt8)
		 * \return (UInt8&) The reference of the time stamp of this Record.
		 *
		 * Returns the reference of the time stamp of this record, whiwh can be later modified.
		 *
		 **/
		Int64& timeStamp ();

		/**
		 * \brief Getter (UInt8)
		 * \return (UInt8) The status of this Record.
		 *
		 * Returns the status of this record.
		 *
		 **/
		UInt8 status () const;

		/**
		 * \brief Getter / Setter (UInt8)
		 * \return (UInt8&) The reference of the status of this Record.
		 *
		 * Returns the reference of the status of this record, whiwh can be later modified.
		 *
		 **/
		UInt8& status ();

		/**
		 * \brief Assignment operator (const Record&)
		 * \param record (const Record&) The acceleration record to copy.
		 * \return (Record&) Reference to 'this' Record once set.
		 *
		 * Sets 'this' Record record time stamp, status and value using a given Record record.
		 *
		 **/
		Record& operator= ( const Record& record );
	
	}; // END CLASS RECORD
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################

	/**
	 * \brief Shared pointer to an Record record.
	 *
	 * Defines the type of a shared pointer to an Record record.
	 *
	 **/
	using RecordPtr = std::shared_ptr< Record >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_RECORD_HPP
// ##################################################