// ##################################################
// ###             TOBII_POINT3_HPP               ###
// ##################################################
#ifndef TOBII_POINT3_HPP
#define TOBII_POINT3_HPP

/**
 * \file Point3.hpp
 * \brief Describes 3 dimentional points with floating-point type.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD DEPENDENCES
#include <cmath>
#include <sstream>
#include <string>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"

// ##################################################
// ###               NAMESPACE DATA               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###                CLASS POINT3                ###
	// ##################################################
	
	/**
	 * \class Point3
	 * \brief Describes 3 dimentional points with floating-point type.
	 *
	 * Describes 3 dimentional points with floating-point type.
	 * 
	 **/
	class Point3
	{
	private:
		float _x; /*!< Coordinate on the X (1.0f, 0.0f, 0.0f) axis. */
		float _y; /*!< Coordinate on the Y (0.0f, 1.0f, 0.0f) axis. */
		float _z; /*!< Coordinate on the Y (0.0f, 0.0f, 1.0f) axis. */

	public:
		/**
		 * \brief Constructor (Default)
		 * \param x (float) Position of the point along the X axis.
		 * \param y (float) Position of the point along the Y axis.
		 * \param z (float) Position of the point along the Z axis.
		 *
		 * Creates and initialize a 3 dimentional point with floating point coordinates.
		 *
		 **/
		Point3 ( float x = 0, float y = 0, float z = 0 );

		/**
		 * \brief Constructor (Copy)
		 * \param point (const Point3&) The Point3 to copy.
		 *
		 * Creates and initialize a copy of an object of type Point3.
		 *
		 **/
		Point3 ( const Point3& point );

		/**
		 * \brief Creates a 3 dimentional point with coordinates (0.0f, 0.0f, 0.0f).
		 * \return (Point3) The Point3 created.
		 *
		 * Creates Creates a 3 dimentional point with coordinates (0.0f, 0.0f, 0.0f).
		 *
		 **/
		static Point3 zero ();

		/**
		 * \brief Getter (float)
		 * \return (float) The X coordinate of this point.
		 *
		 * Return the X coordinate of this point.
		 *
		 **/
		float x () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) The reference of the X coordinates of this point.
		 *
		 * Return the reference of this point X coordinate, which can be used to modify its value.
		 *
		 **/
		float& x ();

		/**
		 * \brief Getter (float)
		 * \return (float) The Y coordinate of this point.
		 *
		 * Return the Y coordinate of this point.
		 *
		 **/
		float y () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) The reference of the Y coordinates of this point.
		 *
		 * Return The reference of this point Y coordinate, which can be used to modify its value.
		 *
		 **/
		float& y ();

		/**
		 * \brief Getter (float)
		 * \return float The Z coordinate of this point.
		 *
		 * Return the Z coordinate of this point.
		 *
		 **/
		float z () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return float& The reference of the Z coordinates of this point.
		 *
		 * Return the reference of this point Z coordinate, which can be used to modify its value.
		 *
		 **/
		float& z ();

		/**
		 * \brief Get the norm of the Point3 (float)
		 * \return (float) The L2 norm of this Point3, i.e. the distance between this point, and the origine (0,0,0).
		 *
		 * Return the L2 norm of this point.
		 *
		 **/
		float norm () const;

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of 'this' Point3
		 *
		 * Creates and return a std::string representation of 'this' Point3.
		 * The representation is in JSON format:
		 * [x,y,z]
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Assignment operator (const Point3&)
		 * \param point (const Point3&) the Point3 to copy.
		 * \return (Point3&) Reference to 'this' Point3 once set.
		 *
		 * Sets 'this' Point3 coordinates from a given Point3.
		 *
		 **/
		Point3& operator= ( const Point3& point );
	
	}; // END CLASS POINT3
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, const Point3& point );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_POINT3_HPP
// ##################################################